/*
    kopetemimetypehandler.h - Kopete Mime-type Handlers

    Copyright (c) 2004      by Richard Smith         <kde@metafoo.co.uk>

    Kopete    (c) 2004      by the Kopete developers <kopete-devel@kde.org>

    *************************************************************************
    *                                                                       *
    * This library is free software; you can redistribute it and/or         *
    * modify it under the terms of the GNU Lesser General Public            *
    * License as published by the Free Software Foundation; either          *
    * version 2 of the License, or (at your option) any later version.      *
    *                                                                       *
    *************************************************************************
*/

#ifndef KOPETEMIMETYPEHANDLER_H
#define KOPETEMIMETYPEHANDLER_H

class KURL;
class TQString;
class TQStringList;

#include "kopete_export.h"

namespace Kopete
{

/**
 * @brief A handler for some set of mime-types
 * A mime type handler is responsible for handling requests to open files of
 * certain mime types presented to the main application.
 */
class KOPETE_EXPORT MimeTypeHandler
{
protected:
	MimeTypeHandler( bool canAcceptRemoteFiles = false );
public:
	virtual ~MimeTypeHandler();

	/**
	 * Finds a MimeTypeHandler for a given URL, and tells that handler to handle it
	 *
	 * @param url the url to dispatch
	 *
	 * @return true if a handler was registered for the mime type, false otherwise
	 */
	static bool dispatchURL( const KURL &url );

	/**
	 * Returns a list of mime types this object is registered to handle
	 */
	const TQStringList mimeTypes() const;

	/**
	 * Returns a list of protocols this object is registered to handle
	 */
	const TQStringList protocols() const;

	/**
	 * Returns true if this handler can accept remote files direcltly;
	 * If false, remote files are downloaded via TDEIO::NetAccess before
	 * being passed to handleURL
	 */
	bool canAcceptRemoteFiles() const;

	/**
	 * Handles the URL @p url
	 *
	 * @param url The url to handle
	 */
	virtual void handleURL( const KURL &url ) const;

	/**
	 * Handles the URL @p url, which has the mime type @p mimeType
	 *
	 * @param mimeType The mime type of the URL
	 * @param url The url to handle
	 */
	virtual void handleURL( const TQString &mimeType, const KURL &url ) const;

protected:
	/**
	 * Register this object as the handler of type @p mimeType.
	 * @param mimeType the mime type to handle
	 * @return true if registration succeeded, false if another handler is
	 *         already set for this mime type.
	 */
	bool registerAsMimeHandler( const TQString &mimeType );

	/**
	 * Register this object as the handler of type @p protocol.
	 * @param protocol the protocol to handle
	 * @return true if registration succeeded, false if another handler is
	 *         already set for this protocol.
	 */
	bool registerAsProtocolHandler( const TQString &protocol );

private:
	/**
	 * Helper function.
	 * Attempts to dispatch a given URL to a given handler
	 *
	 * @param url The url to dispatch
	 * @param mimeType The mime type of the url
	 * @param handler The handler to attempt
	 *
	 * @return true if a handler was able to process the URL, false otherwise
	 */
	static bool dispatchToHandler( const KURL &url, const TQString &mimeType, MimeTypeHandler *handler );

	class Private;
	Private *d;
};

/**
 * Mime-type handler class for Kopete emoticon files
 */
class KOPETE_EXPORT EmoticonMimeTypeHandler : public MimeTypeHandler
{
public:
	EmoticonMimeTypeHandler();

	const TQStringList mimeTypes() const;

	void handleURL( const TQString &mimeType, const KURL &url ) const;
};

} // Kopete

#endif
