/***************************************************************************
*   Copyright (C) 2004-2009 by Thomas Fischer                             *
*   fischer@unix-ag.uni-kl.de                                             *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
***************************************************************************/
#include <tqregexp.h>

#include "entry.h"

#include "fileexporterris.h"

namespace BibTeX
{

    FileExporterRIS::FileExporterRIS() : FileExporter()
    {
        // nothing
    }

    FileExporterRIS::~FileExporterRIS()
    {
        // nothing
    }

    bool FileExporterRIS::save( TQIODevice* iodevice, const Element* element, TQStringList* /*errorLog*/ )
    {
        m_mutex.lock();
        bool result = false;
        TQTextStream stream( iodevice );

        const Entry *entry = dynamic_cast<const Entry*>( element );
        if ( entry != NULL )
            result = writeEntry( stream, entry );

        m_mutex.unlock();
        return result && !m_cancelFlag;
    }

    bool FileExporterRIS::save( TQIODevice* iodevice, const File* bibtexfile, TQStringList* /*errorLog*/ )
    {
        m_mutex.lock();
        bool result = true;
        m_cancelFlag = false;
        TQTextStream stream( iodevice );

        int numElements = (int) bibtexfile->count(), i = 0;
        emit progress( 0, numElements );
        for ( File::ElementList::const_iterator it = bibtexfile->elements.begin(); it != bibtexfile->elements.end() && result && !m_cancelFlag; it++ )
        {
            Entry *entry = dynamic_cast<Entry*>( *it );
            if ( entry != NULL )
            {
                BibTeX::Entry *myEntry = bibtexfile->completeReferencedFieldsConst( entry );
                result &= writeEntry( stream, myEntry );
                delete myEntry;
            }
            emit progress( ++i, numElements );
        }

        m_mutex.unlock();
        return result && !m_cancelFlag;
    }

    void FileExporterRIS::cancel()
    {
        m_cancelFlag = true;
    }

    bool FileExporterRIS::writeEntry( TQTextStream &stream, const Entry* entry )
    {
        bool result = true;
        tqDebug( "Writing Entry" );

        switch ( entry->entryType() )
        {
        case Entry::etBook:
            writeKeyValue( stream, "TY", "BOOK" );
            break;
        case Entry::etInBook:
            writeKeyValue( stream, "TY", "CHAP" );
            break;
        case Entry::etInProceedings:
            writeKeyValue( stream, "TY", "CONF" );
            break;
        case Entry::etArticle:
            writeKeyValue( stream, "TY", "JOUR" );
            break;
        case Entry::etTechReport:
            writeKeyValue( stream, "TY", "RPRT" );
            break;
        case Entry::etPhDThesis:
            writeKeyValue( stream, "TY", "THES" );
            break;
        default:
            writeKeyValue( stream, "TY", "GEN" );
        }

        TQString year = "";
        TQString month = "";

        for ( Entry::EntryFields::ConstIterator it = entry->begin(); result && it != entry->end(); it++ )
        {
            EntryField *field = *it;

            if ( field->fieldType() == EntryField::ftUnknown && field->fieldTypeName().startsWith( "RISfield_" ) )
                result &= writeKeyValue( stream, field->fieldTypeName().right( 2 ), field->value()->simplifiedText() );
            else if ( field->fieldType() == EntryField::ftAuthor )
            {
                TQStringList authors = TQStringList::split( TQRegExp( "\\s+(,|and|&)\\s+", false ), field->value() ->simplifiedText() );
                for ( TQStringList::Iterator it = authors.begin(); result && it != authors.end(); ++it )
                    result &= writeKeyValue( stream, "AU", *it );
            }
            else if ( field->fieldType() == EntryField::ftEditor )
            {
                TQStringList authors = TQStringList::split( TQRegExp( "\\s+(,|and|&)\\s+", false ), field->value() ->simplifiedText() );
                for ( TQStringList::Iterator it = authors.begin(); result && it != authors.end(); ++it )
                    result &= writeKeyValue( stream, "ED", *it );
            }
            else if ( field->fieldType() == EntryField::ftTitle )
                result &= writeKeyValue( stream, "TI", field->value() ->simplifiedText() );
            else if ( field->fieldType() == EntryField::ftJournal )
                result &= writeKeyValue( stream, "JO", field->value() ->simplifiedText() );
            else if ( field->fieldType() == EntryField::ftChapter )
                result &= writeKeyValue( stream, "CP", field->value() ->simplifiedText() );
            else if ( field->fieldType() == EntryField::ftISSN )
                result &= writeKeyValue( stream, "SN", field->value() ->simplifiedText() );
            else if ( field->fieldType() == EntryField::ftISBN )
                result &= writeKeyValue( stream, "SN", field->value() ->simplifiedText() );
            else if ( field->fieldType() == EntryField::ftVolume )
                result &= writeKeyValue( stream, "VL", field->value() ->simplifiedText() );
            else if ( field->fieldType() == EntryField::ftNumber )
                result &= writeKeyValue( stream, "IS", field->value() ->simplifiedText() );
            else if ( field->fieldType() == EntryField::ftNote )
                result &= writeKeyValue( stream, "N1", field->value() ->simplifiedText() );
            else if ( field->fieldType() == EntryField::ftAbstract )
                result &= writeKeyValue( stream, "N2", field->value() ->simplifiedText() );
            else if ( field->fieldType() == EntryField::ftPublisher )
                result &= writeKeyValue( stream, "PB", field->value() ->simplifiedText() );
            else if ( field->fieldType() == EntryField::ftLocation )
                result &= writeKeyValue( stream, "CY", field->value() ->simplifiedText() );
            else if ( field->fieldType() == EntryField::ftKeywords )
                result &= writeKeyValue( stream, "KW", field->value() ->simplifiedText() );
            else if ( field->fieldType() == EntryField::ftYear )
                year = field->value() ->simplifiedText();
            else if ( field->fieldType() == EntryField::ftMonth )
                month = field->value() ->simplifiedText();
            else if ( field->fieldType() == EntryField::ftAddress )
                result &= writeKeyValue( stream, "AD", field->value() ->simplifiedText() );
            else if ( field->fieldType() == EntryField::ftURL )
                result &= writeKeyValue( stream, "UR", field->value() ->simplifiedText() );
            else if ( field->fieldType() == EntryField::ftPages )
            {
                TQStringList pageRange = TQStringList::split( TQRegExp( TQString( "--|-|%1" ).arg( TQChar( 0x2013 ) ) ), field->value() ->simplifiedText() );
                if ( pageRange.count() == 2 )
                {
                    result &= writeKeyValue( stream, "SP", pageRange[ 0 ] );
                    result &= writeKeyValue( stream, "EP", pageRange[ 1 ] );
                }
            }
            else if ( field->fieldTypeName().lower() == "doi" )
                result &= writeKeyValue( stream, "UR", field->value() ->simplifiedText() );
        }

        if ( !year.isEmpty() || !month.isEmpty() )
        {
            result &= writeKeyValue( stream, "PY", TQString( "%1/%2//" ).arg( year ).arg( month ) );
        }

        result &= writeKeyValue( stream, "ER", TQString() );
        stream << endl;

        return result;
    }

    bool FileExporterRIS::writeKeyValue( TQTextStream &stream, const TQString& key, const TQString&value )
    {
        stream << key << "  - ";
        if ( !value.isEmpty() )
            stream << value;
        stream << endl;
        tqDebug( "%s  - %s", key.latin1(), value.latin1() );

        return true;
    }

}
