#!/usr/bin/env python

############################################################################
# Python-TQt template script for Amarok
# (c) 2005 Mark Kretschmann <markey@web.de>
#
# Depends on: Python 3, PyTQt
############################################################################
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
############################################################################

import configparser
import os
import sys
import threading
import signal
from time import sleep

try:
    from PyTQt.tqt import *
except:
    os.popen( "kdialog --sorry 'PyTQt (TQt bindings for Python) is required for this script.'" )
    raise


# Replace with real name
debug_prefix = "[Test Script]"


class ConfigDialog( TQDialog ):
    """ Configuration widget """

    def __init__( self ):
        TQDialog.__init__( self )
        self.setWFlags( TQt.WDestructiveClose )
        self.setCaption( "Test Script - Amarok" )

        foo = None
        try:
            config = configparser.ConfigParser()
            config.read( "testrc" )
            foo = config.get( "General", "foo" )
        except:
            pass

        self.adjustSize()

    def save( self ):
        """ Saves configuration to file """

        self.file = file( "testrc", 'w' )

        self.config = configparser.ConfigParser()
        self.config.add_section( "General" )
        self.config.set( "General", "foo", foovar )
        self.config.write( self.file )
        self.file.close()

        self.accept()


class Notification( TQCustomEvent ):
    __super_init = TQCustomEvent.__init__
    def __init__( self, str ):
        self.__super_init(TQCustomEvent.User + 1)
        self.eventStr = str

class Test( TQApplication ):
    """ The main application, also sets up the TQt event loop """

    def __init__( self, args ):
        TQApplication.__init__( self, args )
        debug( "Started." )

        # Start separate thread for reading data from stdin
        self.stdinReader = threading.Thread( target = self.readStdin )
        self.stdinReader.start()

        self.readSettings()

    def readSettings( self ):
        """ Reads settings from configuration file """

        try:
            foovar = config.get( "General", "foo" )

        except:
            debug( "No config file found, using defaults." )


############################################################################
# Stdin-Reader Thread
############################################################################

    def readStdin( self ):
        """ Reads incoming notifications from stdin """

        while True:
            # Read data from stdin. Will block until data arrives.
            line = sys.stdin.readline()

            if line:
                tqApp.postEvent( self, Notification(line) )
            else:
                break


############################################################################
# Notification Handling
############################################################################

    def customEvent( self, notification ):
        """ Handles notifications """

        eventStr = TQString(notification.eventStr)
        debug( "Received notification: " + str( eventStr ) )

        if eventStr.contains( "configure" ):
            self.configure()

        if eventStr.contains( "engineStateChange: play" ):
            self.engineStatePlay()

        if eventStr.contains( "engineStateChange: idle" ):
            self.engineStateIdle()

        if eventStr.contains( "engineStateChange: pause" ):
            self.engineStatePause()

        if eventStr.contains( "engineStateChange: empty" ):
            self.engineStatePause()

        if eventStr.contains( "trackChange" ):
            self.trackChange()

# Notification callbacks. Implement these functions to react to specific notification
# events from Amarok:

    def configure( self ):
        debug( "configuration" )

        self.dia = ConfigDialog()
        self.dia.show()
        self.connect( self.dia, TQ_SIGNAL( "destroyed()" ), self.readSettings )

    def engineStatePlay( self ):
        """ Called when Engine state changes to Play """
        pass

    def engineStateIdle( self ):
        """ Called when Engine state changes to Idle """
        pass

    def engineStatePause( self ):
        """ Called when Engine state changes to Pause """
        pass

    def engineStateEmpty( self ):
        """ Called when Engine state changes to Empty """
        pass

    def trackChange( self ):
        """ Called when a new track starts """
        pass


############################################################################

def debug( message ):
    """ Prints debug message to stdout """

    print(debug_prefix + " " + message)

def main( ):
    app = Test( sys.argv )

    app.exec_loop()

def onStop(signum, stackframe):
    """ Called when script is stopped by user """
    pass

if __name__ == "__main__":
    mainapp = threading.Thread(target=main)
    mainapp.start()
    signal.signal(15, onStop)
    # necessary for signal catching
    while 1: sleep(120)
