/***************************************************************************
 *   Copyright (C) 2006 by Peter Penz                                      *
 *   peter.penz@gmx.at                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.             *
 ***************************************************************************/

#include "statusbarmessagelabel.h"
#include <tqpainter.h>
#include <tqtimer.h>
#include <tqfontmetrics.h>
#include <kiconloader.h>
#include <tdeglobalsettings.h>

StatusBarMessageLabel::StatusBarMessageLabel(TQWidget* parent) :
    TQWidget(parent),
    m_type(DolphinStatusBar::Default),
    m_state(Default),
    m_illumination(0),
    m_minTextHeight(-1),
    m_timer(0)
{
    setMinimumHeight(TDEIcon::SizeSmall);

    m_timer = new TQTimer(this);
    connect(m_timer, TQ_SIGNAL(timeout()),
            this, TQ_SLOT(timerDone()));
}

StatusBarMessageLabel::~StatusBarMessageLabel()
{
}

void StatusBarMessageLabel::setType(DolphinStatusBar::Type type)
{
    if (type != m_type) {
        m_type = type;

        m_timer->stop();
        m_illumination = 0;
        m_state = Default;

        const char* iconName = 0;
        TQPixmap pixmap;
        switch (type) {
            case DolphinStatusBar::OperationCompleted:
                iconName = "ok";
                break;

            case DolphinStatusBar::Information:
                iconName = "application-vnd.tde.info";
                break;

            case DolphinStatusBar::Error:
                iconName = "error";
                m_timer->start(100);
                m_state = Illuminate;
                break;

            case DolphinStatusBar::Default:
            default: break;
        }

        m_pixmap = (iconName == 0) ? TQPixmap() : SmallIcon(iconName);
        assureVisibleText();
        update();
    }
}

void StatusBarMessageLabel::setText(const TQString& text)
{
    if (text != m_text) {
        if (m_type == DolphinStatusBar::Error) {
            m_timer->start(100);
            m_illumination = 0;
            m_state = Illuminate;
        }
        m_text = text;
        assureVisibleText();
        update();
    }
}

void StatusBarMessageLabel::setMinimumTextHeight(int min)
{
    if (min != m_minTextHeight) {
        m_minTextHeight = min;
        setMinimumHeight(min);
    }
}

void StatusBarMessageLabel::paintEvent(TQPaintEvent* /* event */)
{
    TQPixmap buffer(size());
    TQPainter painter(&buffer);

    // draw background
    TQColor backgroundColor(colorGroup().background());
    TQColor foregroundColor(TDEGlobalSettings::textColor());
    if (m_illumination > 0) {
        backgroundColor = mixColors(backgroundColor, TQColor(255, 255, 64), m_illumination);
        foregroundColor = mixColors(foregroundColor, TQColor(0, 0, 0), m_illumination);
    }
    painter.setBrush(backgroundColor);
    painter.setPen(backgroundColor);
    painter.drawRect(TQRect(0, 0, width(), height()));

    // draw pixmap
    int x = pixmapGap();
    int y = (height() - m_pixmap.height()) / 2;

    if (!m_pixmap.isNull()) {
        painter.drawPixmap(x, y, m_pixmap);
        x += m_pixmap.width() + pixmapGap();
    }

    // draw text
    painter.setPen(foregroundColor);
    painter.drawText(TQRect(x, 0, width() - x, height()), TQt::AlignVCenter | TQt::WordBreak, m_text);
    painter.end();

    bitBlt(this, 0, 0, &buffer);
}

void StatusBarMessageLabel::resizeEvent(TQResizeEvent* event)
{
    TQWidget::resizeEvent(event);
    TQTimer::singleShot(0, this, TQ_SLOT(assureVisibleText()));
}

void StatusBarMessageLabel::timerDone()
{
    switch (m_state) {
        case Illuminate: {
            // increase the illumination
            if (m_illumination < 100) {
                m_illumination += 20;
                update();
            }
            else {
                m_state = Illuminated;
                m_timer->start(1000);
            }
            break;
        }

        case Illuminated: {
            // start desaturation
            m_state = Desaturate;
            m_timer->start(100);
            break;
        }

        case Desaturate: {
            // desaturate
            if (m_illumination > 0) {
                m_illumination -= 5;
                update();
            }
            else {
                m_state = Default;
                m_timer->stop();
            }
            break;
        }

        default:
            break;
    }
}

void StatusBarMessageLabel::assureVisibleText()
{
    if (m_text.isEmpty()) {
        return;
    }


    int availableWidth = width() - m_pixmap.width() - pixmapGap() * 2;

    TQFontMetrics fontMetrics(font());
    TQRect bounds(fontMetrics.boundingRect(0, 0, availableWidth, height(),
                                          TQt::AlignVCenter | TQt::WordBreak,
                                          m_text));
    int requiredHeight = bounds.height();
    if (requiredHeight < m_minTextHeight) {
        requiredHeight = m_minTextHeight;
    }
    setMinimumHeight(requiredHeight);
    updateGeometry();
}

TQColor StatusBarMessageLabel::mixColors(const TQColor& c1,
                                        const TQColor& c2,
                                        int percent) const
{
    const int recip = 100 - percent;
    const int red   = (c1.red()   * recip + c2.red()   * percent) / 100;
    const int green = (c1.green() * recip + c2.green() * percent) / 100;
    const int blue  = (c1.blue()  * recip + c2.blue()  * percent) / 100;
    return TQColor(red, green, blue);
}

#include "statusbarmessagelabel.moc"
