/***************************************************************************
 *   Copyright (C) 2003 by Mario Scalas                                    *
 *   mario.scalas@libero.it                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <tqpopupmenu.h>
#include <kxmlguiclient.h>
#include <kdebug.h>
#include <tdeaction.h>
#include <tdelocale.h>
#include <tdeversion.h>

#include <kdevproject.h>

#include "fileviewpart.h"
#include "filetreewidget.h"
#include "fileitemfactory.h"

#include "filetreeviewwidgetimpl.h"

using namespace filetreeview;

///////////////////////////////////////////////////////////////////////////////
// class FileTreeViewWidgetImpl
///////////////////////////////////////////////////////////////////////////////

FileTreeViewWidgetImpl::FileTreeViewWidgetImpl( FileTreeWidget *parent, const char *name )
    : TQObject( parent, name ), m_branchItemFactory( 0 ),
    m_part( parent->part() ), m_isReloadingTree( false )
{
    kdDebug(9017) << "FileTreeViewWidgetImpl::FileTreeViewWidgetImpl()" << endl;

    // Actions
    m_actionToggleShowNonProjectFiles = new TDEToggleAction( i18n("Show Non Project Files"), TDEShortcut(),
        this, TQ_SLOT(slotToggleShowNonProjectFiles()), this, "actiontoggleshowshownonprojectfiles" );
    m_actionToggleShowNonProjectFiles->setCheckedState(i18n("Hide Non Project Files"));
    m_actionToggleShowNonProjectFiles->setWhatsThis(i18n("<b>Show non project files</b><p>Shows files that do not belong to a project in a file tree."));

    // Reload good ol' settings
    TQDomDocument &dom = *m_part->projectDom();
    m_actionToggleShowNonProjectFiles->setChecked( !DomUtil::readBoolEntry(dom, "/kdevfileview/tree/hidenonprojectfiles") );
}

///////////////////////////////////////////////////////////////////////////////

FileTreeViewWidgetImpl::~FileTreeViewWidgetImpl()
{
    kdDebug(9017) << "FileTreeViewWidgetImpl::~FileTreeViewWidgetImpl()" << endl;

    delete m_branchItemFactory;

    TQDomDocument &dom = *m_part->projectDom();
    DomUtil::writeBoolEntry( dom, "/kdevfileview/tree/hidenonprojectfiles", !showNonProjectFiles() );
}

///////////////////////////////////////////////////////////////////////////////

FileTreeWidget *FileTreeViewWidgetImpl::fileTree() const
{
    return static_cast<FileTreeWidget *>( parent() );
}

///////////////////////////////////////////////////////////////////////////////

TQDomDocument &FileTreeViewWidgetImpl::projectDom() const
{
    return *part()->projectDom();
}

///////////////////////////////////////////////////////////////////////////////

TQString FileTreeViewWidgetImpl::projectDirectory() const
{
    return part()->project()->projectDirectory();
}

///////////////////////////////////////////////////////////////////////////////

bool FileTreeViewWidgetImpl::showNonProjectFiles() const
{
    return m_actionToggleShowNonProjectFiles->isChecked();
}

///////////////////////////////////////////////////////////////////////////////

void FileTreeViewWidgetImpl::fillPopupMenu( TQPopupMenu *popupMenu, TQListViewItem *item ) const
{
    // Show the "reload tree" menu-item only if it is requested for the root object
    // and we don't have a sync-with-repository operation pending (which otherwise will
    // kill the call-back's from working)
    if (item == fileTree()->firstChild() && canReloadTree())
    {
        int id = popupMenu->insertItem( i18n( "Reload Tree"), this, TQ_SLOT( slotReloadTree() ) );
        popupMenu->setWhatsThis( id, i18n("<b>Reload tree</b><p>Reloads the project files tree.") );
    }

    m_actionToggleShowNonProjectFiles->plug( popupMenu );
}

///////////////////////////////////////////////////////////////////////////////

KURL::List FileTreeViewWidgetImpl::selectedPathUrls()
{
    kdDebug(9017) << "FileTreeViewWidgetImpl::selectedPathUrls()" << endl;

	KURL::List urlList;

	TQValueList<TQListViewItem*> list = allSelectedItems( fileTree()->firstChild() );
	TQValueList<TQListViewItem*>::Iterator it = list.begin();
	while( it != list.end() )
	{
		FileTreeViewItem * item = static_cast<FileTreeViewItem*>( *it );
		if ( fileTree()->shouldBeShown( item ) )
		{
			KURL url;
			url.setPath( item->path() );
			urlList << url;
		}
		++it;
	}

	return urlList;
}

///////////////////////////////////////////////////////////////////////////////

TQValueList<TQListViewItem*> FileTreeViewWidgetImpl::allSelectedItems( TQListViewItem * item ) const
{
	TQValueList<TQListViewItem*> list;

	if ( item )
	{
		if ( item->isSelected() )
		{
			list << item;
		}

		TQListViewItem * it = item->firstChild();
		while( it  )
		{
			list += allSelectedItems( it );
			it = it->nextSibling();
		}
	}

	return list;
}

///////////////////////////////////////////////////////////////////////////////

void FileTreeViewWidgetImpl::slotReloadTree()
{
    fileTree()->openDirectory( projectDirectory() );
}

///////////////////////////////////////////////////////////////////////////////

void FileTreeViewWidgetImpl::slotToggleShowNonProjectFiles()
{
    fileTree()->hideOrShow();
}

#include "filetreeviewwidgetimpl.moc"
