                                         /**
 * A class that lets the user draw with the mouse. The
 * window knows how to redraw itself.
 */

import org.trinitydesktop.qt.*;

public class TQTimerTest extends TQWidget {
	
	public static final int COLOR_MENU_ID_BLACK = 0;
	public static final int COLOR_MENU_ID_RED = 1;
	public static final int COLOR_MENU_ID_BLUE = 2;
	public static final int COLOR_MENU_ID_GREEN = 3;
	public static final int COLOR_MENU_ID_YELLOW = 4;
	
	public static int eventCount = 0;
	
	private TQMenuBar _menubar;
	private TQPopupMenu _filemenu;
	private TQPopupMenu _colormenu;
	private TQPopupMenu _helpmenu;
	private TQScrollView _scrollview;
	private ScribbleArea _scribblearea;
	
	public class ScribbleArea extends TQWidget {
	    private TQPoint _last;
	    private TQColor _currentcolor;
	
	    private TQPixmap _buffer;
	    private TQPopupMenu _popupmenu;
	
		/**
		  * The constructor. Initializes the member variables.
		  */
		ScribbleArea()
		{
			// initialize member variables
			_buffer = new TQPixmap();
			_last = new TQPoint();
			_currentcolor = Qt.black();
			
			// don't blank the window before repainting
			setBackgroundMode( NoBackground );
			
			// create a pop-up menu
			_popupmenu = new TQPopupMenu();
			_popupmenu.insertItem( "&Clear", this, Qt.TQ_SLOT( "slotClearArea()" ) );
		}
		
		/**
		  * This slot sets the curren color for the scribble area. It will be
		  * connected with the colorChanged( TQColor ) signal from the
		  * ScribbleWindow.
		  */
		public void setColor( TQColor new_color )
		{
			_currentcolor = new_color;
		}
		
		/**
		 * This slot clears the drawing area by filling the off-screen buffer with
		 * white and copying it over to the window.
		 */
		public void slotClearArea()
		{
			// fill the off screen buffer with plain white
			_buffer.fill( Qt.white() );
			
			// and copy it over to the window
			TQPaintDevice.bitBlt( this, 0, 0, _buffer );
		}
		
		
		/**
		  * This method does the actual loading. It relies on TQPixmap (and the
		  * underlying I/O machinery) to determine the filetype.
		  */
		public void slotLoad( String filename )
		{
			if ( !_buffer.load( filename ) )
				TQMessageBox.warning( null, "Load error", "Could not load file" );
				
			repaint();  // refresh the window
		}
		
		
		/**
		 * This method does the actual saving. We hard-code the file type as
		 * BMP. Unix users might want to replace this with something like XPM.
		 */
		public void slotSave( String filename )
		{
			if( !_buffer.save( filename, "BMP" ) )
				TQMessageBox.warning( null, "Save error", "Could not save file" );
		}
		
		
		/**
		  * This method is called whenever the user presses the
		  * mouse over the window. It just records the position of the mouse
		  * at the time of the click.
		  */
		public void mousePressEvent(TQMouseEvent event)
		{
			if ( event.button() == RightButton )
				_popupmenu.exec( TQCursor.pos() );
			else
			  {
				_last = event.pos();	// retrieve the coordinates from the event
			  }
		}
		
		
		/**
		  * The method is called whenever the usr moves the mouse
		  * while the mouse button is pressed. If we had called
		  * setMouseTracking(true) before, the method would also be called
		  * when the mouse was moved with any button pressed. We know that
		  * we haven't, and thus don't have to check whether any buttons are
		  * pressed.
		  */
		public void mouseMoveEvent(TQMouseEvent event) {
			// create a TQPainter object for drawing onto the window
			TQPainter windowpainter = new TQPainter();
			// and another TQPainter object for drawing int an off-screen pixmap
			TQPainter bufferpainter = new TQPainter();
			
			// start painting
			windowpainter.begin( this ); // this painter paints onto the window
			bufferpainter.begin( _buffer );  // and this one paints in the buffer
		
			// set a standard pen with the currently selected color
			windowpainter.setPen( _currentcolor );
			bufferpainter.setPen( _currentcolor );
		
			// draw a line in both the window and the buffer
			windowpainter.drawLine( _last, event.pos() );
			bufferpainter.drawLine( _last, event.pos() );
		
			// done with painting
			windowpainter.end();
			bufferpainter.end();
		
			// remember the current mouse position
			_last = event.pos();
			eventCount++;						
		}
		/**
		  * This method is called whenever the widget needs
		  * painting, for example when it has been obscured and then revealed again.
		  */
		public void paintEvent(TQPaintEvent event) {
			TQPaintDevice.bitBlt(this, 0, 0, _buffer);
		}
	
		/**
		  * This method get called whenever the widget needs
		  * painting, for example, when it has been obscured and then revealed again.
		  */
		public void resizeEvent(TQResizeEvent event) {
			TQPixmap save = new TQPixmap( _buffer );
			_buffer.resize( event.size() );
			_buffer.fill( Qt.white() );
			TQPaintDevice.bitBlt( _buffer, 0, 0, save );
		}
	}
	
	TQTimerTest()
	{
		/* The next lines build the menu bar. We first create the menus
		* one by one, then add them to the menu bar. */
		_filemenu = new TQPopupMenu();  // create a file menu
		_filemenu.insertItem( "&Load", this, Qt.TQ_SLOT( "slotLoad()" ) );
		_filemenu.insertItem( "&Save", this, Qt.TQ_SLOT( "slotSave()" ) );
		_filemenu.insertSeparator();
		_filemenu.insertItem( "&Quit", Qt.qApp(), Qt.TQ_SLOT( "quit()" ) );
		
		_colormenu = new TQPopupMenu(); // create a color menu
		_colormenu.insertItem( "B&lack", COLOR_MENU_ID_BLACK);
		_colormenu.insertItem( "&Red", COLOR_MENU_ID_RED);
		_colormenu.insertItem( "&Blue", COLOR_MENU_ID_BLUE);
		_colormenu.insertItem( "&Green", COLOR_MENU_ID_GREEN);
		_colormenu.insertItem( "&Yellow", COLOR_MENU_ID_YELLOW);
		TQObject.connect( _colormenu, TQ_SIGNAL( "activated(int)" ),
						 this, TQ_SLOT( "slotColorMenu(int)" ) );
						
		_helpmenu = new TQPopupMenu();  // create a help menu
		_helpmenu.insertItem( "&About QtScribble", this, TQ_SLOT( "slotAbout()" ) );
		_helpmenu.insertItem( "&About Qt", this, TQ_SLOT( "slotAboutQt()" ) );
		
		_menubar = new TQMenuBar( this, "" );  // create a menu bar
		_menubar.insertItem( "&File", _filemenu );
		_menubar.insertItem( "&Color", _colormenu );
		_menubar.insertItem( "&Help", _helpmenu );
		
		/* We create a TQScrollView and a ScribbleArea. The ScribbleArea will
		 * be managed by the scroll view.*/
		_scrollview = new TQScrollView( this );
		_scrollview.setGeometry( 0, _menubar.height(),
		 							width(), height() - _menubar.height() );
		_scribblearea = new ScribbleArea();
		_scribblearea.setGeometry( 0, 0, 1000, 1000 );
		_scrollview.addChild( _scribblearea );
		TQObject.connect( this, TQ_SIGNAL( "colorChanged(TQColor)" ),
						 _scribblearea, TQ_SLOT( "setColor(TQColor)" ) );
		TQObject.connect( this, TQ_SIGNAL( "save(java.lang.String)" ),
						 _scribblearea, TQ_SLOT( "slotSave(java.lang.String)" ) );
		TQObject.connect( this, TQ_SIGNAL( "load(java.lang.String)" ),
						 _scribblearea, TQ_SLOT( "slotLoad(java.lang.String)" ) );
	}
	
	public void resizeEvent( TQResizeEvent event )
	{
		/* When the whole window is resized, we have to rearrange the geometry
		 * in the ScribbleWindow as well. Note that the ScribbleArea does not need
		 * to be changed. */
		 _scrollview.setGeometry( 0, _menubar.height(),
		 							width(), height() - _menubar.height() );
	}

	
	
	public void slotAbout()
	{
		TQMessageBox.information( this, "About QtScribble 5",
										"This is the Scribble 5 application\n" +
										"Copyright 1998 by Mathias Kalle Dalheimer\n"
										);
	}
	
	public void slotAboutQt()
	{
		TQMessageBox.aboutTQt( this, "About Qt" );
	}
	
	public void slotColorMenu( int item )
	{
		switch( item )
		{
			case COLOR_MENU_ID_BLACK:
				emit("colorChanged", Qt.black());
				break;
			case COLOR_MENU_ID_RED:
				emit("colorChanged", Qt.darkRed());
				break;
			case COLOR_MENU_ID_BLUE:
				emit("colorChanged", Qt.darkBlue());
				break;
			case COLOR_MENU_ID_GREEN:
				emit("colorChanged", Qt.darkGreen());
				break;
			case COLOR_MENU_ID_YELLOW:
				emit("colorChanged", Qt.yellow());
				break;
		}
	}
	
	
	/**
	  * This is the slot for the menu item File/Load. It opens a
	  * TQFileDialog to ask the user for a filename, then emits a save()
	  * signal with the filename as parameter.
	  */
	public void slotLoad()
	{
		/* Open a file dialog for loading. The default directory is the
		 * current directory, the filter *.bmp.
		 */
		String filename = TQFileDialog.getOpenFileName( ".", "*.bmp", this );
		if ( !filename.equals("") )
			emit("load", filename);
	}
	
	/**
	  * This is the slot for the menu item File/Load. It opens a
	  * TQFileDialog to ask the user for a filename, then emits a save()
	  * signal with the filename as parameter.
	  */
	public void slotSave()
	{
		/* Open a file dialog for saving. The default directory is the
		 * current directory, the filter *.bmp.
		 */
		String filename = TQFileDialog.getSaveFileName( ".", "*.bmp", this );
		if ( !filename.equals("") )
			emit("save", filename);
	}
	
	public void timerDone()
	{
//		System.out.println("in timerDone...");
	}
	
	public static void main(String[] args)
	{
		TQApplication myapp = new TQApplication(args);
		TQTimerTest mywidget = new TQTimerTest();
		mywidget.setGeometry(50, 500, 400, 400);
    	
    	TQTimer timer = new TQTimer( mywidget );
		connect( timer, Qt.TQ_SIGNAL("timeout()"),
					mywidget, Qt.TQ_SLOT("timerDone()") );
		timer.start( 2000, false );                 // 2 seconds single-shot
		
		myapp.setMainWidget(mywidget);

		AppWizardBase mywizard = new AppWizardBase(mywidget, "", false, 0);
		mywizard.show();
		FilePropsPageBase mypropspage = new FilePropsPageBase(mywidget, "", 0);
		mypropspage.show();
		
		mywidget.show();
		myapp.exec();
		System.out.println("event count: " + eventCount);
		return;
	}
	
	static {
                qtjava.initialize();
	}
}
