/* This file is part of the KDE project
   Copyright (C) 1998, 1999 Reginald Stadlbauer <reggie@kde.org>
   2000, 2001 Werner Trobin <trobin@kde.org>
   2002, 2003 Thomas Nagy <tnagy@eleve.emn.fr>
   2004 David Faure <faure@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
   */

// Description: Template Choose Dialog

/******************************************************************/

#include "KoTemplateChooseDia.h"

#include <tdelocale.h>
#include <tdeversion.h>
#include <tdefiledialog.h>
#include <tdeinstance.h>
#include <KoFilterManager.h>
#include <KoTemplates.h>
#include <KoDocument.h>
#include <tdemainwindow.h>

#include <kdebug.h>
#include <kpushbutton.h>
#include <kjanuswidget.h>
#include <tdeglobalsettings.h>
#include <ktextedit.h>
#include <tdefileiconview.h>
#include <tdefileitem.h>
#include <tdemessagebox.h>
#include <tdeapplication.h>
#include <tdeaboutdata.h>

#include <tqapplication.h>
#include <tqlayout.h>
#include <tqtabwidget.h>
#include <tqcombobox.h>
#include <tqcheckbox.h>
#include <tqpoint.h>
#include <tqobjectlist.h>
#include <tqvgroupbox.h>
#include <tqtooltip.h>

class MyFileDialog : public KFileDialog
{
    public :
        MyFileDialog(
                const TQString& startDir=0,
                const TQString& filter =0,
                TQWidget *parent=0,
                const char *name=0,
                bool modal=0)
            :  KFileDialog (startDir, filter, parent, name, modal),
        m_slotOkCalled( false ) {}

        KURL currentURL()
        {
            setResult( TQDialog::Accepted ); // selectedURL tests for it
            return KFileDialog::selectedURL();
        }

        // Return true if the current URL exists, show msg box if not
        bool checkURL()
        {
            bool ok = true;
            KURL url = currentURL();
            if ( url.isLocalFile() )
            {
                ok = TQFile::exists( url.path() );
                if ( !ok ) {
                    // Maybe offer to create a new document with that name? (see alos KoDocument::openFile)
                    KMessageBox::error( this, i18n( "The file %1 does not exist." ).arg( url.path() ) );
                }
            }
            return ok;
        }
        // Called directly by pressing Return in the location combo
        // (so we need to remember that it got called, to avoid calling it twice)
        // Called "by hand" when clicking on our OK button
        void slotOk() {
            m_slotOkCalled = true;
            KFileDialog::slotOk();
        }
        bool slotOkCalled() const { return m_slotOkCalled; }
    protected:
    // Typing a file that doesn't exist closes the file dialog, we have to
    // handle this case better here.
        virtual void accept() {
            if ( checkURL() )
                KFileDialog::accept();
        }

        virtual void reject() {
		KFileDialog::reject();
		emit cancelClicked();
        }
private:
        bool m_slotOkCalled;
};

/*================================================================*/

/*================================================================*/

class KoTemplateChooseDiaPrivate {
    public:
	KoTemplateChooseDiaPrivate(const TQCString& templateType, TDEInstance* instance,
                                   const TQCString &format,
                                   const TQString &nativeName,
                                   const TQStringList& extraNativeMimeTypes,
                                   const KoTemplateChooseDia::DialogType &dialogType) :
	    m_templateType(templateType), m_instance(instance), m_format(format),
            m_nativeName(nativeName), m_extraNativeMimeTypes( extraNativeMimeTypes ),
            m_dialogType(dialogType), tree(0),
            m_nostartupdlg( false ),
            m_mainwidget(0), m_nodiag( 0 )
	{
	    m_returnType = KoTemplateChooseDia::Empty;
	}

	~KoTemplateChooseDiaPrivate() {}

	TQCString m_templateType;
	TDEInstance* m_instance;
	TQCString m_format;
	TQString m_nativeName;
        TQStringList m_extraNativeMimeTypes;

        KoTemplateChooseDia::DialogType m_dialogType;
	KoTemplateTree *tree;

	TQString m_templateName;
	TQString m_fullTemplateName;
	KoTemplateChooseDia::ReturnType m_returnType;

	bool m_nostartupdlg;

	// the main widget
	TQWidget *m_mainwidget;

	// do not show this dialog at startup
	TQCheckBox *m_nodiag;

	// choose a template
	KJanusWidget * m_jwidget;
	KFileIconView *m_recent;
	TQVGroupBox * boxdescription;
	KTextEdit * textedit;

	// choose a file
	MyFileDialog *m_filedialog;

	// for the layout
	TQTabWidget* tabWidget;
	TQWidget* newTab;
	TQWidget* existingTab;
	TQWidget* recentTab;

};

/******************************************************************/
/* Class: KoTemplateChooseDia                                     */
/******************************************************************/

/*================================================================*/
KoTemplateChooseDia::KoTemplateChooseDia(TQWidget *parent, const char *name, TDEInstance* instance,
                                         const TQCString &format,
                                         const TQString &nativeName,
                                         const TQStringList &extraNativeMimeTypes,
                                         const DialogType &dialogType,
                                         const TQCString& templateType) :
    KDialogBase(parent, name, true, i18n("Open Document"), KDialogBase::Ok | KDialogBase::Cancel,
                KDialogBase::Ok)
{
    d = new KoTemplateChooseDiaPrivate(
        templateType,
        instance,
        format,
        nativeName,
        extraNativeMimeTypes,
        dialogType);

    TQPushButton* ok = actionButton( KDialogBase::Ok );
    TQPushButton* cancel = actionButton( KDialogBase::Cancel );
    cancel->setAutoDefault(false);
    ok->setDefault(true);
    //enableButtonOK(false);

    if (!templateType.isNull() && !templateType.isEmpty() && dialogType!=NoTemplates)
        d->tree = new KoTemplateTree(templateType, instance, true);

    d->m_mainwidget = makeMainWidget();

    d->m_templateName = "";
    d->m_fullTemplateName = "";
    d->m_returnType = Cancel;

    setupDialog();
}

KoTemplateChooseDia::~KoTemplateChooseDia()
{
    delete d->tree;
    delete d;
    d=0L;
}

// Keep in sync with KoMainWindow::chooseNewDocument
static bool cancelQuits() {
    bool onlyDoc = !KoDocument::documentList() || KoDocument::documentList()->count() <= 1;
    bool onlyMainWindow = !TDEMainWindow::memberList || TDEMainWindow::memberList->count() <= 1;
    return onlyDoc && onlyMainWindow && tdeApp->instanceName() != "koshell"; // hack for koshell
}

KoTemplateChooseDia::ReturnType KoTemplateChooseDia::choose(TDEInstance* instance, TQString &file,
                                                            const KoTemplateChooseDia::DialogType &dialogType,
                                                            const TQCString& templateType,
                                                            TQWidget* parent)
{
    const TQString nativeName = instance->aboutData()->programName();
    const TQCString format = KoDocument::readNativeFormatMimeType( instance );
    const TQStringList extraNativeMimeTypes = KoDocument::readExtraNativeMimeTypes( instance );
    // Maybe the above two can be combined into one call, for speed:
    //KoDocument::getNativeMimeTypeInfo( instance, nativeName, extraNativeMimeTypes );
    return choose( instance, file, format, nativeName, extraNativeMimeTypes,
                   dialogType, templateType, parent );
}

KoTemplateChooseDia::ReturnType KoTemplateChooseDia::choose(TDEInstance* instance, TQString &file,
                                       const TQCString &format,
                                       const TQString &nativeName,
                                       const TQStringList& extraNativeMimeTypes,
                                       const DialogType &dialogType,
                                       const TQCString& templateType,
                                       TQWidget* parent )
{
    KoTemplateChooseDia *dlg = new KoTemplateChooseDia(
        parent, "Choose", instance, format,
        nativeName, extraNativeMimeTypes, dialogType, templateType );

    KoTemplateChooseDia::ReturnType rt = Cancel;

    if (dlg->noStartupDlg())
    {
	// start with the default template
	file = dlg->getFullTemplate();
	rt = dlg->getReturnType();
    }
    else
    {
	dlg->resize( 700, 480 );
	if ( dlg->exec() == TQDialog::Accepted )
	{
	    file = dlg->getFullTemplate();
	    rt = dlg->getReturnType();
	}
    }

    delete dlg;
    return rt;
}

bool KoTemplateChooseDia::noStartupDlg() const {
    return d->m_nostartupdlg;
}


TQString KoTemplateChooseDia::getTemplate() const{
    return d->m_templateName;
}

TQString KoTemplateChooseDia::getFullTemplate() const{
    return d->m_fullTemplateName;
}

KoTemplateChooseDia::ReturnType KoTemplateChooseDia::getReturnType() const {
    return d->m_returnType;
}

KoTemplateChooseDia::DialogType KoTemplateChooseDia::getDialogType() const {
    return d->m_dialogType;
}

/*================================================================*/
// private
void KoTemplateChooseDia::setupRecentDialog(TQWidget * widgetbase, TQGridLayout * layout)
{

        d->m_recent = new KoTCDRecentFilesIconView(widgetbase, "recent files");
        // I prefer the icons to be in "most recent first" order (DF)
        d->m_recent->setSorting( static_cast<TQDir::SortSpec>( TQDir::Time | TQDir::Reversed ) );
        layout->addWidget(d->m_recent,0,0);

        TQString oldGroup = d->m_instance->config()->group();
        d->m_instance->config()->setGroup( "RecentFiles" );

        int i = 0;
        TQString value;
        do {
                TQString key=TQString( "File%1" ).arg( i );
                value=d->m_instance->config()->readPathEntry( key );
                if ( !value.isEmpty() ) {
                    // Support for tdelibs-3.5's new RecentFiles format: name[url]
                    TQString s = value;
                    if ( s.endsWith("]") )
                    {
                        int pos = s.find("[");
                        s = s.mid( pos + 1, s.length() - pos - 2);
                    }
                    KURL url(s);

                    if(!url.isLocalFile() || TQFile::exists(url.path())) {
                        KFileItem *item = new KFileItem( KFileItem::Unknown, KFileItem::Unknown, url );
                        d->m_recent->insertItem(item);
                    }
                }
                i++;
        } while ( !value.isEmpty() || i<=10 );

        d->m_instance->config()->setGroup( oldGroup );
        d->m_recent->showPreviews();

	connect(d->m_recent, TQ_SIGNAL( doubleClicked ( TQIconViewItem * ) ),
			this, TQ_SLOT( recentSelected( TQIconViewItem * ) ) );

}

/*================================================================*/
// private
void KoTemplateChooseDia::setupFileDialog(TQWidget * widgetbase, TQGridLayout * layout)
{
    TQString dir = TQString();
    TQPoint point( 0, 0 );

    d->m_filedialog=new MyFileDialog(dir,
	    TQString(),
	    widgetbase,
	    "file dialog",
	    false);

    layout->addWidget(d->m_filedialog,0,0);
    d->m_filedialog->reparent( widgetbase , point );
    //d->m_filedialog->setOperationMode( KFileDialog::Opening);

    TQObjectList *l = d->m_filedialog->queryList( "TQPushButton" );
    TQObjectListIt childit( *l );
    TQObject *obj;
    while ( (obj = childit.current()) != 0 ) {
	++childit;
	((TQPushButton*)obj)->hide();
    }
    delete l;

    d->m_filedialog->setSizeGripEnabled ( false );

    TQStringList mimeFilter = KoFilterManager::mimeFilter( d->m_format, KoFilterManager::Import );
    TQStringList::Iterator mimeFilterIt = mimeFilter.at( 1 );
    for ( TQStringList::ConstIterator it = d->m_extraNativeMimeTypes.begin();
          it != d->m_extraNativeMimeTypes.end(); ++it ) {
        mimeFilterIt = mimeFilter.insert( mimeFilterIt, *it );
        ++mimeFilterIt;
    }
    d->m_filedialog->setMimeFilter( mimeFilter );

    connect(d->m_filedialog, TQ_SIGNAL(  okClicked() ),
	    this, TQ_SLOT (  slotOk() ));

    connect(d->m_filedialog, TQ_SIGNAL( cancelClicked() ),
	    this, TQ_SLOT (  slotCancel() ));

}

/*================================================================*/
// private
void KoTemplateChooseDia::setupTemplateDialog(TQWidget * widgetbase, TQGridLayout * layout)
{

    d->m_jwidget = new KJanusWidget(
	    widgetbase,
	    "kjanuswidget",
	    KJanusWidget::IconList);
    layout->addWidget(d->m_jwidget,0,0);

    d->boxdescription = new TQVGroupBox(
	    i18n("Selected Template"),
	    widgetbase,
	    "boxdescription");
    layout->addWidget(d->boxdescription, 1, 0 );

    // config
    TDEConfigGroup grp( d->m_instance->config(), "TemplateChooserDialog" );
    int templateNum = grp.readNumEntry( "TemplateTab", -1 );
    TQString templateName = grp.readPathEntry( "TemplateName" );
	if ( templateName.isEmpty() && d->tree->defaultTemplate() )
		templateName = d->tree->defaultTemplate()->name(); //select the default template for the app

    // item which will be selected initially
    TQIconViewItem * itemtoselect = 0;

    // count the templates inserted
    int entriesnumber = 0;
	int defaultTemplateGroup = -1;

    for ( KoTemplateGroup *group = d->tree->first(); group!=0L; group=d->tree->next() )
    {
	if (group->isHidden())
	    continue;

	if ( d->tree->defaultGroup() == group )
		defaultTemplateGroup = entriesnumber; //select the default template group for the app

	TQFrame * frame = d->m_jwidget->addPage (
		group->name(),
		group->name(),
		group->first()->loadPicture(d->m_instance));

	TQGridLayout * layout = new TQGridLayout(frame);
	KoTCDIconCanvas *canvas = new KoTCDIconCanvas( frame );
	layout->addWidget(canvas,0,0);

	canvas->setBackgroundColor( colorGroup().base() );
	canvas->setResizeMode(TQIconView::Adjust);
	canvas->setWordWrapIconText( true );
	canvas->show();

	TQIconViewItem * tempitem = canvas->load(group, templateName, d->m_instance);
	if (tempitem)
	    itemtoselect = tempitem;

	canvas->sort();
	canvas->setSelectionMode(TQIconView::Single);

	connect( canvas, TQ_SIGNAL( clicked ( TQIconViewItem * ) ),
		this, TQ_SLOT( currentChanged( TQIconViewItem * ) ) );

	connect( canvas, TQ_SIGNAL( doubleClicked( TQIconViewItem * ) ),
		this, TQ_SLOT( chosen(TQIconViewItem *) ) );

	entriesnumber++;
    }

    d->boxdescription->setInsideMargin ( 3 );
    d->boxdescription->setInsideSpacing ( 3 );

    d->textedit = new KTextEdit( d->boxdescription );
    d->textedit->setReadOnly(1);
    d->textedit->setText(descriptionText(i18n("Empty Document"), i18n("Creates an empty document")));
    d->textedit->setLineWidth(0);
    d->textedit->setMaximumHeight(50);

    // Hide the widget if there is no template available. This should never happen ;-)
    if (!entriesnumber)
	d->m_jwidget->hide();

    // Set the initially shown page, possibly from the last usage of the dialog
    if (entriesnumber >= templateNum && templateNum != -1 )
	d->m_jwidget->showPage(templateNum);
    else if ( defaultTemplateGroup != -1)
	d->m_jwidget->showPage(defaultTemplateGroup);
    

    // Set the initially selected template, possibly from the last usage of the dialog
    currentChanged(itemtoselect);

    // setup the checkbox
    TQString translatedstring = i18n("Always start %1 with the selected template").arg(d->m_nativeName);

    d->m_nodiag = new TQCheckBox ( translatedstring , widgetbase);
    layout->addWidget(d->m_nodiag, 2, 0);
    TQString  startwithoutdialog = grp.readEntry( "NoStartDlg" );
    bool ischecked = startwithoutdialog == TQString("yes");

    // When not starting up, display a tri-state button telling whether
    // the user actually choosed the template to start with next times (bug:77542)
    if (d->m_dialogType == Everything)
    {
        d->m_nodiag->setChecked( ischecked );
    }
    else
    {
        d->m_nodiag->setTristate();
        d->m_nodiag->setNoChange();
    }
}

/*================================================================*/
// private
void KoTemplateChooseDia::setupDialog()
{

    TQGridLayout *maingrid=new TQGridLayout( d->m_mainwidget, 1, 1, 2, 6);
    TDEConfigGroup grp( d->m_instance->config(), "TemplateChooserDialog" );

    if (d->m_dialogType == Everything)
    {

	// the user may want to start with his favorite template
	if (grp.readEntry( "NoStartDlg" ) == TQString("yes") )
	{
	    d->m_nostartupdlg = true;
	    d->m_returnType = Empty;

	    // no default template, just start with an empty document
	    if (grp.readEntry("LastReturnType") == TQString("Empty") )
		return;

	    // start with the default template
	    d->m_templateName = grp.readPathEntry( "TemplateName" );
	    d->m_fullTemplateName = grp.readPathEntry( "FullTemplateName" );

	    // be paranoid : invalid template means empty template
	    if (!TQFile::exists(d->m_fullTemplateName))
		return;

	    if (d->m_fullTemplateName.length() < 2)
		return;

	    d->m_returnType = Template;
	    return;
	}

	if ( cancelQuits() )
	    setButtonCancel( KStdGuiItem::quit() );

	d->tabWidget = new TQTabWidget( d->m_mainwidget, "tabWidget" );
	maingrid->addWidget( d->tabWidget, 0, 0 );

	// new document
	d->newTab = new TQWidget( d->tabWidget, "newTab" );
	d->tabWidget->insertTab( d->newTab, i18n( "&Create Document" ) );
	TQGridLayout * newTabLayout = new TQGridLayout( d->newTab, 1, 1, KDialogBase::marginHint(), KDialogBase::spacingHint());

	// existing document
	d->existingTab = new TQWidget( d->tabWidget, "existingTab" );
	d->tabWidget->insertTab( d->existingTab, i18n( "Open &Existing Document" ) );
	TQGridLayout * existingTabLayout = new TQGridLayout( d->existingTab, 1, 1, 0, KDialog::spacingHint());

        // recent document
        d->recentTab = new TQWidget( d->tabWidget, "recentTab" );
        d->tabWidget->insertTab( d->recentTab, i18n( "Open &Recent Document" ) );
        TQGridLayout * recentTabLayout = new TQGridLayout( d->recentTab, 1, 1, KDialogBase::marginHint(), KDialog::spacingHint());

	setupTemplateDialog(d->newTab, newTabLayout);
	setupFileDialog(d->existingTab, existingTabLayout);
	setupRecentDialog(d->recentTab, recentTabLayout);

	TQString tabhighlighted = grp.readEntry("LastReturnType");
	if ( tabhighlighted == "Template" )
	    d->tabWidget->setCurrentPage(0); // CreateDocument tab
	else if (tabhighlighted == "File" )
	    d->tabWidget->setCurrentPage(2); // RecentDocument tab
	else
		d->tabWidget->setCurrentPage(0); // Default setting: CreateDocument tab
    }
    else
    {

	// open a file
	if (d->m_dialogType == NoTemplates)
	{
	    setupFileDialog(d->m_mainwidget, maingrid);
	}
	// create a new document from a template
	if (d->m_dialogType == OnlyTemplates)
	{
	    setCaption(i18n( "Create Document" ));
	    setupTemplateDialog(d->m_mainwidget, maingrid);
	}
    }
}

/*================================================================*/
// private SLOT
void KoTemplateChooseDia::currentChanged( TQIconViewItem * item)
{
    if (item)
    {
	TQIconView* canvas =  item->iconView();

	// set text in the textarea
	d->textedit->setText( descriptionText(
				item->text(),
				((KoTCDIconViewItem *) item)->getDescr()
				));

	// set the icon in the canvas selected
	if (canvas)
	    canvas->setSelected(item,1,0);

	// register the current template
	d->m_templateName = item->text();
	d->m_fullTemplateName = ((KoTCDIconViewItem *) item)->getFName();
    }
}

/*================================================================*/
// private SLOT
void KoTemplateChooseDia::chosen(TQIconViewItem * item)
{
    // the user double clicked on a template
    if (item)
    {
	currentChanged(item);
	slotOk();
    }
}

/* */
// private SLOT
void KoTemplateChooseDia::recentSelected( TQIconViewItem * item)
{
	if (item)
	{
		slotOk();
	}
}

/*================================================================*/
// protected SLOT
void KoTemplateChooseDia::slotOk()
{
    // Collect info from the dialog into d->m_returnType and d->m_templateName etc.
    if (collectInfo())
    {
	// Save it for the next time
	TDEConfigGroup grp( d->m_instance->config(), "TemplateChooserDialog" );
	static const char* const s_returnTypes[] = { 0 /*Cancel ;)*/, "Template", "File", "Empty" };
	if ( d->m_returnType <= Empty )
	{
	    grp.writeEntry( "LastReturnType", TQString::fromLatin1(s_returnTypes[d->m_returnType]) );
	    if (d->m_returnType == Template)
	    {
		grp.writeEntry( "TemplateTab", d->m_jwidget->activePageIndex() );
		grp.writePathEntry( "TemplateName", d->m_templateName );
		grp.writePathEntry( "FullTemplateName", d->m_fullTemplateName);
	    }

	    if (d->m_nodiag)
	    {
		// The checkbox m_nodiag is in tri-state mode for new documents
		// fixes bug:77542
		if (d->m_nodiag->state() == TQButton::On) {
		    grp.writeEntry( "NoStartDlg", "yes");
		}
		else if (d->m_nodiag->state() == TQButton::Off) {
		    grp.writeEntry( "NoStartDlg", "no");
		}
	    }
	}
	else
	{
	    kdWarning(30003) << "Unsupported template chooser result: " << d->m_returnType << endl;
	    grp.writeEntry( "LastReturnType", TQString() );
	}
	KDialogBase::slotOk();
    }
}

/*================================================================*/
// private
bool KoTemplateChooseDia::collectInfo()
{


    // to determine what tab is selected in "Everything" mode
    bool newTabSelected = false;
    if ( d->m_dialogType == Everything)
	if ( d->tabWidget->currentPage() == d->newTab )
	    newTabSelected = true;

    // is it a template or a file ?
    if ( d->m_dialogType==OnlyTemplates || newTabSelected )
    {
	// a template is chosen
	if (d->m_templateName.length() > 0)
	    d->m_returnType = Template;
	else
	    d->m_returnType=Empty;

	return true;
    }
    else if ( d->m_dialogType != OnlyTemplates )
    {
	// a file is chosen
	if (d->m_dialogType == Everything && d->tabWidget->currentPage() == d->recentTab)
	{
		// Recent file
		KFileItem * item = d->m_recent->currentFileItem();
		if (! item)
			return false;
		KURL url = item->url();
		if(url.isLocalFile() && !TQFile::exists(url.path()))
		{
			KMessageBox::error( this, i18n( "The file %1 does not exist." ).arg( url.path() ) );
			return false;
		}
		d->m_fullTemplateName = url.url();
		d->m_returnType = File;
	}
	else
	{
		// Existing file from file dialog
	        if ( !d->m_filedialog->slotOkCalled() )
	            d->m_filedialog->slotOk();
		KURL url = d->m_filedialog->currentURL();
		d->m_fullTemplateName = url.url();
	        d->m_returnType = File;
	        return d->m_filedialog->checkURL();
	}
	return true;
    }

    d->m_returnType=Empty;
    return false;
}

/*================================================================*/
//private
TQString KoTemplateChooseDia::descriptionText(const TQString &name, const TQString &description)
{
	TQString descrText(i18n("Name:"));
	descrText += " " + name;
	descrText += "\n";
	descrText += i18n("Description:");
	if (description.isEmpty())
	      descrText += " " + i18n("No description available");
	else
              descrText += " " + description;
	return descrText;
}

/*================================================================*/

TQIconViewItem * KoTCDIconCanvas::load( KoTemplateGroup *group, const TQString& name, TDEInstance* instance )
{
    TQIconViewItem * itemtoreturn = 0;

    for (KoTemplate *t=group->first(); t!=0L; t=group->next()) {
	if (t->isHidden())
	    continue;
	TQIconViewItem *item = new KoTCDIconViewItem(
		this,
		t->name(),
		t->loadPicture(instance),
		t->description(),
		t->file());

	if (name == t->name())
	{
	    itemtoreturn = item;
	}

	item->setKey(t->name());
	item->setDragEnabled(false);
	item->setDropEnabled(false);
    }

    return itemtoreturn;
}

/*================================================================*/

KoTCDRecentFilesIconView::~KoTCDRecentFilesIconView()
{
    removeToolTip();
}

void KoTCDRecentFilesIconView::showToolTip( TQIconViewItem* item )
{
    removeToolTip();
    if ( !item )
        return;

    // Mostly duplicated from KFileIconView, because it only shows tooltips
    // for truncated icon texts, and we want tooltips on all icons,
    // with the full path...
    // KFileIconView would need a virtual method for deciding if a tooltip should be shown,
    // and another one for deciding what's the text of the tooltip...
    const KFileItem *fi = ( (KFileIconViewItem*)item )->fileInfo();
    TQString toolTipText = fi->url().prettyURL( 0, KURL::StripFileProtocol );
    toolTip = new TQLabel( TQString::fromLatin1(" %1 ").arg(toolTipText), 0,
                          "myToolTip",
                          WStyle_StaysOnTop | WStyle_Customize | WStyle_NoBorder | WStyle_Tool | WX11BypassWM );
    toolTip->setFrameStyle( TQFrame::Plain | TQFrame::Box );
    toolTip->setLineWidth( 1 );
    toolTip->setAlignment( AlignLeft | AlignTop );
    toolTip->move( TQCursor::pos() + TQPoint( 14, 14 ) );
    toolTip->adjustSize();
    TQRect screen = TQApplication::desktop()->screenGeometry(
        TQApplication::desktop()->screenNumber(TQCursor::pos()));
    if (toolTip->x()+toolTip->width() > screen.right()) {
        toolTip->move(toolTip->x()+screen.right()-toolTip->x()-toolTip->width(), toolTip->y());
    }
    if (toolTip->y()+toolTip->height() > screen.bottom()) {
        toolTip->move(toolTip->x(), screen.bottom()-toolTip->y()-toolTip->height()+toolTip->y());
    }
    toolTip->setFont( TQToolTip::font() );
    toolTip->setPalette( TQToolTip::palette(), true );
    toolTip->show();
}

void KoTCDRecentFilesIconView::removeToolTip()
{
    delete toolTip;
    toolTip = 0;
}

void KoTCDRecentFilesIconView::hideEvent( TQHideEvent *ev )
{
    removeToolTip();
    KFileIconView::hideEvent( ev );
}

#include "KoTemplateChooseDia.moc"
