/* This file is part of the KDE project
   Copyright (C) 2004 Cedric Pasteur <cedric.pasteur@free.fr>
   Copyright (C) 2007 Jaroslaw Staniek <js@iidea.pl>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#include <tqcursor.h>
#include <tqobjectlist.h>
#include <tqtabwidget.h>
#include <tqtabbar.h>

#include <kdebug.h>
#include <kexiutils/utils.h>

#include "form.h"
#include "objecttree.h"
#include "utils.h"

using namespace KFormDesigner;

void
KFormDesigner::removeChildrenFromList(WidgetList &list)
{
	for(WidgetListIterator it(list); it.current() != 0; ++it)  {
		TQWidget *w = it.current();

		// If any widget in the list is a child of this widget, we remove it from the list
		for(WidgetListIterator it2(list); it2.current() != 0; ++it2) {
			TQWidget *widg = it2.current();
			if((w != widg) && (w->child(widg->name())))
			{
				kdDebug() << "Removing the widget " << widg->name() << "which is a child of " << w->name() << endl;
				list.remove(widg);
			}
		}
	}
}

void
KFormDesigner::installRecursiveEventFilter(TQObject *object, TQObject *container)
{
	if(!object || !container|| !object->isWidgetType())
		return;

	kdDebug() << "Installing event filter on widget: " << object->name() << " directed to " << container->name() << endl;
	object->installEventFilter(container);
	if(((TQWidget*)object)->ownCursor())
		((TQWidget*)object)->setCursor(TQCursor(TQt::ArrowCursor));

	TQObjectList list = object->childrenListObject();

	if(list.isEmpty())
		return;

	for(TQObject *obj = list.first(); obj; obj = list.next())
		installRecursiveEventFilter(obj, container);
}

void
KFormDesigner::removeRecursiveEventFilter(TQObject *object, TQObject *container)
{
	object->removeEventFilter(container);
	if(!object->isWidgetType())
		return;
	TQObjectList list = object->childrenListObject();
	if(list.isEmpty())
		return;

	for(TQObject *obj = list.first(); obj; obj = list.next())
		removeRecursiveEventFilter(obj, container);
}

void
KFormDesigner::setRecursiveCursor(TQWidget *w, Form *form)
{
	ObjectTreeItem *tree = form->objectTree()->lookup(w->name());
	if(tree && ((tree->modifiedProperties()->contains("cursor")) || !tree->children()->isEmpty())
		&& !w->inherits("TQLineEdit") && !w->inherits("TQTextEdit")
		) //fix weird behaviour
		return; // if the user has set a cursor for this widget or this is a container, don't change it

	if(w->ownCursor())
		w->setCursor(TQt::ArrowCursor);

	TQObjectList *l = w->queryList( "TQWidget" );
	for(TQObject *o = l->first(); o; o = l->next())
		((TQWidget*)o)->setCursor(TQt::ArrowCursor);
	delete l;
}

TQSize
KFormDesigner::getSizeFromChildren(TQWidget *w, const char *inheritClass)
{
	int tmpw = 0, tmph = 0;
	TQObjectList *list = w->queryList(inheritClass, 0, false, false);
	for(TQObject *o = list->first(); o; o = list->next()) {
		TQRect  r = ((TQWidget*)o)->geometry();
		tmpw = TQMAX(tmpw, r.right());
		tmph = TQMAX(tmph, r.bottom());
	}

	delete list;
	return TQSize(tmpw, tmph) + TQSize(10, 10);
}

// -----------------

HorWidgetList::HorWidgetList(TQWidget *topLevelWidget)
	: WidgetList()
	, m_topLevelWidget(topLevelWidget)
{
}

HorWidgetList::~HorWidgetList()
{
}

int HorWidgetList::compareItems(TQPtrCollection::Item item1, TQPtrCollection::Item item2)
{
	TQWidget *w1 = static_cast<TQWidget*>(item1);
	TQWidget *w2 = static_cast<TQWidget*>(item2);
	return w1->mapTo(m_topLevelWidget, TQPoint(0,0)).x() - w2->mapTo(m_topLevelWidget, TQPoint(0,0)).x();
}

// -----------------

VerWidgetList::VerWidgetList(TQWidget *topLevelWidget)
	: WidgetList()
	, m_topLevelWidget(topLevelWidget)
{
}

VerWidgetList::~VerWidgetList()
{
}

int VerWidgetList::compareItems(TQPtrCollection::Item item1, TQPtrCollection::Item item2)
{
	TQWidget *w1 = static_cast<TQWidget*>(item1);
	TQWidget *w2 = static_cast<TQWidget*>(item2);

	int y1, y2;
	TQObject *page1 = 0;
	TabWidget *tw1 = KFormDesigner::findParent<KFormDesigner::TabWidget>(w1, "KFormDesigner::TabWidget", page1);
	if (tw1) // special case
		y1 = w1->mapTo(m_topLevelWidget, TQPoint(0,0)).y() + tw1->tabBarHeight() -2 -2;
	else
		y1 = w1->mapTo(m_topLevelWidget, TQPoint(0,0)).y();

	TQObject *page2 = 0;
	TabWidget *tw2 = KFormDesigner::findParent<KFormDesigner::TabWidget>(w2, "KFormDesigner::TabWidget", page2);
	if (tw1 && tw2 && tw1 == tw2 && page1 != page2) {
		// this sorts widgets by tabs there're put in
		return tw1->indexOf(static_cast<TQWidget*>(page1)) - tw2->indexOf(static_cast<TQWidget*>(page2));
	}

	if (tw2) // special case
		y2 = w2->mapTo(m_topLevelWidget, TQPoint(0,0)).y() + tw2->tabBarHeight() -2 -2;
	else
		y2 = w2->mapTo(m_topLevelWidget, TQPoint(0,0)).y();
	
	kdDebug() << w1->name() << ": " << y1 << " " 
		<< " | " << w2->name() << ": " << y2 << endl;


	//kdDebug() << w1->name() << ": " << w1->mapTo(m_topLevelWidget, TQPoint(0,0)) << " " << w1->y()
		//<< " | " << w2->name() << ":" /*<< w2->mapFrom(m_topLevelWidget, TQPoint(0,w2->y()))*/ << " " << w2->y() << endl;
	return y1 - y2;
}

#include "utils.moc"
