/***************************************************************************
 *   Copyright (C) 2003 by S�astien Laot                                 *
 *   slaout@linux62.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.             *
 ***************************************************************************/

#ifndef BASKET_H
#define BASKET_H

#include <tqscrollview.h>
#include <tqtooltip.h>
#include <tqvaluelist.h>
#include <tqtimer.h>
#include <tqimage.h>
#include <tqdatetime.h>
#include <tqclipboard.h>
#include <tdeshortcut.h>
#include <kdirwatch.h>
#include <tdeaction.h>
#include <tdeio/job.h>
#include <kdialogbase.h>

#include "filter.h"
#include "note.h" // For Note::Zone

#define BASKET_USE_DRKONQI

class TQVBoxLayout;
class TQDomDocument;
class TQDomElement;

class Basket;
class Note;
class NoteEditor;
class Tag;
#ifdef HAVE_LIBGPGME
class KGpgMe;
#endif

/** Provide a dialog to avert the user the disk is full.
  * This dialog is modal and is shown until the user has made space on the disk.
  * @author S�astien Laot
  */
class DiskErrorDialog : public KDialogBase
{
  TQ_OBJECT
  
  public:
	DiskErrorDialog(const TQString &titleMessage, const TQString &message, TQWidget *parent = 0);
	~DiskErrorDialog();
  protected:
	void closeEvent(TQCloseEvent *event);
	void keyPressEvent(TQKeyEvent*);
};


/** A list of flags to set how notes are inserted/plugged in the basket
  * Declare a varible with the type PlugOptions::Flags and assign a value like PlugOptions::DoSelection...
  * @author S�astien Laot
  */
namespace PlugOptions
{
	enum Flags {
		SelectOnlyNewNotes = 0x01, /// << Unselect every notes in the basket and select the newly inserted ones
		DoTagsInheriting   = 0x02  /// << The new notes inherit the tags of the sibbling note
	};
	// TODO: FocusLastInsertedNote (last visible!), EnsureVisibleAddedNotes, PopupFeebackBaloon (if not called by hand), AnimateNewPosition, FeedbackUnmatched
	// TODO: moveNoteInTree(bool animate);
}

/** This represent a hierarchy of the selected classes.
  * If this is null, then there is no selected note.
  */
class NoteSelection
{
  public:
	NoteSelection()        : note(0), parent(0), firstChild(0), next(0), fullPath() {}
	NoteSelection(Note *n) : note(n), parent(0), firstChild(0), next(0), fullPath() {}

	Note          *note;
	NoteSelection *parent;
	NoteSelection *firstChild;
	NoteSelection *next;
	TQString        fullPath; // Needeed for 'Cut' code to store temporary path of the cutted note.

	NoteSelection* firstStacked();
	NoteSelection* nextStacked();
	void append(NoteSelection *node);
	int count();

	TQValueList<Note*> parentGroups();
};

/** This store all needed information when exporting to HTML
  */
class HtmlExportData
{
  public:
	TQString iconsFolderPath;
	TQString iconsFolderName;
	TQString imagesFolderPath;
	TQString imagesFolderName;
	TQString dataFolderPath;
	TQString dataFolderName;
	bool    formatForImpression;
	bool    embedLinkedFiles;
	bool    embedLinkedFolders;
};

/** This class handle Basket and add a FilterWidget on top of it.
  * @author S�astien Laot
  */
class DecoratedBasket : public TQWidget
{
  TQ_OBJECT
  
  public:
	DecoratedBasket(TQWidget *parent, const TQString &folderName, const char *name = 0, WFlags fl = 0);
	~DecoratedBasket();
	void setFilterBarPosition(bool onTop);
	void resetFilter();
	void setFilterBarShown(bool show, bool switchFocus = true);
	bool isFilterBarShown()        { return m_filter->isShown();    }
	const FilterData& filterData() { return m_filter->filterData(); }
	FilterBar* filterBar()         { return m_filter;               }
	Basket*    basket()            { return m_basket;               }
  private:
	TQVBoxLayout *m_layout;
	FilterBar   *m_filter;
	Basket      *m_basket;
};

class TransparentWidget : public TQWidget
{
  TQ_OBJECT
  
  public:
	TransparentWidget(Basket *basket);
	void setPosition(int x, int y);
	//void reparent(TQWidget *parent, WFlags f, const TQPoint &p, bool showIt = false);
  protected:
	void paintEvent(TQPaintEvent*);
	void mouseMoveEvent(TQMouseEvent *event);
	bool eventFilter(TQObject *object, TQEvent *event);
  private:
	Basket *m_basket;
	int     m_x;
	int     m_y;
};

/**
  * @author S�astien Laot
  */
class Basket : public TQScrollView, public TQToolTip
{
/// CONSTRUCTOR AND DESTRUCTOR:
  TQ_OBJECT
  
	public:
		enum EncryptionTypes {
			NoEncryption         = 0,
			PasswordEncryption   = 1,
			PrivateKeyEncryption = 2
		};

	public:
	Basket(TQWidget *parent, const TQString &folderName);
	~Basket();

/// USER INTERACTION:
  private:
	bool   m_noActionOnMouseRelease;
	bool   m_ignoreCloseEditorOnNextMouseRelease;
	TQPoint m_pressPos;
	bool   m_canDrag;
  public:
	void viewportResizeEvent(TQResizeEvent *);
	void drawContents(TQPainter *painter, int clipX, int clipY, int clipWidth, int clipHeight);
	void enterEvent(TQEvent *);
	void leaveEvent(TQEvent *);
	void contentsMouseMoveEvent(TQMouseEvent *event);
	void contentsMousePressEvent(TQMouseEvent *event);
	void contentsMouseReleaseEvent(TQMouseEvent *event);
	void contentsMouseDoubleClickEvent(TQMouseEvent *event);
	void contentsContextMenuEvent(TQContextMenuEvent *event);
	void updateNote(Note *note);
	void clickedToInsert(TQMouseEvent *event, Note *clicked = 0, int zone = 0);
  private slots:
	void setFocusIfNotInPopupMenu();

/// LAYOUT:
  private:
	Note   *m_firstNote;
	int     m_columnsCount;
	bool    m_mindMap;
	Note   *m_resizingNote;
	int     m_pickedResizer;
	Note   *m_movingNote;
	TQPoint  m_pickedHandle;
  public:
	int tmpWidth;
	int tmpHeight;
  public:
	void unsetNotesWidth();
	void relayoutNotes(bool animate);
	Note* noteAt(int x, int y);
	inline Note* firstNote()       { return m_firstNote;                 }
	inline int   columnsCount()    { return m_columnsCount;              }
	inline bool  isColumnsLayout() { return m_columnsCount > 0;          }
	inline bool  isFreeLayout()    { return m_columnsCount <= 0;         }
	inline bool  isMindMap()       { return isFreeLayout() && m_mindMap; }
	Note* resizingNote()           { return m_resizingNote;              }
	void  deleteNotes();
	Note* lastNote();
	void setDisposition(int disposition, int columnCount);
	void equalizeColumnSizes();

/// NOTES INSERTION AND REMOVAL:
  public:
	/// The following methods assume that the note(s) to insert already all have 'this' as the parent basket:
	void prependNoteIn(   Note *note, Note *in);     /// << Add @p note (and the next linked notes) as the first note(s) of the group @p in.
	void appendNoteIn(    Note *note, Note *in);     /// << Add @p note (and the next linked notes) as the last note(s) of the group @p in.
	void appendNoteAfter( Note *note, Note *after);  /// << Add @p note (and the next linked notes) just after (just below) the note @p after.
	void appendNoteBefore(Note *note, Note *before); /// << Add @p note (and the next linked notes) just before (just above) the note @p before.
	void groupNoteAfter(  Note *note, Note *with);   /// << Add a group at @p with place, move @p with in it, and add @p note (and the next linked notes) just after the group.
	void groupNoteBefore( Note *note, Note *with);   /// << Add a group at @p with place, move @p with in it, and add @p note (and the next linked notes) just before the group.
	void unplugNote(      Note *note);               /// << Unplug @p note (and its child notes) from the basket (and also decrease counts...).
	                                                 /// <<  After that, you should delete the notes yourself. Do not call prepend/append/group... functions two times: unplug and ok
	void ungroupNote(     Note *group);              /// << Unplug @p group but put child notes at its place.
	/// And this one do almost all the above methods depending on the context:
	void insertNote(Note *note, Note *clicked, int zone, const TQPoint &pos = TQPoint(), bool animateNewPosition = false);
	void insertCreatedNote(Note *note);
	/// And working with selections:
	void unplugSelection(NoteSelection *selection);
	void insertSelection(NoteSelection *selection, Note *after);
	void selectSelection(NoteSelection *selection);
  private:
	void preparePlug(Note *note);
  private:
	Note  *m_clickedToInsert;
	int    m_zoneToInsert;
	TQPoint m_posToInsert;
	Note  *m_savedClickedToInsert;
	int    m_savedZoneToInsert;
	TQPoint m_savedPosToInsert;
	bool   m_isInsertPopupMenu;
  public:
	void saveInsertionData();
	void restoreInsertionData();
	void resetInsertionData();
  public slots:
	void insertEmptyNote(int type);
	void insertWizard(int type);
	void insertColor(const TQColor &color);
	void insertImage(const TQPixmap &image);
	void pasteNote(TQClipboard::Mode mode = TQClipboard::Clipboard);
	void delayedCancelInsertPopupMenu();
	void setInsertPopupMenu()    { m_isInsertPopupMenu = true;  }
	void cancelInsertPopupMenu() { m_isInsertPopupMenu = false; }
  private slots:
	void hideInsertPopupMenu();
	void timeoutHideInsertPopupMenu();

/// TOOL TIPS:
  protected:
	void maybeTip(const TQPoint &pos);

/// ANIMATIONS:
  private:
	TQValueList<Note*> m_animatedNotes;
	TQTimer            m_animationTimer;
	int               m_deltaY;
	TQTime             m_lastFrameTime;
	static const int FRAME_DELAY;
  private slots:
	void animateObjects();
  public slots:
	void animateLoad();
  public:
	void addAnimatedNote(Note *note);

/// LOAD AND SAVE:
  private:
	bool m_loaded;
	bool m_loadingLaunched;
	bool m_locked;
	bool m_shouldConvertPlainTextNotes;
	TQFrame* m_decryptBox;
	TQPushButton* m_button;
	int m_encryptionType;
	TQString m_encryptionKey;
#ifdef HAVE_LIBGPGME
	KGpgMe* m_gpg;
#endif
        TQTimer      m_inactivityAutoLockTimer;
        void enableActions();

  private slots:
	void loadNotes(const TQDomElement &notes, Note *parent);
	void saveNotes(TQDomDocument &document, TQDomElement &element, Note *parent);
	void unlock();
protected slots:
        void inactivityAutoLockTimeout();
public slots:
	void load();
	void loadProperties(const TQDomElement &properties);
	void saveProperties(TQDomDocument &document, TQDomElement &properties);
	bool save();
  public:
	bool isEncrypted();
	bool isFileEncrypted();
	bool isLocked()        { return m_locked;          };
	void lock();
	bool isLoaded()        { return m_loaded;          };
	bool loadingLaunched() { return m_loadingLaunched; };
	bool loadFromFile(const TQString &fullPath, TQString* string, bool isLocalEncoding = false);
	bool loadFromFile(const TQString &fullPath, TQByteArray* array);
	bool saveToFile(const TQString& fullPath, const TQByteArray& array);
	bool saveToFile(const TQString& fullPath, const TQByteArray& array, TQ_ULONG length);
	bool saveToFile(const TQString& fullPath, const TQString& string, bool isLocalEncoding = false);
	static bool safelySaveToFile(const TQString& fullPath, const TQByteArray& array);
	static bool safelySaveToFile(const TQString& fullPath, const TQByteArray& array, TQ_ULONG length);
	static bool safelySaveToFile(const TQString& fullPath, const TQString& string, bool isLocalEncoding = false);
	bool setProtection(int type, TQString key);
	int  encryptionType()  { return m_encryptionType;  };
	TQString encryptionKey(){ return m_encryptionKey;   };
	bool saveAgain();

/// BACKGROUND:
  private:
	TQColor   m_backgroundColorSetting;
	TQString  m_backgroundImageName;
	TQPixmap *m_backgroundPixmap;
	TQPixmap *m_opaqueBackgroundPixmap;
	TQPixmap *m_selectedBackgroundPixmap;
	bool     m_backgroundTiled;
	TQColor   m_textColorSetting;
  public:
	inline bool           hasBackgroundImage()     { return m_backgroundPixmap != 0;  }
	inline const TQPixmap* backgroundPixmap()       { return m_backgroundPixmap;       }
	inline bool           isTiledBackground()      { return m_backgroundTiled;        }
	inline TQString        backgroundImageName()    { return m_backgroundImageName;    }
	inline TQColor         backgroundColorSetting() { return m_backgroundColorSetting; }
	inline TQColor         textColorSetting()       { return m_textColorSetting;       }
	TQColor backgroundColor();
	TQColor textColor();
	void setAppearance(const TQString &icon, const TQString &name, const TQString &backgroundImage, const TQColor &backgroundColor, const TQColor &textColor);
	void blendBackground(TQPainter &painter, const TQRect &rect, int xPainter = -1, int yPainter = -1, bool opaque = false, TQPixmap *bg = 0);
	void unbufferizeAll();
	void subscribeBackgroundImages();
	void unsubscribeBackgroundImages();

/// KEYBOARD SHORTCUT:
  public: // Temporar: for deletion purpose
	TDEAction *m_action;
  private:
	int      m_shortcutAction;
  private slots:
	void activatedShortcut();
  public:
	TDEShortcut shortcut() { return m_action->shortcut(); }
	int shortcutAction() { return m_shortcutAction;     }
	void setShortcut(TDEShortcut shortcut, int action);

/// USER INTERACTION:
  private:
	Note  *m_hoveredNote;
	int    m_hoveredZone;
	bool   m_lockedHovering;
	bool   m_underMouse;
	TQRect  m_inserterRect;
	bool   m_inserterShown;
	bool   m_inserterSplit;
	bool   m_inserterTop;
	bool   m_inserterGroup;
	void placeInserter(Note *note, int zone);
	void removeInserter();
  public:
//	bool inserterShown() { return m_inserterShown; }
	bool inserterSplit() { return m_inserterSplit; }
	bool inserterGroup() { return m_inserterGroup; }
  public slots:
	void doHoverEffects(Note *note, Note::Zone zone, const TQPoint &pos = TQPoint(0, 0)); /// << @p pos is optionnal and only used to show the link target in the statusbar
	void doHoverEffects(const TQPoint &pos);
	void doHoverEffects(); // The same, but using the current cursor position
	void mouseEnteredEditorWidget();
  public:
	void popupTagsMenu(Note *note);
	void popupEmblemMenu(Note *note, int emblemNumber);
	void addTagToSelectedNotes(Tag *tag);
	void removeTagFromSelectedNotes(Tag *tag);
	void removeAllTagsFromSelectedNotes();
	void addStateToSelectedNotes(State *state);
	void changeStateOfSelectedNotes(State *state);
	bool selectedNotesHaveTags();
	const TQRect& inserterRect()  { return m_inserterRect;  }
	bool         inserterShown() { return m_inserterShown; }
	void drawInserter(TQPainter &painter, int xPainter, int yPainter);
	DecoratedBasket* decoration();
	State *stateForTagFromSelectedNotes(Tag *tag);
  public slots:
	void activatedTagShortcut(Tag *tag);
	void recomputeAllStyles();
	void removedStates(const TQValueList<State*> &deletedStates);
  private slots:
	void toggledTagInMenu(int id);
	void toggledStateInMenu(int id);
	void unlockHovering();
	void disableNextClick();
	void contentsMoved();
  public:
	Note  *m_tagPopupNote;
  private:
	Tag   *m_tagPopup;
	TQTime  m_lastDisableClick;

/// SELECTION:
  private:
	bool   m_isSelecting;
	bool   m_selectionStarted;
	bool   m_selectionInvert;
	TQPoint m_selectionBeginPoint;
	TQPoint m_selectionEndPoint;
	TQRect  m_selectionRect;
	TQTimer m_autoScrollSelectionTimer;
	void stopAutoScrollSelection();
  private slots:
	void doAutoScrollSelection();
  public:
	inline bool isSelecting() { return m_isSelecting; }
	inline const TQRect& selectionRect() { return m_selectionRect; }
	void selectNotesIn(const TQRect &rect, bool invertSelection, bool unselectOthers = true);
	void resetWasInLastSelectionRect();
	void selectAll();
	void unselectAll();
	void invertSelection();
	void unselectAllBut(Note *toSelect);
	void invertSelectionOf(Note *toSelect);
	TQColor selectionRectInsideColor();
	Note* theSelectedNote();
	NoteSelection* selectedNotes();

/// BLANK SPACES DRAWING:
  private:
	TQValueList<TQRect> m_blankAreas;
	void recomputeBlankRects();
	TQWidget *m_cornerWidget;

/// COMMUNICATION WITH ITS CONTAINER:
  signals:
	void postMessage(const TQString &message);      /// << Post a temporar message in the statusBar.
	void setStatusBarText(const TQString &message); /// << Set the permanent statusBar text or reset it if message isEmpty().
	void resetStatusBarText();                     /// << Equivalent to setStatusBarText("").
	void propertiesChanged(Basket *basket);
	void countsChanged(Basket *basket);
  public slots:
	void linkLookChanged();
	void signalCountsChanged();
  private:
	TQTimer m_timerCountsChanged;
  private slots:
	void countsChangedTimeOut();

/// NOTES COUNTING:
  public:
	void addSelectedNote()    { ++m_countSelecteds;   signalCountsChanged(); }
	void removeSelectedNote() { --m_countSelecteds;   signalCountsChanged(); }
	void resetSelectedNote()  { m_countSelecteds = 0; signalCountsChanged(); } // FIXME: Useful ???
	int count()               { return m_count;          }
	int countFounds()         { return m_countFounds;    }
	int countSelecteds()      { return m_countSelecteds; }
  private:
	int m_count;
	int m_countFounds;
	int m_countSelecteds;

/// PROPERTIES:
  public:
	TQString basketName() { return m_basketName; }
	TQString icon()       { return m_icon;       }
	TQString folderName() { return m_folderName; }
	TQString fullPath();
	TQString fullPathForFileName(const TQString &fileName); // Full path of an [existing or not] note in this basket
	static TQString fullPathForFolderName(const TQString &folderName);
  private:
	TQString m_basketName;
	TQString m_icon;
	TQString m_folderName;

/// ACTIONS ON SELECTED NOTES FROM THE INTERFACE:
  public slots:
	void noteEdit(Note *note = 0L, bool justAdded = false, const TQPoint &clickedPoint = TQPoint());
	void showEditedNoteWhileFiltering();
	void noteDelete();
	void noteDeleteWithoutConfirmation(bool deleteFilesToo = true);
	void noteCopy();
	void noteCut();
	void noteOpen(Note *note = 0L);
	void noteOpenWith(Note *note = 0L);
	void noteSaveAs();
	void noteGroup();
	void noteUngroup();
	void noteMoveOnTop();
	void noteMoveOnBottom();
	void noteMoveNoteUp();
	void noteMoveNoteDown();
	void moveSelectionTo(Note *here, bool below);
  public:
	enum CopyMode { CopyToClipboard, CopyToSelection, CutToClipboard };
	void doCopy(CopyMode copyMode);
	bool selectionIsOneGroup();
	Note* selectedGroup();
	Note* firstSelected();
	Note* lastSelected();

/// NOTES EDITION:
  private:
	NoteEditor *m_editor;
	//TQWidget    *m_rightEditorBorder;
	TransparentWidget *m_leftEditorBorder;
	TransparentWidget *m_rightEditorBorder;
	bool        m_redirectEditActions;
	int         m_editorWidth;
	int         m_editorHeight;
	TQTimer      m_inactivityAutoSaveTimer;
	bool        m_doNotCloseEditor;
	int         m_editParagraph;
	int         m_editIndex;
  public:
	bool isDuringEdit()        { return m_editor;              }
	bool redirectEditActions() { return m_redirectEditActions; }
	bool hasTextInEditor();
	bool hasSelectedTextInEditor();
	bool selectedAllTextInEditor();
	Note* editedNote();
  protected slots:
	void selectionChangedInEditor();
	void contentChangedInEditor();
	void inactivityAutoSaveTimeout();
  public slots:
	void editorCursorPositionChanged();
  private:
	int m_editorX;
	int m_editorY;
  public slots:
	void placeEditor(bool andEnsureVisible = false);
	void placeEditorAndEnsureVisible();
	bool closeEditor();
	void closeEditorDelayed();
	void updateEditorAppearance();
	void editorPropertiesChanged();
	void openBasket();
	void closeBasket();

/// FILTERING:
  public slots:
	void newFilter(const FilterData &data, bool andEnsureVisible = true);
	void cancelFilter();
	void validateFilter();
	void filterAgain(bool andEnsureVisible = true);
	void filterAgainDelayed();
	bool isFiltering();

/// DRAG AND DROP:
  private:
	bool m_isDuringDrag;
	TQValueList<Note*> m_draggedNotes;
  public:
	static void acceptDropEvent(TQDropEvent *event, bool preCond = true);
	void contentsDropEvent(TQDropEvent *event);
	void blindDrop(TQDropEvent* event);
	bool isDuringDrag() { return m_isDuringDrag; }
	TQValueList<Note*> draggedNotes() { return m_draggedNotes; }
  protected:
	void contentsDragEnterEvent(TQDragEnterEvent*);
	void contentsDragMoveEvent(TQDragMoveEvent *event);
	void contentsDragLeaveEvent(TQDragLeaveEvent*);
  public slots:
	void slotCopyingDone2(TDEIO::Job *job);
  public:
	Note* noteForFullPath(const TQString &path);

/// EXPORTATION:
  public:
	TQValueList<State*> usedStates();
	static TQString saveGradientBackground(const TQColor &color, const TQFont &font, const TQString &folder);

  public:
	void listUsedTags(TQValueList<Tag*> &list);

/// MANAGE FOCUS:
  private:
	Note *m_focusedNote;
  public:
	void setFocusedNote(Note *note);
	void focusANote();
	void focusANonSelectedNoteAbove(bool inSameColumn);
	void focusANonSelectedNoteBelow(bool inSameColumn);
	void focusANonSelectedNoteBelowOrThenAbove();
	void focusANonSelectedNoteAboveOrThenBelow();
	Note* focusedNote() { return m_focusedNote; }
	Note* firstNoteInStack();
	Note* lastNoteInStack();
	Note* firstNoteShownInStack();
	Note* lastNoteShownInStack();
	void selectRange(Note *start, Note *end, bool unselectOthers = true); /// FIXME: Not really a focus related method!
	void ensureNoteVisible(Note *note);
	virtual void keyPressEvent(TQKeyEvent *event);
	virtual void focusInEvent(TQFocusEvent*);
	virtual void focusOutEvent(TQFocusEvent*);
	TQRect noteVisibleRect(Note *note); // clipped global (desktop as origin) rectangle
	Note* firstNoteInGroup();
	Note *noteOnHome();
	Note *noteOnEnd();

	enum NoteOn { LEFT_SIDE = 1, RIGHT_SIDE, TOP_SIDE, BOTTOM_SIDE };
	Note* noteOn(NoteOn side);

/// REIMPLEMENTED:
  public:
	void deleteFiles();
	bool convertTexts();


  public:
	void wheelEvent(TQWheelEvent *event);



  public:
	Note *m_startOfShiftSelectionNote;


/// THE NEW FILE WATCHER:
  private:
	KDirWatch           *m_watcher;
	TQTimer               m_watcherTimer;
	TQValueList<TQString>  m_modifiedFiles;
  public:
	void addWatchedFile(const TQString &fullPath);
	void removeWatchedFile(const TQString &fullPath);
  private slots:
	void watchedFileModified(const TQString &fullPath);
	void watchedFileDeleted(const TQString &fullPath);
	void updateModifiedNotes();


/// FROM OLD ARCHITECTURE **********************

public slots:

	void showFrameInsertTo() {}
	void resetInsertTo() {}

	void  computeInsertPlace(const TQPoint &/*cursorPosition*/)    { }
  public:

	friend class SystemTray;

/// SPEED OPTIMIZATION
  private:
	bool m_finishLoadOnFirstShow;
	bool m_relayoutOnNextShow;
  public:
	void aboutToBeActivated();
};









#if 0

#include <tqwidget.h>
#include <tqscrollview.h>
#include <tqclipboard.h>
#include <tqptrlist.h>
#include <tqtimer.h>
#include <tdeio/job.h>
#include <tqcolor.h>

#include "filter.h"

class TQFrame;
class TQVBoxLayout;
class TQCheckBox;
class TQString;
class TQColor;
class TQPixmap;
class TQAction;
class TQStringList;
class TQRect;

class TQDomElement;

class KDirWatch;

class Basket;
class Note;
class NoteEditorBase;

/** Used to enqueue a file path when the Basket receive a file modification / creation / deletion
  * It associate the file name with an event.
  * All this queue will be treated later.
  * TODO: rename to class WatcherEvent ?
  * @author S�astien Laot
  */
class FileEvent
{
  public:
	enum Event { Modified = 1, Created, Deleted, Renamed };
	FileEvent(Event evt, const TQString &path)
	 : event(evt), filePath(path)
	{ }
  public: // Because it must be fast and theire is no need to be private
	Event   event;
	TQString filePath;
};

/** Basket that contain some Notes.
  * @author S�astien Laot
  */
clas   s Bas    ket : public TQScrollView
{
  TQ_OBJECT
  
  public:
	/** Construtor and destructor */
	Bask      et(TQWidget *parent, const TQString &folderName, const char *name = "", WFlags fl = 0);
  public:
  protected:
	virtual void contentsContextMenuEvent(TQContextMenuEvent *event);
	virtual void contentsMousePressEvent(TQMouseEvent *event); // For redirected event !!
	virtual void showEvent(TQShowEvent *);
	/** Drag and drop functions */
	virtual void dragEnterEvent(TQDragEnterEvent*);
	virtual void dragMoveEvent(TQDragMoveEvent* event);
	virtual void dragLeaveEvent(TQDragLeaveEvent*);
  public:
	virtual void dropEvent(TQDropEvent *event);
	static  void acceptDropEvent(TQDropEvent *event, bool preCond = true);
	bool  canDragNote()       { return !isEmpty(); }
	void  computeInsertPlace(const TQPoint &cursorPosition);
	Note* noteAtPosition(const TQPoint &pos);
	Note* duplicatedOf(Note *note);
	void  checkClipboard();
	void  processActionAsYouType(TQKeyEvent *event);
	void  exportToHTML();
  signals:
	void nameChanged(Basket *basket, const TQString &name);
	void iconChanged(Basket *basket, const TQString &icon);
	void notesNumberChanged(Basket *basket);
  public slots:
	void linkLookChanged();
	void showNotesToolTipChanged();
	/** Notes manipulation */
	void insertNote(Note *note);
	void delNote(Note *note, bool askForMirroredFile = true);
	void changeNotePlace(Note *note);
	void pasteNote(TQClipboard::Mode mode = TQClipboard::Clipboard);
	void recolorizeNotes();
	void reloadMirroredFolder();
	void showMirrorOnlyOnceInfo();
	/** Selection of note(s) */
	void selectAll();
	void unselectAll();
	void unselectAllBut(Note *toSelect);
	void invertSelection();
	void selectRange(Note *start, Note *end);
	void clicked(Note *note, bool controlPressed, bool shiftPressed);
	void setFocusedNote(Note *note);
	void focusANote();
	void ensureVisibleNote(Note *note);
	TQRect noteRect(Note *note); // clipped global (desktop as origin) rectangle
	/** Travel the list to find the next shown note, or the previous if step == -1, or the next after 10 if step == 10... */
	Note* nextShownNoteFrom(Note *note, int step);
	/** Actions on (selected) notes */
	void editNote(Note *note, bool editAnnotations = false);
	void editNote();
	void delNote();
	void copyNote();
	void cutNote();
	void openNote();
	void openNoteWith();
	void saveNoteAs();
	void moveOnTop();
	void moveOnBottom();
	void moveNoteUp();
	void moveNoteDown();
  public:
	void    dontCareOfCreation(const TQString &path);
	TQString copyIcon(const TQString &iconName, int size, const TQString &destFolder);
	TQString copyFile(const TQString &srcPath, const TQString &destFolder, bool createIt = false);
  protected slots:
	void slotModifiedFile(const TQString &path);
	void slotCreatedFile(const TQString &path);
	void slotDeletedFile(const TQString &path);
	void slotUpdateNotes();
	void placeEditor();
	void closeEditor(bool save = true);
	void clipboardChanged(bool selectionMode = false);
	void selectionChanged();
  private:
	TQTimer              m_updateTimer;
	TQPtrList<FileEvent> m_updateQueue;
	TQStringList         m_dontCare;
	static const int    c_updateTime;
  private:
	void load(); // Load is performed only once, during contructor
	void loadNotes(const TQDomElement &notes);
	bool importLauncher(const TQString &type, const TQDomElement &content, const TQString &runCommand,
	                    const TQString &annotations/*, bool checked*/);
	void computeShownNotes();

  private:
	KDirWatch      *m_watcher;
	NoteEditorBase *m_editor;
	TQKeyEvent      *m_stackedKeyEvent;
};

#endif // #if 0

#endif // BASKET_H
