/* KPilot
**
** Copyright (C) 2001 by Dan Pilone
** Copyright (C) 2004 by Reinhold Kainhofer
**
** This file defines the log file class, which logs
** sync-messages during a HotSync to a file.
*/

/*
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program in a file called COPYING; if not, write to
** the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
** MA 02110-1301, USA.
*/

/*
** Bug reports and questions can be sent to kde-pim@kde.org.
*/

#include "options.h"

#include <tqfile.h>
#include <tqtextstream.h>
#include <tqdatetime.h>

#include <pi-version.h>

#ifndef PILOT_LINK_PATCH
#define PILOT_LINK_PATCH "unknown"
#endif
#include "logFile.h"
#include "kpilotConfig.h"

#include "logFile.moc"


LogFile::LogFile() : DCOPObject("LogIface"), TQObject(), fOutfile(0L), fSyncing(false)
{
	FUNCTIONSETUP;
}


/* DCOP */ ASYNC LogFile::logStartSync()
{
	FUNCTIONSETUP;
	// If a sync is already running (something went wrong then!), close that old log
	if (fSyncing) logEndSync();
	
	fOutfile = new TQFile(KPilotSettings::logFileName());
	
	if (!fOutfile || !fOutfile->open(IO_WriteOnly)) 
	{
		WARNINGKPILOT << "Unable to open log file " << KPilotSettings::logFileName() << endl;
		KPILOT_DELETE( fOutfile );
		fSyncing = false;
		return;
	}
	
	fSyncing = true;
	fLogStream.setDevice(fOutfile);

	fLogStream<<(CSL1("KPilot HotSync log, %1").arg(TQDateTime::currentDateTime().toString()))<<endl<<endl<<endl;
	fLogStream<<(CSL1("Version: KPilot %1").arg(TQString::fromLatin1(KPILOT_VERSION)))<<endl;
	fLogStream<<(CSL1("Version: pilot-link %1.%2.%3%4" )
		.arg(PILOT_LINK_VERSION).arg(PILOT_LINK_MAJOR).arg(PILOT_LINK_MINOR)
#ifdef PILOT_LINK_PATCH
		.arg(TQString::fromLatin1(PILOT_LINK_PATCH))
#else
		.arg(TQString())
#endif
		)<<endl;
#ifdef TDE_VERSION_STRING
	fLogStream<<(CSL1("Version: KDE %1" ).arg(TQString::fromLatin1(TDE_VERSION_STRING)) )<<endl;
#endif
#ifdef TQT_VERSION_STR
	fLogStream<<(CSL1("Version: TQt %1" ).arg(TQString::fromLatin1(TQT_VERSION_STR)) )<<endl;
#endif
	fLogStream<<endl<<endl;
		
}

/* DCOP */ ASYNC LogFile::logEndSync()
{
	if (fSyncing) 
	{
		logMessage(i18n("HotSync finished."));
		fLogStream.unsetDevice();
		if (fOutfile) fOutfile->close();
		KPILOT_DELETE(fOutfile)
		fSyncing=false;
	}
}

/* DCOP */ ASYNC LogFile::logMessage(TQString s)
{
	addMessage(s);
}

/* DCOP */ ASYNC LogFile::logError(TQString s)
{
	addMessage(s);
}

/* DCOP */ ASYNC LogFile::logProgress(TQString, int)
{
}


void LogFile::addMessage(const TQString & s)
{
	FUNCTIONSETUP;
	if ( fSyncing && !s.isEmpty() ) 
	{
		fLogStream<<TQTime::currentTime().toString()<<"  "<<s<<endl;
	}
}

