// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "ui/ozone/platform/wayland/gpu/wayland_surface_factory.h"

#include <memory>

#include "base/memory/ptr_util.h"
#include "base/memory/raw_ptr.h"
#include "ui/gfx/linux/client_native_pixmap_dmabuf.h"
#include "ui/gl/gl_bindings.h"
#include "ui/gl/gl_surface_egl.h"
#include "ui/ozone/common/egl_util.h"
#include "ui/ozone/common/gl_ozone_egl.h"
#include "ui/ozone/common/native_pixmap_egl_binding.h"
#include "ui/ozone/platform/wayland/common/wayland_object.h"
#include "ui/ozone/platform/wayland/gpu/gl_surface_egl_readback_wayland.h"
#include "ui/ozone/platform/wayland/gpu/gl_surface_wayland.h"
#include "ui/ozone/platform/wayland/gpu/wayland_buffer_manager_gpu.h"
#include "ui/ozone/platform/wayland/gpu/wayland_canvas_surface.h"
#include "ui/ozone/platform/wayland/host/wayland_connection.h"
#include "ui/ozone/platform/wayland/host/wayland_window.h"
#include "ui/ozone/platform/wayland/host/wayland_window_manager.h"

#if defined(WAYLAND_GBM)
#include "ui/ozone/platform/wayland/gpu/gbm_pixmap_wayland.h"
#include "ui/ozone/platform/wayland/gpu/gbm_surfaceless_wayland.h"
#include "ui/ozone/public/ozone_platform.h"
#endif

#if BUILDFLAG(ENABLE_VULKAN)
#include "ui/ozone/platform/wayland/gpu/vulkan_implementation_wayland.h"
#endif

namespace ui {

namespace {

class GLOzoneEGLWayland : public GLOzoneEGL {
 public:
  GLOzoneEGLWayland(WaylandConnection* connection,
                    WaylandBufferManagerGpu* buffer_manager)
      : connection_(connection), buffer_manager_(buffer_manager) {}

  GLOzoneEGLWayland(const GLOzoneEGLWayland&) = delete;
  GLOzoneEGLWayland& operator=(const GLOzoneEGLWayland&) = delete;

  ~GLOzoneEGLWayland() override {}

  bool CanImportNativePixmap() override;

  std::unique_ptr<NativePixmapGLBinding> ImportNativePixmap(
      scoped_refptr<gfx::NativePixmap> pixmap,
      gfx::BufferFormat plane_format,
      gfx::BufferPlane plane,
      gfx::Size plane_size,
      const gfx::ColorSpace& color_space,
      GLenum target,
      GLuint texture_id) override;

  scoped_refptr<gl::GLSurface> CreateViewGLSurface(
      gl::GLDisplay* display,
      gfx::AcceleratedWidget widget) override;

  scoped_refptr<gl::Presenter> CreateSurfacelessViewGLSurface(
      gl::GLDisplay* display,
      gfx::AcceleratedWidget window) override;

  scoped_refptr<gl::GLSurface> CreateOffscreenGLSurface(
      gl::GLDisplay* display,
      const gfx::Size& size) override;

 protected:
  gl::EGLDisplayPlatform GetNativeDisplay() override;
  bool LoadGLES2Bindings(const gl::GLImplementationParts& impl) override;

 private:
  const raw_ptr<WaylandConnection> connection_;
  const raw_ptr<WaylandBufferManagerGpu> buffer_manager_;
};

bool GLOzoneEGLWayland::CanImportNativePixmap() {
  return gl::GLSurfaceEGL::GetGLDisplayEGL()
      ->ext->b_EGL_EXT_image_dma_buf_import;
}

std::unique_ptr<NativePixmapGLBinding> GLOzoneEGLWayland::ImportNativePixmap(
    scoped_refptr<gfx::NativePixmap> pixmap,
    gfx::BufferFormat plane_format,
    gfx::BufferPlane plane,
    gfx::Size plane_size,
    const gfx::ColorSpace& color_space,
    GLenum target,
    GLuint texture_id) {
  return NativePixmapEGLBinding::Create(pixmap, plane_format, plane, plane_size,
                                        color_space, target, texture_id);
}

scoped_refptr<gl::GLSurface> GLOzoneEGLWayland::CreateViewGLSurface(
    gl::GLDisplay* display,
    gfx::AcceleratedWidget widget) {
  // If we run with software GL implementation, use GLSurface which will read
  // pixels back and present via shared memory.
  if (gl::IsSoftwareGLImplementation(gl::GetGLImplementationParts())) {
    return gl::InitializeGLSurface(
        base::MakeRefCounted<GLSurfaceEglReadbackWayland>(
            display->GetAs<gl::GLDisplayEGL>(), widget, buffer_manager_));
  }

  // Only EGLGLES2 is supported with surfaceless view gl.
  if ((gl::GetGLImplementation() != gl::kGLImplementationEGLGLES2) ||
      !connection_)
    return nullptr;

  WaylandWindow* window = connection_->window_manager()->GetWindow(widget);
  if (!window)
    return nullptr;

  // The wl_egl_window needs to be created before the GLSurface so it can be
  // used in the GLSurface constructor.
  auto egl_window = CreateWaylandEglWindow(window);
  if (!egl_window)
    return nullptr;
  return gl::InitializeGLSurface(new GLSurfaceWayland(
      display->GetAs<gl::GLDisplayEGL>(), std::move(egl_window), window));
}

scoped_refptr<gl::Presenter> GLOzoneEGLWayland::CreateSurfacelessViewGLSurface(
    gl::GLDisplay* display,
    gfx::AcceleratedWidget window) {
  if (gl::IsSoftwareGLImplementation(gl::GetGLImplementationParts())) {
    return nullptr;
  } else {
#if defined(WAYLAND_GBM)
  // If there is a gbm device available, use surfaceless gl surface.
  if (!buffer_manager_->GetGbmDevice())
    return nullptr;
  scoped_refptr<gl::Presenter> presenter =
      base::MakeRefCounted<GbmSurfacelessWayland>(
          display->GetAs<gl::GLDisplayEGL>(), buffer_manager_, window);
  if (!presenter->Initialize(gl::GLSurfaceFormat()))
    return nullptr;
  return presenter;
#else
  return nullptr;
#endif
  }
}

scoped_refptr<gl::GLSurface> GLOzoneEGLWayland::CreateOffscreenGLSurface(
    gl::GLDisplay* display,
    const gfx::Size& size) {
  if (display->GetAs<gl::GLDisplayEGL>()->IsEGLSurfacelessContextSupported() &&
      size.width() == 0 && size.height() == 0) {
    return gl::InitializeGLSurface(
        new gl::SurfacelessEGL(display->GetAs<gl::GLDisplayEGL>(), size));
  } else {
    return gl::InitializeGLSurface(
        new gl::PbufferGLSurfaceEGL(display->GetAs<gl::GLDisplayEGL>(), size));
  }
}

gl::EGLDisplayPlatform GLOzoneEGLWayland::GetNativeDisplay() {
  if (connection_)
    return gl::EGLDisplayPlatform(
        reinterpret_cast<EGLNativeDisplayType>(connection_->display()));
  return gl::EGLDisplayPlatform(EGL_DEFAULT_DISPLAY);
}

bool GLOzoneEGLWayland::LoadGLES2Bindings(
    const gl::GLImplementationParts& impl) {
  // TODO: It may not be necessary to set this environment variable when using
  // swiftshader.
  setenv("EGL_PLATFORM", "wayland", 0);
  return LoadDefaultEGLGLES2Bindings(impl);
}

}  // namespace

WaylandSurfaceFactory::WaylandSurfaceFactory(
    WaylandConnection* connection,
    WaylandBufferManagerGpu* buffer_manager)
    : connection_(connection), buffer_manager_(buffer_manager) {
  egl_implementation_ =
      std::make_unique<GLOzoneEGLWayland>(connection_, buffer_manager_);
}

WaylandSurfaceFactory::~WaylandSurfaceFactory() = default;

std::unique_ptr<SurfaceOzoneCanvas>
WaylandSurfaceFactory::CreateCanvasForWidget(gfx::AcceleratedWidget widget) {
  return std::make_unique<WaylandCanvasSurface>(buffer_manager_, widget);
}

std::vector<gl::GLImplementationParts>
WaylandSurfaceFactory::GetAllowedGLImplementations() {
  std::vector<gl::GLImplementationParts> impls;
  if (egl_implementation_) {
    // Add only supported ANGLE implementations. Otherwise, angle-vulkan might
    // be requested, which is not supported with this backend yet.
    impls.emplace_back(gl::ANGLEImplementation::kOpenGL);
    impls.emplace_back(gl::ANGLEImplementation::kOpenGLES);
    impls.emplace_back(gl::ANGLEImplementation::kSwiftShader);
    impls.emplace_back(gl::kGLImplementationEGLGLES2);
  }
  return impls;
}

GLOzone* WaylandSurfaceFactory::GetGLOzone(
    const gl::GLImplementationParts& implementation) {
  switch (implementation.gl) {
    case gl::kGLImplementationEGLGLES2:
    case gl::kGLImplementationEGLANGLE:
      return egl_implementation_.get();
    default:
      return nullptr;
  }
}

#if BUILDFLAG(ENABLE_VULKAN)
std::unique_ptr<gpu::VulkanImplementation>
WaylandSurfaceFactory::CreateVulkanImplementation(bool use_swiftshader,
                                                  bool allow_protected_memory) {
  return std::make_unique<VulkanImplementationWayland>(use_swiftshader);
}
#endif

scoped_refptr<gfx::NativePixmap> WaylandSurfaceFactory::CreateNativePixmap(
    gfx::AcceleratedWidget widget,
    gpu::VulkanDeviceQueue* device_queue,
    gfx::Size size,
    gfx::BufferFormat format,
    gfx::BufferUsage usage,
    absl::optional<gfx::Size> framebuffer_size) {
  if (framebuffer_size &&
      !gfx::Rect(size).Contains(gfx::Rect(*framebuffer_size))) {
    return nullptr;
  }
#if defined(WAYLAND_GBM)
  scoped_refptr<GbmPixmapWayland> pixmap =
      base::MakeRefCounted<GbmPixmapWayland>(buffer_manager_);

  if (!pixmap->InitializeBuffer(widget, size, format, usage, framebuffer_size))
    return nullptr;
  return pixmap;
#else
  return nullptr;
#endif
}

void WaylandSurfaceFactory::CreateNativePixmapAsync(
    gfx::AcceleratedWidget widget,
    gpu::VulkanDeviceQueue* device_queue,
    gfx::Size size,
    gfx::BufferFormat format,
    gfx::BufferUsage usage,
    NativePixmapCallback callback) {
  // CreateNativePixmap is non-blocking operation. Thus, it is safe to call it
  // and return the result with the provided callback.
  std::move(callback).Run(
      CreateNativePixmap(widget, device_queue, size, format, usage));
}

scoped_refptr<gfx::NativePixmap>
WaylandSurfaceFactory::CreateNativePixmapFromHandle(
    gfx::AcceleratedWidget widget,
    gfx::Size size,
    gfx::BufferFormat format,
    gfx::NativePixmapHandle handle) {
#if defined(WAYLAND_GBM)
  scoped_refptr<GbmPixmapWayland> pixmap =
      base::MakeRefCounted<GbmPixmapWayland>(buffer_manager_);

  if (!pixmap->InitializeBufferFromHandle(widget, size, format,
                                          std::move(handle)))
    return nullptr;
  return pixmap;
#else
  return nullptr;
#endif
}

bool WaylandSurfaceFactory::SupportsNativePixmaps() const {
  bool supports_native_pixmaps = false;
#if defined(WAYLAND_GBM)
  supports_native_pixmaps = buffer_manager_->GetGbmDevice() != nullptr;
#endif
  // Native pixmaps are not supported with swiftshader.
  if (gl::IsSoftwareGLImplementation(gl::GetGLImplementationParts())) {
    supports_native_pixmaps = false;
  }
  return supports_native_pixmaps;
}

absl::optional<gfx::BufferFormat>
WaylandSurfaceFactory::GetPreferredFormatForSolidColor() const {
  if (!buffer_manager_->SupportsFormat(gfx::BufferFormat::RGBA_8888))
    return gfx::BufferFormat::BGRA_8888;
  return gfx::BufferFormat::RGBA_8888;
}

}  // namespace ui
