// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

cr.define('cr.icon', function() {
  /**
   * @return {!Array<number>} The scale factors supported by this platform for
   *     webui resources.
   */
  function getSupportedScaleFactors() {
    var supportedScaleFactors = [];
    if (cr.isMac || cr.isChromeOS || cr.isWindows || cr.isLinux) {
      // All desktop platforms support zooming which also updates the
      // renderer's device scale factors (a.k.a devicePixelRatio), and
      // these platforms has high DPI assets for 2.0x. Use 1x and 2x in
      // image-set on these platforms so that the renderer can pick the
      // closest image for the current device scale factor.
      supportedScaleFactors.push(1);
      supportedScaleFactors.push(2);
    } else {
      // For other platforms that use fixed device scale factor, use
      // the window's device pixel ratio.
      // TODO(oshima): Investigate if Android/iOS need to use image-set.
      supportedScaleFactors.push(window.devicePixelRatio);
    }
    return supportedScaleFactors;
  }

  /**
   * Returns the URL of the image, or an image set of URLs for the profile
   * avatar. Default avatars have resources available for multiple scalefactors,
   * whereas the GAIA profile image only comes in one size.
   *
   * @param {string} path The path of the image.
   * @return {string} The url, or an image set of URLs of the avatar image.
   */
  function getProfileAvatarIcon(path) {
    var chromeThemePath = 'chrome://theme';
    var isDefaultAvatar =
        (path.slice(0, chromeThemePath.length) == chromeThemePath);
    return isDefaultAvatar ? imageset(path + '@scalefactorx'): url(path);
  }

  /**
   * Generates a CSS -webkit-image-set for a chrome:// url.
   * An entry in the image set is added for each of getSupportedScaleFactors().
   * The scale-factor-specific url is generated by replacing the first instance
   * of 'scalefactor' in |path| with the numeric scale factor.
   * @param {string} path The URL to generate an image set for.
   *     'scalefactor' should be a substring of |path|.
   * @return {string} The CSS -webkit-image-set.
   */
  function imageset(path) {
    var supportedScaleFactors = getSupportedScaleFactors();

    var replaceStartIndex = path.indexOf('scalefactor');
    if (replaceStartIndex < 0)
      return url(path);

    var s = '';
    for (var i = 0; i < supportedScaleFactors.length; ++i) {
      var scaleFactor = supportedScaleFactors[i];
      var pathWithScaleFactor = path.substr(0, replaceStartIndex) +
          scaleFactor + path.substr(replaceStartIndex + 'scalefactor'.length);

      s += url(pathWithScaleFactor) + ' ' + scaleFactor + 'x';

      if (i != supportedScaleFactors.length - 1)
        s += ', ';
    }
    return '-webkit-image-set(' + s + ')';
  }

  /**
   * A regular expression for identifying favicon URLs.
   * @const {!RegExp}
   */
  var FAVICON_URL_REGEX = /\.ico$/i;

  /**
   * Creates a CSS -webkit-image-set for a favicon request.
   * @param {string} url Either the URL of the original page or of the favicon
   *     itself.
   * @param {number=} opt_size Optional preferred size of the favicon.
   * @param {string=} opt_type Optional type of favicon to request. Valid values
   *     are 'favicon' and 'touch-icon'. Default is 'favicon'.
   * @return {string} -webkit-image-set for the favicon.
   */
  function getFaviconImageSet(url, opt_size, opt_type) {
    var size = opt_size || 16;
    var type = opt_type || 'favicon';

    return imageset(
        'chrome://' + type + '/size/' + size + '@scalefactorx/' +
        // Note: Literal 'iconurl' must match |kIconURLParameter| in
        // components/favicon_base/favicon_url_parser.cc.
        (FAVICON_URL_REGEX.test(url) ? 'iconurl/' : '') + url);
  }

  return {
    getSupportedScaleFactors: getSupportedScaleFactors,
    getProfileAvatarIcon: getProfileAvatarIcon,
    getFaviconImageSet: getFaviconImageSet,
  };
});
