/*
 * Copyright (C) 2014 Google Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above
 * copyright notice, this list of conditions and the following disclaimer
 * in the documentation and/or other materials provided with the
 * distribution.
 *     * Neither the name of Google Inc. nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef WaitableEvent_h
#define WaitableEvent_h

#include "platform/PlatformExport.h"
#include "wtf/Vector.h"
#include <memory>

namespace base {
class WaitableEvent;
}; // namespace base

namespace blink {

// Provides a thread synchronization that can be used to allow one thread to
// wait until another thread to finish some work.
class PLATFORM_EXPORT WaitableEvent {
public:
    // If ResetPolicy::Manual is specified on creation, to set the event state
    // to non-signaled, a consumer must call reset().  Otherwise, the system
    // automatically resets the event state to non-signaled after a single
    // waiting thread has been released.
    enum class ResetPolicy { Auto, Manual };

    // Specify the initial state on creation.
    enum class InitialState { NonSignaled, Signaled };

    explicit WaitableEvent(ResetPolicy = ResetPolicy::Auto, InitialState = InitialState::NonSignaled);

    ~WaitableEvent();

    // Puts the event in the un-signaled state.
    void reset();

    // Waits indefinitely for the event to be signaled.
    void wait();

    // Puts the event in the signaled state. Causing any thread blocked on Wait
    // to be woken up. The event state is reset to non-signaled after
    // a waiting thread has been released.
    void signal();

    // Waits on multiple events and returns the index of the object that
    // has been signaled. Any event objects given to this method must
    // not deleted while this wait is happening.
    static size_t waitMultiple(const WTF::Vector<WaitableEvent*>& events);

private:
    WaitableEvent(const WaitableEvent&) = delete;
    void operator=(const WaitableEvent&) = delete;

    std::unique_ptr<base::WaitableEvent> m_impl;
};

} // namespace blink

#endif // WaitableEvent_h
