#!/usr/bin/python
# Copyright (c) 2014 The Chromium Authors. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

"""Utility to duplicate a config in some subset of our GM expectation files.

Created for http://skbug.com/2752 ('split existing "gpu" GM results into "gl"
and "gles"')

Run with -h to see usage.

Example command lines:
  copy_config.py gl gles '.*Mac10.7.*'

TODO(epoger): Once https://codereview.chromium.org/397103003/ is committed,
we should add a unittest.  Until then, we can test this as follows:

OLD=expectations/gm && NEW=/tmp/expectations && \
  rm -rf $NEW && \
  cp -a $OLD $NEW && \
  gm/copy_config.py 8888 8888-copy '.*Mac10.7.*' \
    --expectations-root $NEW && \
  diff --recursive $OLD $NEW
"""
__author__ = 'Elliot Poger'

import argparse
import os
import re

import gm_json

DEFAULT_EXPECTATIONS_ROOT = os.path.join(
    os.path.dirname(__file__), os.pardir, 'expectations', 'gm')
IMAGE_FILENAME_RE = re.compile(gm_json.IMAGE_FILENAME_PATTERN)


class Copier(object):

  def __init__(self, args):
    """
    Params:
      args: the Namespace object generated by argparse.parse_args()
    """
    self._args = args

  def run(self):
    """Perform all the duplications."""
    for path in self._get_file_list():
      self._duplicate_config(path=path,
                             old=self._args.old_config_name,
                             new=self._args.new_config_name)

  def _duplicate_config(self, path, old, new):
    """Duplicates all instances of a config within a GM expectations file.

    Params:
      path: path to file which will be modified in place
      old: old config name
      new: new config name
    """
    dic = gm_json.LoadFromFile(file_path=path)
    expected_results = dic[gm_json.JSONKEY_EXPECTEDRESULTS]
    orig_keys = expected_results.keys()
    for key in orig_keys:
      result = expected_results[key]
      (testname, config) = IMAGE_FILENAME_RE.match(key).groups()
      if config == old:
        config = new
        key = '%s_%s.png' % (testname, config)
        expected_results[key] = result
    gm_json.WriteToFile(json_dict=dic, file_path=path)

  def _get_file_list(self):
    """Returns the list of files we want to operate on (the complete path
    to each file)."""
    root = self._args.expectations_root
    regex = re.compile(self._args.builder_name_pattern)
    return [os.path.join(root, builder, 'expected-results.json')
            for builder in os.listdir(root)
            if regex.match(builder)]


def main():
  parser = argparse.ArgumentParser()
  parser.add_argument('old_config_name',
                      help=('Config we want to duplicate.'))
  parser.add_argument('new_config_name',
                      help=('Name of the new config we want to create.'))
  parser.add_argument('builder_name_pattern',
                      help=('Regex pattern describing which builders we want '
                            'to make the duplication for; \'.*\' to perform '
                            'the duplication on all builders.'))
  parser.add_argument('--expectations-root',
                      default=DEFAULT_EXPECTATIONS_ROOT,
                      help=('Root of the GM expectations dir; defaults to '
                            '%(default)s'))
  args = parser.parse_args()
  copier = Copier(args)
  copier.run()

if __name__ == '__main__':
  main()
