#!/usr/bin/env python
# Copyright 2014 The Chromium Authors. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

import sys
import string
import json

input_json_path = sys.argv[1]
output_cc_path = sys.argv[2]
output_h_path = sys.argv[3]

header = """\
// Copyright 2014 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

// THIS FILE IS AUTOGENERATED. DO NOT EDIT.
// Generated by
//  content/public/browser/devtools_protocol_handler_generator.py from
//  third_party/WebKit/Source/devtools/protocol.json
"""

template_h = string.Template(header + """\

#ifndef CONTENT_BROWSER_DEVTOOLS_PROTOCOL_DEVTOOLS_PROTOCOL_HANDLER_IMPL_H_
#define CONTENT_BROWSER_DEVTOOLS_PROTOCOL_DEVTOOLS_PROTOCOL_HANDLER_IMPL_H_

#include "content/browser/devtools/devtools_protocol.h"
#include "content/browser/devtools/protocol/devtools_protocol_client.h"

namespace content {

class DevToolsProtocolHandlerImpl;

namespace devtools {

${types}\

}  // namespace devtools

class DevToolsProtocolHandlerImpl : public DevToolsProtocol::Handler {
 public:
  typedef DevToolsProtocolClient::Response Response;
  typedef DevToolsProtocolClient::ResponseStatus ResponseStatus;

  DevToolsProtocolHandlerImpl();
  virtual ~DevToolsProtocolHandlerImpl();

${setters}\

 private:
${friends}\

${methods}\

${fields}\
};

}  // namespace content

#endif  // CONTENT_BROWSER_DEVTOOLS_PROTOCOL_DEVTOOLS_PROTOCOL_HANDLER_IMPL_H_
""")

tmpl_typedef = string.Template("""\
namespace ${domain} {
typedef ${param_type} ${declared_name};
}  // namespace ${domain}
""")

tmpl_struct = string.Template("""\
namespace ${domain} {
struct ${declared_name} {
 public:
  ${declared_name}();

${methods}\

 private:
  friend class ::content::DevToolsProtocolHandlerImpl;

${fields}\
};
}  // namespace ${domain}
""")

tmpl_struct_setter = string.Template("""\
  void set_${param}(${pass_type} ${param});
""")

tmpl_struct_field = string.Template("""\
  ${param_type} ${param}_;
  bool has_${param}_;
""")

tmpl_enum = string.Template("""\
namespace ${domain} {
namespace ${subdomain} {
${values}\
}  // namespace ${subdomain}
}  // namespace ${domain}
""")

tmpl_enum_value = string.Template("""\
extern const char k${Param}${Value}[];
""")

tmpl_enum_value_def = string.Template("""\
const char k${Param}${Value}[] = "${value}";
""")

tmpl_handler = string.Template("""\
namespace ${domain} {
class ${Domain}Handler;
}  // namespace domain
""")

tmpl_client = string.Template("""\
namespace ${domain} {
class Client : public DevToolsProtocolClient {
 public:
  Client(const RawMessageCallback& raw_message_callback);
  virtual ~Client();

${methods}\
};
}  // namespace ${domain}
""")

tmpl_event = string.Template("""\
  void ${Command}(
      const ${Command}Params& params);
""")

tmpl_response = string.Template("""\
  void Send${Command}Response(
      scoped_refptr<DevToolsProtocol::Command> command,
      const ${Command}Response& params);
""")

tmpl_setter = string.Template("""\
  void Set${Domain}Handler(
      devtools::${domain}::${Domain}Handler* ${domain}_handler);
""")

tmpl_friend = string.Template("""\
  friend class devtools::${domain}::Client;
""")

tmpl_callback = string.Template("""\
  scoped_refptr<DevToolsProtocol::Response>
  On${Domain}${Command}(
      scoped_refptr<DevToolsProtocol::Command> command);
""")

tmpl_to_value = string.Template("""\
  static base::DictionaryValue* ToValue(
      const devtools::${domain}::${declared_name}& src);
""")

tmpl_field = string.Template("""\
  devtools::${domain}::${Domain}Handler* ${domain}_handler_;
""")

template_cc = string.Template(header + """\

#include "content/browser/devtools/protocol/devtools_protocol_handler_impl.h"

#include "base/bind.h"
${includes}\

namespace content {

DevToolsProtocolHandlerImpl::DevToolsProtocolHandlerImpl()
    : ${fields_init} {
}

DevToolsProtocolHandlerImpl::~DevToolsProtocolHandlerImpl() {
}

namespace {

typedef DevToolsProtocolClient::ResponseStatus ResponseStatus;

bool CreateCommonResponse(
    scoped_refptr<DevToolsProtocol::Command> command,
    const DevToolsProtocolClient::Response& response,
    scoped_refptr<DevToolsProtocol::Response>* protocol_response) {
  switch (response.status()) {
    case ResponseStatus::RESPONSE_STATUS_FALLTHROUGH:
      *protocol_response = NULL;
      break;
    case ResponseStatus::RESPONSE_STATUS_OK:
      return false;
    case ResponseStatus::RESPONSE_STATUS_INVALID_PARAMS:
      *protocol_response = command->InvalidParamResponse(response.message());
      break;
    case ResponseStatus::RESPONSE_STATUS_INTERNAL_ERROR:
      *protocol_response = command->InternalErrorResponse(response.message());
      break;
    case ResponseStatus::RESPONSE_STATUS_SERVER_ERROR:
      *protocol_response = command->ServerErrorResponse(response.message());
      break;
  }
  return true;
}

}  // namespace

${methods}\

namespace devtools {

${types}\

}  // namespace devtools

}  // namespace content
""")

tmpl_include = string.Template("""\
#include "content/browser/devtools/protocol/${domain}_handler.h"
""")

tmpl_field_init = string.Template("${domain}_handler_(NULL)")

tmpl_setter_impl = string.Template("""\
void DevToolsProtocolHandlerImpl::Set${Domain}Handler(
    devtools::${domain}::${Domain}Handler* ${domain}_handler) {
  DCHECK(!${domain}_handler_);
  ${domain}_handler_ = ${domain}_handler;
${initializations}\
}
""")

tmpl_register = string.Template("""\
  RegisterCommandHandler(
      "${Domain}.${command}",
      base::Bind(
          &DevToolsProtocolHandlerImpl::On${Domain}${Command},
          base::Unretained(this)));
""")

tmpl_init_client = string.Template("""\
  ${domain}_handler_->SetClient(make_scoped_ptr(
      new devtools::${domain}::Client(
          base::Bind(&DevToolsProtocolHandlerImpl::SendRawMessage,
                     base::Unretained(this)))));
""")

tmpl_callback_impl = string.Template("""\
scoped_refptr<DevToolsProtocol::Response>
DevToolsProtocolHandlerImpl::On${Domain}${Command}(
    scoped_refptr<DevToolsProtocol::Command> command) {
${prep}\
  Response response = ${domain}_handler_->${Command}(${args});
  scoped_refptr<DevToolsProtocol::Response> protocol_response;
  if (CreateCommonResponse(command, response, &protocol_response))
    return protocol_response;
  base::DictionaryValue* dict = new base::DictionaryValue();
${wrap}\
  return command->SuccessResponse(dict);
}
""")

tmpl_callback_async_impl = string.Template("""\
scoped_refptr<DevToolsProtocol::Response>
DevToolsProtocolHandlerImpl::On${Domain}${Command}(
    scoped_refptr<DevToolsProtocol::Command> command) {
${prep}\
  return ${domain}_handler_->${Command}(${args});
}
""")

params_prep = """\
  base::DictionaryValue* params = command->params();
"""

tmpl_prep_req = string.Template("""\
  ${param_type} in_${param}${init};
  if (!params ||
      !params->Get${Type}("${proto_param}", &in_${param}))
    return command->InvalidParamResponse("${proto_param}");
""")

tmpl_prep_req_list = string.Template("""\
  base::ListValue* list_${param} = NULL;
  if (!params || !params->GetList("${proto_param}", &list_${param}))
    return command->InvalidParamResponse("${proto_param}");
  ${param_type} in_${param};
  for (base::ListValue::const_iterator it =
      list_${param}->begin(); it != list_${param}->end(); ++it) {
    ${item_type} item${item_init};
    if (!(*it)->GetAs${ItemType}(&item))
      return command->InvalidParamResponse("${proto_param}");
    in_${param}.push_back(item);
  }
""")

tmpl_prep_opt = string.Template("""\
  ${param_type} in_${param}${init};
  bool ${param}_found = params && params->Get${Type}(
      "${proto_param}",
      &in_${param});
""")

tmpl_prep_output = string.Template("""\
  ${param_type} out_${param}${init};
""")

tmpl_arg_req = string.Template("in_${param}")

tmpl_arg_opt = string.Template(
    "${param}_found ?\n          &in_${param} : NULL")

tmpl_arg_output = string.Template("&out_${param}")

tmpl_to_value_impl = string.Template("""\
// static
base::DictionaryValue* DevToolsProtocolHandlerImpl::ToValue(
    const devtools::${domain}::${declared_name}& src) {
  base::DictionaryValue* dict = new base::DictionaryValue();
${dchecks}\
${wrap}\
  return dict;
}
""")

tmpl_dcheck = string.Template("""\
  DCHECK(${cond_name});
""")

tmpl_struct_impl = string.Template("""\
namespace ${domain} {

${declared_name}::${declared_name}()${fields} {
}

${methods}\

}  // namespace ${domain}
""")

tmpl_struct_field_init = string.Template("has_${param}_(false)")

tmpl_struct_setter_impl = string.Template("""\
void ${declared_name}::set_${param}(
    ${pass_type} ${param}) {
  ${param}_ = ${param};
  has_${param}_ = true;
}
""")

tmpl_client_impl = string.Template("""\
namespace ${domain} {

Client::Client(const RawMessageCallback& raw_message_callback)
    : DevToolsProtocolClient(raw_message_callback) {
}

Client::~Client() {
}

${methods}\

}  // namespace ${domain}
""")

tmpl_event_impl = string.Template("""\
void Client::${Command}(
    const ${Command}Params& params) {
  SendNotification("${Domain}.${command}",
      DevToolsProtocolHandlerImpl::ToValue(params));
}
""")

tmpl_response_impl = string.Template("""\
void Client::Send${Command}Response(
    scoped_refptr<DevToolsProtocol::Command> command,
    const ${Command}Response& params) {
  SendAsyncResponse(
      command->SuccessResponse(DevToolsProtocolHandlerImpl::ToValue(params)));
}
""")

tmpl_wrap = string.Template("""\
  dict->Set${Type}("${proto_param}", ${var_name});
""")

tmpl_wrap_dict = string.Template("""\
  dict->Set("${proto_param}",
      DevToolsProtocolHandlerImpl::ToValue(${var_name}));
""")

tmpl_wrap_obj = string.Template("""\
  dict->Set("${proto_param}", ${var_name});
""")

tmpl_wrap_list = string.Template("""\
  base::ListValue* list_${param} = new base::ListValue();
  for (${param_type}::const_iterator it =
      ${var_name}.begin(); it != ${var_name}.end(); ++it) {
${append}\
  }
  dict->Set("${proto_param}", list_${param});
""")

tmpl_append = string.Template("""\
    list_${param}->Append${Type}(*it);
""")

tmpl_append_dict = string.Template("""\
    list_${param}->Append(DevToolsProtocolHandlerImpl::ToValue(*it));
""")

tmpl_append_obj = string.Template("""\
    list_${param}->Append(*it);
""")

tmpl_wrap_opt = string.Template("""\
  if (${cond_name})
    dict->Set${Type}("${proto_param}", ${var_name});
""")

tmpl_typename = string.Template("devtools::${domain}::${declared_name}")

def Capitalize(s):
  return s[:1].upper() + s[1:]

def Decapitalize(s):
  return s.lower()

def Uncamelcase(s):
  result = ""
  for i, c in enumerate(s):
    if c.isupper():
      if (i > 0) and ((i < len(s)-1) and s[i+1].islower() or s[i-1].islower()):
        result += "_"
      result += c.lower()
    else:
      result += c
  return result

types = {}
json_api = json.loads(open(input_json_path, "r").read())
type_decls = []
type_impls = []
handler_methods = []
handler_method_impls = []

for json_domain in json_api["domains"]:
  if "types" in json_domain:
    for json_type in json_domain["types"]:
      types["%s.%s" % (json_domain["domain"], json_type["id"])] = json_type

def DeclareStruct(json_properties, mapping):
  methods = []
  fields = []
  fields_init = []
  method_impls = []
  dchecks = []
  wrap = []
  for json_prop in json_properties:
    prop_map = mapping.copy()
    prop_map["proto_param"] = json_prop["name"]
    prop_map["param"] = Uncamelcase(json_prop["name"])
    prop_map["var_name"] = "src.%s_" % prop_map["param"]
    prop_map["cond_name"] = "src.has_%s_" % prop_map["param"]
    ResolveType(json_prop, prop_map)
    prop_map["declared_name"] = mapping["declared_name"]
    methods.append(tmpl_struct_setter.substitute(prop_map))
    fields.append(tmpl_struct_field.substitute(prop_map))
    fields_init.append(tmpl_struct_field_init.substitute(prop_map))
    method_impls.append(tmpl_struct_setter_impl.substitute(prop_map))
    if json_prop.get("optional"):
      if param_map["Type"] in ["List", "Dictionary"]:
        # TODO(vkuzkokov) Implement.
        raise Exception(
          "Optional array and object properties are not implemented")
      wrap.append(tmpl_wrap_opt.substitute(prop_map))
    else:
      dchecks.append(tmpl_dcheck.substitute(prop_map));
      if not "wrap" in prop_map:
        raise Exception("Arrays of arrays are not implemented")
      wrap.append(prop_map["wrap"])

  type_decls.append(tmpl_struct.substitute(mapping,
      methods = "".join(methods),
      fields = "".join(fields)))
  fields_init_str = ""
  if len(fields_init) > 0:
    fields_init_str = "\n    : " + (",\n      ".join(fields_init))
  type_impls.append(tmpl_struct_impl.substitute(mapping,
      fields = fields_init_str,
      methods = "\n".join(method_impls)))
  handler_methods.append(tmpl_to_value.substitute(mapping))
  handler_method_impls.append(tmpl_to_value_impl.substitute(mapping,
      dchecks = "".join(dchecks),
      wrap = "".join(wrap)))

def ResolveRef(json, mapping):
  dot_pos = json["$ref"].find(".")
  if dot_pos == -1:
    domain_name = mapping["Domain"]
    type_name = json["$ref"]
  else:
    domain_name = json["$ref"][:dot_pos]
    type_name = json["$ref"][dot_pos + 1:]
  json_type = types["%s.%s" % (domain_name, type_name)]
  mapping["declared_name"] = Capitalize(type_name)
  mapping["Domain"] = domain_name
  mapping["domain"] = Decapitalize(domain_name)
  mapping["param_type"] = tmpl_typename.substitute(mapping)
  if json_type.get("enum"):
    # TODO(vkuzkokov) Implement. Approximate template:
    # namespace ${domain} { const char k${declared_name}${Value}; }
    raise Exception("Named enumerations are not implemented")
  ResolveType(json_type, mapping)
  if not "___struct_declared" in json_type:
    json_type["___struct_declared"] = True;
    if (json_type.get("type") == "object") and ("properties" in json_type):
      DeclareStruct(json_type["properties"], mapping)
    else:
      type_decls.append(tmpl_typedef.substitute(mapping))
  mapping["param_type"] = tmpl_typename.substitute(mapping)

def ResolveArray(json, mapping):
  items_map = mapping.copy()
  ResolveType(json["items"], items_map)
  mapping["param_type"] = "std::vector<%s>" % items_map["param_type"]
  mapping["Type"] = "List"
  if "append" in items_map:
    mapping["wrap"] = tmpl_wrap_list.substitute(mapping,
        append = items_map["append"])
  mapping["pass_type"] = "const %s&" % mapping["param_type"]
  mapping["prep_req"] = tmpl_prep_req_list.substitute(mapping,
      item_type = items_map["param_type"],
      item_init = items_map["init"],
      ItemType = items_map["Type"])
  # TODO(vkuzkokov) mapping["append"]: template for array of arrays.

def ResolveObject(json, mapping):
  mapping["Type"] = "Dictionary"
  if "properties" in json:
    if not "declared_name" in mapping:
      mapping["declared_name"] = ("%s%s" %
          (mapping["Command"], Capitalize(mapping["proto_param"])))
      mapping["param_type"] = tmpl_typename.substitute(mapping)
      DeclareStruct(json["properties"], mapping)
    mapping["append"] = tmpl_append_dict.substitute(mapping)
    mapping["wrap"] = tmpl_wrap_dict.substitute(mapping)
    mapping["pass_type"] = "const %s&" % mapping["param_type"]
  else:
    mapping["param_type"] = "base::DictionaryValue*"
    mapping["append"] = tmpl_append_obj.substitute(mapping)
    mapping["wrap"] = tmpl_wrap_obj.substitute(mapping)
    mapping["pass_type"] = mapping["param_type"]

def ResolvePrimitive(json, mapping):
  jsonrpc_type = json["type"]
  if jsonrpc_type == "boolean":
    mapping["param_type"] = "bool"
    mapping["Type"] = "Boolean"
    mapping["init"] = " = false"
  elif jsonrpc_type == "integer":
    mapping["param_type"] = "int"
    mapping["Type"] = "Integer"
    mapping["init"] = " = 0"
  elif jsonrpc_type == "number":
    mapping["param_type"] = "double"
    mapping["Type"] = "Double"
    mapping["init"] = " = 0.0"
  elif jsonrpc_type == "string":
    mapping["param_type"] = "std::string"
    mapping["pass_type"] = "const std::string&"
    mapping["Type"] = "String"
    if "enum" in json:
      values = []
      value_defs = []
      if "declared_name" in mapping:
        mapping["subdomain"] = Uncamelcase(mapping["declared_name"])
      else:
        mapping["subdomain"] = Uncamelcase(mapping["command"])
      mapping["Param"] = Capitalize(mapping["proto_param"])
      for enum_value in json["enum"]:
        values.append(tmpl_enum_value.substitute(mapping,
            Value = Capitalize(enum_value)))
        value_defs.append(tmpl_enum_value_def.substitute(mapping,
            value = enum_value,
            Value = Capitalize(enum_value)))
      type_decls.append(tmpl_enum.substitute(mapping,
          values = "".join(values)))
      type_impls.append(tmpl_enum.substitute(mapping,
          values = "".join(value_defs)))
  else:
    raise Exception("Unknown type: %s" % json_type)
  mapping["wrap"] = tmpl_wrap.substitute(mapping)
  mapping["append"] = tmpl_append.substitute(mapping)
  mapping["prep_req"] = tmpl_prep_req.substitute(mapping)
  if jsonrpc_type != "string":
    mapping["pass_type"] = mapping["param_type"]

def ResolveType(json, mapping):
  mapping["init"] = ""
  if "$ref" in json:
    ResolveRef(json, mapping)
  elif "type" in json:
    jsonrpc_type = json["type"]
    if jsonrpc_type == "array":
      ResolveArray(json, mapping)
    elif jsonrpc_type == "object":
      ResolveObject(json, mapping)
    else:
      ResolvePrimitive(json, mapping)
  else:
    raise Exception("Unknown type at %s.%s %s" %
        (mapping["Domain"], mapping["command"], mapping["proto_param"]))

setters = []
friends = []
fields = []

includes = []
fields_init = []

for json_domain in json_api["domains"]:
  domain_map = {}
  domain_map["Domain"] = json_domain["domain"]
  domain_map["domain"] = Decapitalize(json_domain["domain"])

  initializations = []
  client_methods = []
  client_method_impls = []
  domain_empty = True
  domain_needs_client = False

  if "commands" in json_domain:
    for json_command in json_domain["commands"]:
      if (not ("handlers" in json_command) or
          not ("browser" in json_command["handlers"])):
        continue
      domain_empty = False

      command_map = domain_map.copy()
      command_map["command"] = json_command["name"]
      command_map["Command"] = Capitalize(json_command["name"])

      prep = []
      args = []

      if "parameters" in json_command:
        for json_param in json_command["parameters"]:
          param_map = command_map.copy()
          param_map["proto_param"] = json_param["name"]
          param_map["param"] = Uncamelcase(json_param["name"])
          param_map["var_name"] = "in_%s" % param_map["param"]

          ResolveType(json_param, param_map)
          if len(prep) == 0:
            prep.append(params_prep)
          if json_param.get("optional"):
            if param_map["Type"] in ["List", "Dictionary"]:
              # TODO(vkuzkokov) Implement transformation of base::ListValue
              # to std::vector and base::DictonaryValue to struct.
              raise Exception(
                  "Optional array and object parameters are not implemented")
            prep.append(tmpl_prep_opt.substitute(param_map))
            args.append(tmpl_arg_opt.substitute(param_map))
          else:
            prep.append(param_map["prep_req"])
            args.append(tmpl_arg_req.substitute(param_map))

      if json_command.get("async"):
        domain_needs_client = True
        json_returns = []
        if "returns" in json_command:
          json_returns = json_command["returns"]
        command_map["declared_name"] = "%sResponse" % command_map["Command"]
        DeclareStruct(json_returns, command_map)
        # TODO(vkuzkokov) Pass async callback instance similar to how
        # InspectorBackendDispatcher does it. This, however, can work
        # only if Blink and Chrome are in the same repo.
        args.append("command")
        handler_method_impls.append(
            tmpl_callback_async_impl.substitute(command_map,
                prep = "".join(prep),
                args = "\n      " + ",\n      ".join(args)))
        client_methods.append(tmpl_response.substitute(command_map))
        client_method_impls.append(tmpl_response_impl.substitute(command_map))
      else:
        wrap = []
        if "returns" in json_command:
          for json_param in json_command["returns"]:
            param_map = command_map.copy()
            param_map["proto_param"] = json_param["name"]
            param_map["param"] = Uncamelcase(json_param["name"])
            param_map["var_name"] = "out_%s" % param_map["param"]

            if json_param.get("optional"):
              # TODO(vkuzkokov) Implement Optional<T> for value types.
              raise Exception("Optional return values are not implemented")
            ResolveType(json_param, param_map)
            prep.append(tmpl_prep_output.substitute(param_map))
            args.append(tmpl_arg_output.substitute(param_map))
            if not "wrap" in param_map:
              raise Exception("Arrays of arrays are not implemented")
            wrap.append(param_map["wrap"])

        args_str = ""
        if len(args) > 0:
          args_str = "\n      " + ",\n      ".join(args)
        handler_method_impls.append(tmpl_callback_impl.substitute(command_map,
            prep = "".join(prep),
            args = args_str,
            wrap = "".join(wrap)))

      initializations.append(tmpl_register.substitute(command_map))
      handler_methods.append(tmpl_callback.substitute(command_map))

  if "events" in json_domain:
    for json_event in json_domain["events"]:
      if (not ("handlers" in json_event) or
          not ("browser" in json_event["handlers"])):
        continue
      domain_empty = False
      domain_needs_client = True

      event_map = domain_map.copy()
      event_map["command"] = json_event["name"]
      event_map["Command"] = Capitalize(json_event["name"])

      json_parameters = []
      if "parameters" in json_event:
        json_parameters = json_event["parameters"]
      event_map["declared_name"] = "%sParams" % event_map["Command"]
      DeclareStruct(json_parameters, event_map);

      client_methods.append(tmpl_event.substitute(event_map))
      client_method_impls.append(tmpl_event_impl.substitute(event_map))

  if domain_empty:
    continue
  type_decls.append(tmpl_handler.substitute(domain_map))
  setters.append(tmpl_setter.substitute(domain_map))
  fields.append(tmpl_field.substitute(domain_map))
  includes.append(tmpl_include.substitute(domain_map))
  fields_init.append(tmpl_field_init.substitute(domain_map))
  if domain_needs_client:
    type_decls.append(tmpl_client.substitute(domain_map,
        methods = "".join(client_methods)))
    friends.append(tmpl_friend.substitute(domain_map))
    initializations.append(tmpl_init_client.substitute(domain_map))
    type_impls.append(tmpl_client_impl.substitute(domain_map,
        methods = "\n".join(client_method_impls)))
  handler_method_impls.append(tmpl_setter_impl.substitute(domain_map,
      initializations = "".join(initializations)))


output_h_file = open(output_h_path, "w")
output_cc_file = open(output_cc_path, "w")

output_h_file.write(template_h.substitute({},
    types = "\n".join(type_decls),
    setters = "".join(setters),
    friends = "".join(friends),
    methods = "".join(handler_methods),
    fields = "".join(fields)))
output_h_file.close()

output_cc_file.write(template_cc.substitute({},
    includes = "".join(sorted(includes)),
    fields_init = ",\n      ".join(fields_init),
    methods = "\n".join(handler_method_impls),
    types = "\n".join(type_impls)))
output_cc_file.close()
