/*	$NetBSD: rambo.c,v 1.15 2025/05/01 05:36:02 tsutsui Exp $	*/

/*
 * Copyright (c) 2000 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Wayne Knowles
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: rambo.c,v 1.15 2025/05/01 05:36:02 tsutsui Exp $");

#include <sys/param.h>
#include <sys/kernel.h>
#include <sys/device.h>
#include <sys/systm.h>
#include <sys/timetc.h>

#include <machine/cpu.h>
#include <machine/mainboard.h>
#include <machine/autoconf.h>
#include <machine/sysconf.h>
#include <machine/bus.h>

#include <mipsco/obio/rambo.h>

/*
 * Timer & Interrupt manipulation routines for the Rambo Custom ASIC 
 */

static int	rambo_match(device_t, cfdata_t, void *);
static void	rambo_attach(device_t, device_t, void *);
static unsigned rambo_get_timecount(struct timecounter *);
void rambo_clkintr(struct clockframe *);
static void rambo_tc_init(void);

struct rambo_softc {
	struct evcnt		sc_intrcnt;
	bus_space_tag_t		sc_bst;
	bus_space_handle_t	sc_bsh;
	uint32_t		sc_hzticks;
};

static struct rambo_softc *rambo;

CFATTACH_DECL_NEW(rambo, sizeof(struct rambo_softc),
    rambo_match, rambo_attach, NULL, NULL);

static int
rambo_match(device_t parent, cfdata_t cf, void *aux)
{
	return 1;
}

static void
rambo_attach(device_t parent, device_t self, void *aux)
{
	struct confargs *ca = aux;
	struct rambo_softc *sc = device_private(self);

	sc->sc_bst = ca->ca_bustag;

	if (bus_space_map(ca->ca_bustag, ca->ca_addr, 256,
			  BUS_SPACE_MAP_LINEAR,
			  &sc->sc_bsh) != 0) {
		printf(": cannot map registers\n");
		return;
	}
	evcnt_attach_dynamic(&sc->sc_intrcnt, EVCNT_TYPE_INTR, NULL,
			     "timer", "intr");

	/* Setup RAMBO Timer to generate timer interrupts */
	sc->sc_hzticks = HZ_TO_TICKS(hz);

	bus_space_write_4(sc->sc_bst, sc->sc_bsh, RB_TCOUNT, 0);
	bus_space_write_4(sc->sc_bst, sc->sc_bsh, RB_TBREAK, sc->sc_hzticks);

	bus_space_write_4(sc->sc_bst, sc->sc_bsh, RB_CTLREG,
			  RB_PARITY_EN | RB_BUZZOFF | RB_CLR_IOERR);

	printf(": parity enabled\n");
	rambo = sc;
	platform.clkinit = rambo_tc_init;
}

void
rambo_clkintr(struct clockframe *cf)
{
	register uint32_t tbreak, tcount;
	register int delta;

	rambo->sc_intrcnt.ev_count++;
	tbreak = bus_space_read_4(rambo->sc_bst, rambo->sc_bsh, RB_TBREAK);
	tcount = bus_space_read_4(rambo->sc_bst, rambo->sc_bsh,	RB_TCOUNT);
	delta  = tcount - tbreak;

	if (delta > (rambo->sc_hzticks>>1)) {
		/*
		 * Either tcount may overtake the updated tbreak value
		 * or we have missed several interrupts.
		 */
		int cycles = 10 * hz;
		while (cycles && tbreak < tcount) {
			hardclock(cf);
			tbreak += rambo->sc_hzticks;
			cycles--;
		}
	} else {
		hardclock(cf);
		tbreak += rambo->sc_hzticks;
	}

	bus_space_write_4(rambo->sc_bst, rambo->sc_bsh, RB_TBREAK, tbreak);
}    

static unsigned
rambo_get_timecount(struct timecounter *tc)
{

	return (bus_space_read_4(rambo->sc_bst, rambo->sc_bsh, RB_TCOUNT));
}

static void
rambo_tc_init(void)
{
	static struct timecounter tc = {
		.tc_get_timecount = rambo_get_timecount,
		.tc_frequency = RB_FREQUENCY,
		.tc_quality = 100,
		.tc_name = "rambo_tcount",
		.tc_counter_mask = ~0
	};

	tc_init(&tc);
}
