#	$NetBSD: diskproto2sunlabel.awk,v 1.1.2.2 2026/01/22 20:24:45 martin Exp $
#
# Copyright (c) 2026 Izumi Tsutsui.  All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
# OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
# IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
# INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
# NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
# THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

#
# diskproto2sunlabel.awk
#
# Read a disklabel prototype (${WORKLABEL} generated by Makefile.bootimage),
# and emit sunlabel(8) command script to stdout.
#
# Usage:
#   awk -f diskproto2sunlabel.awk < ${WORKLABEL} | sunlabel -nq ${WORKIMG}

function err(msg) {
	print "diskproto2sunlabel.awk: " msg > "/dev/stderr"
	exit 1
}

BEGIN {
	# Assume MAXPARTITIONS==8 is default on ports that require sunlabel(8)
	if (parts == "")
		parts = "a b c d e f g h"
	n = split(parts, pstr, /[ \t]+/)
	for (i = 1; i <= n; i++)
		validpart[pstr[i]] = 1
	nparts = 0
}

# remove comments and leading/trailing spaces/tabs from all lines
{
	sub(/[ \t]*#.*/, "", $0)
	gsub(/^[ \t]+/, "", $0)
	gsub(/[ \t]+$/, "", $0)
	if ($0 == "")
		next
}

/^sectors\/track:[ \t]*[0-9]+/ {
	if (NF != 2)
		err("Invalid sectors/track line")
	spt = $2
	next
}

/^tracks\/cylinder:[ \t]*[0-9]+/ {
	if (NF != 2)
		err("Invalid tracks/cylinder line")
	head = $2
	next
}

/^sectors\/cylinder:[ \t]*[0-9]+/ {
	if (NF != 2)
		err("Invalid sectors/cylinder line")
	spc = $2
	next
}

/^cylinders:[ \t]*[0-9]+/ {
	if (NF != 2)
		err("Invalid cylinders line")
	ncyl = $2
	next
}

/^[a-p]:[ \t]*[0-9]+[ \t]+[0-9]+/ {
	part = $1
	sub(/:$/, "", part)
	part_size[part] = $2
	part_offset[part] = $3
	parts_in_image[nparts++] = part
	next
}

# ignore all other lines; assume ${WORKLABEL} is a generated file here
{
	next
}

END {
	if (spt == "")
		err("missing geometry (sectors/track)")
	if (head == "")
		err("missing geometry (tracks/cylinder)")
	if (ncyl == "")
		err("missing geometry (cylinders)")
	if (spc == "")
		err("missing geometry (sectors/cylinder)")
	if (spc != spt * head)
		err("inconsistent sectors/track and sectors/cylinder")
	if (nparts == 0)
		err("no valid partition")

	print "V ncyl " ncyl
	print "V nhead " head
	print "V nsect " spt

	for (i = 0; i < nparts; i++) {
		p = parts_in_image[i]
		if (!validpart[p])
			continue
		if (part_size[p] == 0)
			continue

		# sunlabel partition start must be cylinder-aligned.
		if ((part_offset[p] % spc) != 0)
			err("partition " p \
			    " offset " part_offset[p] \
			    " is not cylinder-aligned (spc=" spc ")")

		start = int(part_offset[p] / spc)
		print p " " start " " part_size[p]
	}

	print "W"
}
