#!/bin/sh
# SPDX-License-Identifier: GPL-3.0-or-later
#
# %%NEW_CLAIMING_METHOD%%

set -e

warning() {
    printf "WARNING: %s\n" "${1}" 1>&2
}

error() {
    printf "ERROR: %s\n" "${1}" 1>&2
    exit "${2}"
}

get_templated_value() {
    value="$1"
    default="$2"
    override="$3"

    if [ -n "${override}" ]; then
        echo "${override}"
    elif [ -z "${value}" ]; then
        error "Expected templated value not present"
    elif (echo "${value}" | grep -q '@'); then
        echo "${default}"
    else
        echo "${value}"
    fi
}

config_dir="$(get_templated_value "/opt/local/etc/netdata" "/etc/netdata" "${NETDATA_CLAIM_CONFIG_DIR}")"
claim_config="${config_dir}/claim.conf"
netdatacli="$(get_templated_value "/opt/local/sbin/netdatacli" "$(command -v netdatacli 2>/dev/null)" "${NETDATA_CLAIM_NETDATACLI_PATH}")"
netdata_group="$(get_templated_value "netdata" "netdata" "${NETDATA_CLAIM_CONFIG_GROUP}")"

write_config() {
    config="[global]"
    config="${config}\n    url = ${NETDATA_CLAIM_URL}"
    config="${config}\n    token = ${NETDATA_CLAIM_TOKEN}"
    if [ -n "${NETDATA_CLAIM_ROOMS}" ]; then
        config="${config}\n    rooms = ${NETDATA_CLAIM_ROOMS}"
    fi
    if [ -n "${NETDATA_CLAIM_PROXY}" ]; then
        config="${config}\n    proxy = ${NETDATA_CLAIM_PROXY}"
    fi
    if [ -n "${NETDATA_CLAIM_INSECURE}" ]; then
        config="${config}\n    insecure = ${NETDATA_CLAIM_INSECURE}"
    fi

    touch "${claim_config}.tmp"
    chmod 0660 "${claim_config}.tmp"
    chown "root:${netdata_group}" "${claim_config}.tmp"
    echo "${config}" > "${claim_config}.tmp"
    chmod 0640 "${claim_config}.tmp"
    mv -f "${claim_config}.tmp" "${claim_config}"
}

reload_claiming() {
    if [ -z "${NORELOAD}" ]; then
        "${netdatacli}" reload-claiming-state
    fi
}

parse_args() {
    while [ -n "${1}" ]; do
        case "${1}" in
            --claim-token) NETDATA_CLAIM_TOKEN="${2}"; shift 1 ;;
            -token=*) NETDATA_CLAIM_TOKEN="$(echo "${1}" | sed 's/^-token=//')" ;;
            --claim-rooms) NETDATA_CLAIM_ROOMS="${2}"; shift 1 ;;
            -rooms=*) NETDATA_CLAIM_ROOMS="$(echo "${1}" | sed 's/^-rooms=//')" ;;
            --claim-url) NETDATA_CLAIM_URL="${2}"; shift 1 ;;
            -url=*) NETDATA_CLAIM_URL="$(echo "${1}" | sed 's/^-url=//')" ;;
            --claim-proxy) NETDATA_CLAIM_PROXY="${2}"; shift 1 ;;
            -proxy=*) NETDATA_CLAIM_PROXY="$(echo "${1}" | sed 's/-proxy=//')" ;;
            -noproxy|--noproxy) NETDATA_CLAIM_PROXY="none" ;;
            -noreload|--noreload) NORELOAD=1 ;;
            -insecure|--insecure) NETDATA_CLAIM_INSECURE=yes ;;
            -verbose) true ;;
            -daemon-not-running) true ;;
            -id=*) warning "-id option is no longer supported. Remove the node ID file instead." ;;
            -hostname=*) warning "-hostname option is no longer supported. Update the main netdata configuration manually instead." ;;
            -user=*) warning "-user option is no longer supported." ;;
            *) warning "Ignoring unrecognized option ${1}";;
        esac

        shift 1
    done

    if [ -z "${NETDATA_CLAIM_TOKEN}" ]; then
        error "Claim token must be specified" 1
    fi

    if [ -z "${NETDATA_CLAIM_URL}" ]; then
        NETDATA_CLAIM_URL="https://app.netdata.cloud/"
    fi
}


[ -z "$EUID" ] && EUID="$(id -u)"
if [ "${EUID}" != "0" ] && [ ! -w "${config_dir}" ]; then
    error "Script must be run by a user with write access to ${config_dir}." 32
fi

warning "This script is deprecated and will be officially unsupported in the near future. Please either use the kickstart script with the appropriate '--claim-*' options, or directly write out the claiming configuration instead."
parse_args "${@}"
write_config
reload_claiming
