//////////////////////////////////////////////////////////////////////////////
//    Copyright 2004-2019, SenseGraphics AB
//
//    This file is part of H3D API.
//
//    H3D API is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    H3D API is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with H3D API; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//    A commercial license is also available. Please contact us at 
//    www.sensegraphics.com for more information.
//
//
/// \file ShaderAtomicCounter.h
/// \brief Header file for ShaderAtomicCounter, X3D scene-graph node
///
//
//////////////////////////////////////////////////////////////////////////////
#ifndef __SHADERATOMICCOUNTER_H__
#define __SHADERATOMICCOUNTER_H__

#include <H3D/ShaderChildNode.h>
#include <H3D/SFInt32.h>
#include <H3D/SFString.h>

namespace H3D {

  /// \ingroup H3DNodes
  /// \class ShaderAtomicCounter
  /// \brief A ShaderAtomicCounter is a atomic counter can be used in GLSL shader.
  /// 
  /// It is fast and can only be used to be incremented or decremented by 1.
  /// The initialValue field is used to set the initial value for this counter,
  /// by default this value is zero if not being set. And this counter will be
  /// reset to its initial value once per rendering loop in the scene.
  /// 
  /// <b>Examples:</b>
  ///   - <a href="../../../H3DAPI/examples/All/ShaderAtomicCounter.x3d">ShaderAtomicCounter.x3d</a>
  ///     ( <a href="examples/ShaderAtomicCounter.x3d.html">Source</a> )
  class H3DAPI_API ShaderAtomicCounter : 
    public ShaderChildNode {
  public:
  
    /// Constructor.
    ShaderAtomicCounter( Inst< DisplayList  >  _displayList = 0,
                         Inst< SFNode       >  _metadata = 0,
                         Inst< SFInt32      >  _initialValue = 0
                         );

    /// initial value for the atomic counter 
    auto_ptr<SFInt32> initialValue;

    /// specify the texture_unit will be used for rendering this shader image
    virtual void preRender( unsigned int program );

    virtual void postRender();

    /// update texture and image if necessary, then bind it to shader program
    virtual void render ( );

    virtual void prepareAtomicCounterBuffer ();

    virtual void initialize ( );

    virtual ~ShaderAtomicCounter ( );

    /// traverse function overridden to reset the counter to its initial value
    virtual void traverseSG( TraverseInfo &ti );

    /// The H3DNodeDatabase for this node
    static H3DNodeDatabase database;

  protected:

    /// Set the counter to the value specified in the initialValue field
    ///
    /// This is done once per frame
    void applyInitialValue();

    /// shader program that this shader storage buffer will be attached
    unsigned int program_handle;

    /// buffer id, generated by openGL
    int buffer_id;

    /// the atomic counter binding point for atomic counter buffer
    /// will be used by the shader to access the data in buffer
    /// this value can be directly set through the binding layout in shader
    /// and inquired through glGetActiveAtomicCounterBufferiv function
    unsigned int atomic_counter_binding;

    /// A flag to indicate that the counter should be reset
    bool do_init_value;
  };
}

#endif
