/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { CoreToolScheduler } from '@google/gemini-cli-core';
import { useCallback, useState, useMemo, useEffect, useRef } from 'react';
/**
 * Legacy scheduler implementation based on CoreToolScheduler callbacks.
 *
 * This is currently the default implementation used by useGeminiStream.
 * It will be phased out once the event-driven scheduler migration is complete.
 */
export function useReactToolScheduler(onComplete, config, getPreferredEditor) {
    const [toolCallsForDisplay, setToolCallsForDisplay] = useState([]);
    const [lastToolOutputTime, setLastToolOutputTime] = useState(0);
    const onCompleteRef = useRef(onComplete);
    const getPreferredEditorRef = useRef(getPreferredEditor);
    useEffect(() => {
        onCompleteRef.current = onComplete;
    }, [onComplete]);
    useEffect(() => {
        getPreferredEditorRef.current = getPreferredEditor;
    }, [getPreferredEditor]);
    const outputUpdateHandler = useCallback((toolCallId, outputChunk) => {
        setLastToolOutputTime(Date.now());
        setToolCallsForDisplay((prevCalls) => prevCalls.map((tc) => {
            if (tc.request.callId === toolCallId && tc.status === 'executing') {
                const executingTc = tc;
                return { ...executingTc, liveOutput: outputChunk };
            }
            return tc;
        }));
    }, []);
    const allToolCallsCompleteHandler = useCallback(async (completedToolCalls) => {
        await onCompleteRef.current(completedToolCalls);
    }, []);
    const toolCallsUpdateHandler = useCallback((allCoreToolCalls) => {
        setToolCallsForDisplay((prevTrackedCalls) => {
            const prevCallsMap = new Map(prevTrackedCalls.map((c) => [c.request.callId, c]));
            return allCoreToolCalls.map((coreTc) => {
                const existingTrackedCall = prevCallsMap.get(coreTc.request.callId);
                const responseSubmittedToGemini = existingTrackedCall?.responseSubmittedToGemini ?? false;
                if (coreTc.status === 'executing') {
                    // eslint-disable-next-line @typescript-eslint/no-unsafe-type-assertion
                    const liveOutput = existingTrackedCall
                        ?.liveOutput;
                    return {
                        ...coreTc,
                        responseSubmittedToGemini,
                        liveOutput,
                    };
                }
                else if (coreTc.status === 'success' ||
                    coreTc.status === 'error' ||
                    coreTc.status === 'cancelled') {
                    return {
                        ...coreTc,
                        responseSubmittedToGemini,
                    };
                }
                else {
                    return {
                        ...coreTc,
                        responseSubmittedToGemini,
                    };
                }
            });
        });
    }, [setToolCallsForDisplay]);
    const stableGetPreferredEditor = useCallback(() => getPreferredEditorRef.current(), []);
    const scheduler = useMemo(() => new CoreToolScheduler({
        outputUpdateHandler,
        onAllToolCallsComplete: allToolCallsCompleteHandler,
        onToolCallsUpdate: toolCallsUpdateHandler,
        getPreferredEditor: stableGetPreferredEditor,
        config,
    }), [
        config,
        outputUpdateHandler,
        allToolCallsCompleteHandler,
        toolCallsUpdateHandler,
        stableGetPreferredEditor,
    ]);
    const schedule = useCallback((request, signal) => {
        setToolCallsForDisplay([]);
        return scheduler.schedule(request, signal);
    }, [scheduler, setToolCallsForDisplay]);
    const markToolsAsSubmitted = useCallback((callIdsToMark) => {
        setToolCallsForDisplay((prevCalls) => prevCalls.map((tc) => callIdsToMark.includes(tc.request.callId)
            ? { ...tc, responseSubmittedToGemini: true }
            : tc));
    }, []);
    const cancelAllToolCalls = useCallback((signal) => {
        scheduler.cancelAll(signal);
    }, [scheduler]);
    return [
        toolCallsForDisplay,
        schedule,
        markToolsAsSubmitted,
        setToolCallsForDisplay,
        cancelAllToolCalls,
        lastToolOutputTime,
    ];
}
//# sourceMappingURL=useReactToolScheduler.js.map