#!/bin/bash
#*******************************************************************************
# E.S.O. - VLT project
#
# "@(#) $Id: fposs.sh.tpl 285118 2016-06-25 13:28:58Z pbaksai $"
#
# who       when        what
# --------  ----------  --------------------------------------------------------
# pbaksai   2016/06/23  Validate -f argument is a valid file path.
# pbaksai   2016/06/23  Fixed passthrough of -utdate argument.
# pbaksai   2016/06/22  Added command line options passing through to configure.
# pbaksai   2014/09/21  CCB-000755: Support for 64 bits and portability.
#                       - Corrected execution condition.
# pbaksai   2014/07/28  CCB-000755: Support for 64 bits and portability.
#                       - Start configure in the background and exit.
# pbaksai   2014/07/24  CCB-000755: Support for 64 bits and portability.
#                       - Added library linkage info to debug.
# pbaksai   2014/07/24  CCB-000755: Support for 64 bits and portability.
#                       - Check that ~/.fposs is created for user.
# pbaksai   2014/07/22  CCB-000755: Support for 64 bits and portability.
#                       - Minor corrections.
# pbaksai   2014/07/13  CCB-000755: Support for 64 bits and portability.
#                       - Fixed for differences in svn info output between
#                         svn 1.6 and 1.7
# pbaksai   2014/07/10  CCB-000755: Support for 64 bits and portability.
#                       - Created.
#
#*******************************************************************************
#   NAME
#       fposs.sh - FPOSS startup in user environment
#
#   SYNOPSIS
#       fposs.sh [[<configure opts>]
#                 [-l|--log[=<logpath>]]
#                 [--debug]
#                 [--help]
#                 [--usage]
#                 [--License]
#                 [--Version]
#                 --]
#
#   DESCRIPTION
#       This script starts FPOSS application in the user's environment.
#
#   PARAMETERS
#        -l|--log         : Enables logging mode.
#       --debug           : Enables debugging mode. Hidden from normal users.
#       --help            : Displays a help screen explaining the syntax.
#       --usage           : Displays a help screen explaining the interface. 
#       --License         : Displays copyright and license information.
#       --Version         : Displays program version.
#
#   ARGUMENTS
#       ?<configure opts>?: [configure -- -h] Command line options passed to
#                                             configure.
#                           Default: ""
#       ?<logpath>?       : Path to user defined log file.
#                           Default: ""
#
#   INPUTS
#       None
#
#   OUTPUTS
#       <stdout>
#           <exec log>    : Log of execution.
#       <file>
#           <logfile>     : <$PWD>/fposs.log | <logpath>
#
#   FILES
#       bash_library.sh
#
#   ENVIRONMENT
#       USER
#       HOST
#       HOME
#       DISPLAY
#       SHELL_DEBUG
#
#   RETURN VALUES
#       0: All OK.
#       1: Cancelled due to insufficient conditions.
#       2: Aborted due to errors.
#       3: Rejected due to incorrect arguments.
#       4: Ended with information display only (help, usage, license, version).
#
#   CAUTIONS
#       None
#
#   EXAMPLES
#       fposs.sh -f test/M67.fld -utdate '2017 12 31.5'
#
#   SEE ALSO
#       None
#
#   BUGS
#       None
#
#-------------------------------------------------------------------------------
#        1         2         3         4         5         6         7         8
#23456789 123456789 123456789 123456789 123456789 123456789 123456789 1234567890
#
# Include BASH library
#
bashlib=/opt/local/libexec/eso/bin/bash_library.sh
if [ -f "$bashlib" ]; then
    source $bashlib 2>&-
    status=$?
else
    echo "ERROR: $bashlib not found."
    exit 1
fi
if [ $status -ne 0 ]; then
    echo "ERROR: Code $status sourcing $bashlib."
    exit 1
fi

#################################
# Standard Library Overload BEGIN
#
#
# Standard Library Overload END
###############################

###############################
# Standard Initialization BEGIN
#
# Constants
#
# Text formatting for display and log file
LMARGIN=0 ; # Number of columns on the left margin
RMARGIN=0 ; # Number of columns in the right margin
MARGIN=$(printf "%*s" ${LMARGIN}) ; # String of blanks for left margin
#
# Standard Initialization END
#############################

#####################
# Local Library BEGIN
#
#************#
# Parameters #
#************#
#
# ArgFILE: Prints detailed message about the reasons inputfile
#          argument is not accepted.
#
ArgFILE() {
    # Parameters:
    # 1: Value of inputfile argument
    # Globals:    none
    # Assigns:    nothing
    # Returns:    nothing
    if [ -z "$1" ]; then
        Error   "Argument inputfile is empty."
    else
        Error   "Input file $1 does not exist."
    fi
    Explain "inputfile must be the path to an existing file."
}

#
# ValFILE: Validates that inputfile's value received is acceptable.
#
ValFILE() {
    # Parameters:
    # 1: Value of inputfile argument
    # Globals:    none
    # Assigns:
    # - eflag: Error flag TRUE
    # Returns:    nothing
    if [ -n "$1" -a -f "$1" ]; then
        return
    fi
    eflag=$EC_REJECT
    ArgFILE $1
}


#*************#
# Environment #
#*************#
#

#
# Local Library END
###################

############################
# Local Initialization BEGIN
#
# Local constants
#

# Some options are kept hidden from users but available to developers:
# batch, debug, trace, verbose, screen
HELP_LINE="\
 [[-f|--file <inputfile>]$LIFS\
  [-ut|--utdate '<yyyy> <mm> <dd>']$LIFS\
  [-l|--log[=<logpath>]]$LIFS\
  [--help] [--usage]$LIFS\
  [--License] [--Version]$LIFS\
  --]$LIFS\
          "

USAGE_LINE="\
 [[-f|--file <inputfile>] $LIFS\
  [-ut|--utdate '<yyyy> <mm> <dd>'] $LIFS\
  [-l|--log[=<logpath>]] $LIFS\
  [--help] $LIFS\
  [--usage] $LIFS\
  [--License] $LIFS\
  [--Version] $LIFS\
  --] $LIFS\
 where: $LIFS\
   -f or --file: loads input file $LIFS\
   <inputfile>: input file path (.fld or .sds or .paf) $LIFS\
                No default $LIFS\
   -ut or --utdate: sets the UT date for the configuration $LIFS\
  '<yyyy> <mm> <dd>': UT date of configuration in quotes. $LIFS\
                No default $LIFS\
   -l or --log: enables log to file $LIFS\
   ?logpath?: path to user specified log file $LIFS\
              Default is $PWD/$progname $LIFS\
  --help: displays command line syntax $LIFS\
  --usage: displays this Usage Screen $LIFS\
  --License: displays copyright and license $LIFS\
  --Version: displays the program version $LIFS\
  --: marks end of options, starts positional arguments $LIFS\
          "

LICENSE_LINE="\
  Copyright (C) $(ExpandRange 2014..$yearstamp)$LIFS\
  Pedro Baksai (pbaksai@eso.org)$LIFS\
  $LIFS\
  This program is free software: you can redistribute$LIFS\
  it and/or modify it under the terms of the$LIFS\
  GNU General Public License as published by$LIFS\
  the Free Software Foundation, either version 3 of$LIFS\
  the License, or (at your option) any later version.$LIFS\
  $LIFS\
  This program is distributed in the hope that it$LIFS\
  will be useful, but WITHOUT ANY WARRANTY,$LIFS\
  without even the implied warranty of$LIFS\
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.$LIFS\
  See the GNU General Public License for more details.$LIFS\
  $LIFS\
  You should have received a copy of the$LIFS\
  GNU General Public License along with this program.$LIFS\
  If not, see <http://www.gnu.org/licenses/gpl.html>.$LIFS\
             "

#
# Local variables
#
WORKING_DIR=`pwd`
display=$DISPLAY
CFGOPTS=""
UTDATE=""
utdate=0

#
# Local Initialization END
##########################

################
# MAIN PROGRAM #
################

#########################
# Standard Prologue BEGIN
#
StartLog

# Receive options and parameters
#
# Options may be followed by one colon to indicate they have a required argument,
# or by two colons to indicate they have an optional argument.
options=$(GetOpt -q -o l::d? -a --longoptions file:,\
                                --longoptions utdate:,\
                                --longoptions log::,\
                                --longoptions debug,\
                                --longoptions help,\
                                --longoptions usage,\
                                --longoptions License,\
                                --longoptions Version \
                                -n $progname -- "$@")
status=$?

# Processes valid options
eval set -- "$options"
while [ $# -gt 0 ]; do
    case $1 in
        -f | -file | --file)
            Debug "Processing option $1 with required argument >$2<."
            # options passthrough for 'configure':
            # Cumulate option and its arguments.
            ValFILE $2
            CFGOPTS="$CFGOPTS -f $2"
            shift
            ;;
        -ut | --ut | -utdate | --utdate)
            Debug "Processing option $1 with required argument >$2<."
            # options passthrough for 'configure':
            # Argument to -utdate must be re-quoted in 'configure' command line.
            # Doing CFGOPTS="$CFGOPTS -utdate $2" here does not work for 'configure'.
            utdate=1
            UTDATE="$2"
            shift
            ;;
        -l | -log | --log)
            Debug "Processing option $1 with optional argument >$2<"
            lflag=1
            StartLog "$2"
            shift
            ;;
        -d | -debug | --debug)
            dflag=1
            Debug "Turned ON."
            Debug "Options received: >$options<"
            ;;
        --help | -\?)
            Debug "Processing option $1 without argument."
            Help Syntax "$HELP_LINE"
            Exit $EC_INFO
            ;;
        --usage)
            Debug "Processing option $1 without argument."
            Help Usage "$USAGE_LINE"
            Exit $EC_INFO
            ;;
        --License)
            Debug "Processing option $1 without argument."
            Help License "$LICENSE_LINE"
            Exit $EC_INFO
            ;;
        --Version)
            Debug "Processing option $1 without argument."
            Help Version "$(Version)"
            Exit $EC_INFO
            ;;
        (--)
            Debug "End of options detected."
            shift
            break
            ;;
        (--*)
            Error "invalid long option $1"
            eflag=$EC_REJECT
            ;;
        (-?)
            Error "invalid short option $1"
            eflag=$EC_REJECT
            ;;
        (*)
            Debug "Argument $1 detected."
            break
            ;;
    esac
    shift
done

# Catches invalid options
if [ $status -ne 0 ]; then
    Warning "code $status: unrecognized or invalid option(s) ignored"
fi

# Processes valid non-option arguments
shift $((OPTIND-1))
case $# in
    (0)
        # Use default
        ;;
    (*)
        # ParNum $#
        ;;
esac

# Catches remaining non-option arguments
while [ $# -gt 0 ]; do
    case $1 in
        (*)
            Warning "invalid argument(s) $1 ignored"
            ;;
    esac
    shift
done

#
# Validate argument values
#

#
# Validate environment variables
#

#
# Exit if error
#
if [ $eflag -ne 0 ]; then
    HelpAd
    Exit $eflag
fi

#
# Set debug level
#
dflag=$(( dflag || dbg ))

#
# Set log level
#
lflag=$(( lflag || log ))

#
# Standard Prologue END
#######################

######################
# Main Algorithm BEGIN
#
# Receive installation paths from installer script
FPOSS_BASE_PATH=/opt/local/libexec/eso/lib/fposs
BINPATH=/opt/local/libexec/eso/lib/fposs/bin
LIBPATH=/opt/local/libexec/eso/lib/fposs/lib
BIN_PATH=`echo ${BINPATH} | sed s/[/]/\\\\\\\\\\\\\\\\\\\\\\\\\\\//g`
LIB_PATH=`echo ${LIBPATH} | sed s/[/]/\\\\\\\\\\\\\\\\\\\\\\\\\\\//g`

# Extract binpath if present, then add it again. This way, if it is not present
# then it gets added, else it is not copied. Same for libpath.
CMD="echo $PATH | `echo $BIN_PATH: | awk '{printf(\"sed -e s/%s//g\",$1)}'`"
Debug "CMD BIN = >$CMD<"
SUBPATH=`eval $CMD`
Debug "SUBPATH BIN = >$SUBPATH<"
export PATH=$BINPATH:$SUBPATH

if [ -n "$SHLIB_PATH" ]; then
    # HP-UX library path
    LD_LIBRARY_PATH="$SHLIB_PATH"
fi

CMD="echo $LD_LIBRARY_PATH | `echo $LIB_PATH: | awk '{printf(\"sed -e s/%s//g\",$1)}'`"
Debug "CMD LIB = >$CMD<"
SUBPATH=`eval $CMD`
Debug "SUBPATH LIB = >$SUBPATH<"
export LD_LIBRARY_PATH=$LIBPATH:$SUBPATH
export SHLIB_PATH=$LIBPATH:$SUBPATH

export HOST=$ws
export TCLLIBPATH=@@TCLPATH@@
export VLTROOT=@@VLTPATH@@
export INSROOT=@@INSPATH@@
export INSUSER=.fposs
export FPOSSROOT=$FPOSS_BASE_PATH
export ISF_DIR=$FPOSS_BASE_PATH/../config
export PREP_DIR=$HOME/$INSUSER/PREP
export LOG_DIR=$HOME/$INSUSER/LOG
export SET_DIR=$HOME/$INSUSER/SET
export TCS_ENVNAME=wt0tcs
unset MODROOT

LDD=`ldd -d $FPOSS_BASE_PATH/lib/configure 2>/dev/null`

if [ $dflag -eq 1 ]; then
    Debug "HOST             = >$HOST<"
    Debug "FPOSS_BASE_PATH  = >$FPOSS_BASE_PATH<"
    Debug "TCLLIBPATH       = >$TCLLIBPATH<"
    Debug "VLTROOT          = >$VLTROOT<"
    Debug "TCLROOT          = >$TCLTK_ROOT<"
    Debug "INSROOT          = >$INSROOT<"
    Debug "INSUSER          = >$INSUSER<"
    Debug "PREP_DIR         = >$PREP_DIR<"
    Debug "LOG_DIR          = >$LOG_DIR<"
    Debug "SET_DIR          = >$SET_DIR<"
    Debug "PATH             = >$PATH<"
    Debug "LD_LIBRARY_PATH  = >$LD_LIBRARY_PATH<"
    Debug "SHLIB_PATH       = >$SHLIB_PATH<"
    Debug "Library linkage in 'configure': "
    Debug "$LDD"
fi

# Verification of environment
for d in $PREP_DIR $LOG_DIR $SET_DIR; do
    test -d $d || mkdir -p $d > /dev/null 2>&1
    # 20160623: JPR asked to remove chmod
    # chmod -f 777 $d > /dev/null 2>&1
done

if [ $eflag -ne 0 ]; then
    Exit $eflag
fi

# Find out component versions
if [ -a $FPOSS_BASE_PATH/doc/fposs_svn.txt ]; then
    SVN_INFO=`cat $FPOSS_BASE_PATH/doc/fposs_svn.txt`
else
    SVN_INFO="Path: unknown URL: unknown Repository Root: unknown Repository UUID: 00000000-0000-0000-0000-000000000000 Revision: 000000 Node Kind: directory Schedule: normal Last Changed Author: unknown Last Changed Rev: 000000 Last Changed Date: 0000-00-00 00:00:00 +0000 (Dow, 00 Mon 0000)"
fi

if [ -z "$MOD_VERSION" ]; then
    MOD_DATE=`echo ${SVN_INFO/Path:*Last Changed Date: /} | awk '{printf("%s", $1)}'`
    MOD_TIME=`echo ${SVN_INFO/Path:*Last Changed Date: /} | awk '{printf("%s", $2)}'`
    MOD_VERSION=`echo ${SVN_INFO/Path:*Last Changed Rev: /} | awk '{printf("%s", $1)}'`
    export MOD_VERSION
    Debug "MOD_VERSION = >$MOD_VERSION<"
fi
M_V=${MOD_VERSION:-000000}
line1=`echo "" | awk -v MV=$M_V '{ printf("revision     :   %8s \n", MV) }'`
line2=`echo "" | awk -v MD=${MOD_DATE:-0000-00-00} -v MT=${MOD_TIME:-00:00:00} '{ printf("last checkin : %10s GD %8s UT", MD, MT) }'`

echo
Frame
Banner "  W E L C O M E     T O     F P O S S"
Banner " ------------------------------------"
Filler
Banner "  $line1"
Banner "  $line2"
Banner "  now is       : `date -u +%Y-%m-%d` GD `date -u +%H:%M:%S` UT"
# Banner "  INPUT : $PREP_DIR"
# Banner "  OUTPUT: $SET_DIR"
# Banner "  LOGS  : $LOG_DIR"
Frame
echo
# Restore ^Z signal
trap SIGTSTP
test ! -d $FPOSS_BASE_PATH && Error "configure tool failed." && Exit $EC_ABORT
$FPOSS_BASE_PATH/lib/configure -- $CFGOPTS ${utdate:+-utdate "${UTDATE}"} &
#
# Main Algorithm END
####################

#########################
# Standard Epilogue BEGIN
#
Exit $EC_QUIET
#
# Standard Epilogue END
#######################
#
#___FPOSS___
