# ===========================================================================
# File: convert.tcl
# Target: mediawiki
#                        Created: 2010-10-14 09:51:41
#              Last modification: 2013-12-23 17:34:34
# Author: Bernard Desgraupes
# e-mail: <bdesgraupes@users.sourceforge.net>
# (c) Copyright: Bernard Desgraupes 2010-2013
# All rights reserved.
# Description: Aida callbacks for target mediawiki
# ===========================================================================


namespace eval mediawiki {
	variable styleDepth 0

	# Ensure fallback on base commands
	namespace path ::base
	
}


# Hooks
# -----

proc mediawiki::preConvertHook {} {}
proc mediawiki::postConvertHook {} {}

proc mediawiki::splitHook {file} {}


# Callbacks
# ---------


## 
 # ------------------------------------------------------------------------
 # 
 # "mediawiki::anchorProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc mediawiki::anchorProc {label} {
	set label [string trim $label "\"' "]
	return "<span id=\"$label\"></span>"
}


## 
 # ------------------------------------------------------------------------
 # 
 # "mediawiki::commentProc" --
 # 
 # ------------------------------------------------------------------------
 ##

proc mediawiki::commentProc {str} {
	return "<!-- $str -->"
}



## 
 # ------------------------------------------------------------------------
 # 
 # "mediawiki::horizRuleProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc mediawiki::horizRuleProc {} {
	return "----"
}


##
 # ------------------------------------------------------------------------
 # 
 # "mediawiki::imageProc" --
 # 
 # See http://www.mediawiki.org/wiki/PmWiki/Images
 # 
 # ------------------------------------------------------------------------
 ##
proc mediawiki::imageProc {str attr} {
	set attrDict [aida::getAttr img $attr]
	set result "\[\[File:"
	append result [string trim $str]
	set attributes [list]
	set dims ""
	if {![catch {dict get $attrDict width} wd]} {
		append dims "${wd}"
	} 
	if {![catch {dict get $attrDict height} ht]} {
		append dims "x${ht}px"
	} elseif {$dims ne ""} {
		append dims "px"
	}
	if {$dims ne ""} {
		append result "|$dims"
	} 
	if {![catch {dict get $attrDict alt} txt]} {
		append result "|alt=\"$txt\""
	} 
	if {![catch {dict get $attrDict align} hl]} {
		append result "|${hl}"
	} 
	if {![catch {dict get $attrDict valign} vl]} {
		append result "|${vl}"
	} 
	append result "\]\]"
	return $result
}


##
 # ------------------------------------------------------------------------
 # 
 # "mediawiki::linkProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc mediawiki::linkProc {str url} {
	set str [string trim $str]
	if {$str eq ""} {
		return $url
	} else {
		return "\[$url $str\]"
	} 
}


## 
 # ------------------------------------------------------------------------
 # 
 # "mediawiki::listProc" --
 # 
 # The 'start' and 'type' attributes seem not to be feasible with the
 # Mediawiki syntax.
 # 
 # ------------------------------------------------------------------------
 ##
proc mediawiki::listProc {kind depth attr itemList} {
	set block [list]
	
	switch -- $kind {
		"ol" {set head "#"}
		"ul" {set head "*"}
		"dl" {set head ":"}
	}

	for {set i 0} {$i < [llength $itemList]} {incr i} {
		set itm [lindex $itemList $i]
		if {$kind eq "dl"} {
			lappend block "; [lindex $itm 0]\n[mediawiki::_wrapListItem [string trimleft [lindex $itm 1]] $i $head]"
		} else {
			lappend block "[mediawiki::_wrapListItem [string trimleft $itm] $i $head]"
		} 
	}
	
	return [join $block "\n"]
}


## 
 # ------------------------------------------------------------------------
 # 
 # "mediawiki::navBarProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc mediawiki::navBarProc {curr prev next top} {
	if {[aida::getParam NavBar]} {
		set result [list]
		
		if {$curr ne $top} {
			set top "[mediawiki::_pageName $top]"
			lappend result "\[\[$top| [aida::getParam NavTop]\]\]"		
		} 
		if {$prev ne ""} {
			set prev "[mediawiki::_pageName $prev]"
			lappend result "\[\[$prev| [aida::getParam NavPrev]\]\]"
		} 
		if {$next ne ""} {
			set next "[mediawiki::_pageName $next]"
			lappend result "\[\[$next| [aida::getParam NavNext]\]\]"
		} 

		return "[join $result " | "]\n\n"		
	} else {
		return ""
	} 
}


## 
 # ------------------------------------------------------------------------
 # 
 # "mediawiki::newLineProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc mediawiki::newLineProc {} {
	return "<br/>"
}


## 
 # ------------------------------------------------------------------------
 # 
 # "mediawiki::postambleProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc mediawiki::postambleProc {} {}


## 
 # ------------------------------------------------------------------------
 # 
 # "mediawiki::preambleProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc mediawiki::preambleProc {} {
	set result [list]
	lappend result "__NOTOC__"
	
	# Finally look for preamble data
	set result [concat $result [aida::addPreamble]] 
	
	return [join $result "\n"]
}


##
 # ------------------------------------------------------------------------
 # 
 # "mediawiki::printIndexProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc mediawiki::printIndexProc {} {
	variable aida_head
	
	for {set idx 0} {$idx < [aida::countIndexMarks]} {incr idx} {
		lassign [aida::getIndexMark $idx] fl str
		incr countArr($str)
		set tag "\[\[${fl}#[aida::getParam IndexMark]${idx}| $countArr($str)\]\]"
		lappend indexArr($str) $tag			
	} 
	if {[info exists indexArr]} {
		foreach word [lsort -dict [array names indexArr]] {
			lappend result "* $word\t[join $indexArr($word) ", "]"
		} 
	} 

	return [join $result "\n"]
}


##
 # ------------------------------------------------------------------------
 # 
 # "mediawiki::refProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc mediawiki::refProc {str label {file ""}} {
	set label [string trim $label "\"'"]
	set str [aida::unwrapText $str]
	set page [mediawiki::_pageName $file]
	return "\[\[$page#$label |[string trim $str]\]\]"
}


## 
 # ------------------------------------------------------------------------
 # 
 # "mediawiki::sectionProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc mediawiki::sectionProc {str level {file ""}} {
	variable aida_head
	
	set delim [string repeat "=" $level]
	set title [aida::newSectionNumber $level]
	append title [aida::unwrapText [string trim $str]]
	set sc [aida::setSectionMark $level [mediawiki::_pageName $file] $title]
	set label "<span id=\"[aida::getParam SectionMark]$sc\"></span>"

	return "$label\n$delim $title $delim"
}


## 
 # ------------------------------------------------------------------------
 # 
 # "mediawiki::setIndexProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc mediawiki::setIndexProc {str {file ""}} {
	variable aida_head
	
	set idx [aida::setIndexMark [mediawiki::_pageName $file] $str]
	set mark "<span id=\"[aida::getParam IndexMark]$idx\"></span>"
	
	return $mark
}


## 
 # ------------------------------------------------------------------------
 # 
 # "mediawiki::styleProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc mediawiki::styleProc {style begin} {
	global aida_unwrap
	variable styleDepth
	
	if {$begin} {
		incr styleDepth
	} else {
		incr styleDepth -1
	}
	# Mediawiki does not support style tags across lines, so temporarily turn
	# on line unwrapping: it has the effect of transforming the end-of-line
	# symbols to a space
	set aida_unwrap [expr {$styleDepth > 0}]

	switch -- $style {
		"i" {
			set macro "''"
		}
		"b" {
			set macro "'''"
		}
		"u" {
			if {$begin} {
				set macro "<u>"
			} else {
				set macro "</u>"
			} 
		}
		"y" {
			if {$begin} {
				set macro "<tt>"
			} else {
				set macro "</tt>"
			} 
		}
	}
	return $macro
}


## 
 # ------------------------------------------------------------------------
 # 
 # "mediawiki::tableProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc mediawiki::tableProc {attr rowList} {
	set attrDict [aida::getAttr table $attr]
	
	if {[catch {dict get $attrDict format} frmt]} {
		set frmt "" 
	} 
	regsub -all {\|} $frmt "" posList
	set posList [split $posList ""]

	set tbl [list]
	lappend tbl "{|[aida::dictToAttrString $attrDict]"
	foreach row $rowList {
		set row [split $row "\t"]
		lappend tbl "|-"
		set cnt 0
		foreach cell $row {
			set pos [lindex $posList $cnt]
			switch -- $pos {
				"l" {
					 set otag "|align=\"left\"| "
				}
				"r" {
					 set otag "|align=\"right\"| "
				}
				"c" {
					set otag "|align=\"center\"| "
				}
				default {
					set otag "| "
				}
			}
			lappend tbl "$otag$cell"
			incr cnt
		} 
	} 
	lappend tbl "|}"

	return [join $tbl "\n"]
}


##
 # ------------------------------------------------------------------------
 # 
 # "mediawiki::tocProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc mediawiki::tocProc {} {
	set native [aida::getParam NativeToc mediawiki]
	
	if {$native} {
		return "__TOC__"
	} else {
		variable aida_head
				
		set result [list]
		set depth [aida::getParam TocDepth]
		
		for {set sc 0} {$sc < [aida::countSectionMarks]} {incr sc} {
			lassign [aida::getSectionMark $sc] lv fl title
			if {$lv <= $depth} {
				lappend result "[string repeat "*" $lv] \[\[${fl}#[aida::getParam SectionMark]$sc| $title\]\]"
			} 
		} 
		return [join $result "\n"]
	} 
}


##
 # ------------------------------------------------------------------------
 # 
 # "mediawiki::verbProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc mediawiki::verbProc {str} {
	return "<code><nowiki>[aida::unwrapText $str]</nowiki></code>"
}


##
 # ------------------------------------------------------------------------
 # 
 # "mediawiki::verbatimProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc mediawiki::verbatimProc {str} {
	return "<pre>\n$str\n</pre>"
}



# Target specific utility procs
# =============================


## 
 # ------------------------------------------------------------------------
 # 
 # "mediawiki::_wrapListItem" --
 # 
 # ------------------------------------------------------------------------
 ##
proc mediawiki::_wrapListItem {item num head} {
	set result [list]
	set lines [split $item "\n"]
	set block [list]
	
	for {set i 0} {$i < [llength $lines]} {incr i} {
		set next [lindex $lines $i]
		
		if {[regexp {^[*#;:]+} $next]} {
			if {[llength $block] > 0} {
				lappend result "$head [aida::unwrapText [join $block "\n"]]"
			} 
			lappend result "$head$next"
			set block [list]
		} else {
			lappend block $next
		}
	}
	if {[llength $block] > 0} {
		lappend result "$head [aida::unwrapText [join $block "\n"]]"
	} 
	
	return [join $result "\n"]
}


## 
 # ------------------------------------------------------------------------
 # 
 # "mediawiki::_pageName" --
 # 
 # ------------------------------------------------------------------------
 ##
proc mediawiki::_pageName {file} {
	set ext [aida::getParam NavExtension]
	set result "[file root $file]$ext"
	
	return $result
}



