# ===========================================================================
# File: convert.tcl
#                        Created: 2010-08-29 09:51:41
#              Last modification: 2016-10-04 20:41:38
# Author: Bernard Desgraupes
# e-mail: <bdesgraupes@users.sourceforge.net>
# Copyright (c) 2010-2016 Bernard Desgraupes
# All rights reserved.
# Description: default Aida settings
# ===========================================================================

namespace eval base {
}


# Hooks
# -----

proc base::preConvertHook {} {}
proc base::postConvertHook {} {}
proc base::splitHook {file} {}

# Callbacks
# ---------

## 
 # ------------------------------------------------------------------------
 # 
 # "base::anchorProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc base::anchorProc {label} {}


## 
 # ------------------------------------------------------------------------
 # 
 # "base::commentProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc base::commentProc {str} {
	return "> $str"
}


## 
 # ------------------------------------------------------------------------
 # 
 # "base::horizRuleProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc base::horizRuleProc {} {}


##
 # ------------------------------------------------------------------------
 # 
 # "base::imageProc" --
 # 
 # Build a text for an image: if the 'alt' attribute is specified, use it,
 # otherwise just display the name of the image.
 # 
 # ------------------------------------------------------------------------
 ##
proc base::imageProc {str attr} {
	set attrDict [aida::getAttr img $attr]
	if {[catch {dict get $attrDict alt} txt]} {
		set txt "image: $str" 
	} 
	return "\[$txt\]"
}


##
 # ------------------------------------------------------------------------
 # 
 # "base::linkProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc base::linkProc {str url} {
	set str [string trim $str]
	if {$str eq $url} {
		return "<$url>"
	} else {
		return "[string trim $str] <$url>"
	} 
}


## 
 # ------------------------------------------------------------------------
 # 
 # "base::listProc" --
 # 
 # Build a default list.
 # 
 # The type argument is: ol, ul, dl
 # 
 # The supported list attributes are: 
 # 		start	-> ordered list
 # 		type	-> ordered list: 1 A a I i
 # 				-> unordered list: disc square circle
 # 
 # TODO: the roman numbering is not supported
 # 
 # ------------------------------------------------------------------------
 ##
proc base::listProc {kind depth attr itemList} {
	set block [list]
	
	set attrDict [aida::getAttr $kind $attr]
	if {[catch {dict get $attrDict start} cnt]} {
		set cnt 1
	} 
	if {[catch {dict get $attrDict type} tp]} {
		set tp "1"
	} 
	set indent [string repeat "\t" $depth]

	foreach itm $itemList {
		if {$kind eq "dl"} {
			lappend block "\n$indent[lindex $itm 0]\n$indent[lindex $itm 1]"
		} elseif {$kind eq "ul"} {
			if {[catch {dict get $attrDict type} tp]} {
				set tp "circle"
			} 
			switch -- $tp {
				"circle" {
					set prfx "o "
				}
				"disc" {
					set prfx "\u2022 "
				}
				"square" {
					set prfx "* "
				}
				default {
					error "unknown type '$tp'"
				}
			}
			lappend block "\n$indent$prfx$itm"
		} else {
			set prfx "$cnt. "
			lappend block "\n$indent$prfx$itm"
			incr cnt
		} 
	} 
	
	return [join $block "\n"]
}


## 
 # ------------------------------------------------------------------------
 # 
 # "base::navBarProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc base::navBarProc {curr prev next top} {}


## 
 # ------------------------------------------------------------------------
 # 
 # "base::newLineProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc base::newLineProc {} {
	return "\n"
}


## 
 # ------------------------------------------------------------------------
 # 
 # "base::postambleProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc base::postambleProc {} {}


## 
 # ------------------------------------------------------------------------
 # 
 # "base::preambleProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc base::preambleProc {} {
	return [aida::addPreamble]
}


##
 # ------------------------------------------------------------------------
 # 
 # "base::printIndexProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc base::printIndexProc {} {
	set result [list] 
	for {set idx 0} {$idx < [aida::countIndexMarks]} {incr idx} {
		lassign [aida::getIndexMark $idx] fl str
		if {[info exist indexArr($str)]} {
			lappend indexArr($str) $idx
		} else {
			set indexArr($str) $idx
		} 
	} 
	if {[info exists indexArr]} {
		foreach word [lsort -dict [array names indexArr]] {
			lappend result "$word\t[join $indexArr($word) ", "]"
		} 
	} 
	return [join $result "\n"]
}


##
 # ------------------------------------------------------------------------
 # 
 # "base::refProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc base::refProc {str label {file ""}} {
	return [string trim $str]
}


## 
 # ------------------------------------------------------------------------
 # 
 # "base::sectionProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc base::sectionProc {str level {file ""}} {
	set title [aida::unwrapText [string trim $str]]
	aida::setSectionMark $level $file $title
	return $title
}


## 
 # ------------------------------------------------------------------------
 # 
 # "base::setIndexProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc base::setIndexProc {str {file ""}} {
	 # Put a unique anchor to mark the indexed string
	aida::setIndexMark $file $str
	return ""
}


## 
 # ------------------------------------------------------------------------
 # 
 # "base::styleProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc base::styleProc {style begin} {
	switch -- $style {
		"i" - "u" {
			set mark [aida::getParam ItalMark]
		}
		"b" {
			set mark [aida::getParam BoldMark]
		}
		"y" {
			set mark [aida::getParam TTMark]
		}
	}
	
	return $mark
}


## 
 # ------------------------------------------------------------------------
 # 
 # "base::tableProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc base::tableProc {attr rowList} {
	set pgwd [aida::getParam PageWidth]
	set attrDict [aida::getAttr table $attr]
	if {[catch {dict get $attrDict align} dir]} {
		set dir "left"
	}
	set pos [string index $dir 0]
	set tbl [base::_buildTableRows $attr $rowList]
	
	set result [list]
	foreach row $tbl {
		if {$pos eq "l"} {
			lappend result $row
		} else {
			lappend result [aida::alignString $row $pgwd $pos]
		} 
	} 
	
	return [join $result "\n"]
}


##
 # ------------------------------------------------------------------------
 # 
 # "base::tocProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc base::tocProc {} {

	set result [list]
	set depth [aida::getParam TocDepth ""]
	
	for {set sc 0} {$sc < [aida::countSectionMarks]} {incr sc} {
		lassign [aida::getSectionMark $sc] lv fl title
		if {$lv <= $depth} {
			lappend result "[string repeat "  " $lv]$title"
		} 
	} 
	return [join $result "\n"]
}


##
 # ------------------------------------------------------------------------
 # 
 # "base::verbProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc base::verbProc {str} {
	return $str
}


## 
 # ------------------------------------------------------------------------
 # 
 # "base::verbatimProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc base::verbatimProc {str} {
	return $str
}



# Base utility procs
# ==================

##
 # ------------------------------------------------------------------------
 # 
 # "base::defaultExtension" --
 # 
 # ------------------------------------------------------------------------
 ##
proc base::defaultExtension {} {
	global aida_target

	set ext ""
	if {![catch {aida::getParam Extension} res]} {
		set ext $res
	} elseif {$aida_target ne ""} {
		set ext ".$aida_target"
	} 
	
	return $ext
}


## 
 # ------------------------------------------------------------------------
 # 
 # "base::_buildTableRows" --
 # 
 # Build table rows. It supports the border and format attributes. The align
 # attribute must be handled by the caller. Return a list of rows.
 # Used by text and man targets.
 # 
 # ------------------------------------------------------------------------
 ##
proc base::_buildTableRows {attr rowList {prfx ""}} {
	set tbl [list]
	set splitRows [list]
	set attrDict [aida::getAttr table $attr]
	if {[catch {dict get $attrDict border} bord]} {
		set bord 1
	} 
	
	# Calc the width of each columns
	foreach row $rowList {
		set row [split $row "\t"]
		lappend splitRows $row
		set cnum 0
		foreach cell $row {
			set clen [string length $cell]
			if {![info exists colwd($cnum)] || $clen > $colwd($cnum)} {
				set colwd($cnum) $clen 
			} 
			incr cnum
		} 
	} 
	
	# Adjust the format and the separators
	set ncol [array size colwd]
	set hsep "|"
	set lsep ""
	if {[catch {dict get $attrDict format} frmt]} {
		if {$bord} {
			set frmt "$hsep[string repeat "c$hsep" $ncol]"
		} else {
			set frmt [string repeat "c" $ncol]
		} 
	} else {
		if {$bord} {
			if {[string index $frmt 0] ne $hsep} {
				set frmt "$hsep$frmt"
			} 
			if {[string index $frmt end] ne $hsep} {
				set frmt "$frmt$hsep"
			} 
		} 
	}
	
	# Calc the total width
	set totalWidth 0
	if {[string index $frmt 0] eq $hsep} {
		set lsep $hsep
		set frmt [string range $frmt 1 end]
		incr totalWidth
	} 
	regsub -all {\|} $frmt "" alnList
	set alnList [split $alnList ""]
	regsub -all {[^\|]\|} $frmt "|" tmp
	regsub -all {[^\|]} $tmp " " tmp
	set sepList [split $tmp ""]
	for {set i 0} {$i < $ncol} {incr i} {
		incr totalWidth [expr $colwd($i) + 2]
	}
	foreach itm $sepList {
		incr totalWidth [string length $itm]
	} 
	set hrule [string repeat "-" $totalWidth] 
	
	# Build the table
	if {$bord} {lappend tbl $hrule} 
	foreach row $splitRows {
		set line "$prfx$lsep"
		for {set cnum 0} {$cnum < $ncol} {incr cnum} {
			set cell [lindex $row $cnum]
			if {$cell eq ""} {
				set cell [string repeat " " $colwd($cnum)]
			} 
			set pos [lindex $alnList $cnum]
			set sep [lindex $sepList $cnum]
			if {$pos eq ""} {set pos "c"} 
			append line " [aida::alignString $cell $colwd($cnum) $pos] $sep"
		} 
		lappend tbl $line
		if {$bord} {lappend tbl $hrule} 
	} 
	return $tbl
}




