#![doc(html_root_url = "https://docs.rs/wasm-bindgen-macro/0.2")]
#![cfg_attr(
    any(feature = "coverage", all(not(feature = "std"), feature = "atomics")),
    feature(allow_internal_unstable),
    allow(internal_features)
)]

extern crate proc_macro;

use proc_macro::TokenStream;
use quote::quote;

#[proc_macro_attribute]
#[cfg_attr(feature = "coverage", allow_internal_unstable(coverage_attribute))]
#[cfg_attr(
    all(not(feature = "std"), feature = "atomics"),
    allow_internal_unstable(thread_local)
)]
pub fn wasm_bindgen(attr: TokenStream, input: TokenStream) -> TokenStream {
    match wasm_bindgen_macro_support::expand(attr.into(), input.into()) {
        Ok(tokens) => {
            if cfg!(feature = "xxx_debug_only_print_generated_code") {
                println!("{}", tokens);
            }
            tokens.into()
        }
        Err(diagnostic) => (quote! { #diagnostic }).into(),
    }
}

/// This macro takes a JS module as input and returns a URL that can be used to
/// access it at runtime.
///
/// The module can be specified in a few ways:
/// - You can use `inline_js = "..."` to create an inline JS file.
/// - You can use `module = "/foo/bar"` to reference a file relative to the
///   root of the crate the macro is invoked in.
///
/// The returned URL can be used for things like creating workers/worklets:
/// ```no_run
/// use web_sys::Worker;
/// let worker = Worker::new(&wasm_bindgen::link_to!(module = "/src/worker.js"));
/// ```
#[proc_macro]
#[cfg_attr(feature = "coverage", allow_internal_unstable(coverage_attribute))]
pub fn link_to(input: TokenStream) -> TokenStream {
    match wasm_bindgen_macro_support::expand_link_to(input.into()) {
        Ok(tokens) => {
            if cfg!(feature = "xxx_debug_only_print_generated_code") {
                println!("{}", tokens);
            }
            tokens.into()
        }
        // This `String::clone` is here so that IDEs know this is supposed to be a
        // `String` and can keep type-checking the rest of the program even if the macro
        // fails.
        Err(diagnostic) => (quote! { String::clone(#diagnostic) }).into(),
    }
}

#[proc_macro_attribute]
#[cfg_attr(feature = "coverage", allow_internal_unstable(coverage_attribute))]
pub fn __wasm_bindgen_class_marker(attr: TokenStream, input: TokenStream) -> TokenStream {
    match wasm_bindgen_macro_support::expand_class_marker(attr.into(), input.into()) {
        Ok(tokens) => {
            if cfg!(feature = "xxx_debug_only_print_generated_code") {
                println!("{}", tokens);
            }
            tokens.into()
        }
        Err(diagnostic) => (quote! { #diagnostic }).into(),
    }
}
