// SPDX-FileCopyrightText: 2017 Anton Kochkov <xvilka@gmail.com>
// SPDX-License-Identifier: LGPL-3.0-only

#ifndef WASM_ASM_H
#define WASM_ASM_H
#include <rz_util.h>

/***
 * Core
 **/
typedef enum {

	// Control flow operators
	WASM_OP_TRAP = 0x00,
	WASM_OP_NOP,
	WASM_OP_BLOCK,
	WASM_OP_LOOP,
	WASM_OP_IF,
	WASM_OP_ELSE,
	WASM_OP_END = 0x0b,
	WASM_OP_BR,
	WASM_OP_BRIF,
	WASM_OP_BRTABLE,
	WASM_OP_RETURN,

	// Call operators
	WASM_OP_CALL = 0x10,
	WASM_OP_CALLINDIRECT,

	// Parametric operators
	WASM_OP_DROP = 0x1a,
	WASM_OP_SELECT,

	// Variable access
	WASM_OP_GETLOCAL = 0x20,
	WASM_OP_SETLOCAL,
	WASM_OP_TEELOCAL,
	WASM_OP_GETGLOBAL,
	WASM_OP_SETGLOBAL,

	// Memory-related operators
	WASM_OP_I32LOAD = 0x28,
	WASM_OP_I64LOAD,
	WASM_OP_F32LOAD,
	WASM_OP_F64LOAD,
	WASM_OP_I32LOAD8S,
	WASM_OP_I32LOAD8U,
	WASM_OP_I32LOAD16S,
	WASM_OP_I32LOAD16U,
	WASM_OP_I64LOAD8S,
	WASM_OP_I64LOAD8U,
	WASM_OP_I64LOAD16S,
	WASM_OP_I64LOAD16U,
	WASM_OP_I64LOAD32S,
	WASM_OP_I64LOAD32U,
	WASM_OP_I32STORE,
	WASM_OP_I64STORE,
	WASM_OP_F32STORE,
	WASM_OP_F64STORE,
	WASM_OP_I32STORE8,
	WASM_OP_I32STORE16,
	WASM_OP_I64STORE8,
	WASM_OP_I64STORE16,
	WASM_OP_I64STORE32,
	WASM_OP_CURRENTMEMORY,
	WASM_OP_GROWMEMORY,

	// Constants
	WASM_OP_I32CONST,
	WASM_OP_I64CONST,
	WASM_OP_F32CONST,
	WASM_OP_F64CONST,

	// Comparison operators
	WASM_OP_I32EQZ,
	WASM_OP_I32EQ,
	WASM_OP_I32NE,
	WASM_OP_I32LTS,
	WASM_OP_I32LTU,
	WASM_OP_I32GTS,
	WASM_OP_I32GTU,
	WASM_OP_I32LES,
	WASM_OP_I32LEU,
	WASM_OP_I32GES,
	WASM_OP_I32GEU,
	WASM_OP_I64EQZ,
	WASM_OP_I64EQ,
	WASM_OP_I64NE,
	WASM_OP_I64LTS,
	WASM_OP_I64LTU,
	WASM_OP_I64GTS,
	WASM_OP_I64GTU,
	WASM_OP_I64LES,
	WASM_OP_I64LEU,
	WASM_OP_I64GES,
	WASM_OP_I64GEU,
	WASM_OP_F32EQ,
	WASM_OP_F32NE,
	WASM_OP_F32LT,
	WASM_OP_F32GT,
	WASM_OP_F32LE,
	WASM_OP_F32GE,
	WASM_OP_F64EQ,
	WASM_OP_F64NE,
	WASM_OP_F64LT,
	WASM_OP_F64GT,
	WASM_OP_F64LE,
	WASM_OP_F64GE,

	// Numeric operators
	WASM_OP_I32CLZ,
	WASM_OP_I32CTZ,
	WASM_OP_I32POPCNT,
	WASM_OP_I32ADD,
	WASM_OP_I32SUB,
	WASM_OP_I32MUL,
	WASM_OP_I32DIVS,
	WASM_OP_I32DIVU,
	WASM_OP_I32REMS,
	WASM_OP_I32REMU,
	WASM_OP_I32AND,
	WASM_OP_I32OR,
	WASM_OP_I32XOR,
	WASM_OP_I32SHL,
	WASM_OP_I32SHRS,
	WASM_OP_I32SHRU,
	WASM_OP_I32ROTL,
	WASM_OP_I32ROTR,
	WASM_OP_I64CLZ,
	WASM_OP_I64CTZ,
	WASM_OP_I64POPCNT,
	WASM_OP_I64ADD,
	WASM_OP_I64SUB,
	WASM_OP_I64MUL,
	WASM_OP_I64DIVS,
	WASM_OP_I64DIVU,
	WASM_OP_I64REMS,
	WASM_OP_I64REMU,
	WASM_OP_I64AND,
	WASM_OP_I64OR,
	WASM_OP_I64XOR,
	WASM_OP_I64SHL,
	WASM_OP_I64SHRS,
	WASM_OP_I64SHRU,
	WASM_OP_I64ROTL,
	WASM_OP_I64ROTR,
	WASM_OP_F32ABS,
	WASM_OP_F32NEG,
	WASM_OP_F32CEIL,
	WASM_OP_F32FLOOR,
	WASM_OP_F32TRUNC,
	WASM_OP_F32NEAREST,
	WASM_OP_F32SQRT,
	WASM_OP_F32ADD,
	WASM_OP_F32SUB,
	WASM_OP_F32MUL,
	WASM_OP_F32DIV,
	WASM_OP_F32MIN,
	WASM_OP_F32MAX,
	WASM_OP_F32COPYSIGN,
	WASM_OP_F64ABS,
	WASM_OP_F64NEG,
	WASM_OP_F64CEIL,
	WASM_OP_F64FLOOR,
	WASM_OP_F64TRUNC,
	WASM_OP_F64NEAREST,
	WASM_OP_F64SQRT,
	WASM_OP_F64ADD,
	WASM_OP_F64SUB,
	WASM_OP_F64MUL,
	WASM_OP_F64DIV,
	WASM_OP_F64MIN,
	WASM_OP_F64MAX,
	WASM_OP_F64COPYSIGN,

	// Conversions
	WASM_OP_I32WRAPI64,
	WASM_OP_I32TRUNCSF32,
	WASM_OP_I32TRUNCUF32,
	WASM_OP_I32TRUNCSF64,
	WASM_OP_I32TRUNCUF64,
	WASM_OP_I64EXTENDSI32,
	WASM_OP_I64EXTENDUI32,
	WASM_OP_I64TRUNCSF32,
	WASM_OP_I64TRUNCUF32,
	WASM_OP_I64TRUNCSF64,
	WASM_OP_I64TRUNCUF64,
	WASM_OP_F32CONVERTSI32,
	WASM_OP_F32CONVERTUI32,
	WASM_OP_F32CONVERTSI64,
	WASM_OP_F32CONVERTUI64,
	WASM_OP_F32DEMOTEF64,
	WASM_OP_F64CONVERTSI32,
	WASM_OP_F64CONVERTUI32,
	WASM_OP_F64CONVERTSI64,
	WASM_OP_F64CONVERTUI64,
	WASM_OP_F64PROMOTEF32,

	// Reinterpretations
	WASM_OP_I32REINTERPRETF32,
	WASM_OP_I64REINTERPRETF64,
	WASM_OP_F32REINTERPRETI32,
	WASM_OP_F64REINTERPRETI64,

} WasmOpCodes;

/***
 * Thread extension (0xFE ...)
 **/
typedef enum {

	// Wait and notify
	WASM_OP_ATOMICNOTIFY = 0x00,
	WASM_OP_I32ATOMICWAIT,
	WASM_OP_I64ATOMICWAIT,

	// Load/Store
	WASM_OP_I32ATOMICLOAD = 0x10,
	WASM_OP_I64ATOMICLOAD,
	WASM_OP_I32ATOMICLOAD8U,
	WASM_OP_I32ATOMICLOAD16U,
	WASM_OP_I64ATOMICLOAD8U,
	WASM_OP_I64ATOMICLOAD16U,
	WASM_OP_I64ATOMICLOAD32U,
	WASM_OP_I32ATOMICSTORE = 0x17,
	WASM_OP_I64ATOMICSTORE,
	WASM_OP_I32ATOMICSTORE8,
	WASM_OP_I32ATOMICSTORE16,
	WASM_OP_I64ATOMICSTORE8,
	WASM_OP_I64ATOMICSTORE16,
	WASM_OP_I64ATOMICSTORE32,

	// Read-Modify-Write
	WASM_OP_I32ATOMICRMWADD = 0x1e,
	WASM_OP_I64ATOMICRMWADD,
	WASM_OP_I32ATOMICRMW8UADD,
	WASM_OP_I32ATOMICRMW16UADD,
	WASM_OP_I64ATOMICRMW8UADD,
	WASM_OP_I64ATOMICRMW16UADD,
	WASM_OP_I64ATOMICRMW32UADD,
	WASM_OP_I32ATOMICRMW8USUB = 0x25,
	WASM_OP_I32ATOMICRMW16USUB,
	WASM_OP_I32ATOMICRMWSUB,
	WASM_OP_I64ATOMICRMW8USUB,
	WASM_OP_I64ATOMICRMW16USUB,
	WASM_OP_I64ATOMICRMW32USUB,
	WASM_OP_I64ATOMICRMWSUB,
	WASM_OP_I32ATOMICRMWAND = 0x2c,
	WASM_OP_I64ATOMICRMWAND,
	WASM_OP_I32ATOMICRMW8UAND,
	WASM_OP_I32ATOMICRMW16UAND,
	WASM_OP_I64ATOMICRMW8UAND,
	WASM_OP_I64ATOMICRMW16UAND,
	WASM_OP_I64ATOMICRMW32UAND,
	WASM_OP_I32ATOMICRMWOR = 0x33,
	WASM_OP_I64ATOMICRMWOR,
	WASM_OP_I32ATOMICRMW8UOR,
	WASM_OP_I32ATOMICRMW16UOR,
	WASM_OP_I64ATOMICRMW8UOR,
	WASM_OP_I64ATOMICRMW16UOR,
	WASM_OP_I64ATOMICRMW32UOR,
	WASM_OP_I32ATOMICRMWXOR = 0x3a,
	WASM_OP_I64ATOMICRMWXOR,
	WASM_OP_I32ATOMICRMW8UXOR,
	WASM_OP_I32ATOMICRMW16UXOR,
	WASM_OP_I64ATOMICRMW8UXOR,
	WASM_OP_I64ATOMICRMW16UXOR,
	WASM_OP_I64ATOMICRMW32UXOR,
	WASM_OP_I32ATOMICRMWXCHG = 0x41,
	WASM_OP_I64ATOMICRMWXCHG,
	WASM_OP_I32ATOMICRMW8UXCHG,
	WASM_OP_I32ATOMICRMW16UXCHG,
	WASM_OP_I64ATOMICRMW8UXCHG,
	WASM_OP_I64ATOMICRMW16UXCHG,
	WASM_OP_I64ATOMICRMW32UXCHG,

	// Compare exchange
	WASM_OP_I32ATOMICRMWCMPXCHG = 0x48,
	WASM_OP_I64ATOMICRMWCMPXCHG,
	WASM_OP_I32ATOMICRMW8UCMPXCHG,
	WASM_OP_I32ATOMICRMW16UCMPXCHG,
	WASM_OP_I64ATOMICRMW8UCMPXCHG,
	WASM_OP_I64ATOMICRMW16UCMPXCHG,
	WASM_OP_I64ATOMICRMW32UCMPXCHG,

} WasmOpAtomicCodes;

/***
 * SIMD extension (0xFD ...)
 **/
typedef enum {
	// load ops
	WASM_OP_V128LOAD = 0x00,
	WASM_OP_I16X8LOAD8X8S = 0x01,
	WASM_OP_I16X8LOAD8X8U = 0x02,
	WASM_OP_I32X4LOAD16X4S = 0x03,
	WASM_OP_I32X4LOAD16X4U = 0x04,
	WASM_OP_I64X2LOAD32X2S = 0x05,
	WASM_OP_I64X2LOAD32X2U = 0x06,
	WASM_OP_V8X16LOADSPLAT = 0x07,
	WASM_OP_V16X8LOADSPLAT = 0x08,
	WASM_OP_V32X4LOADSPLAT = 0x09,
	WASM_OP_V64X2LOADSPLAT = 0x0a,
	WASM_OP_V128STORE = 0x0b,
	WASM_OP_V128CONST = 0x0c,

	// lane ops
	WASM_OP_V8X16SHUFFLE = 0x0d,
	WASM_OP_V8X16SWIZZLE = 0x0e,
	WASM_OP_I8X16SPLAT = 0x0f,
	WASM_OP_I16X8SPLAT = 0x10,
	WASM_OP_I32X4SPLAT = 0x11,
	WASM_OP_I64X2SPLAT = 0x12,
	WASM_OP_F32X4SPLAT = 0x13,
	WASM_OP_F64X2SPLAT = 0x14,
	WASM_OP_I8X16EXTRACTLANES = 0x15,
	WASM_OP_I8X16EXTRACTLANEU = 0x16,
	WASM_OP_I8X16REPLACELANE = 0x17,
	WASM_OP_I16X8EXTRACTLANES = 0x18,
	WASM_OP_I16X8EXTRACTLANEU = 0x19,
	WASM_OP_I16X8REPLACELANE = 0x1a,
	WASM_OP_I32X4EXTRACTLANE = 0x1b,
	WASM_OP_I32X4REPLACELANE = 0x1c,
	WASM_OP_I64X2EXTRACTLANE = 0x1d,
	WASM_OP_I64X2REPLACELANE = 0x1e,
	WASM_OP_F32X4EXTRACTLANE = 0x1f,
	WASM_OP_F32X4REPLACELANE = 0x20,
	WASM_OP_F64X2EXTRACTLANE = 0x21,
	WASM_OP_F64X2REPLACELANE = 0x22,

	// cmp ops
	WASM_OP_I8X16EQ = 0x23,
	WASM_OP_I8X16NE = 0x24,
	WASM_OP_I8X16LTS = 0x25,
	WASM_OP_I8X16LTU = 0x26,
	WASM_OP_I8X16GTS = 0x27,
	WASM_OP_I8X16GTU = 0x28,
	WASM_OP_I8X16LES = 0x29,
	WASM_OP_I8X16LEU = 0x2a,
	WASM_OP_I8X16GES = 0x2b,
	WASM_OP_I8X16GEU = 0x2c,
	WASM_OP_I16X8EQ = 0x2d,
	WASM_OP_I16X8NE = 0x2e,
	WASM_OP_I16X8LTS = 0x2f,
	WASM_OP_I16X8LTU = 0x30,
	WASM_OP_I16X8GTS = 0x31,
	WASM_OP_I16X8GTU = 0x32,
	WASM_OP_I16X8LES = 0x33,
	WASM_OP_I16X8LEU = 0x34,
	WASM_OP_I16X8GES = 0x35,
	WASM_OP_I16X8GEU = 0x36,
	WASM_OP_I32X4EQ = 0x37,
	WASM_OP_I32X4NE = 0x38,
	WASM_OP_I32X4LTS = 0x39,
	WASM_OP_I32X4LTU = 0x3a,
	WASM_OP_I32X4GTS = 0x3b,
	WASM_OP_I32X4GTU = 0x3c,
	WASM_OP_I32X4LES = 0x3d,
	WASM_OP_I32X4LEU = 0x3e,
	WASM_OP_I32X4GES = 0x3f,
	WASM_OP_I32X4GEU = 0x40,
	WASM_OP_F32X4EQ = 0x41,
	WASM_OP_F32X4NE = 0x42,
	WASM_OP_F32X4LT = 0x43,
	WASM_OP_F32X4GT = 0x44,
	WASM_OP_F32X4LE = 0x45,
	WASM_OP_F32X4GE = 0x46,
	WASM_OP_F64X2EQ = 0x47,
	WASM_OP_F64X2NE = 0x48,
	WASM_OP_F64X2LT = 0x49,
	WASM_OP_F64X2GT = 0x4a,
	WASM_OP_F64X2LE = 0x4b,
	WASM_OP_F64X2GE = 0x4c,

	// bitwise
	WASM_OP_V128NOT = 0x4d,
	WASM_OP_V128AND = 0x4e,
	WASM_OP_V128ANDNOT = 0x4f,
	WASM_OP_V128OR = 0x50,
	WASM_OP_V128XOR = 0x51,
	WASM_OP_V128BITSELECT = 0x52,

	// arithmetic
	WASM_OP_I8X16ABS = 0x60,
	WASM_OP_I8X16NEG = 0x61,
	WASM_OP_I8X16ANYTRUE = 0x62,
	WASM_OP_I8X16ALLTRUE = 0x63,
	WASM_OP_I8X16BITMASK = 0x64,
	WASM_OP_I8X16NARROWI16X8S = 0x65,
	WASM_OP_I8X16NARROWI16X8U = 0x66,
	WASM_OP_I8X16SHL = 0x6b,
	WASM_OP_I8X16SHRS = 0x6c,
	WASM_OP_I8X16SHRU = 0x6d,
	WASM_OP_I8X16ADD = 0x6e,
	WASM_OP_I8X16ADDSATURATES = 0x6f,
	WASM_OP_I8X16ADDSATURATEU = 0x70,
	WASM_OP_I8X16SUB = 0x71,
	WASM_OP_I8X16SUBSATURATES = 0x72,
	WASM_OP_I8X16SUBSATURATEU = 0x73,
	WASM_OP_I8X16MINS = 0x76,
	WASM_OP_I8X16MINU = 0x77,
	WASM_OP_I8X16MAXS = 0x78,
	WASM_OP_I8X16MAXU = 0x79,
	WASM_OP_I8X16AVGRU = 0x7b,
	WASM_OP_I16X8ABS = 0x80,
	WASM_OP_I16X8NEG = 0x81,
	WASM_OP_I16X8ANYTRUE = 0x82,
	WASM_OP_I16X8ALLTRUE = 0x83,
	WASM_OP_I16X8NARROWI32X4S = 0x85,
	WASM_OP_I16X8NARROWI32X4U = 0x86,
	WASM_OP_I16X8WIDENLOWI8X16S = 0x87,
	WASM_OP_I16X8WIDENHIGHI8X16S = 0x88,
	WASM_OP_I16X8WIDENLOWI8X16U = 0x89,
	WASM_OP_I16X8WIDENHIGHI8X16U = 0x8a,
	WASM_OP_I16X8SHL = 0x8b,
	WASM_OP_I16X8SHRS = 0x8c,
	WASM_OP_I16X8SHRU = 0x8d,
	WASM_OP_I16X8ADD = 0x8e,
	WASM_OP_I16X8ADDSATURATES = 0x8f,
	WASM_OP_I16X8ADDSATURATEU = 0x90,
	WASM_OP_I16X8SUB = 0x91,
	WASM_OP_I16X8SUBSATURATES = 0x92,
	WASM_OP_I16X8SUBSATURATEU = 0x93,
	WASM_OP_I16X8MUL = 0x95,
	WASM_OP_I16X8MINS = 0x96,
	WASM_OP_I16X8MINU = 0x97,
	WASM_OP_I16X8MAXS = 0x98,
	WASM_OP_I16X8MAXU = 0x99,
	WASM_OP_I16X8AVGRU = 0x9b,
	WASM_OP_I32X4ABS = 0xa0,
	WASM_OP_I32X4NEG = 0xa1,
	WASM_OP_I32X4ANYTRUE = 0xa2,
	WASM_OP_I32X4ALLTRUE = 0xa3,
	WASM_OP_I32X4WIDENLOWI16X8S = 0xa7,
	WASM_OP_I32X4WIDENHIGHI16X8S = 0xa8,
	WASM_OP_I32X4WIDENLOWI16X8U = 0xa9,
	WASM_OP_I32X4WIDENHIGHI16X8U = 0xaa,
	WASM_OP_I32X4SHL = 0xab,
	WASM_OP_I32X4SHRS = 0xac,
	WASM_OP_I32X4SHRU = 0xad,
	WASM_OP_I32X4ADD = 0xae,
	WASM_OP_I32X4SUB = 0xb1,
	WASM_OP_I32X4MUL = 0xb5,
	WASM_OP_I32X4MINS = 0xb6,
	WASM_OP_I32X4MINU = 0xb7,
	WASM_OP_I32X4MAXS = 0xb8,
	WASM_OP_I32X4MAXU = 0xb9,
	WASM_OP_I64X2NEG = 0xc1,
	WASM_OP_I64X2SHL = 0xcb,
	WASM_OP_I64X2SHRS = 0xcc,
	WASM_OP_I64X2SHRU = 0xcd,
	WASM_OP_I64X2ADD = 0xce,
	WASM_OP_I64X2SUB = 0xd1,
	WASM_OP_I64X2MUL = 0xd5,
	WASM_OP_F32X4ABS = 0xe0,
	WASM_OP_F32X4NEG = 0xe1,
	WASM_OP_F32X4SQRT = 0xe3,
	WASM_OP_F32X4ADD = 0xe4,
	WASM_OP_F32X4SUB = 0xe5,
	WASM_OP_F32X4MUL = 0xe6,
	WASM_OP_F32X4DIV = 0xe7,
	WASM_OP_F32X4MIN = 0xe8,
	WASM_OP_F32X4MAX = 0xe9,
	WASM_OP_F64X2ABS = 0xec,
	WASM_OP_F64X2NEG = 0xed,
	WASM_OP_F64X2SQRT = 0xef,
	WASM_OP_F64X2ADD = 0xf0,
	WASM_OP_F64X2SUB = 0xf1,
	WASM_OP_F64X2MUL = 0xf2,
	WASM_OP_F64X2DIV = 0xf3,
	WASM_OP_F64X2MIN = 0xf4,
	WASM_OP_F64X2MAX = 0xf5,

	// conversion
	WASM_OP_I32X4TRUNCSATF32X4S = 0xf8,
	WASM_OP_I32X4TRUNCSATF32X4U = 0xf9,
	WASM_OP_F32X4CONVERTI32X4S = 0xfa,
	WASM_OP_F32X4CONVERTI32X4U = 0xfb,
} WasmOpSimdCodes;

typedef enum {
	WASM_TYPE_OP_CORE,
	WASM_TYPE_OP_ATOMIC,
	WASM_TYPE_OP_SIMD,
} WasmTypeOp;

typedef struct {
	union {
		WasmOpCodes core;
		WasmOpAtomicCodes atomic;
		WasmOpSimdCodes simd;
	} op;
	WasmTypeOp type;
	int len;
	char *txt;
} WasmOp;

typedef struct {
	const char *txt;
	size_t min, max;
} WasmOpDef;

RZ_IPI int wasm_asm(const char *str, unsigned char *buf, int buf_len);
RZ_IPI int wasm_dis(WasmOp *op, const unsigned char *buf, int buf_len);

#endif
