import { IChangedArgs } from '@jupyterlab/coreutils';
import { DocumentRegistry } from '@jupyterlab/docregistry';
import * as nbformat from '@jupyterlab/nbformat';
import { IMapChange, ISharedNotebook } from '@jupyter/ydoc';
import { ITranslator } from '@jupyterlab/translation';
import { ISignal } from '@lumino/signaling';
import { CellList } from './celllist';
/**
 * The definition of a model object for a notebook widget.
 */
export interface INotebookModel extends DocumentRegistry.IModel {
    /**
     * The list of cells in the notebook.
     */
    readonly cells: CellList;
    /**
     * The major version number of the nbformat.
     */
    readonly nbformat: number;
    /**
     * The minor version number of the nbformat.
     */
    readonly nbformatMinor: number;
    /**
     * The metadata associated with the notebook.
     *
     * ### Notes
     * This is a copy of the metadata. Changing a part of it
     * won't affect the model.
     * As this returns a copy of all metadata, it is advised to
     * use `getMetadata` to speed up the process of getting a single key.
     */
    readonly metadata: nbformat.INotebookMetadata;
    /**
     * Signal emitted when notebook metadata changes.
     */
    readonly metadataChanged: ISignal<INotebookModel, IMapChange>;
    /**
     * The array of deleted cells since the notebook was last run.
     */
    readonly deletedCells: string[];
    /**
     * Shared model
     */
    readonly sharedModel: ISharedNotebook;
    /**
     * Delete a metadata
     *
     * @param key Metadata key
     */
    deleteMetadata(key: string): void;
    /**
     * Get a metadata
     *
     * ### Notes
     * This returns a copy of the key value.
     *
     * @param key Metadata key
     */
    getMetadata(key: string): any;
    /**
     * Set a metadata
     *
     * @param key Metadata key
     * @param value Metadata value
     */
    setMetadata(key: string, value: any): void;
}
/**
 * An implementation of a notebook Model.
 */
export declare class NotebookModel implements INotebookModel {
    /**
     * Construct a new notebook model.
     */
    constructor(options?: NotebookModel.IOptions);
    /**
     * A signal emitted when the document content changes.
     */
    get contentChanged(): ISignal<this, void>;
    /**
     * Signal emitted when notebook metadata changes.
     */
    get metadataChanged(): ISignal<INotebookModel, IMapChange<any>>;
    /**
     * A signal emitted when the document state changes.
     */
    get stateChanged(): ISignal<this, IChangedArgs<any>>;
    /**
     * Get the observable list of notebook cells.
     */
    get cells(): CellList;
    /**
     * The dirty state of the document.
     */
    get dirty(): boolean;
    set dirty(newValue: boolean);
    /**
     * The read only state of the document.
     */
    get readOnly(): boolean;
    set readOnly(newValue: boolean);
    /**
     * The metadata associated with the notebook.
     *
     * ### Notes
     * This is a copy of the metadata. Changing a part of it
     * won't affect the model.
     * As this returns a copy of all metadata, it is advised to
     * use `getMetadata` to speed up the process of getting a single key.
     */
    get metadata(): nbformat.INotebookMetadata;
    /**
     * The major version number of the nbformat.
     */
    get nbformat(): number;
    /**
     * The minor version number of the nbformat.
     */
    get nbformatMinor(): number;
    /**
     * The default kernel name of the document.
     */
    get defaultKernelName(): string;
    /**
     * A list of deleted cells for the notebook..
     */
    get deletedCells(): string[];
    /**
     * The default kernel language of the document.
     */
    get defaultKernelLanguage(): string;
    /**
     * Whether the model is collaborative or not.
     */
    get collaborative(): boolean;
    /**
     * Dispose of the resources held by the model.
     */
    dispose(): void;
    /**
     * Delete a metadata
     *
     * @param key Metadata key
     */
    deleteMetadata(key: string): void;
    /**
     * Get a metadata
     *
     * ### Notes
     * This returns a copy of the key value.
     *
     * @param key Metadata key
     */
    getMetadata(key: string): any;
    /**
     * Set a metadata
     *
     * @param key Metadata key
     * @param value Metadata value
     */
    setMetadata(key: string, value: any): void;
    /**
     * Serialize the model to a string.
     */
    toString(): string;
    /**
     * Deserialize the model from a string.
     *
     * #### Notes
     * Should emit a [contentChanged] signal.
     */
    fromString(value: string): void;
    /**
     * Serialize the model to JSON.
     */
    toJSON(): nbformat.INotebookContent;
    /**
     * Deserialize the model from JSON.
     *
     * #### Notes
     * Should emit a [contentChanged] signal.
     */
    fromJSON(value: nbformat.INotebookContent): void;
    /**
     * Handle a change in the cells list.
     */
    private _onCellsChanged;
    private _onMetadataChanged;
    private _onStateChanged;
    /**
     * Make sure we have the required metadata fields.
     */
    private _ensureMetadata;
    /**
     * Trigger a state change signal.
     */
    protected triggerStateChange(args: IChangedArgs<any>): void;
    /**
     * Trigger a content changed signal.
     */
    protected triggerContentChange(): void;
    /**
     * Whether the model is disposed.
     */
    get isDisposed(): boolean;
    /**
     * The shared notebook model.
     */
    readonly sharedModel: ISharedNotebook;
    /**
     * Whether the model should disposed the shared model on disposal or not.
     */
    protected standaloneModel: boolean;
    private _dirty;
    private _readOnly;
    private _contentChanged;
    private _stateChanged;
    private _trans;
    private _cells;
    private _deletedCells;
    private _isDisposed;
    private _metadataChanged;
    private _collaborationEnabled;
}
/**
 * The namespace for the `NotebookModel` class statics.
 */
export declare namespace NotebookModel {
    /**
     * An options object for initializing a notebook model.
     */
    interface IOptions extends DocumentRegistry.IModelOptions<ISharedNotebook> {
        /**
         * Default cell type.
         */
        defaultCell?: 'code' | 'markdown' | 'raw';
        /**
         * Language translator.
         */
        translator?: ITranslator;
        /**
         * Defines if the document can be undo/redo.
         *
         * Default: true
         *
         * @experimental
         * @alpha
         */
        disableDocumentWideUndoRedo?: boolean;
    }
}
