import { LanguageSupport, StreamParser } from '@codemirror/language';
import { ITranslator } from '@jupyterlab/translation';
import { IEditorLanguage, IEditorLanguageRegistry } from './token';
/**
 * CodeMirror language registry
 */
export declare class EditorLanguageRegistry implements IEditorLanguageRegistry {
    private _modeList;
    constructor();
    /**
     * Register a new language for CodeMirror
     *
     * @param language Language to register
     */
    addLanguage(language: IEditorLanguage): void;
    /**
     * Ensure a codemirror mode is available by name or Codemirror spec.
     *
     * @param language - The mode to ensure.  If it is a string, uses [findBest]
     *   to get the appropriate spec.
     *
     * @returns A promise that resolves when the mode is available.
     */
    getLanguage(language: string | IEditorLanguage): Promise<IEditorLanguage | null>;
    /**
     * Get the raw list of available modes specs.
     *
     * @returns The available modes
     */
    getLanguages(): IEditorLanguage[];
    /**
     * Find a codemirror mode by MIME.
     *
     * @param mime Mime type to look for
     * @param strict Whether xml and json should be assimilated to the generic mimetype
     * @returns The mode or null
     */
    findByMIME(mime: string | readonly string[], strict?: boolean): IEditorLanguage | null;
    /**
     * Find a codemirror mode by name.
     *
     * @param name The mode name
     * @returns The mode or null
     */
    findByName(name: string): IEditorLanguage | null;
    /**
     * Find a codemirror mode by extension.
     *
     * @param ext The extension name
     * @returns The mode or null
     */
    findByExtension(ext: string | readonly string[]): IEditorLanguage | null;
    /**
     * Find a codemirror mode by filename.
     *
     * @param name File name
     * @returns The mode or null
     */
    findByFileName(name: string): IEditorLanguage | null;
    /**
     * Find a codemirror mode by name or CodeMirror spec.
     *
     * @param language The CodeMirror mode
     * @param fallback Whether to fallback to default mimetype spec or not
     * @returns The mode or null
     */
    findBest(language: string | IEditorLanguage, fallback?: boolean): IEditorLanguage | null;
    /**
     * Parse and style a string.
     *
     * @param code Code to highlight
     * @param language Code language
     * @param el HTML element into which the highlighted code will be inserted
     */
    highlight(code: string, language: IEditorLanguage | null, el: HTMLElement): Promise<void>;
    protected makeSpec(spec: IEditorLanguage): IEditorLanguage;
}
/**
 * EditorLanguageRegistry namespace
 */
export declare namespace EditorLanguageRegistry {
    /**
     * Convert an CodeMirror 5 language parser to CodeMirror 6
     *
     * @param parser Legacy parser
     * @returns Language object
     */
    function legacy(parser: StreamParser<unknown>): LanguageSupport;
    /**
     * Get the default editor languages
     *
     * @param translator Application translator
     * @returns Default CodeMirror 6 languages
     */
    function getDefaultLanguages(translator?: ITranslator | null): ReadonlyArray<IEditorLanguage>;
}
