#!/bin/bash
#
# Creates the API documentation for igraph's Python interface
#
# Usage: ./mkdoc.sh (makes API and tutorial docs)
#        ./mkdoc.sh -d (makes a Dash docset based on standalone docs, requires doc2dash)
#
# Add -c to ensure that the documentation is built from scratch and no cached
# assets from previous builds are used.
#
# Make sure we bail out on build errors
set -e

DOC2DASH=0
LINKCHECK=0
CLEAN=0

while getopts ":cdl" OPTION; do
  case $OPTION in
  c)
    CLEAN=1
    ;;
  d)
    DOC2DASH=1
    ;;
  l)
    LINKCHECK=1
    ;;
  \?)
    echo "Usage: $0 [-cdl]"
    echo ""
    echo "-c: clean and force a full rebuild of the documentation"
    echo "-d: generate Dash docset with doc2dash"
    echo "-l: check the generated documentation for broken links"
    exit 1
    ;;
  esac
done
shift $((OPTIND - 1))

SCRIPTS_FOLDER=$(dirname $0)

cd ${SCRIPTS_FOLDER}/..
ROOT_FOLDER=$(pwd)
DOC_SOURCE_FOLDER=${ROOT_FOLDER}/doc/source
DOC_HTML_FOLDER=${ROOT_FOLDER}/doc/html
DOC_LINKCHECK_FOLDER=${ROOT_FOLDER}/doc/linkcheck
SCRIPTS_FOLDER=${ROOT_FOLDER}/scripts

cd ${ROOT_FOLDER}

# Create a virtual environment
if [ ! -d ".venv" ]; then
  echo "Creating virtualenv..."
  ${PYTHON:-python3} -m venv .venv

  # Install documentation dependencies into the venv.
  # doc2dash is optional; it will be installed when -d is given
  .venv/bin/pip install -q -U pip wheel sphinx==7.4.7 matplotlib pandas scipy pydoctor sphinx-rtd-theme iplotx
else
  # Upgrade pip in the virtualenv
  echo "Upgrading pip in virtualenv..."
  .venv/bin/pip install -q -U pip wheel
fi

# Make sure that documentation dependencies are up-to-date in the virtualenv
echo "Making sure that all dependencies are up-to-date..."
.venv/bin/pip install -q -U sphinx==7.4.7 pydoctor sphinx-gallery sphinxcontrib-jquery sphinx-rtd-theme iplotx
if [ x$DOC2DASH = x1 ]; then
  .venv/bin/pip install -U doc2dash
fi

echo "Removing existing igraph and python-igraph eggs from virtualenv..."
SITE_PACKAGES_DIR=$(.venv/bin/python3 -c 'import sysconfig; print(sysconfig.get_paths()["purelib"])')
rm -rf "${SITE_PACKAGES_DIR}"/igraph*.egg
rm -rf "${SITE_PACKAGES_DIR}"/igraph*.egg-link
rm -rf "${SITE_PACKAGES_DIR}"/python_igraph*.egg
rm -rf "${SITE_PACKAGES_DIR}"/python_igraph*.egg-link

echo "Installing igraph in virtualenv..."
rm -f dist/*.whl && .venv/bin/pip wheel -q -w dist . && .venv/bin/pip install -q --force-reinstall dist/*.whl

echo "Patching modularized Graph methods"
.venv/bin/python3 ${SCRIPTS_FOLDER}/patch_modularized_graph_methods.py

echo "Clean previous docs"
rm -rf "${DOC_HTML_FOLDER}"

if [ "x$CLEAN" = "x1" ]; then
  # This is generated by sphinx-gallery
  rm -rf "${DOC_SOURCE_FOLDER}/tutorials"
fi

if [ "x$LINKCHECK" = "x1" ]; then
  echo "Check for broken links"
  .venv/bin/python -m sphinx \
    -T \
    -b linkcheck \
    -Dtemplates_path='' \
    -Dhtml_theme='alabaster' \
    ${DOC_SOURCE_FOLDER} ${DOC_LINKCHECK_FOLDER}
fi

echo "Generating HTML documentation..."
.venv/bin/pip install -q -U sphinx-rtd-theme
.venv/bin/python -m sphinx -T -b html ${DOC_SOURCE_FOLDER} ${DOC_HTML_FOLDER}

echo "HTML documentation generated in ${DOC_HTML_FOLDER}"

# doc2dash
if [ "x$DOC2DASH" = "x1" ]; then
  PWD=$(pwd)
  # Output folder of sphinx (before Jekyll if requested)
  DOC_API_FOLDER=${ROOT_FOLDER}/doc/html/api
  DOC2DASH=.venv/bin/doc2dash
  DASH_FOLDER=${ROOT_FOLDER}/doc/dash
  if [ "x$DOC2DASH" != x ]; then
    echo "Generating Dash docset..."
    "$DOC2DASH" \
      --online-redirect-url "https://python.igraph.org/en/latest/api/" \
      --name "python-igraph" \
      -d "${DASH_FOLDER}" \
      -f \
      -j \
      -I "index.html" \
      --icon ${ROOT_FOLDER}/doc/source/icon.png \
      --icon-2x ${ROOT_FOLDER}/doc/source/icon@2x.png \
      "${DOC_API_FOLDER}"
    DASH_READY=1
  else
    echo "WARNING: doc2dash not installed, skipping Dash docset generation."
    DASH_READY=0
  fi

  echo ""
  if [ "x${DASH_READY}" = x1 ]; then
    echo "Dash docset generated in ${DASH_FOLDER}/python-igraph.docset"
  fi

  cd "$PWD"
fi
