module language_versions
  character(len=20) :: v_display   ! Display
  character(len=20) :: v_clean     ! CLEAN
  character(len=20) :: v_cal       ! CALIBRATE
  character(len=20) :: v_advanced  ! ADVANCED
  character(len=20) :: v_bundles   ! BUNDLES
  character(len=20) :: v_imager    ! IMAGER
  character(len=20) :: v_simu      ! SIMULATE
  character(len=20) :: v_program   ! UV_SHORT improvement
  integer :: imager_versi=4600     ! Imager version
  integer :: imager_patch=4500     ! last distributed version
end module language_versions
!
subroutine load_display
  use gkernel_interfaces
  use language_versions
  !----------------------------------------------------------------------
  ! @ public
  !*
  ! IMAGER & VIEWER --  Load DISPLAY\ Language
  !!
  !----------------------------------------------------------------------
  external :: run_display
  integer, parameter :: mshow=37+1+1+1+1
  character(len=16) vshow(mshow) 
  data vshow / ' CATALOG', '/SORT', '*COLOR', ' COMPARE', '/DIRECTORY',    &
      & ' EXPLORE', '/DIRECTORY', '/ADD', '/NOPAUSE',             &
      & ' EXTRACT',                                               & 
      & ' FIND', '/CHECK', '/SPECIES',                         & ! Find lines in a given range
      & ' INSPECT_3D', '/DIRECTORY', &
      & ' LOAD', '/FREQUENCY', '/PLANES', '/RANGE',  & ! READ DATA command equivalent
      & ' POPUP', &
      & ' SET', '/DEFAULT',       &  ! The /DEFAULT option is for compatibility with GreG 
      & ' SHOW', '/DIRECTORY', '/OVERLAY', '/SIDE',         &
      & ' SLICE',                 & 
      & ' SPECTRUM', '/CORNER', '/MEAN', '/PLANES', '/SUM', & 
      & ' STATISTIC', '/EDGE', '/FILE', '/NOISE', '/UPDATE',  & ! Image statistic
      & ' VIEW', '/DIRECTORY', '/NOPAUSE', '/OVERLAY' /
  logical, save :: display_loaded=.false.
  logical, parameter :: old=.false.
  !
  if (display_loaded) return
  !
  ! DISPLAY\ Language versions  v_display
  if (old) then
  v_display = '1.0-00 21-Oct-2021' ! First DISPLAY\ Language
  v_display = '1.2-00 23-Oct-2021' ! Also move EXPLORE here
  v_display = '1.3-00 23-Oct-2021' ! Also move STATISTIC here
  v_display = '1.3-01 10-Nov-2021' ! Improve CATALOG command
  v_display = '1.4-00 18-Nov-2021' ! STATISTIC /UPDATE command
  v_display = '1.4-01 23-Nov-2021' ! Allow re-entrant calls for STATISTIC
  v_display = '1.5-00 10-Jan-2022' ! LOAD command
  v_display = '1.5-01 30-Jan-2022' ! Improve CATALOG command
  v_display = '1.5-06 16-May-2022' ! Ensure STATISTIC works correctly on big data
  v_display = '1.6-00 14-Oct-2022' ! Include the SPECTRUM command calling GreG code
  v_display = '1.6-01 15-Oct-2022' ! Include the POPUP command
  v_display = '1.6-02 04-Nov-2022' ! Include the SET command
  v_display = '1.6-03 07-Nov-2022' ! Add the /DEFAULT option to the SET command
  v_display = '1.7-00 08-Nov-2022' ! Move the EXTRACT and SLICE command here
  v_display = '1.7-01 05-Jan-2023' ! SHOW MOSAIC
  v_display = '1.8-00 23-Jan-2023' ! /DIRECTORY option of all effective Display commands
  v_display = '1.8-01 25-Jan-2023' ! SHOW UV-Like data
  v_display = '1.8-02 25-Jan-2023' ! SLICE can call the cursor if no arg.
  v_display = '1.9-00 25-Jan-2023' ! Add the COMPARE command
  v_display = '1.9-01 08-Mar-2023' ! EXTRACT was not handling ANGLE_UNIT correctly
  v_display = '1.9-02 02-Jun-2023' ! FIND /CHECK option
  v_display = '1.9-03 12-Jul-2023' ! CATALOG /SORT option
  v_display = '1.9-04 21-Jul-2023' ! Improved COMPARE command
  v_display = '1.9-05 25-Jul-2023' ! Improve STATISTIC command for Mosaics
  v_display = '1.9-06 25-Jul-2023' ! SHOW /OVERLAY command 
  v_display = '1.9-07 09-Jan-2024' ! Better SHOW & VIEW of non standard data cubes (scripts only)
  v_display = '1.9-08 05-Mar-2024' ! STATISTIC /FILE command
  v_display = '1.9-09 05-Apr-2024' ! SHOW COMPOSITE improvement
  v_display = '1.9-10 05-May-2024' ! LOAD File WRITE
  v_display = '1.9-11 05-Jul-2024' ! SET FORMAT command
  v_display = '1.9-11 18-Jul-2024' ! SHOW PV|NOISE Buffer 
  v_display = '1.9-11 10-Oct-2024' ! VIEW File /NOPAUSE bug correction
  v_display = '1.9-11 25-Jan-2025' ! COMPARE & SHOW behaviour improvements
  endif
  v_display = '1.9-12 15-Apr-2025' ! SHOW & VIEW STATISTIC commands
  !
  call sic_begin('DISPLAY','gag_doc:hlp/imager-help-display.hlp', mshow, vshow, &
    & v_display//'S.Guilloteau', run_display, gr_error)
  display_loaded = .true.
  !
end subroutine load_display
!
subroutine init_clean
  use gkernel_interfaces
  use language_versions
  use gbl_message
  !----------------------------------------------------------------------
  ! @ public
  !*
  ! IMAGER  --  Load the Imager Languages:  
  !   CLEAN\        Basic imaging and deconvolution  
  !   CALIBRATE\    Calibration tools  
  !   ADVANCED\     Advanced processing  
  !   BUNDLES\      Integrated commands  
  !   IMAGER\       Pipeline related commands  
  !   SIMULATE\     Array simulator
  !!
  !----------------------------------------------------------------------
  use clean_def
  use clean_arrays
  use clean_default
  use clean_types
  use preview_mod
  !
  external :: run_advanced, run_bundles, run_clean, run_imager, run_self, run_simu
  !
  ! vocabulaire de CLEAN
  integer, parameter :: mclean=147+2+2
  character(len=16) :: vclean(mclean) 
  data vclean /' DUMP',   &
       ' ALMA', '/FLUX', '/PLOT', '/QUERY', '/NOISE', '/METHOD', &
       ' BUFFERS', &
       ' CCT_CLEAN', ' CCT_CONVERT', ' CCT_MERGE', &
       ' CLEAN', '/FLUX', '/PLOT', '/QUERY', '/NITER', '/ARES', '/RESTART', &  !No conflict with ALMA command
       ' DISCARD',  & 
       ' FIT', '/JVM_FACTOR', '/THRESHOLD', &
       '*LOG', &
       ' MAP_CONVOLVE', &
       ' MAP_COMBINE', '/BLANKING', '/FACTOR', '/THRESHOLD',         &
       ' MAP_COMPRESS', ' MAP_INTEGRATE',                            &
       ' MAP_REPROJECT', '/BLANKING', '/LIKE','/PROJECTION', '/SYSTEM', '/X_AXIS', '/Y_AXIS', & 
       ' MAP_RESAMPLE', '/LIKE',          & 
       ' MAP_SMOOTH', '/ASYMMETRIC', &
       ' MOSAIC', &
       ' MX', '/FLUX', '/PLOT', '/QUERY', &
       ' PRIMARY', '/APPLY', '/COMPUTE', '/CORRECT', '/TRUNCATE', &
       ' READ', '/COMPACT', '/FREQUENCY', '/PLANES', '/RANGE',  '/NOTRAIL',  &
       ' SPECIFY', '/FOR', &
       ' SUPPORT', '/CURSOR', '/MASK', '/PLOT', '/RESET', '/THRESHOLD', '/VARIABLE', &
       ' UV_BASELINE', '/CHANNELS', '/FILE', '/FREQUENCY', '/RANGE', '/VELOCITY', '/WIDTH', &
       ' UV_CHECK', '/FILE', ' UV_CHOP', ' UV_COMPRESS', '/CONTINUUM', '/FILE', & 
       ' UV_CONTINUUM','/INDEX', '/RANGE', &
       ' UV_EXTRACT', '/FIELDS', '/FILE', '/RANGE', &
       ' UV_FLAG', '/ANTENNA', '/DATE', '/FILE', '/RESET', &
       ' UV_FIELDS', '/CENTER', '/FILE', &
       ' UV_FILTER', '/CHANNELS', '/FILE', '/FREQUENCY', '/RANGE', '/VELOCITY', '/WIDTH', '/RESET', '/ZERO', &
       ' UV_HANNING', '/FILE', &
       ' UV_MAP', '/TRUNCATE',  '/FIELDS', '/RANGE', '/CONTINUUM', '/INDEX', &
       ' UV_RESAMPLE', '/FILE', '/LIKE', & 
       ' UV_RESIDUAL', '/FIELDS', '/OUTPUT', ' UV_RESTORE', '/COPY', '/SPEED', &
       ' UV_REWEIGHT', '/FLAG', '/FILE', '/RANGE', &
       ' UV_SHIFT',  '/FILE', & 
       ' UV_SMOOTH', '/ASYMMETRIC', '/FILE', &
       ' UV_SPLIT', '/CHANNELS', '/FILE', '/FREQUENCY', '/RANGE', '/VELOCITY', '/WIDTH', &
       ' UV_STAT',  '/PLOT',  &
       ' UV_TIME', '/FILE', '/WEIGHT',      &
       ' UV_TRIM', '/FILE', ' UV_TRUNCATE', &
       ' UV_SUBTRACT', '/FILE', &
       ' WRITE', '/APPEND ', '/RANGE', '/REPLACE', '/TRIM'/
  !
  integer, parameter :: ncal=21+3
  character(len=16) vcal(ncal)
  data vcal/  & 
       ' AMPLITUDE', '/ANTENNAS', '/FILE', & ! Antenna Amplitude Calibration
       ' APPLY', '/FILE',  '/FLAG',      & ! Apply Self calibration
       ' COLLECT', '/FLAG',              & ! Gather Independent Calibrations
       ' DERIVE',                        & ! Antenna to Baseline gain
       ' MODEL', '/MINVAL',              & ! UV model from UV_FIT or CLEAN
                 '/OUTPUT', '/MODE',     & !  - /OUTPUT must be second option
       ' PLAIT',                         & ! PLAIT command
       ' SCALE_FLUX',                    & ! Verify day-to-day Flux consistency
       ' SOLVE', '/MODE',                & ! Solve for Amplitude and/or Phase
       ' TRANSFORM',                     & ! Image transformations
       ' UV_SELECT',                     & ! Select UV_DATA 
       ' UV_SELF', '/RANGE', '/RESTORE', & ! Image for Self-calibration
       ' UV_SORT', '/FILE'    /            ! Sort UV data
  !
  integer, parameter :: nadv=68+1
  character(len=16) vadv(nadv)
  data vadv/  & 
    & ' FEATHER', '/FILE', '/REPROJECT',                        &
    & '*FLUX', '/CCT', ' HOW_TO',                               &
    & ' MAP_POLAR', '/ANGLE', '/BACKGROUND', '/COMPUTE', '/STEP',       &
    & ' MAP_CONTINUUM', '/METHOD',  '*MASK', '/FROM', ' MFS',   &
    & ' MOMENTS', '/CUT', '/MASK', '/METHOD', '/RANGE', '/THRESHOLD',   &
    & ' PROPER_MOTION', '/FILE',                                &
    & ' STOKES', '/FILE',                                       & 
    & ' UV_ADD', '/FILE',                                       &
    & ' UV_CIRCLE', '/FIT', '/SAMPLING', '/ZERO',               & 
    & ' UV_CORRELATE', '/FILE',                                 &
    & ' UV_DEPROJECT', '/FIT',                                  &
    & '*UV_FIT', '/CONTINUUM', '/INDEX', '/OUTPUT', '/QUIET',   &
    &     '/RANGE', '/RESULT', '/SAVE', '/UVRANGE', '/WIDGET',  &
    & ' UV_GRID',                                               &
    & ' UV_LINEARIZE','/FILE',                                  &
    & ' UV_MERGE', '/FILES', '/IGNORE', '/MODE', '/SCALES', '/WEIGHTS', &
    & ' UV_MOSAIC',                                             &
    & '*UV_PREVIEW', '/BROWSE', '/FILE',                        &  
    & ' UV_RADIAL', '/FIT', '/SAMPLING', '/ZERO',               & 
    & ' UV_SHORT', '/CHECK', '/REMOVE', '/RESAMPLE',            & 
    & ' XY_SHORT', '/RESAMPLE'/
  ! 
  integer, parameter  :: ncomp=17
  character(len=16) vcomp(ncomp)
  data vcomp/ & 
    & ' COMBINE', '/BLANKING', '/FACTOR', '/THRESHOLD', '/RELATIVE',       &
    & ' KEPLER', '/HFS', '/MASK', '/RESET', '/VELOCITY',  '/VSYSTEM',  '/WIDGET', &
    & '*SPECTRAL_CLEAN',                                        &
    & '*SELFCAL', '/WIDGET',                                    &
    & ' UV_DETECT', '/FILE' /                                  
  !
  integer, parameter :: nimag=25
  character(len=16) vimag(nimag)
  data vimag/ &
    & ' CLARK', '/FLUX', &
    & ' GCLEAN', '/FLUX', '/RESTART', &
    & ' HOGBOM', '/FLUX', &
    & ' MRC', '/FLUX', &
    & ' MULTISCALE', '/FLUX', &
    & ' PIPELINE', '/MODE', '/WIDGET', &
    & ' REPORT', &
    & ' SIMULATE','/ARRAY','/CONFIG','/HOURS','/MODEL','/NOISE', &
    & ' SDI', '/FLUX', '/PLOT', &
    & ' TIPS' /
  !
  integer, parameter :: nsimu=13
  character(len=16) :: vsimu(nsimu)
  data vsimu/ &
    & ' ADD_NOISE', '/FILE', &
    & ' ARRAY', &
    & ' UV_TRACKS', '/FRAME', '/HORIZON', '/HOUR_ANGLE', '/INTEGRATION', &
    & '/OFFSET', '/SIZE', '/STATIONS', '/TABLE', '/WEIGHT'/
  ! 
  character(len=2), save :: ansi_crlf
  logical :: error
  integer :: iout
  logical, parameter :: old=.false.
  !
  ! Global IMAGER Version (matching Documentation)
  if (old) then
    v_program = '3.6-01  24-Nov-2022'   ! First release
    v_program = '3.6-02  13-Dec-2022'   ! MODEL works on Mosaic
    v_program = '3.7-00  05-Jan-2023'   ! UV_MAP with Sault method for Mosaics
    v_program = '3.7-02  25-Jan-2023'   ! /DIRECTORY option in DISPLAY commands
    v_program = '3.7-03  24-Feb-2023'   ! Full capabilities of COMBINE command
    v_program = '3.7-04  10-Jun-2023'   ! UV_SHORT, PRIMARY, PLAIT and MAP_POLAR commands
    v_program = '3.7-05  13-Jun-2023'   ! Debug MODEL 
    v_program = '3.8-01  25-Jun-2023'   ! SIMULATE\ language
    v_program = '3.8-02  05-Jul-2023'   ! SIMULATE command
    v_program = '3.8-04  O7-Jul-2023'   ! Bug corrections for Mosaics & UV_SHORT
    v_program = '3.8-05  21-Jul-2023'   ! Bug correction COMBINE - Improve COMPARE
    v_program = '3.9-01  25-Jul-2023'   ! SELFCAL for Mosaics+ SHOW /OVERLAY
    v_program = '3.9-02  29-Aug-2023'   ! UV_SHORT improvement
    v_program = '3.9-03  07-Sep-2023'   ! BEAM_GAP control and UV_RESTORE / COMBINE fixes
    v_program = '3.9-05  18-Sep-2023'   ! SIMULATE & CATALOG bug correction
    v_program = '3.9-06  05-Oct-2023'   ! CATALOG, UV_MERGE & UV_CORRELATE bugs correction
    v_program = '3.9-07  18-Oct-2023'   ! CCT_CONVERT & MASK corrections
    v_program = '3.9-08  20-Oct-2023'   ! CLEAN /RESTART bug correction - SHOW MOMENTS improved
    v_program = '3.9-08  13-Nov-2023'   ! SELFCAL issues, MAP_COMBINE INDEX and UV_FIT parameters bug correction
    v_program = '3.9-09  27-Nov-2023'   ! Installation scripts & Verification procedure
    v_program = '3.9-10  04-Dec-2023'   ! Display scripts on non-standard cubes
    v_program = '3.9-11  09-Jan-2024'   ! Protect for flagged data in Robust weighting & UV_STAT 
    v_program = '4.0-00  05-Mar-2024'   ! PIPELINE /MODE SURVEY bug correction
    v_program = '4.0-01  05-Mar-2024'   ! UV_PREVIEW bug correction
    v_program = '4.0-02  05-Apr-2024'   ! Update PIPELINE scripts
    v_program = '4.0-03  14-May-2024'   ! Better SPECIFY command & MOMENTS  SicVar optional argument
    v_program = '4.0-04  24-May-2024'   ! Use ICRF instead of FK5 for ALMA
    v_program = '4.1-00  13-Jun-2024'   ! Dynamic CCT_LIST use
    v_program = '4.1-01  14-Jun-2024'   ! Remove obsolete code as much as possible (only MX left)
    v_program = '4.2-00  03-Jul-2024'   ! GCLEAN command and GAUSS method for CLEAN
    v_program = '4.2-01  03-Jul-2024'   ! SET FORMAT command and associate WRITE keywords
    v_program = '4.2-02  16-Jul-2024'   ! Optimized and functional GCLEAN command 
    v_program = '4.2-03  18-Jul-2024'   ! PIPELINE improvements
    v_program = '4.2-04  22-Jul-2024'   ! UV_RESTORE for mosaics
    v_program = '4.2-05  24-Jul-2024'   ! UV_GRID command in ADVANCED\ language
    v_program = '4.2-06  29-Aug-2024'   ! AMPLITUDE command in CALIBRATE\ language
    v_program = '4.2-07  15-Sep-2024'   ! REPORT command in IMAGER\, UV_CHOP in CLEAN\ language
    v_program = '4.2-08  15-Oct-2024'   ! UV_STAT /PLOT, Improve PROPER_MOTION, VIEW bug correction
    v_program = '4.2-08  25-Oct-2024'   ! Bug correction in CLEAN /RESTART and READ UV for Stokes 
    v_program = '4.2-08  31-Oct-2024'   ! Bug correction in Polarization handling
    v_program = '4.2-08  05-Nov-2024'   ! MAP_POLAR improvement + casagildas() with no keepflags
    v_program = '4.3-00  15-Nov-2024'   ! UV_LINEAR + Version with no Python
    v_program = '4.4-00  25-Nov-2024'   ! Upgrade of the SELFCAL facility
    v_program = '4.4-01  30-Nov-2024'   ! Minor bug corrections in UV_MERGE, UV_LINEARIZE, UV_PREVIEW
    v_program = '4.4-02  09-Dec-2024'   ! UV_FIT /CONTINUUM first version
    v_program = '4.4-03  25-Jan-2025'   ! AMPLITUDE command operational, improved STATISTIC, and CLEAN bug correction
  endif
  v_program = '4.5-01  15-Apr-2025'   ! New major version
  ! Released above
  v_program = '4.5-02  15-May-2025'   ! New GIO based version
  v_program = '4.5-03  25-May-2025'   ! Major correction in PHASE Mosaics
  v_program = '4.5-04  28-May-2025'   ! Bug correction in CLEAN /RESTART 
  v_program = '4.5-05  05-Jul-2025'   ! UV_SUBTRACT command
  v_program = '4.6-00  15-Sep-2025'   ! Update Major version for easier bug tracking.
  v_program = '4.6-01  25-Sep-2025'   ! Bug correction in UV_MERGE 
  v_program = '4.6-02  15-Oct-2025'   ! Bis + UV_FIT /CONT FieldSize
  v_program = '4.6-03  30-Nov-2025'   ! Support for Hybrid Mosaics
  v_program = '4.6-04  [ Dec-2025 ]'   ! UV_PREVIEW issue
  !
  ! CLEAN Language versions   v_clean
  if (old) then
  v_clean = '4.0-01 08-Mar-2018' ! First IMAGER release with new Variable names (may still change)
  v_clean = '4.1-01 08-Mar-2018' ! IMAGER release with Self Calibration commands
  v_clean = '4.1-02 15-Mar-2018' ! Bug correction in Self Calibration commands
  v_clean = '4.1-02 02-May-2018' ! Split with CALIBRATE\ language 
  v_clean = '4.2-01 03-Jul-2018' ! Add /RANGE options to UV_FILTER and UV_BASELINE
  v_clean = '4.3-01 05-Sep-2018' ! Minor change in SHOW NOISE to identify properly Dirty and Clean data
  v_clean = '4.3-02 05-Nov-2018' ! Debug MX, improve MULTI and cleanup messages  
  v_clean = '4.3-03 14-Nov-2018' ! Add the UV_TRIM command, support READ CCT /RANGE
  v_clean = '4.4-01 14-Jan-2019' ! Add the UV_HANNING command
  v_clean = '4.4-02 17-Jan-2019' ! Add the UV_EXTRACT command
  v_clean = '4.4-03 04-Feb-2019' ! Add the /FILE option to UV_BASELINE,UV_FILTER, UV_SPLIT and UV_SORT
  v_clean = '4.4-04 07-Feb-2019' ! Bug correction in UV_MAP for Phase offset mosaics 
  v_clean = '4.4-05 08-Feb-2019' ! UV_TRIM /FILE syntax
  v_clean = '4.4-06 12-Feb-2019' ! Protect against incomplete UV data (Nvisi < dim(2))
  v_clean = '4.4-07 13-Feb-2019' ! Operational version of UV_SORT /FILE (still slow)
  v_clean = '4.4-08 14-Feb-2019' ! Operational version of UV_SORT /FILE (reasonable now) 
  v_clean = '4.4-09 15-Feb-2019' ! UV_TIME /FILE syntax
  v_clean = '4.4-10 25-Feb-2019' ! UV_RESAMPLE, UV_COMPRESS, UV_HANNING /FILE syntax
  v_clean = '4.5-01 27-Feb-2019' ! Move STOKES command to ADVANCED\ language
  v_clean = '4.5-02 14-Mar-2019' ! Improve the STATISTIC and SUPPORT commands behaviour 
  v_clean = '4.5-03 29-Mar-2019' ! Support for SHOW SOURCES
  v_clean = '5.0-01 05-Apr-2019' ! Remove CLARK, ..., SDI commands, leave only CLEAN
  v_clean = '5.0-02 05-Apr-2019' ! Move COLOR to CLEAN\
  v_clean = '5.0-03 29-Apr-2019' ! Option /CONT for UV_MAP
  v_clean = '5.0-04 02-May-2019' ! BUFFERS command
  v_clean = '5.0-05 06-May-2019' ! /NOPAUSE option to VIEW command
  v_clean = '5.0-06 07-May-2019' ! UV_STAT DEFAULT and associated variables
  v_clean = '5.0-07 25-May-2019' ! /INDEX option for UV_MAP
  v_clean = '5.1-01 05-Jun-2019' ! Command UV_REWEIGHT re-written
  v_clean = '5.1-02 15-Jun-2019' ! UV_EXTRACT /FIELDS
  v_clean = '5.1-03 25-Jun-2019' ! UV_FIELDS command 
  v_clean = '5.1-04 03-Jul-2019' ! UV_COMPRESS /CONTINUUM option
  v_clean = '5.1-05 04-Jul-2019' ! Remove obsolete /SELF option of UV_MAP & UV_RESTORE
  v_clean = '5.2-01 03-Sep-2019' ! Move STATISTIC command out of this language
  v_clean = '5.2-02 13-Sep-2019' ! Move UV_SORT command out of this language
  v_clean = '5.2-03 25-Sep-2019' ! UV_FLAG bug correction
  v_clean = '5.2-04 07-Oct-2019' ! UV_MAP 'Map_Center' bug correction (header not updated)
  v_clean = '5.2-05 10-Oct-2019' ! WRITE does not accept ambiguities
  v_clean = '5.2-06 10-Oct-2019' ! VIEW with no argument just re-displays
  v_clean = '5.3-01 10-Oct-2019' ! Support for the CONTINUUM image
  v_clean = '5.3-02 11-Oct-2019' ! MAP_REPROJECT, MAP_COMBINE commands (+syntax MAP_RESAMPLE)
  v_clean = '5.3-03 13-Oct-2019' ! SHOW /NOPAUSE and VIEW /OVERLAY commands
  v_clean = '5.3-04 16-Oct-2019' ! SHOW /SIDE instead of /NOPAUSE and new MASK THRESHOLD command
  v_clean = '5.3-05 06-Nov-2019' ! /FILE option of UV_CHECK
  v_clean = '5.3-06 20-Nov-2019' ! /FILE option of UV_SHIFT
  v_clean = '5.3-07 06-Feb-2020' ! Bug correction for MAP_COMPRESS syntax
  v_clean = '5.3-08 20-Feb-2020' ! Size computation for Large data set may exceed 2GBytes
  v_clean = '5.3-09 21-Feb-2020' ! Bug correction for Mosaic in Robust weighting 
  v_clean = '5.3-10 24-Feb-2020' ! /LIKE option of MAP_RESAMPLE
  v_clean = '5.3-11 05-Mar-2020' ! Bug correction in MULTI
  v_clean = '5.3-12 05-Mar-2020' ! Bug correction about buffers selected by UV_SELECT and/or UV_MAP /CONT
  v_clean = '5.3-13 11-Mar-2020' ! Bug correction in MAP_RESAMPLE (overflow of channel ranges)
  v_clean = '5.3-14 25-Mar-2020' ! Remove a restriction in MAP_RESAMPLE (overflow of channel ranges)
  v_clean = '5.3-15 25-Mar-2020' ! Improve the /FLUX display in CLEAN command
  v_clean = '5.3-16 13-May-2020' ! Improve the WRITE MODEL command 
  v_clean = '5.4-01 21-May-2020' ! /FOR option of SPECIFY
  v_clean = '5.4-02 09-Jul-2020' ! Command DISCARD
  v_clean = '5.4-03 01-Sep-2020' ! Option /THRESHOLD for command FIT
  v_clean = '5.4-04 03-Sep-2020' ! Improve UV_TIME command : more consistent defaults
  v_clean = '5.4-05 11-Sep-2020' ! Bug correction in MASK APPLY (Blanking was not updated)
  v_clean = '5.4-06 15-Sep-2020' ! Optimize MAP_RESAMPLE if no resampling needed
  v_clean = '5.5-01 26-Oct-2020' ! Command UV_SMOOTH
  v_clean = '5.6-01 07-Dec-2020' ! Command CCT_MERGE and CCT_CONVERT
  v_clean = '5.6-02 14-Dec-2020' ! STATISTIC /EDGE option
  v_clean = '5.6-03 14-Dec-2020' ! /ARES Argument must be Real array
  v_clean = '5.6-04 14-Jan-2021' ! Debug VIEW and SHOW 4-D Beams
  v_clean = '5.6-05 19-Jan-2021' ! Make sure UV_MAP works on Mosaics even in Memory limited conditions
  v_clean = '5.7-01 31-Jan-2021' ! Command CCT_CLEAN
  v_clean = '5.7-02 31-Mar-2021' ! FIT command make one fit per pointing
  v_clean = '5.7-03 01-Apr-2021' ! MULTISCALE is now operational for Mosaics
  v_clean = '5.7-04 01-Jun-2021' ! Bug correction for MULTISCALE in 4-D Beams
  v_clean = '5.7-05 01-Jun-2021' ! More flexible FIT command
  v_clean = '5.8-01 17-Jun-2021' ! SPECIFY FREQUENCIES command for VLA data
  v_clean = '5.8-02 28-Jun-2021' ! UV_SHIFT bug correction for "false" mosaics (ID and MOFF columns)
  v_clean = '5.8-03 28-Jun-2021' ! UV_SORT bug correction for /FILE option
  v_clean = '6.0-00 21-Oct-2021' ! Split Language CLEAN\ into CLEAN\ and DISPLAY\ (no \at end please...)
  v_clean = '6.0-01 09-Nov-2021' ! Bug correction in READ UV: undefine PREVIEW%, avoid write UV data in WRITE *
  v_clean = '6.1-00 17-Nov-2021' ! Add /JVM option to command FIT
  v_clean = '6.1-01 17-Nov-2021' ! CLEAN Variable ? syntax
  v_clean = '6.1-02 17-Nov-2021' ! UV_MAP Variable ? syntax
  v_clean = '6.1-03 19-Nov-2021' ! UV_REWEIGHT syntax bug correction
  v_clean = '6.1-04 23-Nov-2021' ! Use the BEAM_JVM value in UV_RESTORE - (Defaults to 1 if not available)
  v_clean = '6.2-00 02-Dec-2021' ! Add the /LIKE option to UV_RESAMPLE 
  v_clean = '6.2-01 16-Dec-2021' ! Optimize parallel processing
  v_clean = '6.2-02 17-Dec-2021' ! Debug MAP_BEAM_STEP = 1 mode
  v_clean = '6.2-03 17-Dec-2021' ! UV_CHECK beam works better
  v_clean = '6.2-04 18-Dec-2021' ! UV_TIME now works for UV tables with per-channel weights
  v_clean = '6.2-05 20-Dec-2021' ! UV_RESAMPLE must reset BEAM_RANGES variable
  v_clean = '6.3-00 21-Dec-2021' ! Use new way of defining Dirty Beams number and properties.
  v_clean = '6.3-01 03-Jan-2021' ! Parallel programming optimization for CLEAN
  v_clean = '6.3-02 06-Jan-2022' ! Allow direct READ of FITS data cubes.
  v_clean = '6.3-03 07-Jan-2022' ! READ /PLANES option now works as expected
  v_clean = '6.3-04 11-Jan-2022' ! UV_FIELD bug correction 
  v_clean = '6.3-05 18-Jan-2022' ! READ SINGLE bug correction
  v_clean = '6.3-06 20-Jan-2022' ! READ  with FITS format improvement
  v_clean = '6.3-07 08-Feb-2022' ! Allow to use Fixed beamsize in UV_RESTORE
  v_clean = '6.3-08 08-Feb-2022' ! CLEAN_STOP syntax in CLEAN
  v_clean = '6.3-09 07-Mar-2022' ! Allow WRITE BEAM /APPEND in all cases
  v_clean = '6.3-10 07-Mar-2022' ! Bug correction in beam derivation if ANGLE is specified
  v_clean = '6.3-11 09-Mar-2022' ! Bug correction in CCT_MERGE : Concatenation was improper
  v_clean = '6.4-00 18-Mar-2022' ! Implement the /RESTART option of CLEAN
  v_clean = '6.4-01 22-Mar-2022' ! Flag out of band channels in UV_RESAMPLE
  v_clean = '6.4-02 03-May-2022' ! /RESET option of UV_FILTER
  v_clean = '6.4-03 10-May-2022' ! Bug correction in UV_SHIFT and PROPER_MOTION
  v_clean = '6.4-04 02-Jun-2022' ! Make SPECIFY TELESCOPE work on Images too
  v_clean = '6.4-05 09-Jun-2022' ! Debug UV_FLAG command
  v_clean = '6.4-06 29-Jun-2022' ! Make UV_EXTRACT work with Random Frequency Axis
  v_clean = '6.4-07 29-Jun-2022' ! Make READ handle properly Random Frequency Axis in UVFITS format
  v_clean = '6.5-00 05-Sep-2022' ! Add the /ASYMMETRIC option of MAP_SMOOTH and UV_SMOOTH
  v_clean = '6.5-01 27-Sep-2022' ! Add the UNIT specifier to MAP_REPROJECT
  v_clean = '6.5-02 07-Oct-2022' ! Bug correction in Beam Consistency checking
  v_clean = '6.5-03 10-Oct-2022' ! Use a Channel mask array to hide line-emitting parts of the spectrum
  v_clean = '6.5-04 07-Nov-2022' ! Make sure MAP_CENTER is also used for Mosaics
  v_clean = '6.5-05 14-Nov-2022' ! Bug in UV_SHIFT: Pointing center was incorrectly reset in /FILE option for single fields
  v_clean = '6.5-06 14-Nov-2022' ! Bug in UV_MOSAIC: Wrong identification of arguments
  v_clean = '6.5-07 14-Nov-2022' ! Make sure MAP_CENTER is never forgotten (Mosaic stills needs checking)
  v_clean = '6.5-08 15-Nov-2022' ! UV_TIME crashed on Mosaics
  v_clean = '6.6-00 19-Dec-2022' ! Prepare UV_MAP with Sault Method for Mosaics, using the MOSAIC command
  v_clean = '6.6-01 05-Jan-2023' ! Working version of UV_MAP with Sault Method for Mosaics, using the MOSAIC command
  v_clean = '6.6-02 05-Jan-2023' ! all "TRUNCATE" values are now in [0,1] (instead of % in some options before)
  v_clean = '6.6-03 20-Feb-2023' ! MAP_CONVOLVE now preserves flux when possible. MAP_RESAMPLE handles blanking.
  v_clean = '6.6-04 23-Feb-2023' ! UV_STAT better handles Recommended vs User specified sizes
  v_clean = '6.6-05 03-Mar-2023' ! UV_CHECK uses BEAM_TOLERANCE
  v_clean = '6.6-06 03-Mar-2023' ! Bug correction in UV_SHIFT for degenerate 1 field Mosaic
  v_clean = '6.6-07 08-Mar-2023' ! Bug correction in READ for FITS files (file left open in optimized case)
  v_clean = '6.6-08 08-Mar-2023' ! UV_CHECK BEAM uses an improved method (ignore per-channel scale factors)
  v_clean = '6.6-09 15-May-2023' ! PRIMARY command syntax improved, backward compatible
  v_clean = '6.6-10 25-May-2023' ! UV_REWEIGHT bug correction
  v_clean = '6.7-00 06-Jun-2023' ! LOG command
  v_clean = '6.7-01 10-Jun-2023' ! Bug correction for UV_RESTORE in Mosaics.
  v_clean = '6.7-02 07-Jul-2023' ! READ UV in UVFITS format was broken on Mosaics.
  v_clean = '6.7-03 07-Jul-2023' ! SPECIFY /FOR was not working
  v_clean = '6.7-04 13-Jul-2023' ! Improvements and Bug corrections in UV_STAT
  v_clean = '6.7-05 13-Jul-2023' ! READ Item FileFilter opens a file search window
  v_clean = '6.7-06 28-Aug-2023' ! SPECIFY command syntax: BLANKING, FREQUENCY and SPECUNIT
  v_clean = '6.7-07 04-Sep-2023' ! Introduce BEAM_GAP and verify Weight channel more clearly
  v_clean = '6.7-08 06-Sep-2023' ! DUMP MEM command and IMAGER_RAM variable
  v_clean = '6.7-09 07-Sep-2023' ! Proper handling of Memory limit in UV_RESTORE and UV_RESIDUAL - Sault Mosaic to be done
  v_clean = '6.7-10 04-Oct-2023' ! UV_CHECK improvements and UV_CONTINUUM using the Code_uvt_IF flag
  v_clean = '6.7-11 18-Oct-2023' ! CCT_CONVERT flux scale correction
  v_clean = '6.7-12 20-Oct-2023' ! CLEAN /RESTART bug correction
  v_clean = '6.7-13 13-Nov-2023' ! MAP_COMBINE INDEX bug correction
  v_clean = '6.7-14 09-Jan-2024' ! Ensure flagged data are not used in Robust weighting estimate
  v_clean = '6.8-00 05-Mar-2024' ! Improved PRIMARY command, WRITE CCT /APPEND support
  v_clean = '6.8-01 08-Mar-2024' ! Debug UV_PREVIEW for Confused spectra
  v_clean = '6.8-02 10-Mar-2024' ! Update fits-to-uvt.ima and debug UV_REWEIGHT /FILE
  v_clean = '6.8-03 07-May-2024' ! Make SPECIFY /FOR obsolete
  v_clean = '6.8-04 24-May-2024' ! Use ICRF for ALMA data (READ and UV_MAP minor changes)
  v_clean = '6.8-05 13-Jun-2024' ! Dynamic CCT_LIST use
  v_clean = '6.9-00 03-Jul-2024' ! GAUSS Method for CLEAN
  v_clean = '6.9-02 22-Jul-2024' ! UV_RESTORE for Mosaics and GAUSS method
  v_clean = '6.9-03 09-Sep-2024' ! UV_MAP for small mosaics (and READ modificatioons)
  v_clean = '6.9-04 09-Sep-2024' ! UV_CHOP command
  v_clean = '6.9-05 01-Oct-2024' ! /PLOT option to UV_STAT
  v_clean = '6.9-06 25-Oct-2024' ! Bug correction in CLEAN /RESTART & READ for Stokes 
  v_clean = '7.0-01 22-Nov-2024' ! Upgrade of SELFCAL facility
  v_clean = '7.0-02 09-Jan-2025' ! Improved STATISTIC command, for use with 3-D Masks.
  v_clean = '7.0-03 09-Jan-2025' ! Improved READ command, with warning for Multi-Arrays in UVFITS files.
  v_clean = '7.0-04 13-Jan-2025' ! Bug correction - CLEAN would occasionally SegFault in parallel mode.
  v_clean = '7.1-01 15-Apr-2025' ! READ command engines re-written for Mosaics
  v_clean = '7.1-02 20-Apr-2025' ! Properly handle PHASE Mosaics by conversion to POINTING ones.
  endif
  v_clean = '7.2-01 01-Jul-2025' ! Properly handle PHASE Mosaics by conversion to POINTING ones.
  v_clean = '7.2-02 01-Sep-2025' ! UV_SUBTRACT command added
  v_clean = '7.2-03 08-Sep-2025' ! UV_CONTINUUM baseline extrema bug correction
  v_clean = '7.3-01 27-Nov-2025' ! UV_MOSAIC et al for more than 1 telescope
  !
  ! ADVANCED\ Language versions   v_advanced
  if (old) then
  v_advanced = '1.0-01 01-Dec-2017' ! Command UV_PREVIEW 
  v_advanced = '1.0-02 08-Mar-2018' ! Commands UV_DEPROJECT  and UV_RADIAL
  v_advanced = '1.1-02 08-Mar-2018' ! Commands MODEL, SOLVE /MODE, CALIBRATE and UV_MAP /SELF 
  v_advanced = '1.2-01 10-Apr-2018' ! Template for Command UV_SHORT
  v_advanced = '1.2-02 26-Apr-2018' ! Operational Command UV_SHORT - Still problems with repetitions.
  v_advanced = '1.3-01 03-Jul-2018' ! Rename CALIBRATE into APPLY -- Split with CALIBRATE\ language
  v_advanced = '1.3-02 26-Sep-2018' ! Add SLICE and EXTRACT commands
  v_advanced = '1.3-03 26-Sep-2018' ! allow support for VIEW (or SHOW) AnySicImage
  v_advanced = '1.3-04 26-Sep-2018' ! UV_PREVIEW bug correction
  v_advanced = '1.4-01 23-Nov-2018' ! Rename language to ADVANCED, and insert SELFCAL command
  v_advanced = '1.4-02 26-Nov-2018' ! Add UV_FIT command 
  v_advanced = '1.4-03 28-Nov-2018' ! Add MOMENTS command 
  v_advanced = '1.4-04 28-Nov-2018' ! Add CATALOG command 
  v_advanced = '1.4-05 15-Jan-2019' ! Better version of the CATALOG command - Drives a script
  v_advanced = '1.4-06 23-Jan-2019' ! Correction of a bug in UV_PREVIEW
  v_advanced = '1.5-01 24-Jan-2019' ! Add the ADVANCED\MASK command
  v_advanced = '1.5-02 04-Feb-2019' ! Debug the UV_FIT command -- Add the /FILE option for UV_PREVIEW
  v_advanced = '1.6-01 26-Feb-2019' ! Add the UV_ADD command (similar to UV_DOPPLER in ASTRO)
  v_advanced = '1.6-02 27-Feb-2019' ! Move the STOKES command from the Clean\ language
  v_advanced = '1.6-03 19-Mar-2019' ! Add the UV_MERGE command (simpler than the task...)
  v_advanced = '1.6-04 20-Mar-2019' ! Add the COLOR and FLUX commands  
  v_advanced = '1.6-05 28-Mar-2019' ! /SAVE option of UV_FIT 
  v_advanced = '1.6-06 14-May-2019' ! HOW_TO command
  v_advanced = '1.7-01 19-May-2019' ! FEATHER command
  v_advanced = '1.7-02 24-May-2019' ! XY_SHORT command
  v_advanced = '1.8-01 20-Jul-2019' ! UV_CIRCLE command
  v_advanced = '1.8-02 22-Jul-2019' ! /RESULT option of UV_FIT command
  v_advanced = '1.8-03 31-Jul-2019' ! PROPER_MOTION command
  v_advanced = '1.8-04 23-Sep-2019' ! Improved UV_FIT command
  v_advanced = '1.8-05 16-Oct-2019' ! COMBINE command
  v_advanced = '1.8-06 17-Oct-2019' ! FLUX and MAP_CONTINUUM commands bug correction
  v_advanced = '1.9-00 29-Nov-2019' ! Move the COMBINE and SELFCAL to the BUNDLES\ language
  v_advanced = '1.9-01 20-Jan-2020' ! Speed-up (considerably) UV_PREVIEW
  v_advanced = '1.9-02 20-Jan-2020' ! Add the UV_CORRELATE command
  v_advanced = '1.9-03 06-Feb-2020' ! Change name of SLICE buffer to SLICED for Python
  v_advanced = '1.9-04 12-Feb-2020' ! Allow UV_MERGE to merge Line tables in CONTINUUM mode
  v_advanced = '1.9-05 17-Feb-2020' ! UV_PREVIEW bug correction. Also affects MAP_CONTINUUM
  v_advanced = '1.9-06 18-Feb-2020' ! Improved STATISTIC command
  v_advanced = '2.0-00 19-Feb-2020' ! MOMENTS /CUTS option and improved SHOW MOMENTS
  v_advanced = '2.0-01 20-Feb-2020' ! UV_SHORT ? proper support
  v_advanced = '2.0-02 20-Feb-2020' ! UV_SHORT with non-circular Short-Spacing beam
  v_advanced = '2.0-03 26-Feb-2020' ! UV_SHORT /CHECK option
  v_advanced = '2.1-00 27-Feb-2020' ! UV_SHORT with minimum baseline and single-fields
  v_advanced = '2.1-01 29-Mar-2020' ! Improvements in MOMENTS
  v_advanced = '2.1-02 13-May-2020' ! Improvements in UV_CORRELATE
  v_advanced = '2.1-03 14-May-2020' ! Bug correction in UV_PREVIEW and UV_CORRELATE
  v_advanced = '2.1-04 28-May-2020' ! UV_PREVIEW /BROWSE command
  v_advanced = '2.1-05 10-Jun-2020' ! UV_PREVIEW /FILE Fich [Drop] command
  v_advanced = '2.1-07 26-Oct-2020' ! Debug completely UV_MERGE command (or 2021 ?)
  v_advanced = '2.2-00 26-Oct-2021' ! Move CATALOG and FIND to DISPLAY\ Language
  v_advanced = '2.2-01 18-Nov-2021' ! UV_SHORT V 3.9 syntax and Help improvement
  v_advanced = '2.3-00 26-Nov-2021' ! MAP_CONTINUUM /METHOD option
  v_advanced = '2.4-00 02-Dec-2021' ! UV_MOSAIC command
  v_advanced = '2.4-01 19-Dec-2021' ! UV_MERGE /MODE CONCATENATE command
  v_advanced = '2.4-02 19-Dec-2021' ! EXTRACT and SLICE syntax improvements.
  v_advanced = '2.4-03 19-Dec-2021' ! EXTRACT now allows expansion
  v_advanced = '2.4-04 08-Feb-2022' ! UV_FIT prints primary beam corrected flux
  v_advanced = '2.4-05 17-Apr-2022' ! STOKES can now work on internal buffer
  v_advanced = '2.4-06 22-Apr-2022' ! UV_PREVIEW syntax change & improvement
  v_advanced = '2.4-07 25-May-2022' ! MOMENTS SMOOTH keyword
  v_advanced = '2.4-08 07-Jun-2022' ! UV_FIT /UVRANGE Min Max
  v_advanced = '2.4-09 29-Jun-2022' ! STOKES: Correct derivation from H-V polarization
  v_advanced = '2.5-00 05-Jul-2022' ! First operational version of MAP_POLAR
  v_advanced = '2.5-01 08-Sep-2022' ! Bug correction in EXTRACT command
  v_advanced = '2.5-02 13-Oct-2022' ! Allow adding columns in UV_ADD /FILE
  v_advanced = '2.6-00 08-Nov-2022' ! Move EXTRACT and SLICE to DISPLAY\ language
  v_advanced = '2.6-01 24-Nov-2022' ! Reload the MASK after a user specified modification if in use
  v_advanced = '2.6-02 24-Jan-2023' ! SOLVE uses the SELF_GAIN convergence factor
  v_advanced = '2.6-03 08-Mar-2023' ! MASK works again on 2-D data cubes (p_mask.ima)
  v_advanced = '2.6-04 27-Mar-2023' ! MAP_CONTINUUM  SIZE and CENTER units correction
  v_advanced = '2.6-05 02-Jun-2023' ! MAP_POLAR /ANGLE option
  v_advanced = '2.6-06 07-Jul-2023' ! UV_SHORT bug corrections
  v_advanced = '2.6-06 29-Aug-2023' ! UV_SHORT command improvement (handle BLANKING and resampling)
  v_advanced = '2.6-07 07-Sep-2023' ! Fixed COMBINE memory leak
  v_advanced = '2.6-07 18-Sep-2023' ! Bug correction in frequency units in CATALOG
  v_advanced = '2.6-08 26-Sep-2023' ! Bug correction in CSV CATALOG line names
  v_advanced = '2.6-09 03-Oct-2023' ! UV_MERGE, UV_CORRELATE & UV_DETECT bugs correction
  v_advanced = '2.6-10 18-Oct-2023' ! MASK LABEL command
  v_advanced = '2.7-00 20-Oct-2023' ! MASK Improvement and UV_FIT /SAVE modification
  v_advanced = '2.7-09 13-Nov-2023' ! Correct UV_FIT for many parameters
  v_advanced = '2.7-10 14-May-2024' ! MOMENTS  SicVar  optional argument
  v_advanced = '2.7-11 17-Jul-2024' ! UV_PREVIEW after UV_CHECK BEAM bug correction
  v_advanced = '2.8-01 24-Jul-2024' ! UV_GRID command
  v_advanced = '2.8-02 12-Oct-2024' ! UV_MERGE /IGNORE option
  v_advanced = '2.8-03 05-Nov-2024' ! MAP_POLAR improvement
  v_advanced = '2.9-01 04-Nov-2024' ! UV_LINEARIZE and UV_MERGE with STITCH
  v_advanced = '2.9-02 30-Nov-2024' ! Minor bug corrections in UV_MERGE, UV_LINEARIZE, UV_PREVIEW
  v_advanced = '2.9-03 09-Dec-2024' ! UV_FIT /CONTINUUM
  endif
  v_advanced = '2.9-04 15-May-2025' ! UV_FIT protection against improper input parameters
  v_advanced = '2.9-05 17-Sep-2025' ! MASK THRESHOLD REGION improvement
  v_advanced = '2.9-06 21-Sep-2025' ! UV_MERGE /CONT Index Frequency bug correction
  v_advanced = '2.9-07 14-Oct-2025' ! UV_FIT /CONT FieldSize + UV_MERGE /CONT bug correction
  v_advanced = '2.9-08 01-Dec-2025' ! Bug correction in UV_PREVIEW /FILE (array overflow)
  !
  ! CALIBRATE\ Language versions  v_cal
  if (old) then
  v_cal = '1.0-01 03-Jul-2018' ! Split with CALIBRATE\ language 
  v_cal = '1.1-01 07-Jul-2018' ! /MINVAL option instead of /MINFLUX for MODEL 
  v_cal = '1.2-01 20-Mar-2019' ! Rename command FLUX_SCALE into SCALE_FLUX 
  v_cal = '1.3-01 16-Jul-2019' ! Command COLLECT 
  v_cal = '1.3-02 16-Jul-2019' ! Command DERIVE  
  v_cal = '1.4-01 03-Sep-2019' ! Move STATISTIC command to this language
  v_cal = '1.4-02 06-Sep-2019' ! Implement UV_SELECT selection command
  v_cal = '1.4-03 13-Sep-2019' ! Move UV_SORT command to this language
  v_cal = '1.4-04 26-Sep-2019' ! APPLY command modification ((weights and flags, r1.11)
  v_cal = '1.5-01 08-Nov-2019' ! COLLECT /FLAG option and bug correction
  v_cal = '1.6-01 17-Jan-2020' ! FIND Fmin Fmax command
  v_cal = '1.7-01 19-Mar-2020' ! MODEL /MODE IMAGE  implemented. 
  v_cal = '1.7-04 13-May-2020' ! MODEL command defines UV_MODEL
  v_cal = '1.7-05 26-May-2020' ! Finish MODEL /MODE IMAGE implementation
  v_cal = '1.7-06 15-Sep-2020' ! Add time tolerance in command APPLY
  v_cal = '1.8-00 28-Oct-2021' ! Move STATISTIC command to DISPLAY\ language
  v_cal = '1.8-01 11-Jan-2022' ! Protect MODEL against undefined data units.
  v_cal = '1.8-02 17-Jan-2022' ! Add the Frequency keyword to MODEL /MODE 
  v_cal = '1.8-03 13-Dec-2022' ! Make MODEL works on Mosaics
  v_cal = '1.8-04 15-Dec-2022' ! Make UV_MAP /FIELD works for Mosaics
  v_cal = '1.9-00 03-May-2023' ! Add PLAIT command
  v_cal = '1.9-02 24-Jul-2023' ! UV_SELF supports Mosaics now
  v_cal = '1.9-03 03-Oct-2023' ! UV_SELF no longer works on "pseudo-continuum" data
  v_cal = '1.9-04 09-Nov-2023' ! UV_SELF should force weight recomputation when starting.
  v_cal = '2.0-00 04-Sep-2024' ! AMPLITUDE command
  v_cal = '2.0-01 09-Jan-2025' ! AMPLITUDE command now functional.
  endif
  v_cal = '2.1-01 15-Apr-2025' ! New MODEL engine
  !
  ! BUNDLES\  Language versions  v_bundles
  if (old) then
  v_bundles = '1.0-00 29-Nov-2019' ! Move the COMBINE and SELFCAL to the BUNDLES\ language
  v_bundles = '1.1-00 11-Mar-2020' ! Add the EXPLORE tool
  v_bundles = '1.2-00 25-Mar-2020' ! Add the KEPLER tool
  v_bundles = '1.2-01 29-Mar-2020' ! Add the KEPLER_VELO variable
  v_bundles = '1.2-02 13-May-2020' ! Add the KEPLER /RESET option
  v_bundles = '1.3-01 22-May-2020' ! Add the UV_DETECT /COMPRESS command
  v_bundles = '1.3-02 26-May-2020' ! Better UV_DETECT command
  v_bundles = '1.3-03 30-May-2020' ! First correct UV_DETECT command
  v_bundles = '1.3-04 10-Jun-2020' ! Handle KEPLER_X0 and KEPLER_Y0 properly
  v_bundles = '1.3-05 09-Jun-2020' ! Re-size CONTINUUM data if needed in UV_RESTORE
  v_bundles = '1.4-01 30-Jul-2020' ! /VDISK option for KEPLER
  v_bundles = '1.4-02 01-Sep-2020' ! /VELOCITY option for KEPLER
  v_bundles = '1.4-03 09-Sep-2020' ! /MASK option for KEPLER
  v_bundles = '1.4-04 11-Sep-2020' ! /VDISK renamed to /VSYSTEM in KEPLER
  v_bundles = '1.5-05 14-Dec-2020' ! SPECTRAL_CLEAN command
  v_bundles = '1.5-06 14-Dec-2020' ! /HFS option to KEPLER SHOW
  v_bundles = '1.6-00 28-Oct-2021' ! Move command EXPLORE to DISPLAY\  Language
  v_bundles = '1.6-01 18-Nov-2021' ! Finish moving command EXPLORE to DISPLAY\  Language
  v_bundles = '1.6-02 29-Sep-2022' ! /WIDGET option to KEPLER
  v_bundles = '1.6-03 17-Feb-2023' ! COMBINE now support automatic spatial and spectral resampling
  v_bundles = '1.6-04 25-Jun-2023' ! COMBINE /RELATIVE to ignore absolute position
  v_bundles = '1.6-05 11-Jul-2023' ! COMBINE bug correction when spectral resampling 
  endif
  v_bundles = '1.6-06 24-Jul-2023' ! SELFCAL supports Mosaic
  v_bundles = '1.6-07 31-Aug-2023' ! SELFCAL temporary file handling changes
  !
  ! IMAGER\ Language version v_imager
  v_imager =  '1.0-00 23-Mar-2022' ! First built-in release of IMAGER\ language
  v_imager =  '1.0-01 10-May-2022' ! Insert the PIPELINE command in stack
  v_imager =  '1.0-02 30-May-2022' ! Add the /PLOT option to SDI
  v_imager =  '1.1-00 17-Oct-2022' ! Move the POPUP command to DISPLAY\
  v_imager =  '1.1-01 02-Nov-2022' ! Add the /MODE option to PIPELINE
  v_imager =  '1.2-00 22-Jun-2023' ! Add the SIMULATE command
  v_imager =  '1.2-01 05-Jul-2023' ! SIMULATE FIDELITY and SHOW command
  v_imager =  '1.2-02 13-Jul-2023' ! PIPELINE EXPLORE action
  v_imager =  '1.3-01 05-Jul-2024' ! GCLEAN command
  v_imager =  '1.3-02 17-Jul-2024' ! PIPELINE improvements (UV_PREVIEW, Resampling and line filtering behaviours)
  v_imager =  '1.4-00 04-Sep-2024' ! REPORT command
  !
  ! SIMULATE\ Language version v_simu
  v_simu   =  '1.0-02 07-Jul-2023' ! SIMULATE\ language - ADD_NOISE command
  v_simu   =  '1.0-03 29-Aug-2023' ! Small bug correction in UV_TRACK initialization
  v_simu   =  '1.0-04 14-Sep-2023' ! Bug in the NOISE handling 
  v_simu   =  '1.0-05 05-Sep-2025' ! Better messages in UV_TRACK + better noise guess
  !
  call load_display                ! Load DISPLAY\ language
  !
  ! Load the other languages
  call sic_begin('CLEAN','gag_doc:hlp/imager-help-clean.hlp',mclean, vclean, v_clean//'S.Guilloteau', run_clean, gr_error)
  call sic_begin('CALIBRATE','gag_doc:hlp/imager-help-calib.hlp',ncal, vcal, v_cal//'S.Guilloteau', run_self, gr_error)
  call sic_begin('ADVANCED','gag_doc:hlp/imager-help-news.hlp',nadv, vadv, v_advanced//'S.Guilloteau', run_advanced, gr_error)
  call sic_begin('BUNDLES','gag_doc:hlp/imager-help-bundles.hlp',ncomp, vcomp, v_bundles//'S.Guilloteau', run_bundles, gr_error)
  call sic_begin('IMAGER','gag_doc:hlp/imager-help-imager.hlp',nimag,vimag,v_imager//'S.Guilloteau', run_imager, gr_error)
  call sic_begin('SIMULATE','gag_doc:hlp/imager-help-simu.hlp',nsimu,vsimu,v_simu//'S.Guilloteau', run_simu, gr_error)
  !
  call setup_display  ! Setup the DISPLAY\ language
  !
  call sic_def_inte('IMAGER%VERSION',imager_versi,0,0,.true.,error)
  call sic_def_inte('IMAGER%PATCH',imager_patch,0,0,.true.,error)
  ! 
  call omp_setup      ! Setup Open-MP context
  !
  call sic_def_logi('CALL_TREE',call_debug,.false.,error)
  !
  user_method%do_mask = .true.  ! Unclear why ...
  call sic_def_logi('METHOD_DOMASK',user_method%do_mask,.false.,error) ! To check
  nullify(duvr)
  nullify(duvs)
  call define_imager_vars(error)
  !
  call sic_def_inte('MAP_VERSION',map_version,0,0,.false.,error)
  call sic_def_real('MAP_ROUNDING',map_rounding,0,0,.false.,error)
  call sic_def_inte('MAP_POWER   ',map_power,0,0,.false.,error)
  call gi4_round_forfft(64,iout,error,map_rounding,map_power)
  call sic_def_logi('CLEAN_DYNAMIC',cct_dynamic,.false.,error)
  !
  save_data = .false.  ! Nothing to save at start
  call sic_def_login('SAVE_DATA',save_data,1,mtype,.false.,error)
  call gprompt_set('IMAGER')
  !
  call sic_get_inte('SIC%RAMSIZE',sys_ramsize,error)
  call sic_def_inte('IMAGER_RAM',sys_ramsize,0,0,.false.,error)
  call feather_comm(' ',error) ! Initialize
  !
  call sic_def_inte('BROWSE_LAST',browse_last,0,0,.false.,error)
  !
  ansi_crlf = char(13)//char(10)
  call sic_def_char('ANSI%CRLF',ansi_crlf,.true.,error)
  !
end subroutine init_clean
!
subroutine imager_version
  use gbl_ansicodes
  use language_versions
  use gkernel_types
  use gkernel_interfaces, only : gagout, sic_findfile, sic_getlog, sic_setlog
  !
  !! IMAGER -- Test last IMAGER usage
  !
  character(len=filename_length) :: chain,file,fich
  character(len=20) :: v
  logical :: exist
  integer :: ier
  !
  call gagout(' ')
  call gagout(c_blue//c_bold//'I-IMAGER,  Program version is '//v_program//c_clear)
  call gagout(' ')
  !
  chain = '.imager_last'
  exist = sic_findfile(chain,file,'$HOME/',' ')
  if (exist) then
    open(unit=1,file=file,action='READWRITE')
    read(1,'(A)')
    read(1,'(A)') v
    if (v.ne.v_display) Call gagout(c_red//'W-IMAGER,  DISPLAY\ Language version now '//v_display &
      & //' (last version used '//v//')'//c_clear)
    read(1,'(A)') v
    if (v.ne.v_clean) Call gagout(c_red//'W-IMAGER,  CLEAN\ Language version now '//v_clean &
      & //' (last version used '//v//')'//c_clear)
    read(1,'(A)') v
    if (v.ne.v_cal) Call gagout(c_red//'W-IMAGER,  CALIBRATE\ Language version now '//v_cal &
      & //' (last version used '//v//')'//c_clear)
    read(1,'(A)') v
    if (v.ne.v_advanced) Call gagout(c_red//'W-IMAGER,  ADVANCED\ Language version now '//v_advanced &
      & //' (last version used '//v//')'//c_clear)
    read(1,'(A)') v
    if (v.ne.v_bundles) Call gagout(c_red//'W-IMAGER,  BUNDLES\ Language version now '//v_bundles &
      & //' (last version used '//v//')'//c_clear)
    read(1,'(A)',iostat=ier) v
    if (ier.ne.0) then
      if (v.ne.v_bundles) Call gagout(c_red//'W-IMAGER,  IMAGER\ Language version now '//v_bundles &
        & //' (last version used '//v//')'//c_clear)
    endif
    read(1,'(A)',iostat=ier) 
    if (ier.ne.0) then
      if (v.ne.v_program) Call gagout(c_red//'W-IMAGER,  Main Program version changed, now '//v_program &
        & //' (last version used '//v//')'//c_clear)
    endif
    rewind(1)
  else
    open(unit=1,file=file,status='NEW')  
  endif
  write(1,'(A)') ' '
  write(1,'(A)') v_display
  write(1,'(A)') v_clean
  write(1,'(A)') v_cal
  write(1,'(A)') v_advanced
  write(1,'(A)') v_bundles
  write(1,'(A)') v_imager
  write(1,'(A)') v_program
  close(unit=1)
  !
  write(fich,'(A,I4)') '$HOME/.imager/v',imager_patch
  exist = sic_findfile(fich,file,' ',' ')
  if (exist) then
    ier = sic_getlog('MACRO#DIR:',chain)
    if (chain(1:3).eq.'./;') then
      chain = chain(1:3)//trim(fich)//'/;'//trim(chain(4:))
    else
      chain = trim(fich)//'/;'//trim(chain)
    endif
    ier = sic_setlog('MACRO#DIR:',chain)
    call gagout(c_red//'W-IMAGER,  Using updated procedures from '//trim(file)//c_clear)
  endif
end subroutine imager_version
!
