subroutine init_display 
  !! Initialize the DISPLAY language from VIEWER
  call load_display
  call setup_display
  call gprompt_set('VIEWER')
end subroutine init_display
!
subroutine setup_display
  use gkernel_interfaces
  use clean_def
  use clean_arrays
  use clean_default
  use clean_types
  !----------------------------------------------------------------------
  ! @ public
  !*
  ! IMAGER -- Initialize the DISPLAY\ language associated variables
  !!
  !----------------------------------------------------------------------
  !
  logical :: error
  !
  error = .false.
  ImagerError = .false.
  call sic_def_logi('MAPPING_ERROR',ImagerError,.false.,error)
  !
  call sic_def_char('GTV%ROOT',gtv_root,.false.,error)
  !
  ! Re-define the DO_LOOP variable for VIEW
  call sic_delvariable('DO_LOOP',.false.,error)
  call sic_def_logi('DO_LOOP',view_do_loop,.false.,error)
  view_do_loop = .true.
  !
  call sic_delvariable('SHOW_SIDE',.false.,error)
  call sic_def_logi('SHOW_SIDE',show_side,.false.,error)
  show_side = .false.
  !
  ! Angular Unit of Angle-like axes
  call sic_def_inte('ANGLE_UNIT',iangle_unit,0,0,.false.,error)
  call sic_delvariable('SIZE',.false.,error)
  call sic_delvariable('CENTER',.false.,error)
  call sic_def_dble('SIZE',area_size,1,2,.false.,error)
  call sic_def_dble('CENTER',area_center,1,2,.false.,error)
  !
  ! Last type of active Image (CLEAN or SKY)
  call sic_def_char('LAST_SHOWN',last_shown,.false.,error)
end subroutine setup_display
!
subroutine run_self (line,comm,error)
  use clean_arrays
  use clean_default
  use gbl_message
  use imager_interfaces, except_this=>run_self
  use gkernel_interfaces, only : sic_present, sic_mapgildas, sic_delvariable, sic_lire
  !----------------------------------------------------------------------
  ! @ public
  !*
  ! IMAGER  -- Dispatching routine for Language CALIBRATE\
  !!
  !----------------------------------------------------------------------
  character(len=*), intent(inout) :: line    !! Command line
  character(len=*), intent(in)    :: comm    !! Command name
  logical,          intent(out)   :: error   !! Logical error flag
  !
  ! Local ---
  integer, save :: icall=0
  integer :: i
  logical :: quiet, tmod
  real(8) :: elapsed
  !
  ! Code ----
  if (icall.ne.0) then
    Print *,'Rentrant call to RUN_SELF ',comm
    read(5,*) i
  endif
  icall = icall+1
  !
  call report_init(elapsed)
  call quiet_message(line,quiet)
  !
  ImagerError = .false.
  call map_message(seve%c,'CALIBRATE',line)
  !
  select case (comm)
  case ('UV_SELF')
    call uv_map_comm(line,comm,error)
  case ('APPLY')
    call apply_comm(line,error)
  case ('MODEL') 
    tmod = .false.
    call sic_get_logi('TEST_MODEL',tmod,error)
    if (tmod) then
      Print *,'Calling uv_model_comm_bis '
      call uv_model_comm_bis(line,error)
    else
      call uv_model_comm(line,error)
    endif
  case ('SOLVE') 
    call solve_comm(line,error)
  case ('COLLECT')
    call collect_comm(line,comm,error)
  case ('DERIVE')
    call derive_comm(line,comm,error)
  case ('SCALE_FLUX')
    call flux_dispatch (line,error)
  case ('UV_SELECT')
    call uv_select_comm(line,comm,error)
  case ('UV_SORT')
    if (sic_present(1,0)) then
      call uv_sort_file(line,error)
    else
      call uv_sort_comm(line,error)
    endif
  case ('FIND')
    call catalog_find(line,error)
    quiet = .true.
  case ('TRANSFORM')
    call transform_comm(line,error)
  case ('PLAIT')
    call map_message(seve%w,'CALIBRATE',comm//' is experimental')
    call plait_comm(line,comm,error)
  case ('AMPLITUDE')
    call amplitude_comm(line,error)
  case default
    call map_message(seve%e,'CALIBRATE','No code for '//comm)
    error = .true.
  end select
  icall = icall-1  
  if (ImagerError) error = .true.
  !
  call end_message(comm,elapsed,quiet,error)
  !
end subroutine run_self
!
subroutine run_clean (line,comm,error)
  use clean_arrays
  use clean_default
  use gbl_message
  use imager_interfaces, except_this=>run_clean
  use gkernel_types
  use gkernel_interfaces, only : sic_present, sic_mapgildas, sic_delvariable
  !----------------------------------------------------------------------
  ! @ public
  !*
  ! IMAGER   -- Dispatching routine for Language CLEAN\
  !!
  !----------------------------------------------------------------------
  character(len=*), intent(inout) :: line    !! Command line
  character(len=*), intent(in)    :: comm    !! Command name
  logical,          intent(out)   :: error   !! Logical error flag
  !
  ! Local ---
  integer, save :: icall=0
  type(sic_descriptor_t) :: desc   ! Descriptor
  integer :: i
  logical :: err, found, quiet
  real(8) :: elapsed
  !
  ! Code ---
  if (icall.ne.0) then
     Print *,'Rentrant call to RUN_CLEAN ',comm
     read(5,*) i
  endif
  icall = icall+1
  !
  call report_init(elapsed)
  call quiet_message(line,quiet)
  !
  ImagerError = .false.
  call map_message(seve%c,'CLEAN',line)
  !
  ! Analyze command
  ! First cases do not require to update variables
  select case (comm)
  case ('FIT')
    call fit_comm(line,error)
    quiet = .true.
  case ('DISCARD')
    call discard_comm(line,comm,error)
    quiet = .true.
  case ('COLOR')
    call color_comm(line,error)
    quiet = .true.
  case ('WRITE')
    call write_comm(line,error)
  case ('SUPPORT')
    call support_comm(line,error)
    quiet = .true.
  case ('UV_STAT')
    call uv_stat_comm(line,error)
  case ('DUMP')
    call debug_all(line,error)
    quiet = .true.
  case ('LOG')
    call log_stack_comm(line,error)
    quiet = .true.
  case ('BUFFERS')
    call buffers_comm(line,error)
    quiet = .true.
  case ('MOSAIC')
    call mosaic_comm (line,error)
    quiet = .true.
  case ('CCT_MERGE')
    call cct_merge_comm(line,error)
  !
  ! Other cases should update their own variables
  case default
    select case (comm)
    case ('READ')
      call read_comm(line,error)
    case ('CLEAN')
      call clean_comm(line,error)
    case ('MX')
      if (themap%nfields.ne.0) then
        call map_message(seve%e,comm,'UV data is a Mosaic, not supported')
        error = .true.
      else 
        call map_message(seve%w,comm,'MX is deprecated, use at your own risk',1)
        call mx_uvmap_clean('MX',line,error)
      endif
!OBSOLETE!    case ('ALMA')
!OBSOLETE!      call alma_clean(line,error)
    case ('UV_CHOP') 
      call uv_chop_comm(line,comm,error)
    case ('UV_FLAG')
      call uv_flag_comm(line,error)
    case ('UV_MAP')
      call uv_map_comm(line,comm,error)
    case ('UV_RESAMPLE','UV_COMPRESS','UV_HANNING','UV_SMOOTH')
      call uv_resample_comm(line,comm,error)
    case ('UV_RESTORE')
      call uv_map_comm(line,comm,error)
    case ('UV_SHIFT')
      call uv_shift_comm(line,comm,error)
    case ('UV_CONTINUUM')
      call uv_line2cont (line,error)
    case ('UV_FILTER')
      call uv_filter_comm (line,error)
    case ('UV_BASELINE')
      call uv_baseline_comm (line,error)
    case ('UV_TIME')
      if (themap%nfields.ne.0) call map_message(seve%w,comm,'UV data is a Mosaic')
      call uv_time_comm (line,error)
    case ('UV_TRUNCATE')
      call uv_truncate_comm (line,error)
    case ('UV_TRIM')
      call uv_trim_comm (line,error)
    case ('UV_REWEIGHT')
      call uv_reweight_comm (line,comm,error)
    case ('UV_CHECK')
      call uv_check_comm (line,error)
    case ('UV_FIELDS')
      call uv_fields_comm (line,comm,error)
    case ('UV_SUBTRACT') 
      call uv_subtract_comm (line,comm,error)
    case ('PRIMARY')
      call primary_comm (line,error)
    case ('SPECIFY')
      call specify_comm (line,error)
      quiet = .true.
    case ('UV_RESIDUAL')
      call uv_residual_comm(line,error)
    case ('MAP_RESAMPLE','MAP_COMPRESS','MAP_INTEGRATE','MAP_SMOOTH') 
      call map_resample_comm(line,comm,error)
    case ('MAP_COMBINE')
      call map_combine_comm(line,error)
    case ('MAP_REPROJECT')
      call map_reproject_comm(line,error)
    case ('UV_SPLIT')
      call uv_split_comm(line,error)
    case ('UV_EXTRACT')
      call uv_extract_comm(line,error)
    case ('CCT_CONVERT')
      call cct_convert_comm(line,error)
    case ('CCT_CLEAN')
      call cct_clean_comm(line,error)
    case ('MAP_CONVOLVE')
      call map_convolve_comm(line,error)
    case default
      call map_message(seve%i,'CLEAN',comm//' not yet implemented')
      icall = icall-1
      error = .true.
      return
    end select
    ! Delete variables that may have been affected
    call sic_descriptor('SPECTRA',desc,found)  
    err = .false.
    if (found) call sic_delvariable('SPECTRA',.false.,err)
  end select
  icall = icall-1
  if (ImagerError) error = .true.
  !
  call end_message(comm,elapsed,quiet,error)
end subroutine run_clean
!
subroutine run_display (line,comm,error)
  use clean_arrays
  use clean_default
  use gbl_message
  use imager_interfaces, except_this=>run_display
  use gkernel_types
  use gkernel_interfaces
  !----------------------------------------------------------------------
  ! @ public
  !*
  ! IMAGER   -- Dispatching routine for Language DISPLAY\
  !!
  !----------------------------------------------------------------------
  character(len=*), intent(inout) :: line    !! Command line
  character(len=*), intent(in)    :: comm    !! Command name
  logical,          intent(out)   :: error   !! Logical error flag
  !
  ! Memory -- 
  integer, save :: icall=0
  character(len=16), save :: last_comm=' '
  character(len=64), save :: last_arg =' '
  character(len=16), save :: last_root='<GREG'
  logical, save :: force_clear=.false.    ! Patch for the Static Pen LUT in VIEW CCT
  ! Local ---
  integer :: i, ia
  logical :: quiet
  real(8) :: elapsed
  character(len=8) :: key
  character(len=64) :: arg
  !
  ! Code ----
  if (icall.ne.0) then
    if  ( (icall.eq.1) .and. ( &
      & (comm.eq.'FIND').or.(comm.eq.'STATISTIC').or. &
      & (comm.eq.'LOAD').or.(comm.eq.'SPECTRUM').or. &
      & (comm.eq.'SET') .or.(comm.eq.'EXTRACT').or. &
      & (comm.eq.'SLICE') ) ) then
      continue
    else
      Print *,'Unforeseen Rentrant call to RUN_DISPLAY ',comm
      read(5,*) i
    endif
  endif
  icall = icall+1
  !
  ImagerError = .false.
  quiet = .true.  ! In general here, as these are interactive commands
  call report_init(elapsed)
  call map_message(seve%c,'DISPLAY',line)
  !
  ! Dispatch command
  select case (comm)
  case ('CATALOG')
    call catalog_comm(line,quiet,error)
  case ('COLOR')
    call color_comm(line,error)
  case ('LOAD')
    call view_load_comm (line,error)
    quiet = sic_lire().ne.0 .or. icall.gt.1
  case ('STATISTIC')
    call statistic(line,error)
    quiet = sic_lire().ne.0 .or. icall.gt.1
  case ('EXTRACT') 
    call extract_comm(line,error)
  case ('FIND')
    call catalog_find(line,error)
  case ('POPUP')
    ia = index(line,' ')
    call exec_program('@ p_popup_all '//line(ia:)) !SG
    quiet = .true.
  case ('SPECTRUM')
    call greg3_spectrum_compute(line,error)
    quiet = sic_lire().ne.0 .or. icall.gt.1
  case ('SET')
    call display_set_comm(line,comm,error)
    quiet = .true.
  case ('SLICE') 
    call slice_comm(line,error)
    quiet = sic_lire().ne.0 .or. icall.gt.1
  case default
    if (force_clear) call gtv_destroy_all(error)
    error = .false.
    !
    ! Reset the LUT to dynamic
    !
    if ((comm.eq.'SHOW').and.sic_present(0,1)) then
      call sic_ke(line,0,1,key,ia,.true.,error)
      if (key.eq.'MOSAIC') then
        call mosaic_show(huv%mos%fields)
        icall = icall-1
        return
      endif
    endif
    ! All effective plotting commands with a /DIR option
    quiet = .true.
    call sic_let_char('GTV%ROOT','<GREG',error)
    if (comm.eq.last_comm .and. gtexist(last_root)) then
      if (comm.eq.'SHOW' .or. comm.eq.'VIEW' .or. comm.eq.'INSPECT') then
        if (sic_present(0,1)) then
          call sic_ke(line,0,1,arg,ia,.true.,error)
          if (arg.eq.last_arg) then
            call sic_let_char('GTV%ROOT',last_root,error)
          else
            last_arg = arg
          endif
        else
          call sic_let_char('GTV%ROOT',last_root,error)
        endif
      endif
    endif
    !
    select case (comm)
    case ('COMPARE')
      call compare_comm(line,error)
    case ('SHOW','VIEW','INSPECT_3D')
      if (sic_present(0,1)) call sic_ke(line,0,1,last_arg,ia,.true.,error)
      call display_buffer(comm,line,error)
      force_clear = comm.eq."VIEW".and.last_arg.eq."CCT"
    case ('EXPLORE')
      call explore_comm(line,comm,error)
    case default
      call map_message(seve%i,'DISPLAY',comm//' not yet implemented')
      icall = icall-1
      error = .true.
      return
    end select
    last_comm = comm 
    call sic_get_char('GTV%ROOT',last_root,ia,error)
    call sic_upper(last_root)
  end select
  icall = icall-1
  if (ImagerError) error = .true.
  !
  call end_message(comm,elapsed,quiet,error)
end subroutine run_display
!
subroutine end_message(comm,elapsed,quiet,error)
  use gbl_ansicodes
  use gbl_message
  use imager_interfaces, only : map_message
  !-----------------------------------------------------------
  !@ private
  !
  !! IMAGER -- Print a Timing and Status message if needed
  !-----------------------------------------------------------
  character(len=*), intent(in) :: comm   !! Command line name
  logical, intent(in) :: error           !! Error flag
  logical, intent(in) :: quiet           !! Quiet flag
  real(8), intent(inout) :: elapsed      !! Elapsed time
  !
  character(len=80) :: chain
  !
  if (error) then
    call map_message(seve%e,comm,'failed to complete',1)  
  else if (.not.quiet) then
    call report_time(elapsed)
    if (elapsed.gt.0) then
      write(chain,'(a,a,f9.2,a)') 'Successful completion','  - Elapsed time ',elapsed,' sec'
    else
      chain = 'Successful completion '
    endif
    call map_message(seve%i,comm,chain,2)
  endif
  !
end subroutine end_message
!
module time_support
  use gsys_types
  !! Timing support module
  real(8), save :: elapsed_s    !! Start time
  real(8), save :: elapsed_e    !! End (current) time
  logical, save :: nopara       !! Parallel mode
  type(cputime_t), save :: time   !! Timing structure
end module time_support
!
subroutine report_time(elapsed)
  !$ use omp_lib
  use time_support
  ! @ private
  !! IMAGER -- Timing support routine: compute Elapsed time
  real(8), intent(inout) :: elapsed !! Elapsed time
  !
  if (nopara) then
    call gag_cputime_get(time)
    elapsed = time%diff%elapsed
  endif
  !$ elapsed_e = omp_get_wtime()
  !$ elapsed = elapsed_e - elapsed_s
end subroutine report_time
!
subroutine report_init(elapsed)
  !$ use omp_lib
  use time_support
  ! @ private
  !! IMAGER -- Timing support routine: Initialize Elapsed starting time
  real(8), intent(inout) :: elapsed !! Elapsed time
  !
  elapsed = 0
  !
  nopara = .true.
  !$ nopara = .false.
  if (nopara) call gag_cputime_init(time)
  !$ elapsed_s = omp_get_wtime()
end subroutine report_init
!
subroutine run_advanced(line,comm,error)
  use clean_arrays
  use clean_default
  use gbl_message
  use imager_interfaces, except_this=>run_advanced
  !----------------------------------------------------------------------
  ! @ public
  !*
  ! IMAGER  -- Dispatching routine for Language ADVANCED\
  !!
  !----------------------------------------------------------------------
  character(len=*), intent(inout) :: line    !! Command line
  character(len=*), intent(in)    :: comm    !! Command name
  logical,          intent(out)   :: error   !! Logical error flag
  !
  ! Local ---
  integer, save :: icall=0
  integer :: i, icode
  real(8) :: elapsed
  logical :: quiet
  !
  ! Code ----
  if (icall.ne.0) then
    Print *,'Rentrant call to RUN_ADVANCED ',comm
    read(5,*) i
  endif
  icall = icall+1
  !
  quiet = .true.
  call report_init(elapsed)
  !
  ImagerError = .false.
  call map_message(seve%c,'ADVANCED',line)
  !
  ! Analyze command
  select case (comm)
  case ('UV_PREVIEW')
    call uv_preview(line,error)
    quiet = .false.
  case ('PROPER_MOTION')
    call comm_proper_motion(line,comm,error)
    quiet = .false.
  case ('MAP_CONTINUUM')
    call map_continuum(line,error)
    quiet = .false.
  case ('MAP_POLAR')
    call map_polar(line,comm,error)
  case ('MFS')
    call clean_mfs (line,error)
  case ('UV_DEPROJECT','UV_RADIAL','UV_CIRCLE') 
    call uv_radial(line,comm,error)
    quiet = .false.
  case ('UV_SHORT')
    call uvshort_comm(line,comm,error,quiet)
  case ('XY_SHORT')
    call uvshort_comm(line,comm,error,quiet)
  case ('UV_FIT')
    call uvfit_comm(line,error)
    quiet = .false.
  case ('STOKES')
    call stokes_comm(line,error)
    quiet = .false.
  case ('EXTRACT') 
    call extract_comm(line,error)
  case ('MOMENTS')
    call moments_comm(line,error)
    quiet = .false.
  case ('MASK') 
    call mask_comm(line,error)
  case ('UV_ADD') 
    call uv_add_comm(line,error)
    quiet = .false.
  case ('UV_MERGE') 
    call uv_merge_comm(line,error)
    quiet = .false.
  case ('FLUX')
    call flux_comm(line,error)
  case ('HOW_TO')
    icode = 0
    call howto_comm(line,error,icode)
    if (icode.eq.1) call howto_comm(line,error,icode)
  case ('FEATHER')
    call feather_comm(line,error)
    quiet = .false.
  case ('UV_CORRELATE')
    call correlate_comm(line,comm,error)
    quiet = .false.
  case ('UV_MOSAIC')
    call uv_mosaic_comm(line,comm,error)
    quiet = .false.
  case ('UV_GRID')
    call uv_grid_comm (line,comm,error)      
  case ('UV_LINEARIZE')
    call uv_linear_comm (line,comm,error)
    quiet = .false.
  case default
    call map_message(seve%i,'ADVANCED',comm//' not yet implemented')
  end select
  icall = icall-1
  if (ImagerError) error = .true.
  !
  call end_message(comm,elapsed,quiet,error)
  !
end subroutine run_advanced
!
subroutine run_bundles(line,comm,error)
  use clean_arrays
  use clean_default
  use gbl_message
  use imager_interfaces, except_this=>run_bundles
  !----------------------------------------------------------------------
  ! @ public
  !*
  ! IMAGER -- Dispatching routine for Language BUNDLES\
  !!
  !----------------------------------------------------------------------
  character(len=*), intent(inout) :: line    !! Command line
  character(len=*), intent(in)    :: comm    !! Command name
  logical,          intent(out)   :: error   !! Logical error flag
  !
  ! Local ---
  integer, save :: icall=0
  integer :: i
  real(8) :: elapsed
  logical :: quiet
  !
  ! Code ----
  quiet = .true.
  call report_init(elapsed)
  !
  if (icall.ne.0) then
    Print *,'Rentrant call to RUN_BUNDLES ',comm
    read(5,*) i
  endif
  icall = icall+1
  !
  ImagerError = .false.
  call map_message(seve%c,'BUNDLES',line)
  !
  ! Analyze command
  select case (comm)
  case ('SELFCAL') 
    call selfcal_comm(line,comm,error)
  case ('COMBINE')
    call combine_comm(line,error)
    quiet = .false.
!  case ('EXPLORE')
!    call explore_comm(line,comm,error)
  case ('KEPLER')
    call kepler_comm(line,comm,quiet,error)
  case ('UV_DETECT')
    call detect_comm(line,comm,error)
    quiet = .false.
  case ('SPECTRAL_CLEAN')
    call spectral_comm(line,comm,error)
  case default
    call map_message(seve%i,'BUNDLES',comm//' not yet implemented')
  end select
  icall = icall-1
  if (ImagerError) error = .true.
  !
  call end_message(comm,elapsed,quiet,error)
  !
end subroutine run_bundles
!
subroutine run_imager (line,comm,error)
  use clean_arrays
  use clean_default
  use gbl_message
  use imager_interfaces, except_this=>run_imager
  use gkernel_types
  use gkernel_interfaces, only : sic_present, sic_lire, sic_ambigs, sic_narg
  !----------------------------------------------------------------------
  ! @ public
  !*
  ! IMAGER  -- Dispatching routine for Language IMAGER\
  !!
  !----------------------------------------------------------------------
  character(len=*), intent(inout) :: line    !! Command line
  character(len=*), intent(in)    :: comm    !! Command name
  logical,          intent(out)   :: error   !! Logical error flag
  !
  ! Constants
  integer(4), parameter :: mvoc=4
  character(len=12) :: voca(mvoc)
  data voca/'ALL', 'CONTINUUM', 'SPLIT','SURVEY'/
  integer(4), parameter :: msim=7
  character(len=12) :: vsim(msim)
  data vsim/'COVERAGE','FIDELITY','IMAGE','MODEL','NOISE','SETUP','SHOW'/
  !
  ! Local ---
  integer, save :: icall=0
  character(len=16), save :: ocomm
  character(len=12) :: argum,name
  integer(4) :: i, ia,ie, ja,n
  logical :: do_insert
  logical :: quiet
  real(8) :: elapsed
  !
  ! Code ----
  ImagerError = .false.
  call map_message(seve%c,'IMAGER',line)
  !
  if (icall.ne.0) then
    if  ( (icall.eq.1) .and. (ocomm.eq.'REPORT') ) then
      continue
    else
      Print *,'OCOMM ',ocomm
      Print *,'Rentrant call to RUN_IMAGER ',comm
      read(5,*) i
    endif
  endif
  icall = icall+1
  if (icall.eq.1) ocomm = comm
  !
  ImagerError = .false.
  call map_message(seve%c,'IMAGER',line)
  quiet = .true.
  !
  ! Dispatch command
  select case (comm)
  case ('CLARK','HOGBOM','MRC','MULTISCALE','SDI','GCLEAN') 
    call report_init(elapsed)
    call quiet_message(line,quiet)
    !
    if (comm.eq.'MULTISCALE') then
      call sic_let_char('METHOD','MULTI',error)
    else if (comm.eq.'GCLEAN') then
      call sic_let_char('METHOD','GAUSS',error)
    else
      call sic_let_char('METHOD',comm,error)
    endif
    call clean_comm(line,error)
  case ('PIPELINE')
    do_insert = sic_lire().eq.0
    if (sic_present(2,0)) then        ! /WIDGET option
      if (sic_present(1,0)) then
        call map_message(seve%e,comm,'/MODE and /WIDGET options are conflicting')
        icall = icall-1
        error = .true.
        return
      endif
      call exec_program('@ all-widget')
    else if (sic_present(1,0)) then   ! /MODE option
      !
      icall = icall-1
      call sic_ke(line,1,1,argum,n,.true.,error)
      if (error) return
      call sic_ambigs (comm,argum,name,n,voca,mvoc,error)
      if (error) return
      argum = ' '
      call sic_ke(line,0,1,argum,n,.false.,error)
      n = max(n,1)
      !
      icall = icall+1
      call exec_program('@ all-pipeline "'//argum(1:n)//'" '//name)
    else
      ia = index(line,' ')
      call exec_program('@ all-pipeline'//line(ia:))
    endif
    if (do_insert) call sic_insert_log(line)
  case ('REPORT')
    if (sic_narg(0).ne.0) then
      call sic_ch(line,0,1,argum,n,.false.,error)
      call exec_program('@ all-report '//argum(1:n))
    else
      call exec_program('@ all-report')
    endif    
  case ('TIPS')
    if (sic_narg(0).ne.0) then
      call sic_ch(line,0,1,argum,n,.false.,error)
      call exec_program('@ gag_pro:tips '//argum(1:n))
    else
      call exec_program('@ gag_pro:tips')
    endif    
  case ('SIMULATE')
    do_insert = sic_lire().eq.0
    ia = index(line,'/')
    ie = len_trim(line)
    !
    ! Parse the arguments to lift ambiguities
    if (sic_present(0,1)) then
      icall = icall-1
      call sic_ke(line,0,1,argum,n,.true.,error)
      if (error) return
      if (argum.eq.'*') then
        continue
      else if (argum.eq.'?') then
        continue
      else
        call sic_ambigs (comm,argum,name,n,vsim,msim,error)
        if (error) return
        argum = name
        ja = index(line,' ')+1
        if (ia.ne.0) then
          line(ja:) = trim(argum)//line(ia-1:ie)  ! Need the Space before the "/"
        else if (argum.ne.'SHOW') then
          line(ja:) = argum
        else
          name = ' '
          call sic_ke(line,0,2,name,n,.false.,error)
          line(ja:) = trim(argum)//' '//trim(name)
        endif
        ie = len_trim(line)
      endif
      icall = icall+1
    endif
    !
    ia = index(line,' ')
    ie = max(ia,ie)
    call exec_program('@ p_simulate '//line(ia:ie))
    if (do_insert) call sic_insert_log(line)
  case default
    call map_message(seve%i,'IMAGER',comm//' not yet implemented')
    icall = icall-1
    error = .true.
    return
  end select
  icall = icall-1
  if (ImagerError) error = .true.
  if (icall.eq.0) ocomm = ' '
  !
  call end_message(comm,elapsed,quiet,error)
end subroutine run_imager
!
subroutine run_simu (line,comm,error)
  use clean_arrays
  use clean_default
  use gbl_message
  use imager_interfaces, except_this=>run_simu
  use gkernel_types
  use gkernel_interfaces, only : sic_present, sic_lire, sic_ambigs
  !----------------------------------------------------------------------
  ! @ public
  !*
  ! IMAGER  -- Dispatching routine for Language SIMULATE\
  !!
  !----------------------------------------------------------------------
  character(len=*), intent(inout) :: line    !! Command line
  character(len=*), intent(in)    :: comm    !! Command name
  logical,          intent(out)   :: error   !! Logical error flag
  !
  ! Constants
  integer(4), parameter :: mvoc=4
  character(len=12) :: voca(mvoc)
  data voca/'ALL', 'CONTINUUM', 'SPLIT','SURVEY'/
  !
  ! Local ---
  integer, save :: icall=0
  integer(4) :: i
  logical :: quiet
  real(8) :: elapsed
  !
  ! Code ----
  if (icall.ne.0) then
    Print *,'Unforeseen Rentrant call to RUN_SIMULATE ',comm
    read(5,*) i
  endif
  icall = icall+1
  !
  ImagerError = .false.
  call map_message(seve%c,'SIMULATE',line)
  quiet = .true.
  !
  ! Dispatch command
  select case (comm)
  case ('UV_TRACKS')
    call astro_uv(line,error)
  case ('ARRAY')
    call astro_observatory_command(line,error)
  case ('ADD_NOISE')
    call uv_addnoise(line,comm,error)
  case default
    call map_message(seve%i,'SIMULATE',comm//' not yet implemented')
    icall = icall-1
    error = .true.
    return
  end select
  icall = icall-1
  if (ImagerError) error = .true.
  !
  call end_message(comm,elapsed,quiet,error)
end subroutine run_simu
!
subroutine save_result(error)
  use gkernel_interfaces
  use gildas_def
  use clean_types
  use gbl_message
  use imager_interfaces, only : map_message
  !-------------------------------------------------------------------
  !! IMAGER - Exit routine: ask user whether to save data or not
  !-------------------------------------------------------------------
  logical error !! Logical error flag
  !
  character(len=filename_length) :: ans
  character(len=filename_length) :: autosave='autosave'  ! Default name when saving
  character(len=commandline_length) line
  character(len=message_length) :: chain
  character(len=*), parameter :: rname = 'IMAGER'
  integer :: n, icode
  logical :: do_write,noprompt
  !
  ans = ' '
  error = .false.
  do_write = .false.
  call gmaster_get_exitnoprompt(noprompt)  ! Check for SIC\EXIT /NOPROMPT
  !
  if (any(save_data) .and. .not.noprompt) then
    call map_message(seve%w,rname,'There are unsaved data, should we save them?')
    do while (.true.)
      call sic_wprn('Y)es, N)o, C)ancel exit (default No): ',ans,n)
      call sic_upper(ans)
      if (ans(1:1).eq.'C') then
        error = .true.
        return
      elseif (ans(1:1).eq.'N' .or. ans(1:1).eq.' ') then
        do_write = .false.
      elseif (ans(1:1).ne.'Y') then
        call map_message(seve%e,rname,'Unrecognized answer '//ans)
        cycle
      else 
        do_write = .true.
      endif
      exit
    enddo
  endif
  !
  call sic_delvariable('DIRTY',.false.,error)
  call sic_delvariable('CLEAN',.false.,error)
  call sic_delvariable('RESIDUAL',.false.,error)
  call sic_delvariable('PRIMARY',.false.,error)
  call sic_delvariable('SKY',.false.,error)
  if (.not.do_write) return
  !
  do icode=1,mtype
    ans = ' '
    error = .false.
    do while (save_data(icode))
       write(chain,100) vtype(icode)
       call sic_wprn(trim(chain)//' ',ans,n)
       if (ans.eq.' ') ans = autosave
       line = 'CLEAN'//char(92)//'WRITE '//vtype(icode)//' "'//trim(ans)//'"'
       call exec_command (line,error)
    enddo
  enddo
  !
100 format('Enter a filename for ',a,' (default ''autosave''): ')
end subroutine save_result
!
subroutine debug_all(line,error)
  !$ use omp_lib
  use gkernel_interfaces
  use imager_interfaces, except_this => debug_all
  use clean_def
  use clean_arrays
  use clean_types
  use clean_default
  use gbl_message
  !----------------------------------------------------------------------
  ! @  private
  !
  !! IMAGER  -- Support routine for command DEBUG
  !----------------------------------------------------------------------
  character(len=*), intent(inout) :: line  !! Command line
  logical, intent(out) :: error            !! Logical error flag
  !
  ! Constants
  integer, parameter :: mvoc=6
  character(len=8) :: voca(mvoc)
  data voca /'BUFFERS', 'MEMORY', 'METHOD', 'SG', 'THREADS ', 'USER    '/
  ! Local ---
  character(len=8) :: name,argum
  integer :: n, nc, ier, mthread
  integer, allocatable :: mic(:)
  !
  ! Code ----
  argum = '?'
  call sic_ke(line,0,1,argum,n,.false.,error)
  if (error) return
  call sic_ambigs ('MOSAIC',argum,name,n,voca,mvoc,error)
  if (error) return
  !
  select case(name)
  case ('MEMORY')
    call dump_memory(error)
  case ('USER') 
     call mapping_print_debug(user_method)
  case ('METHOD') 
     call mapping_print_debug(the_method)
  case ('BUFFERS') 
    call uv_dump_buffers('DUMP')
  case ('SG') 
    !
    nc = huv%gil%nchan
    if (nc.ne.hcct%gil%dim(2)) then
      Print *,'Mismatch number of channels between HUV ',nc,' and CCT ',hcct%gil%dim(2)
    endif
    !
    allocate(mic(nc),stat=ier)
    call uv_clean_sizes(hcct,dcct,mic,1,nc)
    Print *,'SG_DEBUG -- MIC ',mic
    deallocate(mic)
  case ('THREADS') 
    mthread = 0
    !$ mthread = omp_get_max_threads()
    Print *,'Number of Threads ',mthread
  end select
  !
end subroutine debug_all
!
subroutine quiet_message(line,quiet)
  use gildas_def
  use gkernel_interfaces
  !----------------------------------------------------------------------
  ! @ private
  !! IMAGER -- Test if command should be "Quiet", i.e. is only a Help-like command
  !----------------------------------------------------------------------
  character(len=*), intent(in) :: line    !! Command line
  logical, intent(out) :: quiet           !! Is command quiet ?
  !
  logical :: error
  character(len=filename_length) :: ctest
  integer :: i
  !
  if (sic_present(0,1)) then
    ctest = ' '    
    call sic_ch(line,0,1,ctest,i,.false.,error)
    quiet = (ctest(1:1).eq.'?')
    if (.not.quiet) then
      if (sic_present(0,2)) then
        call sic_ch(line,0,2,ctest,i,.false.,error)
        quiet = (ctest(1:1).eq.'?')
      endif
    endif
  else
    quiet = .false.
  endif
end subroutine quiet_message
