!
subroutine amplitude_comm(line,error)
  use gildas_def
  use gkernel_interfaces
  use imager_interfaces, except_this => amplitude_comm
  use gbl_message
  use clean_arrays
  !---------------------------------------------------------------------
  ! @ private
  !*
  !   IMAGER -- Support routine for 
  !     CALIBRATE\AMPLITUDE Factor [/ANTENNA Number]
  !         [/FILE UvIN [UvOut]] 
  ! 
  ! Scale the gain of (all or) the specified Antenna by the
  ! given Factor. 
  ! Work on current UV data if no /FILE option.
  ! Work in place if /FILE is specified, and no UvOut argument is given
  !!
  !---------------------------------------------------------------------
  character(len=*), intent(in) :: line  !! Command line
  logical, intent(inout) :: error       !! Logical error flag
  !
  ! Global --
  character(len=*), parameter :: rname='AMPLITUDE'
  real(8), parameter :: pi=3.141592653589793d0
  integer, parameter :: o_ant=1, o_file=2
  !
  ! Local ---
  character(len=80) :: mess
  real, pointer :: duv_previous(:,:), duv_next(:,:)
  integer(4) :: nvis, ncol, nvg, ndc, ier
  real(4) :: gain
  real(4), allocatable :: gains(:)
  integer(4), allocatable :: iant(:)
  real(4), allocatable :: luv(:,:)
  character(len=filename_length) :: file
  type (gildas) :: uvin, uvou
  logical :: dofile, err
  integer :: nblock, ib, nf, narg, nant, ngain, i, j
  !
  ! Code ----
  error = .false.
  err = .false.
  !
  nant = sic_narg(o_ant)
  ngain = max(sic_narg(0),1)
  if (nant.gt.0) then
    allocate(iant(nant),gains(nant),stat=ier)
    do i=1,nant
      call sic_i4(line,o_ant,i,iant(i),.false.,error)
      if (error) return
      j = min(i,ngain)
      call sic_r4(line,0,i,gains(j),.true.,error)
      if (error) return    
    enddo
  else
    call sic_r4(line,0,1,gain,.true.,error)
    if (error) return
    write(mess,'(A,1PG11.3)') 'Scaling all visibilities by ',gain
    call map_message(seve%i,rname,mess)
  endif
  !
  ! Now do the job
  !
  dofile = sic_present(o_file,0)
  if (dofile) then
    narg = sic_narg(o_file)
    if (narg.lt.1 .or. narg.gt.2) then
      call map_message(seve%e,rname,'/FILE option requires 1 or 2 arguments')
      error = .true.
      return
    endif
    !
    ! Layout for a by-block operation, used by the /FILE option
    !
    call gildas_null (uvin, type = 'UVT')     ! Define a UVTable gildas header
    call gildas_null (uvou, type = 'UVT')     ! Define a UVTable gildas header
    !
    call sic_ch(line,o_file,1,file,nf,.true.,error)
    if (error)  return
    call gdf_read_gildas (uvin, file, '.uvt', error, data=.false.)
    if (error) return
    call gdf_copy_header(uvin,uvou,error)
    !
    if (narg.eq.2) then
      call sic_ch(line,o_file,2,file,nf,.true.,error)
      if (error) return
      call sic_parse_file(file,' ','.uvt',uvou%file)
      if (error) return
      call gdf_create_image(uvou,error)
      if (error) then
        call gdf_close_image(uvin,err)
        return    
      endif
    endif
    !
    ! Define blocking factor on data file
    !
    ! Set the Blocking Factor
    nblock = space_nitems('SPACE_IMAGER',uvin,1)
    ! Allocate respective space for each file
    ncol = uvin%gil%dim(1)
    ndc =  uvin%gil%nchan
    allocate (luv(ncol,nblock), stat=ier)
    if (failed_allocate(rname,'UV input data',ier,error)) then
      call gdf_close_image(uvin,err)
      if (narg.eq.2) call gdf_close_image(uvou,err)
      return
    endif
    !
    ! Loop over line table
    uvin%blc = 0
    uvin%trc = 0
    !
    do ib = 1,uvin%gil%dim(2),nblock
      write(mess,*) ib,' / ',uvin%gil%dim(2),nblock
      call map_message(seve%i,rname,mess)
      uvin%blc(2) = ib
      uvin%trc(2) = min(uvin%gil%dim(2),ib-1+nblock)
      call gdf_read_data(uvin,luv,error)
      if (error) exit
      !
      nvis = uvin%trc(2)-uvin%blc(2)+1
      !
      ! Here do the job 
      call sub_uvamp(ncol,nvis, ndc, luv, gain, nant, iant, gains)
      !
      if (sic_ctrlc()) then
        error = .true.
        call map_message(seve%e,rname,'Aborted by ^C')
      else
        call gdf_write_data(uvou,luv,error)
      endif
      if (error) exit
    enddo
    !
    deallocate(luv,stat=ier)
    call gdf_close_image(uvin,err)
    error = err.or.error
    if (narg.eq.2) then
      call gdf_close_image(uvou,err)
      error = err.or.error
    endif
    !
  else
    !
    ! In memory version
    nvis = huv%gil%nvisi ! not %dim(2)
    ncol = huv%gil%dim(1)
    ndc =  huv%gil%nchan
    !
    nullify (duv_previous, duv_next)  
    call uv_find_buffers (rname,ncol,nvis,duv_previous, duv_next,error)
    if (error) return
    !
    duv_next(:,:) = duv_previous(:,:)
    !
    nvg = hbgain%gil%dim(2)
    call sub_uvamp(ncol,nvis, ndc, duv_next, gain, nant, iant, gains)
    if (error) return
    !
    call uv_clean_buffers (duv_previous, duv_next,error)
    !
    ! Signal the change in UV data - Amplitude Calibration also affects Weights
    call uv_new_data (weight=.true.)
  endif
end subroutine amplitude_comm
!
subroutine sub_uvamp(ncol,nvis,ndc,cal, gain, nant, iant, gains)
  use imager_interfaces, only : map_message
  use gbl_message
  use gkernel_interfaces
  !---------------------------------------------------------------------
  ! @ public
  !*
  !   IMAGER -- Support routine for command AMPLITUDE
  !
  !   Apply Amplitude calibration factor to the "raw" UV data,
  !   for all or a list of antennas. 
  !!
  !---------------------------------------------------------------------
  integer, intent(in) :: ncol  !! Visibility size
  integer, intent(in) :: nvis  !! Number of visibilities
  integer, intent(in) :: ndc   !! Number of channels
  real, intent(inout) :: cal(ncol,nvis)  !! Calibrated visibilities
  real, intent(in) :: gain     !! Common gain
  integer, intent(in) :: nant  !! Number of specified antennas
  integer, intent(in) :: iant(:)  !! Antenna numbers
  real(4), intent(in) :: gains(:) !! Gains per antenna
  !
  ! Global --
  character(len=*), parameter :: rname='AMPLITUDE'
  integer, parameter :: mant=256  ! ALMA maximum, after renumbering
  !
  ! Local ---
  integer :: ia,jv,k, ncold
  real :: gg, wg
  !
  ! Code ----
  ncold = 7+3*ndc   ! Last data column
  !
  ! Gain in both cases - Explained in the HELP
  if (nant.le.0) then
    !
    ! All antennas are modified, so the Gain is global
    gg = gain
    wg = 1.0/gg**2
    !$OMP PARALLEL DEFAULT(none) &
    !$OMP & SHARED(cal,nvis,ncol,ncold,gg,wg) &
    !$OMP & PRIVATE(jv,k)
    !$OMP DO
    do jv=1,nvis
      do k=8, ncold, 3
        cal(k,jv) = cal(k,jv)*gg
        cal(k+1,jv) = cal(k+1,jv)*gg
        cal(k+2,jv) = cal(k+2,jv)*wg
      enddo
    enddo
    !$OMP ENDDO
    !$OMP END PARALLEL
  else
    !$OMP PARALLEL DEFAULT(none) &
    !$OMP & SHARED(cal,nvis,ncol,ncold) &
    !$OMP & SHARED(iant,gains,nant) & 
    !$OMP & PRIVATE(jv,k,gg,wg,ia)
    !$OMP DO
    do jv=1,nvis
      gg = 1.0
      wg = 1.0
      do ia=1,nant
        if (iant(ia).eq.cal(6,jv).or.iant(ia).eq.cal(7,jv)) then
          gg = gg*gains(ia)          
          wg = wg/gains(ia)**2
        endif
      enddo
      !
      if (gg.ne.1.0) then
        do k=8, ncold, 3
          cal(k,jv) = cal(k,jv)*gg
          cal(k+1,jv) = cal(k+1,jv)*gg
          cal(k+2,jv) = cal(k+2,jv)*wg
        enddo
      endif
    enddo
    !$OMP ENDDO
    !$OMP END PARALLEL
  endif
  !
end subroutine sub_uvamp
!
