# -*- coding: utf-8 -*- #
# Copyright 2025 Google LLC. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""'vmware datastores describe' command."""


from googlecloudsdk.api_lib.vmware import datastores
from googlecloudsdk.calliope import base
from googlecloudsdk.command_lib.vmware import flags

DatastoresClient = datastores.DatastoresClient
DETAILED_HELP = {
    'DESCRIPTION': """
          Describe a datastore.
        """,
    'EXAMPLES': """
          To describe a datastore named `my-datastore` in location `us-west2-a`, run:

          $ {command} my-datastore --location=us-west2-a --project=my-project

          Or:

          $ {command} my-datastore

          In the second example, the project and location are taken from gcloud properties core/project and compute/zone.
    """,
}


@base.DefaultUniverseOnly
@base.ReleaseTracks(base.ReleaseTrack.GA)
class Describe(base.DescribeCommand):
  """Describe a datastore."""

  detailed_help = DETAILED_HELP

  @staticmethod
  def Args(parser):
    """Register flags for this command."""
    flags.AddDatastoreArgToParser(parser, positional=True)

  def Run(self, args):
    datastore = args.CONCEPTS.datastore.Parse()
    client = DatastoresClient()
    return client.Get(datastore)
