// Copyright (c) 2024-2026 Carsen Klock under MIT License
// ioreport.go - Go wrappers for IOReport power/thermal metrics
package app

/*
#cgo CFLAGS: -x objective-c
#cgo LDFLAGS: -framework CoreFoundation -framework IOKit -framework Foundation -framework CoreWLAN -lIOReport
#include <mach/mach_host.h>
#include <mach/processor_info.h>
#include <mach/mach_init.h>
#include <CoreFoundation/CoreFoundation.h>
#include <IOKit/IOKitLib.h>
#include <stdint.h>
#include <string.h>
#include <stdlib.h>

typedef struct IOReportSubscriptionRef* IOReportSubscriptionRef;

extern CFDictionaryRef IOReportCopyChannelsInGroup(CFStringRef group, CFStringRef subgroup, uint64_t a, uint64_t b, uint64_t c);
extern void IOReportMergeChannels(CFDictionaryRef a, CFDictionaryRef b, CFTypeRef unused);
extern IOReportSubscriptionRef IOReportCreateSubscription(void* a, CFMutableDictionaryRef channels, CFMutableDictionaryRef* out, uint64_t d, CFTypeRef e);
extern CFDictionaryRef IOReportCreateSamples(IOReportSubscriptionRef sub, CFMutableDictionaryRef channels, CFTypeRef unused);
extern CFDictionaryRef IOReportCreateSamplesDelta(CFDictionaryRef a, CFDictionaryRef b, CFTypeRef unused);
extern int64_t IOReportSimpleGetIntegerValue(CFDictionaryRef item, int32_t idx);
extern CFStringRef IOReportChannelGetGroup(CFDictionaryRef item);
extern CFStringRef IOReportChannelGetSubGroup(CFDictionaryRef item);
extern CFStringRef IOReportChannelGetChannelName(CFDictionaryRef item);
extern CFStringRef IOReportChannelGetUnitLabel(CFDictionaryRef item);
extern int32_t IOReportStateGetCount(CFDictionaryRef item);
extern CFStringRef IOReportStateGetNameForIndex(CFDictionaryRef item, int32_t idx);
extern int64_t IOReportStateGetResidency(CFDictionaryRef item, int32_t idx);

typedef void* IOHIDEventSystemClientRef;
typedef void* IOHIDServiceClientRef;
typedef void* IOHIDEventRef;

extern IOHIDEventSystemClientRef IOHIDEventSystemClientCreate(CFAllocatorRef allocator);
extern int IOHIDEventSystemClientSetMatching(IOHIDEventSystemClientRef client, CFDictionaryRef matching);
extern CFArrayRef IOHIDEventSystemClientCopyServices(IOHIDEventSystemClientRef client);
extern CFStringRef IOHIDServiceClientCopyProperty(IOHIDServiceClientRef service, CFStringRef key);
extern IOHIDEventRef IOHIDServiceClientCopyEvent(IOHIDServiceClientRef service, int64_t type, int32_t options, int64_t timeout);
extern double IOHIDEventGetFloatValue(IOHIDEventRef event, int64_t field);

typedef struct {
    double cpuPower;
    double gpuPower;
    double anePower;
    double dramPower;
    double gpuSramPower;
    double systemPower;
    int gpuFreqMHz;
    double gpuActive;
    double eClusterActive;
    double pClusterActive;
    int eClusterFreqMHz;
    int pClusterFreqMHz;
    float socTemp;
    float cpuTemp;
    float gpuTemp;
} PowerMetrics;

int initIOReport();
PowerMetrics samplePowerMetrics(int durationMs);
void cleanupIOReport();
int getThermalState();
extern void debugIOReport(void);
extern void printAllChannels(void);
extern void debugMonitorChannels(int durationMs);

// Wi-Fi link info structure (defined in ioreport.m)
typedef struct {
    char interface_name[32];
    char phy_mode[32];
    char wifi_generation[16];
    int tx_rate_mbps;
    int is_connected;
} wifi_link_info_t;

int get_wifi_link_info(wifi_link_info_t *info);
*/
import "C"

type SocMetrics struct {
	CPUPower        float64 `json:"cpu_power"`
	GPUPower        float64 `json:"gpu_power"`
	ANEPower        float64 `json:"ane_power"`
	DRAMPower       float64 `json:"dram_power"`
	GPUSRAMPower    float64 `json:"gpu_sram_power"`
	SystemPower     float64 `json:"system_power"`
	TotalPower      float64 `json:"total_power"`
	GPUFreqMHz      int32   `json:"gpu_freq_mhz"`
	GPUActive       float64 `json:"gpu_active"`
	EClusterActive  float64 `json:"e_cluster_active"`
	PClusterActive  float64 `json:"p_cluster_active"`
	EClusterFreqMHz int32   `json:"e_cluster_freq_mhz"`
	PClusterFreqMHz int32   `json:"p_cluster_freq_mhz"`
	SocTemp         float32 `json:"soc_temp"`
	CPUTemp         float32 `json:"cpu_temp"`
	GPUTemp         float32 `json:"gpu_temp"`
}

func initSocMetrics() error {
	if ret := C.initIOReport(); ret != 0 {
		return nil
	}
	return nil
}

func sampleSocMetrics(durationMs int) SocMetrics {
	pm := C.samplePowerMetrics(C.int(durationMs))
	return SocMetrics{
		CPUPower:        float64(pm.cpuPower),
		GPUPower:        float64(pm.gpuPower),
		ANEPower:        float64(pm.anePower),
		DRAMPower:       float64(pm.dramPower),
		GPUSRAMPower:    float64(pm.gpuSramPower),
		SystemPower:     float64(pm.systemPower),
		TotalPower:      float64(pm.cpuPower) + float64(pm.gpuPower) + float64(pm.anePower) + float64(pm.dramPower) + float64(pm.gpuSramPower),
		GPUFreqMHz:      int32(pm.gpuFreqMHz),
		GPUActive:       float64(pm.gpuActive),
		EClusterActive:  float64(pm.eClusterActive),
		PClusterActive:  float64(pm.pClusterActive),
		EClusterFreqMHz: int32(pm.eClusterFreqMHz),
		PClusterFreqMHz: int32(pm.pClusterFreqMHz),
		SocTemp:         float32(pm.socTemp),
		CPUTemp:         float32(pm.cpuTemp),
		GPUTemp:         float32(pm.gpuTemp),
	}
}

func cleanupSocMetrics() {
	C.cleanupIOReport()
}

func getSocThermalState() int {
	return int(C.getThermalState())
}

// DebugIOReport prints all available IOReport channels and groups to stdout
func DebugIOReport() {
	C.debugIOReport()
}

// WiFiLinkInfo represents Wi-Fi interface link information
type WiFiLinkInfo struct {
	InterfaceName  string // Interface name (en0, en1, etc.)
	PHYMode        string // "802.11n", "802.11ac", "802.11ax", etc.
	WiFiGeneration string // "Wi-Fi 4", "Wi-Fi 5", "Wi-Fi 6", etc.
	TxRateMbps     int    // Current transmit rate in Mbps
	IsConnected    bool   // True if associated to a network
}

// GetWiFiLinkInfo returns Wi-Fi link information
func GetWiFiLinkInfo() *WiFiLinkInfo {
	var info C.wifi_link_info_t
	ret := C.get_wifi_link_info(&info)
	if ret != 0 {
		return nil
	}
	return &WiFiLinkInfo{
		InterfaceName:  C.GoString(&info.interface_name[0]),
		PHYMode:        C.GoString(&info.phy_mode[0]),
		WiFiGeneration: C.GoString(&info.wifi_generation[0]),
		TxRateMbps:     int(info.tx_rate_mbps),
		IsConnected:    info.is_connected != 0,
	}
}
