// Copyright 2026 Google LLC.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Code generated file. DO NOT EDIT.

// Package solar provides access to the Solar API.
//
// For product documentation, see: https://developers.google.com/maps/documentation/solar
//
// # Library status
//
// These client libraries are officially supported by Google. However, this
// library is considered complete and is in maintenance mode. This means
// that we will address critical bugs and security issues but will not add
// any new features.
//
// When possible, we recommend using our newer
// [Cloud Client Libraries for Go](https://pkg.go.dev/cloud.google.com/go)
// that are still actively being worked and iterated on.
//
// # Creating a client
//
// Usage example:
//
//	import "google.golang.org/api/solar/v1"
//	...
//	ctx := context.Background()
//	solarService, err := solar.NewService(ctx)
//
// In this example, Google Application Default Credentials are used for
// authentication. For information on how to create and obtain Application
// Default Credentials, see https://developers.google.com/identity/protocols/application-default-credentials.
//
// # Other authentication options
//
// To use an API key for authentication (note: some APIs do not support API
// keys), use [google.golang.org/api/option.WithAPIKey]:
//
//	solarService, err := solar.NewService(ctx, option.WithAPIKey("AIza..."))
//
// To use an OAuth token (e.g., a user token obtained via a three-legged OAuth
// flow, use [google.golang.org/api/option.WithTokenSource]:
//
//	config := &oauth2.Config{...}
//	// ...
//	token, err := config.Exchange(ctx, ...)
//	solarService, err := solar.NewService(ctx, option.WithTokenSource(config.TokenSource(ctx, token)))
//
// See [google.golang.org/api/option.ClientOption] for details on options.
package solar // import "google.golang.org/api/solar/v1"

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"log/slog"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	"github.com/googleapis/gax-go/v2/internallog"
	googleapi "google.golang.org/api/googleapi"
	internal "google.golang.org/api/internal"
	gensupport "google.golang.org/api/internal/gensupport"
	option "google.golang.org/api/option"
	internaloption "google.golang.org/api/option/internaloption"
	htransport "google.golang.org/api/transport/http"
)

// Always reference these packages, just in case the auto-generated code
// below doesn't.
var _ = bytes.NewBuffer
var _ = strconv.Itoa
var _ = fmt.Sprintf
var _ = json.NewDecoder
var _ = io.Copy
var _ = url.Parse
var _ = gensupport.MarshalJSON
var _ = googleapi.Version
var _ = errors.New
var _ = strings.Replace
var _ = context.Canceled
var _ = internaloption.WithDefaultEndpoint
var _ = internal.Version
var _ = internallog.New

const apiId = "solar:v1"
const apiName = "solar"
const apiVersion = "v1"
const basePath = "https://solar.googleapis.com/"
const basePathTemplate = "https://solar.UNIVERSE_DOMAIN/"
const mtlsBasePath = "https://solar.mtls.googleapis.com/"

// OAuth2 scopes used by this API.
const (
	// See, edit, configure, and delete your Google Cloud data and see the email
	// address for your Google Account.
	CloudPlatformScope = "https://www.googleapis.com/auth/cloud-platform"
)

// NewService creates a new Service.
func NewService(ctx context.Context, opts ...option.ClientOption) (*Service, error) {
	scopesOption := internaloption.WithDefaultScopes(
		"https://www.googleapis.com/auth/cloud-platform",
	)
	// NOTE: prepend, so we don't override user-specified scopes.
	opts = append([]option.ClientOption{scopesOption}, opts...)
	opts = append(opts, internaloption.WithDefaultEndpoint(basePath))
	opts = append(opts, internaloption.WithDefaultEndpointTemplate(basePathTemplate))
	opts = append(opts, internaloption.WithDefaultMTLSEndpoint(mtlsBasePath))
	opts = append(opts, internaloption.EnableNewAuthLibrary())
	client, endpoint, err := htransport.NewClient(ctx, opts...)
	if err != nil {
		return nil, err
	}
	s := &Service{client: client, BasePath: basePath, logger: internaloption.GetLogger(opts)}
	s.BuildingInsights = NewBuildingInsightsService(s)
	s.DataLayers = NewDataLayersService(s)
	s.GeoTiff = NewGeoTiffService(s)
	if endpoint != "" {
		s.BasePath = endpoint
	}
	return s, nil
}

// New creates a new Service. It uses the provided http.Client for requests.
//
// Deprecated: please use NewService instead.
// To provide a custom HTTP client, use option.WithHTTPClient.
// If you are using google.golang.org/api/googleapis/transport.APIKey, use option.WithAPIKey with NewService instead.
func New(client *http.Client) (*Service, error) {
	if client == nil {
		return nil, errors.New("client is nil")
	}
	return NewService(context.TODO(), option.WithHTTPClient(client))
}

type Service struct {
	client    *http.Client
	logger    *slog.Logger
	BasePath  string // API endpoint base URL
	UserAgent string // optional additional User-Agent fragment

	BuildingInsights *BuildingInsightsService

	DataLayers *DataLayersService

	GeoTiff *GeoTiffService
}

func (s *Service) userAgent() string {
	if s.UserAgent == "" {
		return googleapi.UserAgent
	}
	return googleapi.UserAgent + " " + s.UserAgent
}

func NewBuildingInsightsService(s *Service) *BuildingInsightsService {
	rs := &BuildingInsightsService{s: s}
	return rs
}

type BuildingInsightsService struct {
	s *Service
}

func NewDataLayersService(s *Service) *DataLayersService {
	rs := &DataLayersService{s: s}
	return rs
}

type DataLayersService struct {
	s *Service
}

func NewGeoTiffService(s *Service) *GeoTiffService {
	rs := &GeoTiffService{s: s}
	return rs
}

type GeoTiffService struct {
	s *Service
}

// BuildingInsights: Response message for `Solar.FindClosestBuildingInsights`.
// Information about the location, dimensions, and solar potential of a
// building.
type BuildingInsights struct {
	// AdministrativeArea: Administrative area 1 (e.g., in the US, the state) that
	// contains this building. For example, in the US, the abbreviation might be
	// "MA" or "CA."
	AdministrativeArea string `json:"administrativeArea,omitempty"`
	// BoundingBox: The bounding box of the building.
	BoundingBox *LatLngBox `json:"boundingBox,omitempty"`
	// Center: A point near the center of the building.
	Center *LatLng `json:"center,omitempty"`
	// ImageryDate: Date that the underlying imagery was acquired. This is
	// approximate.
	ImageryDate *Date `json:"imageryDate,omitempty"`
	// ImageryProcessedDate: When processing was completed on this imagery.
	ImageryProcessedDate *Date `json:"imageryProcessedDate,omitempty"`
	// ImageryQuality: The quality of the imagery used to compute the data for this
	// building.
	//
	// Possible values:
	//   "IMAGERY_QUALITY_UNSPECIFIED" - No quality is known.
	//   "HIGH" - Solar data is derived from aerial imagery captured at
	// low-altitude and processed at 0.1 m/pixel.
	//   "MEDIUM" - Solar data is derived from enhanced aerial imagery captured at
	// high-altitude and processed at 0.25 m/pixel.
	//   "LOW" - Solar data is derived from enhanced satellite imagery processed at
	// 0.25 m/pixel.
	//   "BASE" - Solar data is derived from enhanced satellite imagery processed
	// at 0.25 m/pixel.
	ImageryQuality string `json:"imageryQuality,omitempty"`
	// Name: The resource name for the building, of the format
	// `buildings/{place_id}`.
	Name string `json:"name,omitempty"`
	// PostalCode: Postal code (e.g., US zip code) this building is contained by.
	PostalCode string `json:"postalCode,omitempty"`
	// RegionCode: Region code for the country (or region) this building is in.
	RegionCode string `json:"regionCode,omitempty"`
	// SolarPotential: Solar potential of the building.
	SolarPotential *SolarPotential `json:"solarPotential,omitempty"`
	// StatisticalArea: Statistical area (e.g., US census tract) this building is
	// in.
	StatisticalArea string `json:"statisticalArea,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AdministrativeArea") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AdministrativeArea") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s BuildingInsights) MarshalJSON() ([]byte, error) {
	type NoMethod BuildingInsights
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CashPurchaseSavings: Cost and benefit of an outright purchase of a
// particular configuration of solar panels with a particular electricity
// usage.
type CashPurchaseSavings struct {
	// OutOfPocketCost: Initial cost before tax incentives: the amount that must be
	// paid out-of-pocket. Contrast with `upfront_cost`, which is after tax
	// incentives.
	OutOfPocketCost *Money `json:"outOfPocketCost,omitempty"`
	// PaybackYears: Number of years until payback occurs. A negative value means
	// payback never occurs within the lifetime period.
	PaybackYears float64 `json:"paybackYears,omitempty"`
	// RebateValue: The value of all tax rebates.
	RebateValue *Money `json:"rebateValue,omitempty"`
	// Savings: How much is saved (or not) over the lifetime period.
	Savings *SavingsOverTime `json:"savings,omitempty"`
	// UpfrontCost: Initial cost after tax incentives: it's the amount that must be
	// paid during first year. Contrast with `out_of_pocket_cost`, which is before
	// tax incentives.
	UpfrontCost *Money `json:"upfrontCost,omitempty"`
	// ForceSendFields is a list of field names (e.g. "OutOfPocketCost") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "OutOfPocketCost") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CashPurchaseSavings) MarshalJSON() ([]byte, error) {
	type NoMethod CashPurchaseSavings
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *CashPurchaseSavings) UnmarshalJSON(data []byte) error {
	type NoMethod CashPurchaseSavings
	var s1 struct {
		PaybackYears gensupport.JSONFloat64 `json:"paybackYears"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.PaybackYears = float64(s1.PaybackYears)
	return nil
}

// DataLayers: Information about the solar potential of a region. The actual
// data are contained in a number of GeoTIFF files covering the requested
// region, for which this message contains URLs: Each string in the
// `DataLayers` message contains a URL from which the corresponding GeoTIFF can
// be fetched. These URLs are valid for a few hours after they've been
// generated. Most of the GeoTIFF files are at a resolution of 0.1m/pixel, but
// the monthly flux file is at 0.5m/pixel, and the hourly shade files are at
// 1m/pixel. If a `pixel_size_meters` value was specified in the
// `GetDataLayersRequest`, then the minimum resolution in the GeoTIFF files
// will be that value.
type DataLayers struct {
	// AnnualFluxUrl: The URL for the annual flux map (annual sunlight on roofs) of
	// the region. Values are kWh/kW/year. This is *unmasked flux*: flux is
	// computed for every location, not just building rooftops. Invalid locations
	// are stored as -9999: locations outside our coverage area will be invalid,
	// and a few locations inside the coverage area, where we were unable to
	// calculate flux, will also be invalid.
	AnnualFluxUrl string `json:"annualFluxUrl,omitempty"`
	// DsmUrl: The URL for an image of the DSM (Digital Surface Model) of the
	// region. Values are in meters above EGM96 geoid (i.e., sea level). Invalid
	// locations (where we don't have data) are stored as -9999.
	DsmUrl string `json:"dsmUrl,omitempty"`
	// HourlyShadeUrls: Twelve URLs for hourly shade, corresponding to
	// January...December, in order. Each GeoTIFF will contain 24 bands,
	// corresponding to the 24 hours of the day. Each pixel is a 32 bit integer,
	// corresponding to the (up to) 31 days of that month; a 1 bit means that the
	// corresponding location is able to see the sun at that day, of that hour, of
	// that month. Invalid locations are stored as -9999 (since this is negative,
	// it has bit 31 set, and no valid value could have bit 31 set as that would
	// correspond to the 32nd day of the month). An example may be useful. If you
	// want to know whether a point (at pixel location (x, y)) saw sun at 4pm on
	// the 22nd of June you would: 1. fetch the sixth URL in this list
	// (corresponding to June). 1. look up the 17th channel (corresponding to 4pm).
	// 1. read the 32-bit value at (x, y). 1. read bit 21 of the value
	// (corresponding to the 22nd of the month). 1. if that bit is a 1, then that
	// spot saw the sun at 4pm 22 June. More formally: Given `month` (1-12), `day`
	// (1...month max; February has 28 days) and `hour` (0-23), the shade/sun for
	// that month/day/hour at a position `(x, y)` is the bit ```
	// (hourly_shade[month - 1])(x, y)[hour] & (1 << (day - 1)) ``` where `(x, y)`
	// is spatial indexing, `[month - 1]` refers to fetching the `month - 1`st URL
	// (indexing from zero), `[hour]` is indexing into the channels, and a final
	// non-zero result means "sunny". There are no leap days, and DST doesn't exist
	// (all days are 24 hours long; noon is always "standard time" noon).
	HourlyShadeUrls []string `json:"hourlyShadeUrls,omitempty"`
	// ImageryDate: When the source imagery (from which all the other data are
	// derived) in this region was taken. It is necessarily somewhat approximate,
	// as the images may have been taken over more than one day.
	ImageryDate *Date `json:"imageryDate,omitempty"`
	// ImageryProcessedDate: When processing was completed on this imagery.
	ImageryProcessedDate *Date `json:"imageryProcessedDate,omitempty"`
	// ImageryQuality: The quality of the result's imagery.
	//
	// Possible values:
	//   "IMAGERY_QUALITY_UNSPECIFIED" - No quality is known.
	//   "HIGH" - Solar data is derived from aerial imagery captured at
	// low-altitude and processed at 0.1 m/pixel.
	//   "MEDIUM" - Solar data is derived from enhanced aerial imagery captured at
	// high-altitude and processed at 0.25 m/pixel.
	//   "LOW" - Solar data is derived from enhanced satellite imagery processed at
	// 0.25 m/pixel.
	//   "BASE" - Solar data is derived from enhanced satellite imagery processed
	// at 0.25 m/pixel.
	ImageryQuality string `json:"imageryQuality,omitempty"`
	// MaskUrl: The URL for the building mask image: one bit per pixel saying
	// whether that pixel is considered to be part of a rooftop or not.
	MaskUrl string `json:"maskUrl,omitempty"`
	// MonthlyFluxUrl: The URL for the monthly flux map (sunlight on roofs, broken
	// down by month) of the region. Values are kWh/kW/year. The GeoTIFF pointed to
	// by this URL will contain twelve bands, corresponding to January...December,
	// in order.
	MonthlyFluxUrl string `json:"monthlyFluxUrl,omitempty"`
	// RgbUrl: The URL for an image of RGB data (aerial photo) of the region.
	RgbUrl string `json:"rgbUrl,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AnnualFluxUrl") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AnnualFluxUrl") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DataLayers) MarshalJSON() ([]byte, error) {
	type NoMethod DataLayers
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Date: Represents a whole or partial calendar date, such as a birthday. The
// time of day and time zone are either specified elsewhere or are
// insignificant. The date is relative to the Gregorian Calendar. This can
// represent one of the following: * A full date, with non-zero year, month,
// and day values. * A month and day, with a zero year (for example, an
// anniversary). * A year on its own, with a zero month and a zero day. * A
// year and month, with a zero day (for example, a credit card expiration
// date). Related types: * google.type.TimeOfDay * google.type.DateTime *
// google.protobuf.Timestamp
type Date struct {
	// Day: Day of a month. Must be from 1 to 31 and valid for the year and month,
	// or 0 to specify a year by itself or a year and month where the day isn't
	// significant.
	Day int64 `json:"day,omitempty"`
	// Month: Month of a year. Must be from 1 to 12, or 0 to specify a year without
	// a month and day.
	Month int64 `json:"month,omitempty"`
	// Year: Year of the date. Must be from 1 to 9999, or 0 to specify a date
	// without a year.
	Year int64 `json:"year,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Day") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Day") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Date) MarshalJSON() ([]byte, error) {
	type NoMethod Date
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// FinancedPurchaseSavings: Cost and benefit of using a loan to buy a
// particular configuration of solar panels with a particular electricity
// usage.
type FinancedPurchaseSavings struct {
	// AnnualLoanPayment: Annual loan payments.
	AnnualLoanPayment *Money `json:"annualLoanPayment,omitempty"`
	// LoanInterestRate: The interest rate on loans assumed in this set of
	// calculations.
	LoanInterestRate float64 `json:"loanInterestRate,omitempty"`
	// RebateValue: The value of all tax rebates (including Federal Investment Tax
	// Credit (ITC)).
	RebateValue *Money `json:"rebateValue,omitempty"`
	// Savings: How much is saved (or not) over the lifetime period.
	Savings *SavingsOverTime `json:"savings,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AnnualLoanPayment") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AnnualLoanPayment") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FinancedPurchaseSavings) MarshalJSON() ([]byte, error) {
	type NoMethod FinancedPurchaseSavings
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *FinancedPurchaseSavings) UnmarshalJSON(data []byte) error {
	type NoMethod FinancedPurchaseSavings
	var s1 struct {
		LoanInterestRate gensupport.JSONFloat64 `json:"loanInterestRate"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.LoanInterestRate = float64(s1.LoanInterestRate)
	return nil
}

// FinancialAnalysis: Analysis of the cost and benefits of the optimum solar
// layout for a particular electric bill size.
type FinancialAnalysis struct {
	// AverageKwhPerMonth: How much electricity the house uses in an average month,
	// based on the bill size and the local electricity rates.
	AverageKwhPerMonth float64 `json:"averageKwhPerMonth,omitempty"`
	// CashPurchaseSavings: Cost and benefit of buying the solar panels with cash.
	CashPurchaseSavings *CashPurchaseSavings `json:"cashPurchaseSavings,omitempty"`
	// DefaultBill: Whether this is the bill size selected to be the default bill
	// for the area this building is in. Exactly one `FinancialAnalysis` in
	// `BuildingSolarPotential` should have `default_bill` set.
	DefaultBill bool `json:"defaultBill,omitempty"`
	// FinancedPurchaseSavings: Cost and benefit of buying the solar panels by
	// financing the purchase.
	FinancedPurchaseSavings *FinancedPurchaseSavings `json:"financedPurchaseSavings,omitempty"`
	// FinancialDetails: Financial information that applies regardless of the
	// financing method used.
	FinancialDetails *FinancialDetails `json:"financialDetails,omitempty"`
	// LeasingSavings: Cost and benefit of leasing the solar panels.
	LeasingSavings *LeasingSavings `json:"leasingSavings,omitempty"`
	// MonthlyBill: The monthly electric bill this analysis assumes.
	MonthlyBill *Money `json:"monthlyBill,omitempty"`
	// PanelConfigIndex: Index in solar_panel_configs of the optimum solar layout
	// for this bill size. This can be -1 indicating that there is no layout. In
	// this case, the remaining submessages will be omitted.
	PanelConfigIndex int64 `json:"panelConfigIndex,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AverageKwhPerMonth") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AverageKwhPerMonth") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FinancialAnalysis) MarshalJSON() ([]byte, error) {
	type NoMethod FinancialAnalysis
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *FinancialAnalysis) UnmarshalJSON(data []byte) error {
	type NoMethod FinancialAnalysis
	var s1 struct {
		AverageKwhPerMonth gensupport.JSONFloat64 `json:"averageKwhPerMonth"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.AverageKwhPerMonth = float64(s1.AverageKwhPerMonth)
	return nil
}

// FinancialDetails: Details of a financial analysis. Some of these details are
// already stored at higher levels (e.g., out of pocket cost). Total money
// amounts are over a lifetime period defined by the panel_lifetime_years field
// in SolarPotential. Note: The out of pocket cost of purchasing the panels is
// given in the out_of_pocket_cost field in CashPurchaseSavings.
type FinancialDetails struct {
	// CostOfElectricityWithoutSolar: Total cost of electricity the user would have
	// paid over the lifetime period if they didn't install solar.
	CostOfElectricityWithoutSolar *Money `json:"costOfElectricityWithoutSolar,omitempty"`
	// FederalIncentive: Amount of money available from federal incentives; this
	// applies if the user buys (with or without a loan) the panels.
	FederalIncentive *Money `json:"federalIncentive,omitempty"`
	// InitialAcKwhPerYear: How many AC kWh we think the solar panels will generate
	// in their first year.
	InitialAcKwhPerYear float64 `json:"initialAcKwhPerYear,omitempty"`
	// LifetimeSrecTotal: Amount of money the user will receive from Solar
	// Renewable Energy Credits over the panel lifetime; this applies if the user
	// buys (with or without a loan) the panels.
	LifetimeSrecTotal *Money `json:"lifetimeSrecTotal,omitempty"`
	// NetMeteringAllowed: Whether net metering is allowed.
	NetMeteringAllowed bool `json:"netMeteringAllowed,omitempty"`
	// PercentageExportedToGrid: The percentage (0-100) of solar electricity
	// production we assumed was exported to the grid, based on the first quarter
	// of production. This affects the calculations if net metering is not allowed.
	PercentageExportedToGrid float64 `json:"percentageExportedToGrid,omitempty"`
	// RemainingLifetimeUtilityBill: Utility bill for electricity not produced by
	// solar, for the lifetime of the panels.
	RemainingLifetimeUtilityBill *Money `json:"remainingLifetimeUtilityBill,omitempty"`
	// SolarPercentage: Percentage (0-100) of the user's power supplied by solar.
	// Valid for the first year but approximately correct for future years.
	SolarPercentage float64 `json:"solarPercentage,omitempty"`
	// StateIncentive: Amount of money available from state incentives; this
	// applies if the user buys (with or without a loan) the panels.
	StateIncentive *Money `json:"stateIncentive,omitempty"`
	// UtilityIncentive: Amount of money available from utility incentives; this
	// applies if the user buys (with or without a loan) the panels.
	UtilityIncentive *Money `json:"utilityIncentive,omitempty"`
	// ForceSendFields is a list of field names (e.g.
	// "CostOfElectricityWithoutSolar") to unconditionally include in API requests.
	// By default, fields with empty or default values are omitted from API
	// requests. See https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields
	// for more details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CostOfElectricityWithoutSolar")
	// to include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FinancialDetails) MarshalJSON() ([]byte, error) {
	type NoMethod FinancialDetails
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *FinancialDetails) UnmarshalJSON(data []byte) error {
	type NoMethod FinancialDetails
	var s1 struct {
		InitialAcKwhPerYear      gensupport.JSONFloat64 `json:"initialAcKwhPerYear"`
		PercentageExportedToGrid gensupport.JSONFloat64 `json:"percentageExportedToGrid"`
		SolarPercentage          gensupport.JSONFloat64 `json:"solarPercentage"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.InitialAcKwhPerYear = float64(s1.InitialAcKwhPerYear)
	s.PercentageExportedToGrid = float64(s1.PercentageExportedToGrid)
	s.SolarPercentage = float64(s1.SolarPercentage)
	return nil
}

// HttpBody: Message that represents an arbitrary HTTP body. It should only be
// used for payload formats that can't be represented as JSON, such as raw
// binary or an HTML page. This message can be used both in streaming and
// non-streaming API methods in the request as well as the response. It can be
// used as a top-level request field, which is convenient if one wants to
// extract parameters from either the URL or HTTP template into the request
// fields and also want access to the raw HTTP body. Example: message
// GetResourceRequest { // A unique request id. string request_id = 1; // The
// raw HTTP body is bound to this field. google.api.HttpBody http_body = 2; }
// service ResourceService { rpc GetResource(GetResourceRequest) returns
// (google.api.HttpBody); rpc UpdateResource(google.api.HttpBody) returns
// (google.protobuf.Empty); } Example with streaming methods: service
// CaldavService { rpc GetCalendar(stream google.api.HttpBody) returns (stream
// google.api.HttpBody); rpc UpdateCalendar(stream google.api.HttpBody) returns
// (stream google.api.HttpBody); } Use of this type only changes how the
// request and response bodies are handled, all other features will continue to
// work unchanged.
type HttpBody struct {
	// ContentType: The HTTP Content-Type header value specifying the content type
	// of the body.
	ContentType string `json:"contentType,omitempty"`
	// Data: The HTTP request/response body as raw binary.
	Data string `json:"data,omitempty"`
	// Extensions: Application specific response metadata. Must be set in the first
	// response for streaming APIs.
	Extensions []googleapi.RawMessage `json:"extensions,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "ContentType") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ContentType") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s HttpBody) MarshalJSON() ([]byte, error) {
	type NoMethod HttpBody
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// LatLng: An object that represents a latitude/longitude pair. This is
// expressed as a pair of doubles to represent degrees latitude and degrees
// longitude. Unless specified otherwise, this object must conform to the WGS84
// standard. Values must be within normalized ranges.
type LatLng struct {
	// Latitude: The latitude in degrees. It must be in the range [-90.0, +90.0].
	Latitude float64 `json:"latitude,omitempty"`
	// Longitude: The longitude in degrees. It must be in the range [-180.0,
	// +180.0].
	Longitude float64 `json:"longitude,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Latitude") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Latitude") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s LatLng) MarshalJSON() ([]byte, error) {
	type NoMethod LatLng
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *LatLng) UnmarshalJSON(data []byte) error {
	type NoMethod LatLng
	var s1 struct {
		Latitude  gensupport.JSONFloat64 `json:"latitude"`
		Longitude gensupport.JSONFloat64 `json:"longitude"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Latitude = float64(s1.Latitude)
	s.Longitude = float64(s1.Longitude)
	return nil
}

// LatLngBox: A bounding box in lat/lng coordinates.
type LatLngBox struct {
	// Ne: The northeast corner of the box.
	Ne *LatLng `json:"ne,omitempty"`
	// Sw: The southwest corner of the box.
	Sw *LatLng `json:"sw,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Ne") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Ne") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s LatLngBox) MarshalJSON() ([]byte, error) {
	type NoMethod LatLngBox
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// LeasingSavings: Cost and benefit of leasing a particular configuration of
// solar panels with a particular electricity usage.
type LeasingSavings struct {
	// AnnualLeasingCost: Estimated annual leasing cost.
	AnnualLeasingCost *Money `json:"annualLeasingCost,omitempty"`
	// LeasesAllowed: Whether leases are allowed in this juristiction (leases are
	// not allowed in some states). If this field is false, then the values in this
	// message should probably be ignored.
	LeasesAllowed bool `json:"leasesAllowed,omitempty"`
	// LeasesSupported: Whether leases are supported in this juristiction by the
	// financial calculation engine. If this field is false, then the values in
	// this message should probably be ignored. This is independent of
	// `leases_allowed`: in some areas leases are allowed, but under conditions
	// that aren't handled by the financial models.
	LeasesSupported bool `json:"leasesSupported,omitempty"`
	// Savings: How much is saved (or not) over the lifetime period.
	Savings *SavingsOverTime `json:"savings,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AnnualLeasingCost") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AnnualLeasingCost") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s LeasingSavings) MarshalJSON() ([]byte, error) {
	type NoMethod LeasingSavings
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Money: Represents an amount of money with its currency type.
type Money struct {
	// CurrencyCode: The three-letter currency code defined in ISO 4217.
	CurrencyCode string `json:"currencyCode,omitempty"`
	// Nanos: Number of nano (10^-9) units of the amount. The value must be between
	// -999,999,999 and +999,999,999 inclusive. If `units` is positive, `nanos`
	// must be positive or zero. If `units` is zero, `nanos` can be positive, zero,
	// or negative. If `units` is negative, `nanos` must be negative or zero. For
	// example $-1.75 is represented as `units`=-1 and `nanos`=-750,000,000.
	Nanos int64 `json:"nanos,omitempty"`
	// Units: The whole units of the amount. For example if `currencyCode` is
	// "USD", then 1 unit is one US dollar.
	Units int64 `json:"units,omitempty,string"`
	// ForceSendFields is a list of field names (e.g. "CurrencyCode") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CurrencyCode") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Money) MarshalJSON() ([]byte, error) {
	type NoMethod Money
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// RoofSegmentSizeAndSunshineStats: Information about the size and sunniness
// quantiles of a roof segment.
type RoofSegmentSizeAndSunshineStats struct {
	// AzimuthDegrees: Compass direction the roof segment is pointing in. 0 =
	// North, 90 = East, 180 = South. For a "flat" roof segment (`pitch_degrees`
	// very near 0), azimuth is not well defined, so for consistency, we define it
	// arbitrarily to be 0 (North).
	AzimuthDegrees float64 `json:"azimuthDegrees,omitempty"`
	// BoundingBox: The bounding box of the roof segment.
	BoundingBox *LatLngBox `json:"boundingBox,omitempty"`
	// Center: A point near the center of the roof segment.
	Center *LatLng `json:"center,omitempty"`
	// PitchDegrees: Angle of the roof segment relative to the theoretical ground
	// plane. 0 = parallel to the ground, 90 = perpendicular to the ground.
	PitchDegrees float64 `json:"pitchDegrees,omitempty"`
	// PlaneHeightAtCenterMeters: The height of the roof segment plane, in meters
	// above sea level, at the point designated by `center`. Together with the
	// pitch, azimuth, and center location, this fully defines the roof segment
	// plane.
	PlaneHeightAtCenterMeters float64 `json:"planeHeightAtCenterMeters,omitempty"`
	// Stats: Total size and sunlight quantiles for the roof segment.
	Stats *SizeAndSunshineStats `json:"stats,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AzimuthDegrees") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AzimuthDegrees") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s RoofSegmentSizeAndSunshineStats) MarshalJSON() ([]byte, error) {
	type NoMethod RoofSegmentSizeAndSunshineStats
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *RoofSegmentSizeAndSunshineStats) UnmarshalJSON(data []byte) error {
	type NoMethod RoofSegmentSizeAndSunshineStats
	var s1 struct {
		AzimuthDegrees            gensupport.JSONFloat64 `json:"azimuthDegrees"`
		PitchDegrees              gensupport.JSONFloat64 `json:"pitchDegrees"`
		PlaneHeightAtCenterMeters gensupport.JSONFloat64 `json:"planeHeightAtCenterMeters"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.AzimuthDegrees = float64(s1.AzimuthDegrees)
	s.PitchDegrees = float64(s1.PitchDegrees)
	s.PlaneHeightAtCenterMeters = float64(s1.PlaneHeightAtCenterMeters)
	return nil
}

// RoofSegmentSummary: Information about a roof segment on the building, with
// some number of panels placed on it.
type RoofSegmentSummary struct {
	// AzimuthDegrees: Compass direction the roof segment is pointing in. 0 =
	// North, 90 = East, 180 = South. For a "flat" roof segment (`pitch_degrees`
	// very near 0), azimuth is not well defined, so for consistency, we define it
	// arbitrarily to be 0 (North).
	AzimuthDegrees float64 `json:"azimuthDegrees,omitempty"`
	// PanelsCount: The total number of panels on this segment.
	PanelsCount int64 `json:"panelsCount,omitempty"`
	// PitchDegrees: Angle of the roof segment relative to the theoretical ground
	// plane. 0 = parallel to the ground, 90 = perpendicular to the ground.
	PitchDegrees float64 `json:"pitchDegrees,omitempty"`
	// SegmentIndex: Index in roof_segment_stats of the corresponding
	// `RoofSegmentSizeAndSunshineStats`.
	SegmentIndex int64 `json:"segmentIndex,omitempty"`
	// YearlyEnergyDcKwh: How much sunlight energy this part of the layout captures
	// over the course of a year, in DC kWh, assuming the panels described above.
	YearlyEnergyDcKwh float64 `json:"yearlyEnergyDcKwh,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AzimuthDegrees") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AzimuthDegrees") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s RoofSegmentSummary) MarshalJSON() ([]byte, error) {
	type NoMethod RoofSegmentSummary
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *RoofSegmentSummary) UnmarshalJSON(data []byte) error {
	type NoMethod RoofSegmentSummary
	var s1 struct {
		AzimuthDegrees    gensupport.JSONFloat64 `json:"azimuthDegrees"`
		PitchDegrees      gensupport.JSONFloat64 `json:"pitchDegrees"`
		YearlyEnergyDcKwh gensupport.JSONFloat64 `json:"yearlyEnergyDcKwh"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.AzimuthDegrees = float64(s1.AzimuthDegrees)
	s.PitchDegrees = float64(s1.PitchDegrees)
	s.YearlyEnergyDcKwh = float64(s1.YearlyEnergyDcKwh)
	return nil
}

// SavingsOverTime: Financial information that's shared between different
// financing methods.
type SavingsOverTime struct {
	// FinanciallyViable: Indicates whether this scenario is financially viable.
	// Will be false for scenarios with poor financial viability (e.g.,
	// money-losing).
	FinanciallyViable bool `json:"financiallyViable,omitempty"`
	// PresentValueOfSavingsLifetime: Using the assumed discount rate, what is the
	// present value of the cumulative lifetime savings?
	PresentValueOfSavingsLifetime *Money `json:"presentValueOfSavingsLifetime,omitempty"`
	// PresentValueOfSavingsYear20: Using the assumed discount rate, what is the
	// present value of the cumulative 20-year savings?
	PresentValueOfSavingsYear20 *Money `json:"presentValueOfSavingsYear20,omitempty"`
	// SavingsLifetime: Savings in the entire panel lifetime.
	SavingsLifetime *Money `json:"savingsLifetime,omitempty"`
	// SavingsYear1: Savings in the first year after panel installation.
	SavingsYear1 *Money `json:"savingsYear1,omitempty"`
	// SavingsYear20: Savings in the first twenty years after panel installation.
	SavingsYear20 *Money `json:"savingsYear20,omitempty"`
	// ForceSendFields is a list of field names (e.g. "FinanciallyViable") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "FinanciallyViable") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SavingsOverTime) MarshalJSON() ([]byte, error) {
	type NoMethod SavingsOverTime
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SizeAndSunshineStats: Size and sunniness quantiles of a roof, or part of a
// roof.
type SizeAndSunshineStats struct {
	// AreaMeters2: The area of the roof or roof segment, in m^2. This is the roof
	// area (accounting for tilt), not the ground footprint area.
	AreaMeters2 float64 `json:"areaMeters2,omitempty"`
	// GroundAreaMeters2: The ground footprint area covered by the roof or roof
	// segment, in m^2.
	GroundAreaMeters2 float64 `json:"groundAreaMeters2,omitempty"`
	// SunshineQuantiles: Quantiles of the pointwise sunniness across the area. If
	// there are N values here, this represents the (N-1)-iles. For example, if
	// there are 5 values, then they would be the quartiles (min, 25%, 50%, 75%,
	// max). Values are in annual kWh/kW like max_sunshine_hours_per_year.
	SunshineQuantiles []float64 `json:"sunshineQuantiles,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AreaMeters2") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AreaMeters2") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SizeAndSunshineStats) MarshalJSON() ([]byte, error) {
	type NoMethod SizeAndSunshineStats
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *SizeAndSunshineStats) UnmarshalJSON(data []byte) error {
	type NoMethod SizeAndSunshineStats
	var s1 struct {
		AreaMeters2       gensupport.JSONFloat64   `json:"areaMeters2"`
		GroundAreaMeters2 gensupport.JSONFloat64   `json:"groundAreaMeters2"`
		SunshineQuantiles []gensupport.JSONFloat64 `json:"sunshineQuantiles"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.AreaMeters2 = float64(s1.AreaMeters2)
	s.GroundAreaMeters2 = float64(s1.GroundAreaMeters2)
	s.SunshineQuantiles = make([]float64, len(s1.SunshineQuantiles))
	for i := range s1.SunshineQuantiles {
		s.SunshineQuantiles[i] = float64(s1.SunshineQuantiles[i])
	}
	return nil
}

// SolarPanel: SolarPanel describes the position, orientation, and production
// of a single solar panel. See the panel_height_meters, panel_width_meters,
// and panel_capacity_watts fields in SolarPotential for information on the
// parameters of the panel.
type SolarPanel struct {
	// Center: The centre of the panel.
	Center *LatLng `json:"center,omitempty"`
	// Orientation: The orientation of the panel.
	//
	// Possible values:
	//   "SOLAR_PANEL_ORIENTATION_UNSPECIFIED" - No panel orientation is known.
	//   "LANDSCAPE" - A `LANDSCAPE` panel has its long edge perpendicular to the
	// azimuth direction of the roof segment that it is placed on.
	//   "PORTRAIT" - A `PORTRAIT` panel has its long edge parallel to the azimuth
	// direction of the roof segment that it is placed on.
	Orientation string `json:"orientation,omitempty"`
	// SegmentIndex: Index in roof_segment_stats of the
	// `RoofSegmentSizeAndSunshineStats` which corresponds to the roof segment that
	// this panel is placed on.
	SegmentIndex int64 `json:"segmentIndex,omitempty"`
	// YearlyEnergyDcKwh: How much sunlight energy this layout captures over the
	// course of a year, in DC kWh.
	YearlyEnergyDcKwh float64 `json:"yearlyEnergyDcKwh,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Center") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Center") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SolarPanel) MarshalJSON() ([]byte, error) {
	type NoMethod SolarPanel
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *SolarPanel) UnmarshalJSON(data []byte) error {
	type NoMethod SolarPanel
	var s1 struct {
		YearlyEnergyDcKwh gensupport.JSONFloat64 `json:"yearlyEnergyDcKwh"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.YearlyEnergyDcKwh = float64(s1.YearlyEnergyDcKwh)
	return nil
}

// SolarPanelConfig: SolarPanelConfig describes a particular placement of solar
// panels on the roof.
type SolarPanelConfig struct {
	// PanelsCount: Total number of panels. Note that this is redundant to (the sum
	// of) the corresponding fields in roof_segment_summaries.
	PanelsCount int64 `json:"panelsCount,omitempty"`
	// RoofSegmentSummaries: Information about the production of each roof segment
	// that is carrying at least one panel in this layout.
	// `roof_segment_summaries[i]` describes the i-th roof segment, including its
	// size, expected production and orientation.
	RoofSegmentSummaries []*RoofSegmentSummary `json:"roofSegmentSummaries,omitempty"`
	// YearlyEnergyDcKwh: How much sunlight energy this layout captures over the
	// course of a year, in DC kWh, assuming the panels described above.
	YearlyEnergyDcKwh float64 `json:"yearlyEnergyDcKwh,omitempty"`
	// ForceSendFields is a list of field names (e.g. "PanelsCount") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "PanelsCount") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SolarPanelConfig) MarshalJSON() ([]byte, error) {
	type NoMethod SolarPanelConfig
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *SolarPanelConfig) UnmarshalJSON(data []byte) error {
	type NoMethod SolarPanelConfig
	var s1 struct {
		YearlyEnergyDcKwh gensupport.JSONFloat64 `json:"yearlyEnergyDcKwh"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.YearlyEnergyDcKwh = float64(s1.YearlyEnergyDcKwh)
	return nil
}

// SolarPotential: Information about the solar potential of a building. A
// number of fields in this are defined in terms of "panels". The fields
// panel_capacity_watts, panel_height_meters, and panel_width_meters describe
// the parameters of the model of panel used in these calculations.
type SolarPotential struct {
	// BuildingStats: Size and sunlight quantiles for the entire building,
	// including parts of the roof that were not assigned to some roof segment.
	// Because the orientations of these parts are not well characterised, the roof
	// area estimate is unreliable, but the ground area estimate is reliable. It
	// may be that a more reliable whole building roof area can be obtained by
	// scaling the roof area from whole_roof_stats by the ratio of the ground areas
	// of `building_stats` and `whole_roof_stats`.
	BuildingStats *SizeAndSunshineStats `json:"buildingStats,omitempty"`
	// CarbonOffsetFactorKgPerMwh: Equivalent amount of CO2 produced per MWh of
	// grid electricity. This is a measure of the carbon intensity of grid
	// electricity displaced by solar electricity.
	CarbonOffsetFactorKgPerMwh float64 `json:"carbonOffsetFactorKgPerMwh,omitempty"`
	// FinancialAnalyses: A FinancialAnalysis gives the savings from going solar
	// assuming a given monthly bill and a given electricity provider. They are in
	// order of increasing order of monthly bill amount. This field will be empty
	// for buildings in areas for which the Solar API does not have enough
	// information to perform financial computations.
	FinancialAnalyses []*FinancialAnalysis `json:"financialAnalyses,omitempty"`
	// MaxArrayAreaMeters2: Size, in square meters, of the maximum array.
	MaxArrayAreaMeters2 float64 `json:"maxArrayAreaMeters2,omitempty"`
	// MaxArrayPanelsCount: Size of the maximum array - that is, the maximum number
	// of panels that can fit on the roof.
	MaxArrayPanelsCount int64 `json:"maxArrayPanelsCount,omitempty"`
	// MaxSunshineHoursPerYear: Maximum number of sunshine hours received per year,
	// by any point on the roof. Sunshine hours are a measure of the total amount
	// of insolation (energy) received per year. 1 sunshine hour = 1 kWh per kW
	// (where kW refers to kW of capacity under Standard Testing Conditions).
	MaxSunshineHoursPerYear float64 `json:"maxSunshineHoursPerYear,omitempty"`
	// PanelCapacityWatts: Capacity, in watts, of the panel used in the
	// calculations.
	PanelCapacityWatts float64 `json:"panelCapacityWatts,omitempty"`
	// PanelHeightMeters: Height, in meters in portrait orientation, of the panel
	// used in the calculations.
	PanelHeightMeters float64 `json:"panelHeightMeters,omitempty"`
	// PanelLifetimeYears: The expected lifetime, in years, of the solar panels.
	// This is used in the financial calculations.
	PanelLifetimeYears int64 `json:"panelLifetimeYears,omitempty"`
	// PanelWidthMeters: Width, in meters in portrait orientation, of the panel
	// used in the calculations.
	PanelWidthMeters float64 `json:"panelWidthMeters,omitempty"`
	// RoofSegmentStats: Size and sunlight quantiles for each roof segment.
	RoofSegmentStats []*RoofSegmentSizeAndSunshineStats `json:"roofSegmentStats,omitempty"`
	// SolarPanelConfigs: Each SolarPanelConfig describes a different arrangement
	// of solar panels on the roof. They are in order of increasing number of
	// panels. The `SolarPanelConfig` with panels_count=N is based on the first N
	// panels in the `solar_panels` list. This field is only populated if at least
	// 4 panels can fit on a roof.
	SolarPanelConfigs []*SolarPanelConfig `json:"solarPanelConfigs,omitempty"`
	// SolarPanels: Each SolarPanel describes a single solar panel. They are listed
	// in the order that the panel layout algorithm placed this. This is usually,
	// though not always, in decreasing order of annual energy production.
	SolarPanels []*SolarPanel `json:"solarPanels,omitempty"`
	// WholeRoofStats: Total size and sunlight quantiles for the part of the roof
	// that was assigned to some roof segment. Despite the name, this may not
	// include the entire building. See building_stats.
	WholeRoofStats *SizeAndSunshineStats `json:"wholeRoofStats,omitempty"`
	// ForceSendFields is a list of field names (e.g. "BuildingStats") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "BuildingStats") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SolarPotential) MarshalJSON() ([]byte, error) {
	type NoMethod SolarPotential
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *SolarPotential) UnmarshalJSON(data []byte) error {
	type NoMethod SolarPotential
	var s1 struct {
		CarbonOffsetFactorKgPerMwh gensupport.JSONFloat64 `json:"carbonOffsetFactorKgPerMwh"`
		MaxArrayAreaMeters2        gensupport.JSONFloat64 `json:"maxArrayAreaMeters2"`
		MaxSunshineHoursPerYear    gensupport.JSONFloat64 `json:"maxSunshineHoursPerYear"`
		PanelCapacityWatts         gensupport.JSONFloat64 `json:"panelCapacityWatts"`
		PanelHeightMeters          gensupport.JSONFloat64 `json:"panelHeightMeters"`
		PanelWidthMeters           gensupport.JSONFloat64 `json:"panelWidthMeters"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.CarbonOffsetFactorKgPerMwh = float64(s1.CarbonOffsetFactorKgPerMwh)
	s.MaxArrayAreaMeters2 = float64(s1.MaxArrayAreaMeters2)
	s.MaxSunshineHoursPerYear = float64(s1.MaxSunshineHoursPerYear)
	s.PanelCapacityWatts = float64(s1.PanelCapacityWatts)
	s.PanelHeightMeters = float64(s1.PanelHeightMeters)
	s.PanelWidthMeters = float64(s1.PanelWidthMeters)
	return nil
}

type BuildingInsightsFindClosestCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// FindClosest: Locates the building whose centroid is closest to a query
// point. Returns an error with code `NOT_FOUND` if there are no buildings
// within approximately 50m of the query point.
func (r *BuildingInsightsService) FindClosest() *BuildingInsightsFindClosestCall {
	c := &BuildingInsightsFindClosestCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// ExactQualityRequired sets the optional parameter "exactQualityRequired":
// Whether to require exact quality of the imagery. If set to false, the
// `required_quality` field is interpreted as the minimum required quality,
// such that HIGH quality imagery may be returned when `required_quality` is
// set to MEDIUM. If set to true, `required_quality` is interpreted as the
// exact required quality and only `MEDIUM` quality imagery is returned if
// `required_quality` is set to `MEDIUM`.
func (c *BuildingInsightsFindClosestCall) ExactQualityRequired(exactQualityRequired bool) *BuildingInsightsFindClosestCall {
	c.urlParams_.Set("exactQualityRequired", fmt.Sprint(exactQualityRequired))
	return c
}

// Experiments sets the optional parameter "experiments": Specifies the pre-GA
// experiments to enable. Requests using this field are classified as a pre-GA
// offering under the Google Maps Platform Service Specific Terms
// (https://cloud.google.com/maps-platform/terms/maps-service-terms). See
// launch stage descriptions
// (https://cloud.google.com/maps-platform/terms/launch-stages) for more
// details.
//
// Possible values:
//
//	"EXPERIMENT_UNSPECIFIED" - No experiments are specified.
//	"EXPANDED_COVERAGE" - Expands the geographic region available for querying
//
// solar data. For more information, see [Expanded
// Coverage](https://developers.google.com/maps/documentation/solar/expanded-cov
// erage).
func (c *BuildingInsightsFindClosestCall) Experiments(experiments ...string) *BuildingInsightsFindClosestCall {
	c.urlParams_.SetMulti("experiments", append([]string{}, experiments...))
	return c
}

// LocationLatitude sets the optional parameter "location.latitude": The
// latitude in degrees. It must be in the range [-90.0, +90.0].
func (c *BuildingInsightsFindClosestCall) LocationLatitude(locationLatitude float64) *BuildingInsightsFindClosestCall {
	c.urlParams_.Set("location.latitude", fmt.Sprint(locationLatitude))
	return c
}

// LocationLongitude sets the optional parameter "location.longitude": The
// longitude in degrees. It must be in the range [-180.0, +180.0].
func (c *BuildingInsightsFindClosestCall) LocationLongitude(locationLongitude float64) *BuildingInsightsFindClosestCall {
	c.urlParams_.Set("location.longitude", fmt.Sprint(locationLongitude))
	return c
}

// RequiredQuality sets the optional parameter "requiredQuality": The minimum
// quality level allowed in the results. No result with lower quality than this
// will be returned. Not specifying this is equivalent to restricting to HIGH
// quality only.
//
// Possible values:
//
//	"IMAGERY_QUALITY_UNSPECIFIED" - No quality is known.
//	"HIGH" - Solar data is derived from aerial imagery captured at
//
// low-altitude and processed at 0.1 m/pixel.
//
//	"MEDIUM" - Solar data is derived from enhanced aerial imagery captured at
//
// high-altitude and processed at 0.25 m/pixel.
//
//	"LOW" - Solar data is derived from enhanced satellite imagery processed at
//
// 0.25 m/pixel.
//
//	"BASE" - Solar data is derived from enhanced satellite imagery processed
//
// at 0.25 m/pixel.
func (c *BuildingInsightsFindClosestCall) RequiredQuality(requiredQuality string) *BuildingInsightsFindClosestCall {
	c.urlParams_.Set("requiredQuality", requiredQuality)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *BuildingInsightsFindClosestCall) Fields(s ...googleapi.Field) *BuildingInsightsFindClosestCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *BuildingInsightsFindClosestCall) IfNoneMatch(entityTag string) *BuildingInsightsFindClosestCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *BuildingInsightsFindClosestCall) Context(ctx context.Context) *BuildingInsightsFindClosestCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *BuildingInsightsFindClosestCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *BuildingInsightsFindClosestCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/buildingInsights:findClosest")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "solar.buildingInsights.findClosest", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "solar.buildingInsights.findClosest" call.
// Any non-2xx status code is an error. Response headers are in either
// *BuildingInsights.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *BuildingInsightsFindClosestCall) Do(opts ...googleapi.CallOption) (*BuildingInsights, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &BuildingInsights{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "solar.buildingInsights.findClosest", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type DataLayersGetCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets solar information for a region surrounding a location. Returns an
// error with code `NOT_FOUND` if the location is outside the coverage area.
func (r *DataLayersService) Get() *DataLayersGetCall {
	c := &DataLayersGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// ExactQualityRequired sets the optional parameter "exactQualityRequired":
// Whether to require exact quality of the imagery. If set to false, the
// `required_quality` field is interpreted as the minimum required quality,
// such that HIGH quality imagery may be returned when `required_quality` is
// set to MEDIUM. If set to true, `required_quality` is interpreted as the
// exact required quality and only `MEDIUM` quality imagery is returned if
// `required_quality` is set to `MEDIUM`.
func (c *DataLayersGetCall) ExactQualityRequired(exactQualityRequired bool) *DataLayersGetCall {
	c.urlParams_.Set("exactQualityRequired", fmt.Sprint(exactQualityRequired))
	return c
}

// Experiments sets the optional parameter "experiments": Specifies the pre-GA
// experiments to enable. Requests using this field are classified as a pre-GA
// offering under the Google Maps Platform Service Specific Terms
// (https://cloud.google.com/maps-platform/terms/maps-service-terms). See
// launch stage descriptions (
// https://cloud.google.com/maps-platform/terms/launch-stages) for more
// details.
//
// Possible values:
//
//	"EXPERIMENT_UNSPECIFIED" - No experiments are specified.
//	"EXPANDED_COVERAGE" - Expands the geographic region available for querying
//
// solar data. For more information, see [Expanded
// Coverage](https://developers.google.com/maps/documentation/solar/expanded-cov
// erage).
func (c *DataLayersGetCall) Experiments(experiments ...string) *DataLayersGetCall {
	c.urlParams_.SetMulti("experiments", append([]string{}, experiments...))
	return c
}

// LocationLatitude sets the optional parameter "location.latitude": The
// latitude in degrees. It must be in the range [-90.0, +90.0].
func (c *DataLayersGetCall) LocationLatitude(locationLatitude float64) *DataLayersGetCall {
	c.urlParams_.Set("location.latitude", fmt.Sprint(locationLatitude))
	return c
}

// LocationLongitude sets the optional parameter "location.longitude": The
// longitude in degrees. It must be in the range [-180.0, +180.0].
func (c *DataLayersGetCall) LocationLongitude(locationLongitude float64) *DataLayersGetCall {
	c.urlParams_.Set("location.longitude", fmt.Sprint(locationLongitude))
	return c
}

// PixelSizeMeters sets the optional parameter "pixelSizeMeters": The minimum
// scale, in meters per pixel, of the data to return. Values of 0.1 (the
// default, if this field is not set explicitly), 0.25, 0.5, and 1.0 are
// supported. Imagery components whose normal resolution is less than
// `pixel_size_meters` will be returned at the resolution specified by
// `pixel_size_meters`; imagery components whose normal resolution is equal to
// or greater than `pixel_size_meters` will be returned at that normal
// resolution.
func (c *DataLayersGetCall) PixelSizeMeters(pixelSizeMeters float64) *DataLayersGetCall {
	c.urlParams_.Set("pixelSizeMeters", fmt.Sprint(pixelSizeMeters))
	return c
}

// RadiusMeters sets the optional parameter "radiusMeters": Required. The
// radius, in meters, defining the region surrounding that centre point for
// which data should be returned. The limitations on this value are: * Any
// value up to 100m can always be specified. * Values over 100m can be
// specified, as long as `radius_meters` <= `pixel_size_meters * 1000`. *
// However, for values over 175m, the `DataLayerView` in the request must not
// include monthly flux or hourly shade.
func (c *DataLayersGetCall) RadiusMeters(radiusMeters float64) *DataLayersGetCall {
	c.urlParams_.Set("radiusMeters", fmt.Sprint(radiusMeters))
	return c
}

// RequiredQuality sets the optional parameter "requiredQuality": The minimum
// quality level allowed in the results. No result with lower quality than this
// will be returned. Not specifying this is equivalent to restricting to HIGH
// quality only.
//
// Possible values:
//
//	"IMAGERY_QUALITY_UNSPECIFIED" - No quality is known.
//	"HIGH" - Solar data is derived from aerial imagery captured at
//
// low-altitude and processed at 0.1 m/pixel.
//
//	"MEDIUM" - Solar data is derived from enhanced aerial imagery captured at
//
// high-altitude and processed at 0.25 m/pixel.
//
//	"LOW" - Solar data is derived from enhanced satellite imagery processed at
//
// 0.25 m/pixel.
//
//	"BASE" - Solar data is derived from enhanced satellite imagery processed
//
// at 0.25 m/pixel.
func (c *DataLayersGetCall) RequiredQuality(requiredQuality string) *DataLayersGetCall {
	c.urlParams_.Set("requiredQuality", requiredQuality)
	return c
}

// View sets the optional parameter "view": The desired subset of the data to
// return.
//
// Possible values:
//
//	"DATA_LAYER_VIEW_UNSPECIFIED" - Equivalent to FULL.
//	"DSM_LAYER" - Get the DSM only.
//	"IMAGERY_LAYERS" - Get the DSM, RGB, and mask.
//	"IMAGERY_AND_ANNUAL_FLUX_LAYERS" - Get the DSM, RGB, mask, and annual
//
// flux.
//
//	"IMAGERY_AND_ALL_FLUX_LAYERS" - Get the DSM, RGB, mask, annual flux, and
//
// monthly flux.
//
//	"FULL_LAYERS" - Get all data.
func (c *DataLayersGetCall) View(view string) *DataLayersGetCall {
	c.urlParams_.Set("view", view)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *DataLayersGetCall) Fields(s ...googleapi.Field) *DataLayersGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *DataLayersGetCall) IfNoneMatch(entityTag string) *DataLayersGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *DataLayersGetCall) Context(ctx context.Context) *DataLayersGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *DataLayersGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *DataLayersGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/dataLayers:get")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "solar.dataLayers.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "solar.dataLayers.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *DataLayers.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *DataLayersGetCall) Do(opts ...googleapi.CallOption) (*DataLayers, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &DataLayers{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "solar.dataLayers.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type GeoTiffGetCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Returns an image by its ID.
func (r *GeoTiffService) Get() *GeoTiffGetCall {
	c := &GeoTiffGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// Id sets the optional parameter "id": Required. The ID of the asset being
// requested.
func (c *GeoTiffGetCall) Id(id string) *GeoTiffGetCall {
	c.urlParams_.Set("id", id)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *GeoTiffGetCall) Fields(s ...googleapi.Field) *GeoTiffGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *GeoTiffGetCall) IfNoneMatch(entityTag string) *GeoTiffGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *GeoTiffGetCall) Context(ctx context.Context) *GeoTiffGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *GeoTiffGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *GeoTiffGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/geoTiff:get")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "solar.geoTiff.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "solar.geoTiff.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *HttpBody.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *GeoTiffGetCall) Do(opts ...googleapi.CallOption) (*HttpBody, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &HttpBody{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "solar.geoTiff.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}
