package proposallineage_test

import (
	"errors"
	"fmt"
	"strconv"
	"strings"
	"testing"

	"github.com/git-town/git-town/v22/internal/config/configdomain"
	"github.com/git-town/git-town/v22/internal/forge/forgedomain"
	"github.com/git-town/git-town/v22/internal/git/gitdomain"
	"github.com/git-town/git-town/v22/internal/proposallineage"
	"github.com/git-town/git-town/v22/internal/subshell/subshelldomain"
	. "github.com/git-town/git-town/v22/pkg/prelude"
	"github.com/shoenig/test/must"
)

// a double that implements the forgedomain.ProposalFinder interface
type testFinder struct {
	count int
}

func (tf *testFinder) BrowseRepository(_ subshelldomain.Runner) error {
	return nil
}

func (tf *testFinder) CreateProposal(_ forgedomain.CreateProposalArgs) error {
	return nil
}

func (tf *testFinder) DefaultProposalMessage(_ forgedomain.ProposalData) string {
	return ""
}

func (tf *testFinder) FindProposal(source, _ gitdomain.LocalBranchName) (Option[forgedomain.Proposal], error) {
	if strings.HasPrefix(source.String(), "no-proposal") {
		return None[forgedomain.Proposal](), nil
	}
	tf.count++
	return Some(forgedomain.Proposal{
		Data: forgedomain.ProposalData{
			URL: "https://www.github.com/git-town/git-town/pull/" + strconv.Itoa(tf.count),
		},
	}), nil
}

// a Connector double that simulates connection errors
type failingFinder struct{}

func (ff *failingFinder) BrowseRepository(_ subshelldomain.Runner) error {
	return errors.New("simulated error browsing repository")
}

func (ff *failingFinder) CreateProposal(_ forgedomain.CreateProposalArgs) error {
	return errors.New("simulated error creating proposal")
}

func (ff *failingFinder) DefaultProposalMessage(_ forgedomain.ProposalData) string {
	return ""
}

func (ff *failingFinder) FindProposal(branch, _ gitdomain.LocalBranchName) (Option[forgedomain.Proposal], error) {
	return None[forgedomain.Proposal](), fmt.Errorf("simulated error finding proposal for %s", branch)
}

func TestRender(t *testing.T) {
	t.Parallel()

	t.Run("all branches have proposals", func(t *testing.T) {
		t.Parallel()
		lineage := configdomain.NewLineageWith(configdomain.LineageData{
			"feature-a":  "main",
			"feature-a1": "feature-a",
			"feature-a2": "feature-a",
			"feature-b":  "main",
			"feature-b1": "feature-b",
			"feature-b2": "feature-b",
		})
		var connector forgedomain.Connector = &testFinder{}
		have := proposallineage.RenderSection(lineage, "feature-a", configdomain.OrderAsc, forgedomain.ProposalBreadcrumbBranches, Some(connector))
		want := `

-------------------------
- main
  - https://www.github.com/git-town/git-town/pull/1 :point_left:
    - https://www.github.com/git-town/git-town/pull/2
    - https://www.github.com/git-town/git-town/pull/3

<sup>[Stack](https://www.git-town.com/how-to/proposal-breadcrumb.html) generated by [Git Town](https://github.com/git-town/git-town)</sup>
`[1:]
		must.Eq(t, want, have)
	})

	t.Run("connector errors", func(t *testing.T) {
		t.Parallel()
		lineage := configdomain.NewLineageWith(configdomain.LineageData{
			"feature-a": "main",
			"feature-b": "feature-a",
			"feature-c": "feature-b",
		})
		var connector forgedomain.Connector = &failingFinder{}
		have := proposallineage.RenderSection(lineage, "feature-a", configdomain.OrderAsc, forgedomain.ProposalBreadcrumbBranches, Some(connector))
		want := `

-------------------------
- main
  - feature-a :point_left:

<sup>[Stack](https://www.git-town.com/how-to/proposal-breadcrumb.html) generated by [Git Town](https://github.com/git-town/git-town)</sup>
`[1:]
		must.Eq(t, want, have)
	})

	t.Run("some branches have proposals", func(t *testing.T) {
		t.Parallel()
		lineage := configdomain.NewLineageWith(configdomain.LineageData{
			"no-proposal-a": "main",
			"no-proposal-b": "no-proposal-a",
			"feature-c":     "no-proposal-b",
		})
		var connector forgedomain.Connector = &testFinder{}
		have := proposallineage.RenderSection(lineage, "no-proposal-a", configdomain.OrderAsc, forgedomain.ProposalBreadcrumbStacks, Some(connector))
		want := `

-------------------------
- main
  - no-proposal-a :point_left:
    - no-proposal-b
      - https://www.github.com/git-town/git-town/pull/1

<sup>[Stack](https://www.git-town.com/how-to/proposal-breadcrumb.html) generated by [Git Town](https://github.com/git-town/git-town)</sup>
`[1:]
		must.Eq(t, want, have)
	})
}
