package cmd

import (
	"bufio"
	"bytes"
	"context"
	"errors"
	"fmt"
	"os"
	"os/exec"
	"strings"
	"text/template"
	"time"

	"github.com/nickawilliams/diffscribe/internal/llm"
	"github.com/spf13/viper"
)

type gitContext struct {
	Branch string
	Paths  []string
	Diff   string
}

func collectContext() (gitContext, error) {
	if oid := strings.TrimSpace(os.Getenv("DIFFSCRIBE_STASH_COMMIT")); oid != "" {
		return collectStashContext(oid), nil
	}

	return gitContext{
		Branch: strings.TrimSpace(run("git", "rev-parse", "--abbrev-ref", "HEAD")),
		Paths:  nonEmptyLines(run("git", "diff", "--cached", "--name-only")),
		Diff:   capString(run("git", "diff", "--cached", "--unified=0"), 8000),
	}, nil
}

func collectStashContext(oid string) gitContext {
	return gitContext{
		Branch: strings.TrimSpace(run("git", "rev-parse", "--abbrev-ref", "HEAD")),
		Paths:  nonEmptyLines(run("git", "stash", "show", "--include-untracked", "--name-only", oid)),
		Diff:   capString(run("git", "stash", "show", "--include-untracked", "--patch", oid), 8000),
	}
}

func generateCandidates(c gitContext, prefix string) []string {
	if len(c.Paths) == 0 {
		return nil
	}

	tplData := templateData{
		Branch:     c.Branch,
		Paths:      c.Paths,
		Diff:       c.Diff,
		FileCount:  len(c.Paths),
		Summary:    joinLimit(c.Paths, 3),
		DiffLength: len(c.Diff),
		Prefix:     prefix,
		Format:     viper.GetString("format"),
		Timestamp:  time.Now(),
	}

	cfg := newLLMConfig(tplData)
	if err := requireLLMConfig(cfg); err != nil {
		fmt.Fprintln(os.Stderr, err)
		return nil
	}

	msgs, err := llm.GenerateCommitMessages(context.Background(), llm.Context{
		Branch: c.Branch,
		Paths:  c.Paths,
		Diff:   c.Diff,
		Prefix: prefix,
	}, cfg)
	if err != nil {
		fmt.Fprintln(os.Stderr, "diffscribe: LLM error:", err)
	} else if len(msgs) > 0 {
		return msgs
	}

	return stubCandidates(c, prefix)
}

type templateData struct {
	Branch     string
	Paths      []string
	Diff       string
	FileCount  int
	Summary    string
	DiffLength int
	Prefix     string
	Format     string
	Timestamp  time.Time
}

type systemPromptData struct {
	templateData
	Model       string
	Provider    string
	Quantity    int
	Temperature float64
}

type userPromptData struct {
	templateData
	Quantity int
}

func newLLMConfig(data templateData) llm.Config {
	cfg := llm.Config{
		APIKey:              strings.TrimSpace(viper.GetString("llm.api_key")),
		Provider:            strings.TrimSpace(viper.GetString("llm.provider")),
		Model:               strings.TrimSpace(viper.GetString("llm.model")),
		BaseURL:             strings.TrimSpace(viper.GetString("llm.base_url")),
		Temperature:         viper.GetFloat64("llm.temperature"),
		Quantity:            viper.GetInt("quantity"),
		MaxCompletionTokens: viper.GetInt("llm.max_completion_tokens"),
	}

	sysData := systemPromptData{
		templateData: data,
		Model:        cfg.Model,
		Provider:     "openai",
		Quantity:     cfg.Quantity,
		Temperature:  cfg.Temperature,
	}
	userData := userPromptData{
		templateData: data,
		Quantity:     cfg.Quantity,
	}

	cfg.SystemPrompt = renderTemplate(viper.GetString("system_prompt"), sysData)
	cfg.UserPrompt = renderTemplate(viper.GetString("user_prompt"), userData)
	return cfg
}

func renderTemplate(raw string, data any) string {
	raw = strings.TrimSpace(raw)
	if raw == "" {
		return ""
	}
	tmpl, err := template.New("prompt").Parse(raw)
	if err != nil {
		fmt.Fprintf(os.Stderr, "diffscribe: bad system prompt template: %v\n", err)
		return raw
	}
	var buf bytes.Buffer
	if err := tmpl.Execute(&buf, data); err != nil {
		fmt.Fprintf(os.Stderr, "diffscribe: system prompt render error: %v\n", err)
		return raw
	}
	return buf.String()
}

func requireLLMConfig(cfg llm.Config) error {
	if strings.TrimSpace(cfg.APIKey) == "" {
		return errors.New("diffscribe: api_key is required (set --api-key or DIFFSCRIBE_API_KEY/OPENAI_API_KEY)")
	}
	return nil
}

func stubCandidates(c gitContext, prefix string) []string {
	summary := joinLimit(c.Paths, 3)
	suggestions := []string{
		"feat: " + summary,
		"fix: address issues in " + c.Branch,
		"chore: update " + summary,
		"refactor: simplify " + summary,
		"docs: update docs for " + summary,
	}

	trimmed := strings.TrimSpace(prefix)
	if trimmed == "" {
		return suggestions
	}

	withPrefix := make([]string, 0, len(suggestions))
	lowerPrefix := strings.ToLower(trimmed)
	for _, cand := range suggestions {
		if strings.HasPrefix(strings.ToLower(cand), lowerPrefix) {
			withPrefix = append(withPrefix, cand)
			continue
		}
		remainder := strings.TrimLeft(cand, " ")
		if strings.HasSuffix(trimmed, " ") {
			withPrefix = append(withPrefix, trimmed+remainder)
		} else {
			withPrefix = append(withPrefix, trimmed+" "+remainder)
		}
	}
	return withPrefix
}

// runFunc is the function used to execute shell commands. It can be replaced in tests.
var runFunc = runCommand

func runCommand(name string, args ...string) string {
	cmd := exec.Command(name, args...)
	cmd.Env = os.Environ()
	var out bytes.Buffer
	var errBuf bytes.Buffer
	cmd.Stdout, cmd.Stderr = &out, &errBuf
	_ = cmd.Start()
	done := make(chan struct{})
	go func() {
		_ = cmd.Wait()
		close(done)
	}()
	select {
	case <-done:
	case <-time.After(1500 * time.Millisecond):
		_ = cmd.Process.Kill()
	}
	return out.String()
}

func run(name string, args ...string) string {
	return runFunc(name, args...)
}

func nonEmptyLines(s string) []string {
	sc := bufio.NewScanner(strings.NewReader(s))
	var out []string
	for sc.Scan() {
		x := strings.TrimSpace(sc.Text())
		if x != "" {
			out = append(out, x)
		}
	}
	return out
}

func capString(s string, n int) string {
	if len(s) > n {
		return s[:n] + "\n…"
	}
	return s
}

func joinLimit(ss []string, n int) string {
	if len(ss) == 0 {
		return "changes"
	}
	if len(ss) <= n {
		return strings.Join(ss, ", ")
	}
	return strings.Join(ss[:n], ", ") + "…"
}
