import { jsx as _jsx } from "react/jsx-runtime";
/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { render } from '../../test-utils/render.js';
import { Table } from './Table.js';
import { Text } from 'ink';
describe('Table', () => {
    it('should render headers and data correctly', async () => {
        const columns = [
            { key: 'id', header: 'ID', width: 5 },
            { key: 'name', header: 'Name', flexGrow: 1 },
        ];
        const data = [
            { id: 1, name: 'Alice' },
            { id: 2, name: 'Bob' },
        ];
        const renderResult = render(_jsx(Table, { columns: columns, data: data }), 100);
        const { lastFrame, waitUntilReady } = renderResult;
        await waitUntilReady?.();
        const output = lastFrame();
        expect(output).toContain('ID');
        expect(output).toContain('Name');
        expect(output).toContain('1');
        expect(output).toContain('Alice');
        expect(output).toContain('2');
        expect(output).toContain('Bob');
        await expect(renderResult).toMatchSvgSnapshot();
    });
    it('should support custom cell rendering', async () => {
        const columns = [
            {
                key: 'value',
                header: 'Value',
                flexGrow: 1,
                renderCell: (item) => (_jsx(Text, { color: "green", children: item.value * 2 })),
            },
        ];
        const data = [{ value: 10 }];
        const renderResult = render(_jsx(Table, { columns: columns, data: data }), 100);
        const { lastFrame, waitUntilReady } = renderResult;
        await waitUntilReady?.();
        const output = lastFrame();
        expect(output).toContain('20');
        await expect(renderResult).toMatchSvgSnapshot();
    });
    it('should handle undefined values gracefully', async () => {
        const columns = [{ key: 'name', header: 'Name', flexGrow: 1 }];
        const data = [{ name: undefined }];
        const { lastFrame, waitUntilReady } = render(_jsx(Table, { columns: columns, data: data }), 100);
        await waitUntilReady?.();
        const output = lastFrame();
        expect(output).toContain('undefined');
    });
    it('should support inverse text rendering', async () => {
        const columns = [
            {
                key: 'status',
                header: 'Status',
                flexGrow: 1,
                renderCell: (item) => (_jsx(Text, { inverse: true, children: item.status })),
            },
        ];
        const data = [{ status: 'Active' }];
        const renderResult = render(_jsx(Table, { columns: columns, data: data }), 100);
        const { lastFrame, waitUntilReady } = renderResult;
        await waitUntilReady?.();
        const output = lastFrame();
        expect(output).toContain('Active');
        await expect(renderResult).toMatchSvgSnapshot();
    });
});
//# sourceMappingURL=Table.test.js.map