/**
 * @license
 * Copyright 2026 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { MessageType } from '../types.js';
import { formatDuration } from '../utils/formatters.js';
import { UserAccountManager, getG1CreditBalance, } from '@google/gemini-cli-core';
import { CommandKind, } from './types.js';
function getUserIdentity(context) {
    const selectedAuthType = context.services.settings.merged.security.auth.selectedType || '';
    const userAccountManager = new UserAccountManager();
    const cachedAccount = userAccountManager.getCachedGoogleAccount();
    const userEmail = cachedAccount ?? undefined;
    const tier = context.services.config?.getUserTierName();
    const paidTier = context.services.config?.getUserPaidTier();
    const creditBalance = getG1CreditBalance(paidTier) ?? undefined;
    return { selectedAuthType, userEmail, tier, creditBalance };
}
async function defaultSessionView(context) {
    const now = new Date();
    const { sessionStartTime } = context.session.stats;
    if (!sessionStartTime) {
        context.ui.addItem({
            type: MessageType.ERROR,
            text: 'Session start time is unavailable, cannot calculate stats.',
        });
        return;
    }
    const wallDuration = now.getTime() - sessionStartTime.getTime();
    const { selectedAuthType, userEmail, tier, creditBalance } = getUserIdentity(context);
    const currentModel = context.services.config?.getModel();
    const statsItem = {
        type: MessageType.STATS,
        duration: formatDuration(wallDuration),
        selectedAuthType,
        userEmail,
        tier,
        currentModel,
        creditBalance,
    };
    if (context.services.config) {
        const [quota] = await Promise.all([
            context.services.config.refreshUserQuota(),
            context.services.config.refreshAvailableCredits(),
        ]);
        if (quota) {
            statsItem.quotas = quota;
            statsItem.pooledRemaining = context.services.config.getQuotaRemaining();
            statsItem.pooledLimit = context.services.config.getQuotaLimit();
            statsItem.pooledResetTime = context.services.config.getQuotaResetTime();
        }
    }
    context.ui.addItem(statsItem);
}
export const statsCommand = {
    name: 'stats',
    altNames: ['usage'],
    description: 'Check session stats. Usage: /stats [session|model|tools]',
    kind: CommandKind.BUILT_IN,
    autoExecute: false,
    action: async (context) => {
        await defaultSessionView(context);
    },
    subCommands: [
        {
            name: 'session',
            description: 'Show session-specific usage statistics',
            kind: CommandKind.BUILT_IN,
            autoExecute: true,
            action: async (context) => {
                await defaultSessionView(context);
            },
        },
        {
            name: 'model',
            description: 'Show model-specific usage statistics',
            kind: CommandKind.BUILT_IN,
            autoExecute: true,
            action: (context) => {
                const { selectedAuthType, userEmail, tier } = getUserIdentity(context);
                const currentModel = context.services.config?.getModel();
                const pooledRemaining = context.services.config?.getQuotaRemaining();
                const pooledLimit = context.services.config?.getQuotaLimit();
                const pooledResetTime = context.services.config?.getQuotaResetTime();
                context.ui.addItem({
                    type: MessageType.MODEL_STATS,
                    selectedAuthType,
                    userEmail,
                    tier,
                    currentModel,
                    pooledRemaining,
                    pooledLimit,
                    pooledResetTime,
                });
            },
        },
        {
            name: 'tools',
            description: 'Show tool-specific usage statistics',
            kind: CommandKind.BUILT_IN,
            autoExecute: true,
            action: (context) => {
                context.ui.addItem({
                    type: MessageType.TOOL_STATS,
                });
            },
        },
    ],
};
//# sourceMappingURL=statsCommand.js.map