/*+
                           t d F  P a r s e

 *  Module name:
      tdFparse

 *  Function:
      Handles input of the ascii input target list for configure (FPOSS).

 *  Description:
      This module contains utility routines used by the AAO fibre configuration
      program, 'configure' (known at ESO as FPOSS). The 'tdF' prefix comes 
      from the original use of the program for the AAO's 2dF (two degree
      field) system, but it has since been extended to support the AAO's
      6dF (six degree field) system and the FLAMES fibre positioner used by
      ESO.
      
      This module handles the reading of the user-supplied input ascii file
      (usually a .fld file). They parse the field data and target list in
      this file and create C data structures containing the necessary
      information for the main configuration code.
      
      There are only three external routines in this module:
      
      void tdFparseFieldData (const FpilType *instrument,
          FILE *asciiFile, fieldType *fieldData, double uttime,
          short *fracDayFlag, int *lineCount, StatusType *status)
          
      Is passed an already open file (usually a '.fld' file) and reads
      the field information at the start of it, returning this field
      information in the fieldData structure.  It leaves the file positioned
      ready for:
      
      void tdFparseObjectData (const FpilType *instrument, FILE *asciiFile,
               tdFfieldType *field, short *options, float maglimits[],
               int *lineCount, StatusType *status)
               
      to be called. This continues the processing, reading the target list
      from the file and returning details of all the targets in the 
      structure given by the 'field' argument.
      
      The remaining routine:
      
      int tdFparseObjectCount (const FpilType *instrument, FILE *asciiFile,
                float maglimits[], StatusType *status)
                
      is intended to be called as a preliminary step to check that the file
      does not contain more targets in the specified magnitude range than
      the program can handle. It reads through the already open file and
      returns the number of targets within the specified magnitude range. It
      then rewinds the file. It should be called directly after a call to
      tdFparseFieldData(). 
      
      Note that this means that the usual sequence is:
      1) Call tdFparseFieldData() to position past the field data.
      2) Call tdFparseObjectCount() to check on the number of targets.
      3) Call tdFparseFieldData() to read the field data.
      4) Call tdFparseObjectData() to read the target object data.

 *  Language:
      C

 *  Authors: James Wilcox, AAO (JW)
             Jeremy Bailey, AAO (JAB)
             Keith Shortridge, AAO (KS).
 *-

 *  History:
      05-Nov-93  JW  Original version
      15-Dec-93 JAB  Handle conversion of positions from input equinox
      16-Dec-93 JAB  Handle conversion of positions to xy
      04-Jul-00  KS  Add support for type 'G' objects - VLT guide stars.
                     Added casts to int in islower() calls to placate compiler.
      10-Jul-00  KS  Support decision that EQUINOX should not be specified for
                     FLAMES, which always uses J2000. Allow FLAMES type codes.
                     Test for single character object types added.
      18-Jul-00  KS  Modify to use Fpil routines for object and pivot type.
      29-Nov-00  KS  Add magnitude limits to tdfParseObjectData().
      21-May-01  KS  Allow a fractional part for the UTDATE specification.
      10-Jun-01  KS  Added tdFparseObjectCount().
      20-Aug-01  KS  Minor changes to get a clean compilation from gcc -Wall.
      29-Aug-01  KS  Now allows '#' as well as '*' as a comment character. Made
                     all internal routines static and moved prototypes from
                     tdFconvert.h. Removed tdFparseFileName() as it's no longer
                     used. Comments extended.
      30-Oct-01  KS  In tdFparseObjectData(), corrected explicit test against
                     'S' for sky. Magnitude filter no longer affects sky
                     targets.
      20-Nov-02  KS  For FLAMES, UTDATE is allowed to default to current date.
      15-NOV-02  AK  In tdFparseObjectData() make cols 1-9 mandatory, cols >10 optional
                     (cols 1-11 were mandatory)
      {@change entry@}

 *  Sccs Id: tdFparse.c, Release 1.9, 11/21/02 (mm/dd/yy)
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <time.h>
#include "status.h"
#include "slalib.h"
#include "Ers.h"
#include "sds.h"
#include "tdFconvert.h"
#include "conf_err.h"
#include "tcl.h"

/*  DEBUG is used just to make diagnostic lines easy to find and remove */

#ifdef DEBUG
#undef DEBUG
#endif
#define DEBUG printf

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif

/*  Forward declarations of internal routines used in this module. */

static void tdFtoJ2000(double *ra, double *dec, char *sys, double eqnx);

static void ShowPos(double a, double b);

static void strToUpper (char buff[]);

/*  Some systems (Linux) don't seem to have a proper prototype for index(),
 *  so we provide one.
 */
 
char *index(const char *s, int c);

/*  Globals used to handle coordinate conversion. These are set once the
 *  equinox is determined from the header section of the input file (in
 *  tdFparseFieldData()) and then used by the coordinate conversion routines.
 */

static double geqnx;   /* Equinox of input star positions  */
static char gsys[2];   /* System of input star positions (B = FK4, J = FK5) */
static int gflag = 0;  /* set to 1 if positions need to be converted to J2000 */
static double gmap[21];/* Mean to apparent parameters  */

/*  ------------------------------------------------------------------------- */

/*                    s t r  T o  U p p e r
 *
 *  strToUpper is a simple utility routine that folds a string into upper case.
 */
 
static void strToUpper (
    char buff[])
{
    int i = 0;

    /*
     *  If character is lower case, convert to upper case.  (Without the islower
     *  check, we run into problems on some computers).
     */
    while (buff[i] != '\0') {
        if (islower((int)buff[i]))
            buff[i] = toupper(buff[i]);
        i++;
    }
}

/*  ------------------------------------------------------------------------- */

/*                      T r i m  S t r i n g
 *
 *   TrimString is a simple utility routine that inserts a terminating nul
 *   after the last non-blank character in a string. This trims it for use
 *   in output error messages, for example.
 */

static void TrimString (char* StringPtr)
 
{
   char* LastNonBlank = 0;
   for (;;) {
      char theChar = *StringPtr;
      if (theChar == '\0') {
         if (LastNonBlank) *(LastNonBlank + 1) = '\0';
         break;
      }
      if (!isprint((int)theChar)) {
         theChar = ' ';
         *StringPtr = ' ';
      }
      if (theChar != ' ') LastNonBlank = StringPtr;
      StringPtr++;
   }
}

/*  ------------------------------------------------------------------------- */

/*                t d F  p a r s e  F i e l d  D a t a
 *
 *  Description:
 *
 *    This is passed an already open file (usually a '.fld' file) and reads
 *    the field information at the start of it, returning this field
 *    information in the fieldData structure.  It leaves the file positioned
 *    just before the first line it was unable to parse, on the assumption
 *    that this is the first line containing target data.
 *
 *  Parameters:
 *
 *    instrument   (FpilType *)  Pointer to FPIL structure descibing the
 *                               instrument in use.
 *    asciiFile    (FILE *)      The input text file, already opened.
 *    fieldData    (fieldType *) Structure to receive the field details.
 *    uttime       (double)      A value to be added to the value given by
 *                               the UTDATE value in the field data.
 *    fracDayFlag  (short *)     Set to zero if the UTDATE value in the
 *                               field data did not specify a fractional part
 *                               and to 1 if it did. (Specifying a fractional
 *                               part has the effect of specifying a time of
 *                               day as well as the date.)
 *    lineCount    (int*)        Returned with the number of lines read in
 *                               the file - the intention is that this can be
 *                               passed on to tdFparseObjectData() so that
 *                               it can generate error messages that contain
 *                               a proper line count.
 *    status       (StatusType*) Inherited status value.
 *
 *  Note:
 *     The value given in uttime is added to the UTDATE value in the field
 *     data if that data does not explicitly specify a fractional part. If 
 *     there is a '.' in the Utdate given in the field data then that 
 *     fractional part is used. If not, the value of the uttime argument 
 *     passed to this routine is added to the calculated uttime. (We expect 
 *     this will have been chosen to give an approximation to the local 
 *     midnight for the observatory, so about 0.5 for the AAO, -0.2 for 
 *     the VLT.).
 *  
 */
 
void tdFparseFieldData (
    const FpilType *instrument,
    FILE           *asciiFile,
    fieldType      *fieldData,
    double          uttime,
    short          *fracDayFlag,
    int            *lineCount,
    StatusType     *status)
{
    char   dataline[LINE_LENGTH+2];
    char   dataName[NAME_LENGTH];
    short  lineNo = 0;
    int    linesRead = 0;
    short  labelFound = NO,
           utdateFound = NO,
           equinoxFound = NO,
           centreFound = NO;
    int    jstat;
    short  usingFlames;
    short  lineRecognised;

    if (*status != STATUS__OK) return;

    /*
     *  Initialise certain values.
     */
    fieldData->unallocObj = fieldData->unallocGui = fieldData->unallocSky = 0;
    sprintf(fieldData->mode,"NORMAL");
    sprintf(fieldData->progId," ");
    fieldData->posangle = 0.0;
    fieldData->argusSpec = NO;
    strcpy(fieldData->scale,"1:1");
    
    *fracDayFlag = FALSE;
    
    /*
     *  FLAMES doesn't allow EQUINOX to be specified, insisting on J2000
     *  in all cases.  Really, we should abstract this into an Fpil routine.
     */
    if (strcmp(FpilGetInstName(*instrument),"FLAMES")) {
        usingFlames = NO;
    } else {
        geqnx = 2000.0;
        equinoxFound = YES;
        usingFlames = YES;
    }

    /*
     *  Continue reading lines until we reach a line that we don't
     *  recognise. We assume this is the start of the actual target list.
     */
    while (1) {
        lineRecognised = NO;
        if (fgets(dataline,LINE_LENGTH,asciiFile) == NULL) {
            *status = CONF__FIELD;
            ErsRep(ERS_M_BELL,status,"Error reading field details");
            goto Exit;
        }
        else {
            linesRead++;
            /*
             *  Ignore comment lines.
             */
            if ((dataline[0] == '*') || (dataline[0] == '#')) {
                lineRecognised = YES;
                continue;
            }
            /*
             *  Parse non-comment line looking for field details.
             */
            else {
                TrimString(dataline);
                lineNo++;
                sscanf(dataline," %s ",dataName);
                strToUpper(dataName);

                /*
                 *  LABEL found.
                 */
                 
                if (strcmp(dataName,"LABEL") == 0) {
                    sscanf(dataline," %*s %[^\n] ",fieldData->label);
                    labelFound = YES;
                    lineRecognised = YES;
                }

                /*
                 *  UTDATE found. Note that we allow a fractional part for
                 *  the actual day field. If the day field has a fractional
                 *  part specified - if there is a '.' in the field - then
                 *  the fractional part is used. If not, the value of the
                 *  uttime argument passed to this routine is added to
                 *  the calculated uttime. (We expect this will have been 
                 *  chosen to give the local midnight for the observatory.)
                 *  Note also that although appEpoch is calculated, this is
                 *  no longer used by the configure program.
                 */
                 
                else if (strcmp(dataName,"UTDATE") == 0) {
                    int uty,utm,utd;
                    char daySpec[128];
                    double day,fracDay;
                    sscanf(dataline," %*s %d %d %s ",&uty,&utm,daySpec);
                    if (index(daySpec,'.')) {
                       day = atof(daySpec);
                       utd = (int)day;
                       fracDay = day - ((float)utd);
                       *fracDayFlag = TRUE;
                    } else {
                       utd = atoi(daySpec);
                       fracDay = uttime;
                    }
                    slaCldj(uty,utm,utd,&(fieldData->configMjd),&jstat);
                    if (jstat != 0) {
                        *status = CONF__UTDATE;
                        ErsRep(ERS_M_BELL,status,
                           "Illegal UT date - %s",dataline);
                        goto Exit;
                    }
                    fieldData->configMjd = fieldData->configMjd + fracDay;
                    fieldData->appEpoch = fieldData->configMjd; 
                    utdateFound = YES;
                    lineRecognised = YES;
                }

                /*
                 *  EQUINOX found.
                 */
                 
                else if (strcmp(dataName,"EQUINOX") == 0) {
                    char  eqnx[LINE_LENGTH];
                    Tcl_Channel    stdoutChannel;

                    int ns = 1;
                    int j2;
                    sscanf(dataline," %*s %s ",eqnx);
                    slaDbjin(eqnx,&ns,&geqnx,&jstat,&j2);
                    if (jstat != 0) {
                        *status = CONF__EQUINOX;
                        ErsRep(ERS_M_BELL,status,
                           "Illegal Equinox - %s",dataline);
                        goto Exit;
                    }
                    slaKbj(j2,geqnx,gsys,&jstat);
                    if (jstat != 0) {
                        *status = CONF__EQUINOX;
                        ErsRep(ERS_M_BELL,status,
                           "Illegal Equinox - %s",dataline);
                        goto Exit;
                    }
                    if (usingFlames) {
                       if ((gsys[0] != 'J') || (geqnx != 2000.0)) {
                          *status = CONF__EQUINOX;
                          ErsRep (ERS_M_BELL,status,
                             "EQUINOX, if specified, must be J2000 for FLAMES");
                          goto Exit;
                        }
                    }

                    stdoutChannel = Tcl_GetStdChannel(TCL_STDOUT);
                    if (stdoutChannel)
                    {
                        if ((gsys[0] != 'J') || (geqnx != 2000.0)) {
                            char buff[100];
                            sprintf(buff,"Equinox is %c%f\n",gsys[0],geqnx);
                            Tcl_Write(stdoutChannel,buff,-1);
                            Tcl_Write(stdoutChannel,
                                    "Positions will be converted to J2000",-1);
                            gflag = 1;
                        } else {
                            gflag = 0;
                        }
                    }
                        
                    equinoxFound = YES;
                    lineRecognised = YES;
                }

                /*
                 *  CENTRE found.
                 */
                 
                else if (strcmp(dataName,"CENTRE") == 0) {
                    int raHH, raMM, decMM;
	                 long decDD;
                    double  raSS, decSS;
                    char degrees[20];
                    double temp;
                    int ns = 1;
                    sscanf(dataline," %*s %d %d %lf %s %d %lf ",
                           &raHH,&raMM,&raSS,degrees,&decMM,&decSS);
                    slaDtf2r(raHH,raMM,raSS,&(fieldData->cenRa),&jstat);
                    if ((jstat != 0) || (raHH > 23) || (raHH < 0)) {
                        *status = CONF__CENTRE; 
                        ErsRep(ERS_M_BELL,status,
                            "Illegal Centre Right Ascension - %s",dataline);
                        goto Exit;
                    }
                    strcat(degrees," ");
                    decDD = 0;
                    slaIntin(degrees,&ns,&decDD,&jstat);
                    if (decDD > 90 || decDD < -90) {
                        *status = CONF__CENTRE;
                        ErsRep(ERS_M_BELL,status,
                            "Illegal Centre Declination - %s",dataline);
                        goto Exit;
                    }
                    if (jstat == -1) {
                        slaDaf2r(-decDD,decMM,decSS,&temp,&jstat);
                        fieldData->cenDec = -temp;
                    }
                    else if (jstat == 0) {
                        slaDaf2r(decDD,decMM,decSS,
                                  &(fieldData->cenDec),&jstat);
                    } else {
                        *status = CONF__CENTRE;
                        ErsRep(ERS_M_BELL,status,
                            "Illegal Centre Declination - %s",dataline);
                        goto Exit;
                    }
                    if (jstat != 0) {
                        *status = CONF__CENTRE;
                        ErsRep(ERS_M_BELL,status,
                            "Illegal Centre Declination - %s",dataline);
                        goto Exit;
                    }
                       
                    centreFound = YES;
                    lineRecognised = YES;
                }
                
                /*
                 *  ARGUS found. This is followed by the ARGUS position angle,
                 *  and the scale. Note that we specify the angle in degrees.
                 *  The scale must be 1:1 or 1:1.67.
                 */
                 
                else if (strcmp(dataName,"ARGUS") == 0) {
                    double Degrees;
                    char Scale[80];
                    if (!usingFlames) {
                        *status = CONF__FIELD;
                        ErsRep(ERS_M_BELL,status,"%s %s",
                             "An ARGUS specification is only valid for a",
                              "FLAMES configuration");
                        goto Exit;
                    }
                    sscanf(dataline," %*s %lf %s",&Degrees,Scale);
                    fieldData->argusSpec = YES;
                    lineRecognised = YES;
                    fieldData->posangle = Degrees;
                    if (!strcmp(Scale,"1:1.67") || !strcmp(Scale,"1:1")) {
                        strcpy (fieldData->scale,Scale);
                    } else {
                        *status = CONF__FIELD;
                        ErsRep(ERS_M_BELL,status,
                          "%s is invalid. Scale must be '1:1.67' or '1:1'",
                             Scale);
                        goto Exit;
                    }
                }

                /*  If we haven't recognised the line, we assume this is the
                 *  start of the actual field target data. We reposition
                 *  the file at the start of the current line, and return.
                 *  This means that most syntactic errors will be picked up
                 *  in tdFparseObjectData.
                 */
                 
                if (!lineRecognised) {
                    int line;
                    rewind (asciiFile);
                    for (line = 0; line < linesRead - 1; line++) {
                       if (fgets(dataline,LINE_LENGTH,asciiFile) == NULL) {
                          *status = CONF__FIELD;
                          ErsRep(ERS_M_BELL,status,
                                        "Error skipping field details");
                          goto Exit;
                       }
                    }
                    *lineCount = linesRead - 1;
                    goto Exit;
                }
            }
        }
    }
    
Exit:;

    /*
     *  On the way out, check if error has occured or all field details
     *  were found.
     */
    if (*status != 0) {
        ErsRep(ERS_M_BELL,status,"Error was in line %d",linesRead);
        *lineCount = linesRead;
    }
    else {
    
        /*  FLAMES allows UTDATE to be treated as optional. If it wasn't
         *  specified, use today's date.
         */
        
       if (usingFlames && !utdateFound) {
           int uty,utm,utd;
           struct tm* tmPtr;
           time_t theTime = time(0);
           tmPtr = localtime(&theTime);
           if (tmPtr) {
               uty = tmPtr->tm_year + 1900;
               utm = tmPtr->tm_mon + 1;
               utd = tmPtr->tm_mday;
               slaCldj(uty,utm,utd,&(fieldData->configMjd),&jstat);
               if (jstat == 0) {
  	             fieldData->appEpoch = fieldData->configMjd; 
		     utdateFound = YES;
	       }
           }
       }
               
       if (labelFound && utdateFound && equinoxFound && centreFound) {

           /*
            *   Convert to J2000 if required
            */

           if (gflag) {
               tdFtoJ2000(&(fieldData->cenRa),&(fieldData->cenDec),
                   gsys,geqnx);
           }

           /*
            *   Convert to apparent
            */

           slaMappa(2000.0,fieldData->appEpoch,gmap);
           slaMapqkz(fieldData->cenRa,fieldData->cenDec,
                    gmap,&(fieldData->appRa),&(fieldData->appDec));
       } else {
           *status = CONF__FIELD;
           ErsRep (ERS_M_BELL,status,"Not all field details supplied");
       }
    }
    
    /*  This call is never made, but prevents compilers complaining that
     *  the ShowPos() diagnostic routine is static and unreferenced.
     */
   
    if (0) ShowPos (fieldData->cenRa,fieldData->cenDec);
}

/*  ------------------------------------------------------------------------- */

/*                t d F  p a r s e  O b j e c t  D a t a
 *
 *  Description:
 *    This is passed an already open file (usually a '.fld' file),
 *    that has just had the field details read by tdFparseFieldData().
 *    It reads the details of all the objects in the file within the
 *    specified magnitude range and returns their details in the structure
 *    passed as 'field'.
 *
 *  Parameters:
 *
 *    instrument   (FpilType *)     Pointer to FPIL structure descibing the
 *                                  instrument in use.
 *    asciiFile    (FILE *)         The input text file, already opened.
 *    field        (tdFfieldType *) Structure to receive the field details.
 *    options      (short *)        Flags word in which the flags given by
 *                                  PRIORITY, MAGNITUDE, PROG_ID and COMMENT
 *                                  are set if any of the target lines includes
 *                                  the corresponding data. This should be
 *                                  passed as zero, and the return value can
 *                                  then be used to indicate which items have
 *                                  been specified.
 *    maglimits    (float[])        An array of two values. maglimits[0] gives
 *                                  the lower magnitude limit and maglimits[1]
 *                                  gives the upper magnitude limit. If these
 *                                  are the same, no magnitude filtering is
 *                                  performed. 'Lower' here means numerically 
 *                                  lower (ie objects are brighter).
 *    lineCount    (int*)           Passed as the number of lines already
 *                                  read in the file and updated by this 
 *                                  routine.
 *    status       (StatusType*)    Inherited status value.
 *
 *  Note: 
 *    The structure whose address is given by 'field' is really just the
 *    starting point for two linked lists of structures of type 
 *    objectType. One list is for guide targets, one for ordinary targets.
 *    Each objectType structure holds the details for one target. These
 *    are malloc'ed individually by this routine, and have to be free'd 
 *    once they are no longer required.
 */
 
void tdFparseObjectData (
    const FpilType *instrument,
    FILE           *asciiFile,
    tdFfieldType   *field,
    short          *options,
    float          maglimits[],
    int            *lineCount,
    StatusType     *status)
{
    objectType  *thisObject, *tmp;
    char        dataline[LINE_LENGTH+2];
    short       magfilter;

    if (*status != STATUS__OK) return;

    /*
     *  Initialise lists.
     */
    field->unallocGuide = field->unallocObject = NULL;
    field->fieldData.unallocGui = 0;
    field->fieldData.unallocSky = 0;
    field->fieldData.unallocObj = 0;
    magfilter = (maglimits[0] != maglimits[1]);
    
    /*
     *  Keep parsing untill end of file is reached.
     */
    while (fgets(dataline,LINE_LENGTH,asciiFile) != NULL) {
        char        weight[LINE_LENGTH] = "\0",
                    name[LINE_LENGTH] = "\0",
                    magnitude[LINE_LENGTH] = "\0",
                    pId[LINE_LENGTH] = "\0",
                    comment[LINE_LENGTH] = "\0";
        char        type[10];
        int         raHH, raMM,
                    decMM;
        long        decDD;
        double      raSS, decSS;
        double      temp;
        float       magnitude_value;
        int         jstat;
        char        degrees[20];
        int         ns;
        int         i,isblank;
        int         items;

        /*
         *  Ignore comment lines.
         */
        (*lineCount)++;
        if ((dataline[0] == '*') || (dataline[0] == '#'))
            continue;
            
        /* Check if completely blank */
        
        isblank = 1;
        for(i=0; i<LINE_LENGTH && dataline[i]!=0; i++) {
           if (!isspace( (int) dataline[i]) && dataline[i]!=0) 
             isblank = 0;
        }
                
        if (isblank) 
           continue;           
        /*
         *  Read data from next line from file.
         */
        TrimString(dataline);
        items = sscanf(dataline," %s %d %d %lf %s %d %lf %s %s %s %s %[^\n] ",
               name,
               &raHH,&raMM,&raSS,degrees,&decMM,&decSS,type,
               weight,magnitude,pId,comment);
        if (items < 9) {
	    char LineStart[32];
            *status = CONF__FIELD;
            strncpy (LineStart,dataline,sizeof(LineStart));
            LineStart[sizeof(LineStart) - 1] = '\0';
            TrimString(LineStart);
            ErsRep(ERS_M_BELL,status,"Syntax error in input file line");
            ErsRep(ERS_M_BELL,status,
             "Not all required fields provided, or some specified incorrectly");
            ErsRep(ERS_M_BELL,status, "Line starts '%s'",LineStart);
            goto Exit;
	}
        /*
         *  Filter on magnitude value if required. Note that we don't filter
         *  out guide objects or sky targets.
         */
        if (magfilter) {
            char tchar,schar;
            magnitude_value = 0.0;
            (void) FpilEncodeTargetType(*instrument,type,&tchar,&schar);
            if (!FpilIsTargetGuide(*instrument,tchar) &&
                   !FpilIsTargetSky(*instrument,tchar)) {
                if (magnitude[0] && magnitude[0] != '*') {
                    magnitude_value = atof(magnitude);
                    if ((magnitude_value < maglimits[0]) || 
                       (magnitude_value > maglimits[1])) continue;
                }
            }
        }

        /*
         *  Create new object structure.
         */
         
        if ((thisObject = (objectType *)malloc(sizeof(objectType))) == NULL) {
            *status = CONF__MALLOC;
            ErsRep(ERS_M_BELL,status,"malloc error while parsing file");
            goto Exit;
        }
        strcpy (thisObject->name,name);

        /*
         *  Convert ra and dec values to doubles and x,y values.
         */
        slaDtf2r(raHH,raMM,raSS,&(thisObject->ra),&jstat);
        if (jstat != 0) {
            free (thisObject);
            *status = CONF__ILLRA;
            ErsRep(ERS_M_BELL,status,"Illegal Right Ascension - %s",dataline);
            goto Exit;
        }
        
        /* Gracefully handle lousy observers' crap seconds (APM) */
        
        if (raSS == 60.0) 
           raSS = 59.9999999;
        
        if (decSS == 60.0) 
           decSS = 59.9999999;
           
        strcat(degrees," ");
        ns = 1;
        slaIntin(degrees,&ns,&decDD,&jstat);
        if (jstat == -1) {
            slaDaf2r(-decDD,decMM,decSS,&temp,&jstat);
            thisObject->dec = -temp;
        }
        else if (jstat == 0) {
            slaDaf2r(decDD,decMM,decSS,
                    &(thisObject->dec),&jstat);
        } else {
            free (thisObject);
            *status = CONF__ILLDEC;
            ErsRep(ERS_M_BELL,status,"Illegal Declination - %s",dataline);
            goto Exit;
        }
        if (jstat != 0) {
            free (thisObject);
            *status = CONF__ILLDEC;
            ErsRep(ERS_M_BELL,status,"Illegal Declination - %s",dataline);
            goto Exit;
        }

        /*
         *  Convert to J2000 if required
         */

        if (gflag) {
           tdFtoJ2000(&(thisObject->ra),&(thisObject->dec),gsys,geqnx);
        }
                    
        thisObject->x = 0;
        thisObject->y = 0;

        /*
         *  Assign type value. Allowed types depend on the instrument,
         *  so this is part of Fpil.
         */
         
        if (!FpilEncodeTargetType(*instrument,type,&(thisObject->type),
                                         &(thisObject->spectrograph))) {
            ErsRep(ERS_M_BELL,status,"Invalid object type - '%s' in '%s'",type,
                                                                  dataline);
            thisObject->type = *type;
            thisObject->spectrograph = 0;
            *status = CONF__INVTTYPE;
        }
        /*
         *  Assign priority value.
         */
        if (weight[0] && weight[0] != '*') 
            thisObject->priority = (short)atoi(weight);
        else
            thisObject->priority = 0;
        if (!(*options & PRIORITY))
	    *options += PRIORITY;

        /*
         *  Assign magnitude value.
         */
        if (magnitude[0] && magnitude[0] != '*') 
            thisObject->magnitude = atof(magnitude);
        else
            thisObject->magnitude = 0.0;
        if (!(*options & MAGNITUDE))
	    *options += MAGNITUDE;

        /*
         *  Assign program id value.
         */
        if (pId[0] && pId[0] != '*')
            thisObject->pId = atol(pId);
        else
            thisObject->pId = 0;
        if (!(*options & PROG_ID))
	    *options += PROG_ID;

        /*
         *  Record comment.
         */
        if (comment[0])
           sprintf(thisObject->comment,"%s",comment);
        else
           sprintf(thisObject->comment," ");
        if (!(*options & COMMENT))
           *options += COMMENT;

        /*
         *  Add object to appropriate list and update counters.
         */
        if (FpilIsTargetGuide(*instrument,thisObject->type)) {
            tmp = field->unallocGuide;
            if (tmp)  thisObject->next = tmp;
            else      thisObject->next = NULL;
            field->unallocGuide = thisObject;
            field->fieldData.unallocGui++;
        }
        else {
            tmp = field->unallocObject;
            if (tmp)  thisObject->next = tmp;
            else      thisObject->next = NULL;
            field->unallocObject = thisObject;
            if (FpilIsTargetSky(*instrument,thisObject->type)) 
                field->fieldData.unallocSky++;
            else
                field->fieldData.unallocObj++;
        }
    }
    
Exit:;

    if (*status) {
        ErsRep(ERS_M_BELL,status,"Error was in line %d",*lineCount);
    }
       
}

/*  ------------------------------------------------------------------------- */

/*
 *               t d F  p a r s e  O b j e c t  C o u n t
 *
 *  This is a cut-down version of tdFparseObjectData() which doesn't do
 *  anything with the data in the file other than count the number of
 *  objects that will be included, given the magnitude filter (if one
 *  was specified). This can be used as a fast pre-reading check to see
 *  if a magnitude filter is actually needed. The file is rewound after
 *  being read.
 *
 *  Call:
 *    Count = tdFparseObjectCount(instrument,asciiFile,maglimits,status);
 *
 *  Parameters:
 *
 *    instrument   (FpilType *)  Pointer to FPIL structure descibing the
 *                               instrument in use.
 *    asciiFile    (FILE *)      The input text file, already opened.
 *    maglimits    (float[])     An array of two values. maglimits[0] gives
 *                               the lower magnitude limit and maglimits[1]
 *                               gives the upper magnitude limit. If these
 *                               are the same, no magnitude filtering is
 *                               performed. 'Lower' here means numerically 
 *                               lower (ie objects are brighter).
 *    status       (StatusType*) Inherited status value.
 *
 *  Returns:
 *    Count        (int)         The number of targets in the file in the
 *                               specified magnitude range.
 *    
 */
  
int tdFparseObjectCount (
    const FpilType *instrument,
    FILE           *asciiFile,
    float          maglimits[],
    StatusType     *status)
{
    char        dataline[LINE_LENGTH+2];
    short       magfilter;
    int         ObjectCount;

    if (*status != STATUS__OK) return 0;

    ObjectCount = 0;
    magfilter = (maglimits[0] != maglimits[1]);
    
    /*
     *  Keep parsing untill end of file is reached.
     */
     
    while (fgets(dataline,LINE_LENGTH,asciiFile) != NULL) {
        char        weight[LINE_LENGTH] = "\0",
                    name[LINE_LENGTH] = "\0",
                    magnitude[LINE_LENGTH] = "\0",
                    pId[LINE_LENGTH] = "\0",
                    comment[LINE_LENGTH] = "\0";
        char        type[10];
        int         raHH, raMM,
                    decMM;
        double      raSS, decSS;
        float       magnitude_value;
        char        degrees[20];
        int         i,isblank;

        /*
         *  Ignore comment lines.
         */
        if (dataline[0] == '*')
            continue;
            
        /* Check if completely blank */
        
        isblank = 1;
        for(i=0; i<LINE_LENGTH && dataline[i]!=0; i++) {
           if (!isspace( (int) dataline[i]) && dataline[i]!=0) 
             isblank = 0;
        }
                
        if (isblank) 
           continue;
           
        /*
         *  Read data from next line from file.
         */
        TrimString(dataline);
        sscanf(dataline," %s %d %d %lf %s %d %lf %s %s %s %s %[^\n] ",
               name,
               &raHH,&raMM,&raSS,degrees,&decMM,&decSS,type,
               weight,magnitude,pId,comment);
               
        /*
         *  Filter on magnitude value if required. Note that we don't filter
         *  guide objects out.
         */
        if (magfilter) {
            char tchar,schar;
            magnitude_value = 0.0;
            (void) FpilEncodeTargetType(*instrument,type,&tchar,&schar);
            if (!FpilIsTargetGuide(*instrument,tchar)) {
                if (magnitude[0] && magnitude[0] != '*') {
                    magnitude_value = atof(magnitude);
                    if ((magnitude_value < maglimits[0]) || 
                       (magnitude_value > maglimits[1])) continue;
                }
            }
        }
        ObjectCount++;
    }
    
    rewind (asciiFile);
    
    return ObjectCount;
}

/*  ------------------------------------------------------------------------- */

/*                       t d F  t o  J 2 0 0 0
 *
 *   Converts an RA and Dec to J2000 FK5 from any equinox in either 
 *   FK4 or FK5 system.
 *
 *    ra   (double *)     The Right Ascension
 *    dec  (double *)     The declination
 *    sys  (char *)       The system (B for FK4, J for FK5)
 *    eqnx (double)       The equinox of the original coordinates
 *                        expressed as a Besselian epoch for the FK4 
 *                        system or a Julian epoch for the FK5 system.
 */
 
static void tdFtoJ2000(double *ra, double *dec, char *sys, double eqnx)


{
   double tra, tdec, tra2, tdec2;

   if (sys[0] == 'J') {

/*
 *   FK5 case - simply precess from original epoch to J2000.0
 */
      slaPreces("FK5",eqnx,2000.0,ra,dec);
   } else {

/*
 *   FK4 case - precess to B1950.0, then convert to J2000 FK5
 */
      slaSubet(*ra,*dec,eqnx,&tra,&tdec);
      slaPreces("FK4",eqnx,1950.0,&tra,&tdec);
      slaAddet(tra,tdec,1950.0,&tra2,&tdec2);
      slaFk45z(tra2,tdec2,1950.0,ra,dec);
   }
}

/*  ------------------------------------------------------------------------- */

/*                         S h o w  P o s
 *
 *  ShowPos() is a diagnostic that formats an RA,Dec pair and prints them out.
 */
 
static void ShowPos(double ra, double dec)

{
   char sign;
   int ihmsf[4];

   slaDr2tf(3,ra,&sign,ihmsf);
   printf("%d %d %d.%d ",ihmsf[0],ihmsf[1],ihmsf[2],ihmsf[3]);
   slaDr2af(2,dec,&sign,ihmsf);
   printf("%c%d %d %d.%d\n",(char)sign,ihmsf[0],ihmsf[1],ihmsf[2],ihmsf[3]);
}
