/*
 *                      E r s _ i m p
 *
 *  Description:
 *     This is a very simplified standalone version of the Ers
 *     library. It implements the various functions at least to
 *     the point that they can be used in a program that uses
 *     IMP without needing the whole of the DRAMA system. For
 *     serious use it is probably better to use the proper DRAMA
 *     version of Ers, but this will do if it is only needed
 *     from within IMP.
 *
 *  Limitations:
 *     o  Code is relatively crude.
 *     o  The flags options in ErsOut and ErsRep are not supported.
 *     o  Commenting is minimal.
 *
 *  Compiling:
 *     Two pre-processor variables are used and need to be set properly:
 *     -  VxWorks  needs to be defined if the code has to run under VxWorks
 *     -  VMS  needs to be defined if the code has to run under VMS.
 *     No particular definitions are needed for any of the varieties
 *     of UNIX supported by IMP, although note that:
 *     -  The SunOS standard C compiler cannot handle this code. You
 *        have to use gcc instead.
 *
 *  Authors:
 *     Allan Brighton, ESO
 *     Keith Shortridge, AAO.
 *
 *  History:
 *     23rd Apr 1997. Original version, AB/ESO.
 *      6th Jun 1997. Modified to support context levels properly,
 *                    and to run under VMS and VxWorks as well as
 *                    under UNIX. KS/AAO.
 *     12th Jun 1997. Added ErsClear() call to ErsStop() in the non-
 *                    VxWorks case. KS/AAO.
 *     19th Aug 1997. Added current pid to output format - temporarily. KS/AAO.
 *      1st Sep 1997. Reversed above change. KS/AAO.
 *     20th Oct 1997. ErsAnnul now zeros the status argument, as it ought to
 *                    do. KS/AAO.
 *     29th May 2001. Added support for redirection of output - the output
 *                    routine argument of ErsStart() is now supported. Also
 *                    modified to work properly in cases where ErsOut() or
 *                    ErsRep() are called before ErsStart(). This version
 *                    needs testing under VxWorks. KS/AAO.
 *      6th Jun 2001. Modified status type to IMPZ_INT4 so definition of
 *                    ErsRep() and ErsOut() match that in Ers.h (which wasn't
 *                    being used until now). Fixed bug in ErsAnnul which
 *                    left the count set incorrectly if all messages were
 *                    annulled. KS/AAO.
 *     16th Jul 2001. Modified forward declaration of ErsRep() and ErsOut()
 *                    to match that in Ers.h. (The Sun cc compiler picked 
 *                    this, together with a missing cast.) KS/AAO.
 *     13th Aug 2001. Fixed bug in ErsFlush() code that sometimes confused the
 *                    message entry to use, generating random error text, Also
 *                    added Ers___GlobalsInit(). KS/AAO.
 *     16th Aug 2001. Allowed for DEBUG already being defined. KS/AAO.
 */

#ifdef DEBUG
#undef DEBUG
#endif
#define DEBUG printf

#include "Ers.h"

#ifdef VMS
#include <varargs.h>
#else
#include <stdarg.h>
#ifndef macintosh
#include <sys/types.h>
#endif
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#ifdef VxWorks
#include <taskVarLib.h>
#endif

/* Global structure holding error messages */

#define MAX_MESSAGES 1024

typedef struct {
   char *address;
   int context;
} message_struct;

typedef struct {
   message_struct messages[MAX_MESSAGES];
   int current_context;
   int count;
   ErsOutRoutineType outRoutine;
   void* outArg;
} ers_global_struct;

/*  Note that under VxWorks, global_init_ is always set true, but globals_
 *  is initially set to a null address and only set to a non-null address
 *  when ErsStart() is called successfully.
 *
 *  Conversely, under other systems, globals_ is always set to a non-null
 *  address, but global_init_ is initially set false and only set to true
 *  after ErsStart() has been called successfully.
 *
 *  This means that on all systems, testing for BOTH globals_ and global_init_
 *  being non-zero is a test for the system having been initialised 
 *  successfully.
 */
 
#ifdef VxWorks

static ers_global_struct *globals_ = 0;

static int global_init_ = 1;

#else

static ers_global_struct global_values_;

static ers_global_struct *globals_ = &global_values_;

static int global_init_ = 0;

#endif

/*  Start of actual code */

/*  Strictly speaking, Ers___GlobalsInit() is unnecessary, since the code
 *  makes sure that uninitialised message entries are never used. However,
 *  it seems a lot cleaner to ensure that all entries are set to sensible
 *  values on initialisation.
 */
 
void Ers___GlobalsInit (void)
{
   int i;
   if (globals_) {
      for (i = 0; i < MAX_MESSAGES; i++) {
         globals_->messages[i].address = NULL;
         globals_->messages[i].context = 0;
      }
   }
}

/*  On most systems ErsStart does very little other than initialise
 *  the global variables. On VxWorks, however, we need to make sure
 *  each task has its private version of these. It is only the need
 *  to allow VxWorks to save only one task variable that is responsible
 *  for the rather complicated scheme of having globals_ work
 *  the way it does.
 */

#ifdef VxWorks

void ErsStart (void *outRoutine, void *outArg, void *logRoutine,
      void *logArg, long *status)                  /* VxWorks version */
{
   globals_ = (ers_global_struct *) malloc (sizeof(ers_global_struct));

   taskVarAdd (0,(int *) &globals_);

   globals_->current_context = 0;
   globals_->count = 0;
   globals_->outRoutine = (ErsOutRoutineType) outRoutine;
   globals_->outArg = outArg;
   
   Ers___GlobalsInit();
   
}

void ErsStop (long *status)                        /* VxWorks version */
{
   ErsClear (status);
   if (globals_) free(globals_);
   taskVarDelete (0,(int *) &globals_);
}

#else

void ErsStart (void *outRoutine, void *outArg, void *logRoutine,
      void *logArg, long *status)
{
   globals_->current_context = 0;
   globals_->count = 0;
   globals_->outRoutine = (ErsOutRoutineType) outRoutine;
   globals_->outArg = outArg;
   global_init_ = 1;
   
   Ers___GlobalsInit();
}

void ErsStop (long *status)
{
   ErsClear (status);
   global_init_ = 0;
}

#endif

/*  There follow different versions of ErsRep() and ErsOut() for VMS.
 *  All the other systems on which IMP runs support the stdarg.h 
 *  style of using variable argument lists. VMS is an odd one out.
 *  Also, VMS does not support the gmtime() function, so we omit that.
 */

#ifdef VMS

void ErsRep(va_alist)              /* VMS version */
{
    char buf[1024];
    int size;
    char *fmt; 
    va_list ap;

    va_start(ap);
    va_arg (ap,int);
    va_arg (ap,long *);
    fmt = va_arg(ap,char *);
    vsprintf(buf, fmt, ap);

    if (globals_ && global_init_) {
       if (globals_->count < MAX_MESSAGES) {
          size = strlen(buf) + 1;
          if ((globals_->messages[globals_->count].address = 
                                             (char *) malloc(size))) {
             strcpy (globals_->messages[globals_->count].address,buf);
             globals_->messages[globals_->count].context = 
                                                  globals_->current_context;
             globals_->count++;
          }
       }
    } else {
       fprintf(stderr, "[ERROR] %s\n",buf);
    }
    va_end(ap);
}

void ErsOut(va_alist)              /* VMS version */
{
    char buf[1024];
    char *fmt;
    long status = 0;
    ErsMessageType ErsMessage;
    short printIt;

    va_list ap;
    va_start (ap);
    va_arg (ap,int);
    va_arg (ap,long *);
    fmt = va_arg(ap,char *);
    vsprintf(buf, fmt, ap);
    va_end(ap);

    if (strlen(buf)) {
       printIt = 1;
       if (globals_ && global_init_) {
          if (globals_->outRoutine) {
             ErsMessage.mesStatus = ErsMessage.context = ErsMessage.flags = 0;
             strncpy (ErsMessage.message,buf,ERS_C_LEN);
             ErsMessage.message[ERS_C_LEN - 1] = '\0';
             (globals_->outRoutine)(globals->outArg,1,&ErsMessage,&status);
             printIt = 0;
          }
       } 
       if (printIt) fprintf(stderr, "[ERROR] %s\n",buf);
    }
}

#else

#ifdef __GNUC__
void ErsRep (int flags, IMPZ_INT4 *status, const char *fmt, ...)
#else
void ErsRep(int flags, IMPZ_INT4* status, char* fmt, ...)
#endif
{
    char buf[1024];
    int size;
    va_list ap;

    va_start(ap, fmt);
    vsprintf(buf, fmt, ap);
    if (globals_ && global_init_) {
       if (globals_->count < MAX_MESSAGES) {
          size = strlen(buf) + 1;
          if ((globals_->messages[globals_->count].address = 
                                                  (char *) malloc(size))) {
             strcpy (globals_->messages[globals_->count].address,buf);
             globals_->messages[globals_->count].context = 
                                               globals_->current_context;
             globals_->count++;
          }
       }
    } else {
       fprintf(stderr, "[ERROR] %s\n", buf);
    }
    va_end(ap);
}

#ifdef __GNUC__
void ErsOut (int flags, IMPZ_INT4 *status, const char *fmt, ...)
#else
void ErsOut(int flags, IMPZ_INT4* status, char* fmt, ...)
#endif   
{
    char isotime[256];
    time_t t = time(0);
    struct tm* tm = gmtime(&t);
    ErsMessageType ErsMessage;
    char buf[1024];
    short printIt;

    va_list ap;
    va_start(ap, fmt);
    vsprintf(buf, fmt, ap);
    va_end(ap);

    if (strlen(buf)) {
       printIt = 1;
       if (globals_ && global_init_) {
          if (globals_->outRoutine) {
             ErsMessage.mesStatus = ErsMessage.context = ErsMessage.flags = 0;
             strncpy (ErsMessage.message,buf,ERS_C_LEN);
             ErsMessage.message[ERS_C_LEN - 1] = '\0';
             (globals_->outRoutine)(globals_->outArg,1,&ErsMessage,status);
             printIt = 0;
          }
       }
       if (printIt) {
          if (tm) {
             strftime(isotime, sizeof(isotime)-1, "%T", tm);
             fprintf(stderr, "%s [ERROR] %s\n", isotime, buf);
          } else {
             fprintf(stderr, "[ERROR] %s\n", buf);
          }
       }
    }
}

#endif

void ErsFlush(long* status)
{
    short allOutput;
    int i;
    int flags = 0;
    int max_count;
    int msgCount;
    IMPZ_INT4 ignore;
    ErsMessageType* ErsMessages;

    *status = 0;
    
    if ((globals_ == 0) || (global_init_ == 0)) return;

    max_count = globals_->count - 1;
    if (globals_->count) {
       allOutput = 0;
       if (globals_->outRoutine) {
       
          /*  This section gets invoked if an output routine was specified in
           *  the ErsStart() call. We go through all the messages we have and
           *  for any that are at the current context level or greater we
           *  add them to an array of ErsMessageType structures that we
           *  eventually pass to the output routine.
           */
           
          ErsMessages = malloc (globals_->count * sizeof(ErsMessageType));
          if (ErsMessages) {
             msgCount = 0;
             for (i=0; i<globals_->count; i++) {
                if (globals_->messages[i].address) {
                   if (globals_->messages[i].context >= 
                                             globals_->current_context) {
                      if (strlen(globals_->messages[i].address)) {
                         ErsMessages[msgCount].mesStatus = 0;
                         ErsMessages[msgCount].context = 0;
                         ErsMessages[msgCount].flags = 0;
                         strncpy (ErsMessages[msgCount].message,
                                    globals_->messages[i].address,ERS_C_LEN);
                         ErsMessages[msgCount].message[ERS_C_LEN - 1] = '\0';
                         msgCount++;
                      }
                      free(globals_->messages[i].address);
                      globals_->messages[i].address = NULL;
                   } else {
                      max_count = i;
                   }
                }
	          }
             ignore = 0;
             if (msgCount > 0) {
                (*(globals_->outRoutine))(globals_->outArg,
                                             msgCount,ErsMessages,&ignore);
             }
             allOutput = 1;
             globals_->count = max_count + 1;
             free(ErsMessages);
          }
       }
       if (!allOutput) {
       
          /*  This section gets invoked if no output routine was specified in
           *  the ErsStart() call. We go through all the messages we have and
           *  call ErsOut() for any that are at the current context level
           *  or greater.
           */
           
          for (i=0; i<globals_->count; i++) {
             if (globals_->messages[i].address) {
                if (globals_->messages[i].context >= 
                                            globals_->current_context) {
                   if (strlen(globals_->messages[i].address)) {
                      ignore = 0;
                      ErsOut(flags, &ignore, globals_->messages[i].address);
                   }
                   free(globals_->messages[i].address);
                   globals_->messages[i].address = NULL;
                } else {
                   max_count = i;
                }
             }
          }
          globals_->count = max_count + 1;
	    }
   }
   *status = 0;
}

void ErsAnnul(long* status)
{
    int i;
    int max_count = 0;

    *status = 0;
    
    if ((globals_ == 0) || (global_init_ == 0)) return;

    if (globals_->count) {
       for(i=0; i<globals_->count; i++) {
          if (globals_->messages[i].address) {
             if (globals_->messages[i].context >= globals_->current_context) {
                free(globals_->messages[i].address);
                globals_->messages[i].address = NULL;
             } else {
                max_count = i + 1;
             }
          }
	    }
	    globals_->count = max_count;
    }
}

void ErsClear(long* status)
{
   *status = 0;
   if ((globals_ == 0) || (global_init_ == 0)) return;

   globals_->current_context = 0;
   ErsFlush(status);
}

void ErsPush()
{
   if ((globals_ == 0) || (global_init_ == 0)) return;

   globals_->current_context++;
}

void ErsPop(long* status)
{
    int i;

    if ((globals_ == 0) || (global_init_ == 0)) return;

    if (globals_->current_context > 0) globals_->current_context--;
    if (globals_->count) {
       for(i=0; i<globals_->count; i++) {
          if (globals_->messages[i].address) {
             if (globals_->messages[i].context > globals_->current_context) {
                globals_->messages[i].context = globals_->current_context;
             }
          }
	    }
    }
}
