@ModuleInfo { minPklVersion = "0.27.2" }
module hk.Config
min_hk_version = "{{version | truncate(length=1)}}.0.0"

class Script {
    linux: String?
    macos: String?
    windows: String?
    other: String?
}

class Step {
    _type = "step"
    /// Which profiles (HK_PROFILES) need to be active for the step to run
    profiles: List<String>?
    
    /// Which file patterns to run the step on
    glob: (String | List<String>)?

    /// Which file patterns to exclude from the step
    exclude: (String | List<String>)?

    /// files to stage after running the fix step
    stage: (String | List<String>)?
    
    /// If true, the step will run by itself, waiting for previous steps to finish before starting and making later steps wait for it to finish
    exclusive = false

    /// Connects stdin/stdout/stderr to the hk's execution. Implies `exclusive`.
    interactive = false
    
    /// Wait for sibling steps to finish before starting this step.
    depends: (String | List<String>) = List()

    /// Which shell to use for running commands
    shell: (String | Script)?
    
    /// a shell command to check. Any edits will be ignored.
    check: (String | Script)?
    check_list_files: (String | Script)?
    check_diff: (String | Script)?

    /// a shell command that checks and edits files. Edits will be automatically added to the index.
    fix: (String | Script)?

    /// if set, run the linter on workspaces only which are parent directories containing this filename
    workspace_indicator: String?

    /// if set, run the linter scripts with this prefix, e.g.: "mise exec --" or "npm run"
    prefix: String?

    /// if set, run the linter scripts in this directory
    dir: String?

    /// if set, run the step only if this condition is true
    condition: String?

    /// If true, when intending to run the fix command for this step, if any other steps in the same group need to read to the same files, this step will run the check command instead
    /// if it fails, the fix command will be run after
    /// Note that this behavior is only used if multiple steps would be in contention. If all fix steps edit disjoint files, the fix step will be chosen first despite this config.
    check_first = true

    /// Run check/fix step on batches of files instead of all files at once.
    /// This takes advantage of parallel processing for otherwise single-threaded linters like eslint and prettier.
    batch = false

    /// fetches a read lock instead of a write lock when running fix/fix_all. Use if the tool has its own locking mechanism or you simply don't care if files may be written to
    /// by multiple steps.
    stomp = false

    hide = false
    
    /// run the linter scripts with these environment variables
    env = new Mapping<String, String>{}
}

typealias StashMethod = Boolean | "git" | "patch-file" | "none"

class Group {
    _type = "group"
    steps: Mapping<String, Step> = new Mapping<String, Step>{}
}

class Hook {
    fix: Boolean?
    stash: StashMethod?
    steps: Mapping<String, Step | Group> = new Mapping<String, Step>{}
}

hooks: Mapping<String, Hook> = new Mapping<String, Hook>{}
env: Mapping<String, String> = new Mapping<String, String>{}

output {
  renderer {
    converters {
      [Step] = (s) -> new Step {
            ...s
            .toMap()
            .mapValues((k, v) ->
                if ((k == "glob" || k == "exclude" || k == "depends" || k == "stage") && v is String)
                    List(v) // permits "s" instead of List("s")
                else if (k == "stash" && v is Boolean)
                    if (v) "git" else "none"
                else
                    v
            )
            .toDynamic()
        }
    }
  }
}
