/*#############################################################################
#                                                                             #
# Pakfire - The IPFire package management system                              #
# Copyright (C) 2017 Pakfire development team                                 #
#                                                                             #
# This program is free software: you can redistribute it and/or modify        #
# it under the terms of the GNU General Public License as published by        #
# the Free Software Foundation, either version 3 of the License, or           #
# (at your option) any later version.                                         #
#                                                                             #
# This program is distributed in the hope that it will be useful,             #
# but WITHOUT ANY WARRANTY; without even the implied warranty of              #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               #
# GNU General Public License for more details.                                #
#                                                                             #
# You should have received a copy of the GNU General Public License           #
# along with this program.  If not, see <http://www.gnu.org/licenses/>.       #
#                                                                             #
#############################################################################*/

#ifndef PAKFIRE_PAKFIRE_H
#define PAKFIRE_PAKFIRE_H

#include <stdarg.h>
#include <stddef.h>
#include <stdio.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <time.h>

// libsolv
#include <solv/pool.h>

struct pakfire;

#include <pakfire/config.h>
#include <pakfire/ctx.h>
#include <pakfire/filelist.h>
#include <pakfire/key.h>
#include <pakfire/packagelist.h>
#include <pakfire/parser.h>
#include <pakfire/progress.h>
#include <pakfire/pwd.h>
#include <pakfire/repo.h>
#include <pakfire/repolist.h>
#include <pakfire/transaction.h>

enum pakfire_flags {
	PAKFIRE_FLAGS_STUB              = (1 << 0),
	PAKFIRE_FLAGS_BUILD             = (1 << 1),
	PAKFIRE_FLAGS_BUILD_LOCAL       = (1 << 2),
	PAKFIRE_USE_TMPFS               = (1 << 3),
	PAKFIRE_USE_SNAPSHOT            = (1 << 4),
};

// Callbacks
typedef void (*pakfire_status_callback)(struct pakfire* pakfire, void* data,
	int progress, const char* status);

int pakfire_create(struct pakfire** pakfire, struct pakfire_ctx* ctx,
	struct pakfire_config* config, const char* path, const char* arch, int flags);

struct pakfire* pakfire_ref(struct pakfire* pakfire);
struct pakfire* pakfire_unref(struct pakfire* pakfire);

int pakfire_has_flag(struct pakfire* pakfire, const int flag);

const char* pakfire_get_path(struct pakfire* pakfire);

int pakfire_clean(struct pakfire* pakfire, int flags);

const char* pakfire_get_arch(struct pakfire* pakfire);

int pakfire_version_compare(struct pakfire* pakfire, const char* evr1, const char* evr2);

struct pakfire_repolist* pakfire_get_repos(struct pakfire* pakfire);
struct pakfire_repo* pakfire_get_repo(struct pakfire* pakfire, const char* name);

int pakfire_whatprovides(struct pakfire* pakfire, const char* what, int flags,
	struct pakfire_packagelist* list);
int pakfire_whatrequires(struct pakfire* pakfire, const char* what, int flags,
	struct pakfire_packagelist* list);

// Search

enum pakfire_search_flags {
	PAKFIRE_SEARCH_NAME_ONLY        = (1 << 0),
};

int pakfire_search(struct pakfire* pakfire, const char* what, int flags,
	struct pakfire_packagelist* list);

// Check

int pakfire_check(struct pakfire* pakfire, struct pakfire_filelist* errors);

// Snapshots
int pakfire_update_snapshot(struct pakfire* pakfire);

struct pakfire_ctx* pakfire_ctx(struct pakfire* pakfire);

const char* pakfire_get_effective_arch(struct pakfire* pakfire);

int pakfire_on_root(struct pakfire* pakfire);

uid_t pakfire_uid(struct pakfire* pakfire);
gid_t pakfire_gid(struct pakfire* pakfire);

const struct pakfire_subid* pakfire_subuid(struct pakfire* pakfire);
const struct pakfire_subid* pakfire_subgid(struct pakfire* pakfire);

struct pakfire_config* pakfire_get_config(struct pakfire* pakfire);

int pakfire_confirm(struct pakfire* pakfire, const char* message, const char* question);
int pakfire_pick_solution(struct pakfire* pakfire, struct pakfire_transaction* transaction);
int pakfire_setup_progress(struct pakfire* pakfire, struct pakfire_progress* progress);

const char* pakfire_get_distro_name(struct pakfire* pakfire);
const char* pakfire_get_distro_id(struct pakfire* pakfire);
const char* pakfire_get_distro_vendor(struct pakfire* pakfire);
const char* pakfire_get_distro_version(struct pakfire* pakfire);
const char* pakfire_get_distro_version_id(struct pakfire* pakfire);
const char* pakfire_get_distro_tag(struct pakfire* pakfire);

int pakfire_openat(struct pakfire* pakfire, const char* path, int flags) __nonnull((2));

#define pakfire_path(pakfire, path, format, ...) \
	__pakfire_path(pakfire, path, sizeof(path), format, __VA_ARGS__)
int __pakfire_path(struct pakfire* pakfire, char* path, const size_t length,
	const char* format, ...) __attribute__((format(printf, 4, 5)));

const char* pakfire_relpath(struct pakfire* pakfire, const char* path);

#define pakfire_cache_path(pakfire, path, format, ...) \
	__pakfire_cache_path(pakfire, path, sizeof(path), format, __VA_ARGS__)
int __pakfire_cache_path(struct pakfire* pakfire, char* path, size_t length,
	const char* format, ...) __attribute__((format(printf, 4, 5)));

void pakfire_pool_has_changed(struct pakfire* pakfire);
void pakfire_pool_internalize(struct pakfire* pakfire);

Pool* pakfire_get_solv_pool(struct pakfire* pakfire);

struct pakfire_repo* pakfire_get_installed_repo(struct pakfire* pakfire);

int pakfire_commandline_add(struct pakfire* pakfire, const char* path,
	struct pakfire_package** package);

typedef int (*pakfire_repo_walk_callback)
	(struct pakfire* pakfire, struct pakfire_repo* repo, void* p);
int pakfire_repo_walk(struct pakfire* pakfire,
	pakfire_repo_walk_callback callback, void* p);

// Archive helpers
struct archive* pakfire_get_disk_reader(struct pakfire* pakfire);
struct archive* pakfire_get_disk_writer(struct pakfire* pakfire);

int pakfire_install(struct pakfire* self, const char** packages);

#endif /* PAKFIRE_PAKFIRE_H */
