/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */
import {
  HttpStatusCode,
  InstanceConfigurationMessage,
  TalerMerchantApi,
} from "@gnu-taler/taler-util";
import {
  useChallengeHandler,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { Fragment, VNode, h } from "preact";
import { useState } from "preact/hooks";
import { NotificationCard } from "../../../components/menu/index.js";
import { SolveMFAChallenges } from "../../../components/SolveMFA.js";
import { useSessionContext } from "../../../context/session.js";
import { Notification } from "../../../utils/types.js";
import { FOREVER_REFRESHABLE_TOKEN } from "../../login/index.js";
import { CreatePage } from "./CreatePage.js";

interface Props {
  onBack?: () => void;
  onConfirm: () => void;
  forceId?: string;
}
export type Entity = TalerMerchantApi.InstanceConfigurationMessage;

export default function Create({ onBack, onConfirm, forceId }: Props): VNode {
  const [notif, setNotif] = useState<Notification | undefined>(undefined);
  const { i18n } = useTranslationContext();
  const { lib, state, logIn } = useSessionContext();

  const { doCancelChallenge, pendingChallenge, withMfaHandler } =
    useChallengeHandler();

  const [onFirstCall, repeatCall] = withMfaHandler(
    ({ challengeIds, onChallengeRequired }) =>
      async function createInstaceImpl(d: InstanceConfigurationMessage) {
        if (state.status !== "loggedIn") return;
        try {
          const resp = await lib.instance.createInstance(state.token, d, {
            challengeIds: challengeIds,
          });
          if (resp.type === "fail") {
            if (resp.case === HttpStatusCode.Accepted) {
              onChallengeRequired(resp.body);
              return;
            }
            setNotif({
              message: i18n.str`Failed to create instance`,
              type: "ERROR",
              description: resp.detail?.hint,
            });
            return;
          }
          if (d.auth.password) {
            //if auth has been updated, request a new access token
            const result = await lib.instance.createAccessToken(
              d.id,
              d.auth.password,
              FOREVER_REFRESHABLE_TOKEN(i18n.str`Instance created`),
            );
            if (result.type === "ok") {
              const { access_token: token } = result.body;
              logIn(state.instance, token);
            }
          }
          onConfirm();
        } catch (error) {
          setNotif({
            message: i18n.str`Failed to create instance`,
            type: "ERROR",
            description: error instanceof Error ? error.message : String(error),
          });
        }
      },
  );

  if (pendingChallenge) {
    return (
      <SolveMFAChallenges
        currentChallenge={pendingChallenge}
        onCompleted={repeatCall}
        onCancel={doCancelChallenge}
      />
    );
  }

  return (
    <Fragment>
      <NotificationCard notification={notif} />

      <CreatePage onBack={onBack} forceId={forceId} onCreate={onFirstCall} />
    </Fragment>
  );
}
