/**
 *    Copyright (C) 2018-present MongoDB, Inc.
 *
 *    This program is free software: you can redistribute it and/or modify
 *    it under the terms of the Server Side Public License, version 1,
 *    as published by MongoDB, Inc.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    Server Side Public License for more details.
 *
 *    You should have received a copy of the Server Side Public License
 *    along with this program. If not, see
 *    <http://www.mongodb.com/licensing/server-side-public-license>.
 *
 *    As a special exception, the copyright holders give permission to link the
 *    code of portions of this program with the OpenSSL library under certain
 *    conditions as described in each individual source file and distribute
 *    linked combinations including the program with the OpenSSL library. You
 *    must comply with the Server Side Public License in all respects for
 *    all of the code used other than as permitted herein. If you modify file(s)
 *    with this exception, you may extend this exception to your version of the
 *    file(s), but you are not obligated to do so. If you do not wish to do so,
 *    delete this exception statement from your version. If you delete this
 *    exception statement from all source files in the program, then also delete
 *    it in the license file.
 */

#include <memory>


#include "mongo/base/status.h"
#include "mongo/base/string_data.h"
#include "mongo/bson/bsonelement.h"
#include "mongo/bson/bsonobj.h"
#include "mongo/bson/bsonobjbuilder.h"
#include "mongo/bson/bsontypes.h"
#include "mongo/db/logical_time.h"
#include "mongo/db/operation_context.h"
#include "mongo/db/operation_time_tracker.h"
#include "mongo/db/service_context.h"
#include "mongo/db/vector_clock.h"
#include "mongo/db/vector_clock_gen.h"
#include "mongo/db/vector_clock_metadata_hook.h"
#include "mongo/transport/session.h"
#include "mongo/util/assert_util.h"

namespace mongo {

namespace rpc {

namespace {
const char kOperationTimeFieldName[] = "operationTime";
}

VectorClockMetadataHook::VectorClockMetadataHook(ServiceContext* service) : _service(service) {}

Status VectorClockMetadataHook::writeRequestMetadata(OperationContext* opCtx,
                                                     BSONObjBuilder* metadataBob) {
    VectorClock::get(_service)->gossipOut(opCtx, metadataBob, true /* forceInternal */);
    return Status::OK();
}

Status VectorClockMetadataHook::readReplyMetadata(OperationContext* opCtx,
                                                  const BSONObj& metadataObj) {
    if (!VectorClock::get(_service)->isEnabled()) {
        return Status::OK();
    }

    if (opCtx) {
        auto timeTracker = OperationTimeTracker::get(opCtx);
        auto operationTime = metadataObj[kOperationTimeFieldName];
        if (!operationTime.eoo()) {
            tassert(4457010,
                    "operationTime must be a timestamp if present",
                    operationTime.type() == BSONType::bsonTimestamp);
            timeTracker->updateOperationTime(LogicalTime(operationTime.timestamp()));
        }
    }

    auto receivedComponents = GossipedVectorClockComponents::parse(
        IDLParserContext("VectorClockComponents"), metadataObj);
    VectorClock::get(_service)->gossipIn(opCtx,
                                         receivedComponents,
                                         false /* couldBeUnauthorized */,
                                         true /* defaultIsInternalClient */);
    return Status::OK();
}

}  // namespace rpc
}  // namespace mongo
