/**
 * Tests the compatibility of v1 and v2 resume tokens across server version upgrade / downgrade.
 * @tags: [uses_change_streams]
 */

import "jstests/multiVersion/libs/multi_cluster.js";
import {
    assertCreateCollection,
    assertDropCollection
} from "jstests/libs/collection_drop_recreate.js";

const kLargeStringSize = 15 * 1024 * 1024;

const st = new ShardingTest({
    shards: 2,
    config: 1,
    other: {
        mongosOptions: {binVersion: "last-lts"},
        configOptions: {
            binVersion: "last-lts",
        },
        rsOptions: {
            binVersion: "last-lts",
        },
        rs: {
            nodes: 2,
            // Reserving enough of oplog space to accommodate 4 nearly 16MB-large changes.
            oplogSize: 16 * 5
        }
    }
});

let testDB = st.s.getDB(jsTestName());
let testColl = testDB["test"];

// Helper function to retrieve high-watermark change stream tokens.
function getHighWaterMarkToken(collection, pipeline = [], options = {}) {
    const csCursor = collection.watch(pipeline, {batchSize: 0, ...options});
    const result = csCursor.getResumeToken();
    csCursor.close();
    return result;
}

// Record a high-watermark resume token marking the start point of the test.
const testStartV1HWMToken = getHighWaterMarkToken(testColl, [], {$_generateV2ResumeTokens: false});

// An array which will list the expected sequence of change events generated by the test.
const expectedEvents = [];

//
// Below, we generate one of each type of change event so that we can later test resuming from a v1
// token representing each such event.
//
testColl = assertCreateCollection(testDB, testColl.getName());
expectedEvents.push({operationType: "create"});

assert.commandWorked(testColl.createIndexes([{shard: 1}, {shard: 1, _id: 1}, {largeField: 1}]));
expectedEvents.push({operationType: "createIndexes"},
                    {operationType: "createIndexes"},
                    {operationType: "createIndexes"});

// The 'modify' event has to come before sharding the collection, otherwise we get different number
// of events on 'last-lts' and on 'latest'.
assert.commandWorked(testDB.runCommand(
    {collMod: testColl.getName(), changeStreamPreAndPostImages: {enabled: true}}));
expectedEvents.push({operationType: "modify"});

// Shard the test collection and split it into two chunks: one that contains all {shard: 1}
// documents and one that contains all {shard: 2} documents.
st.shardColl(testColl, {shard: 1} /* shard key */, {shard: 2} /* split at */);
expectedEvents.push({operationType: "shardCollection"});

assert.commandWorked(testColl.insertMany([
    {_id: "a", shard: 1, largeField: ""},
    {_id: "b", shard: 2, largeField: ""},
    {_id: "c", shard: 2, largeField: ""}
]));
expectedEvents.push({operationType: "insert", documentKey: {shard: 1, _id: "a"}},
                    {operationType: "insert", documentKey: {shard: 2, _id: "b"}},
                    {operationType: "insert", documentKey: {shard: 2, _id: "c"}});

// This high watermark token will be at the same clusterTime as the subsequent update event that
// needs to be split.
const v1HwmTokenBeforeUpdateWithSplit = getHighWaterMarkToken(
    testColl, [{$changeStreamSplitLargeEvent: {}}], {$_generateV2ResumeTokens: false});

assert.commandWorked(
    testColl.update({_id: "a", shard: 1}, {$set: {largeField: "x".repeat(kLargeStringSize)}}));
expectedEvents.push({operationType: "update", documentKey: {_id: "a", shard: 1}});

assert.commandWorked(
    testColl.update({_id: "b", shard: 2}, {$set: {largeField: "x".repeat(kLargeStringSize)}}));
expectedEvents.push({operationType: "update", documentKey: {_id: "b", shard: 2}});

assert.commandWorked(testColl.replaceOne(
    {_id: "a", shard: 1}, {_id: "a", shard: 1, largeField: "y".repeat(kLargeStringSize)}));
expectedEvents.push({operationType: "replace", documentKey: {_id: "a", shard: 1}});

assert.commandWorked(testColl.replaceOne(
    {_id: "b", shard: 2}, {_id: "b", shard: 2, largeField: "y".repeat(kLargeStringSize)}));
expectedEvents.push({operationType: "replace", documentKey: {_id: "b", shard: 2}});

assert.commandWorked(testColl.remove({_id: "a"}));
expectedEvents.push({operationType: "delete", documentKey: {_id: "a", shard: 1}});

assert.commandWorked(testColl.remove({_id: "b"}));
expectedEvents.push({operationType: "delete", documentKey: {_id: "b", shard: 2}});

assert.commandWorked(
    st.s.adminCommand({refineCollectionShardKey: testColl.getFullName(), key: {shard: 1, _id: 1}}));
expectedEvents.push({operationType: "refineCollectionShardKey"});

assert.commandWorked(st.s.adminCommand(
    {reshardCollection: testColl.getFullName(), key: {_id: 1}, numInitialChunks: 1}));
expectedEvents.push({operationType: "reshardCollection"});

assert.commandWorked(testColl.dropIndex({largeField: 1}));
expectedEvents.push({operationType: "dropIndexes"});

const newTestCollectionName = "test_";
assert.commandWorked(testColl.renameCollection(newTestCollectionName));
expectedEvents.push({operationType: "rename"});

assertDropCollection(testDB, newTestCollectionName);
expectedEvents.push({operationType: "drop"});

assert.commandWorked(testDB.dropDatabase());
// A whole-DB stream will be invalidated by the dropDatabase event. We include a second dropDatabase
// event because one such event is generated on each shard, and will be reported if we resume after
// the invalidate. This second dropDatabase acts as a sentinel here, signifying that we have reached
// the end of the test stream.
expectedEvents.push({operationType: "dropDatabase"},
                    {operationType: "invalidate"},
                    {operationType: "dropDatabase"});

// Helper function to assert on the given event fields.
function assertEventMatches(event, expectedEvent, errorMsg) {
    for (const k in expectedEvent) {
        assert.docEq(expectedEvent[k], event[k], errorMsg + `: value mismatch for field '${k}'`);
    }
}

// Asserts the next change event with the given pipeline and options matches the expected event.
// Returns the resume token of the matched event on success.
function assertNextChangeEvent(expectedEvent, pipeline, options) {
    const csCursor = testDB.watch([...pipeline], {showExpandedEvents: true, ...options});
    const errorMsg = "could not retrieve the expected event matching " + tojson(expectedEvent);
    let event;
    do {
        assert.doesNotThrow(() => assert.soon(() => csCursor.hasNext()), [], errorMsg);
        event = csCursor.next();
        // When using '$_generateV2ResumeTokens: false', the change stream will additionally
        // emit non-public 'kNewShardDetected' event. Unfortunately, it cannot be filtered out
        // by adding a '$match' stage, so we fast-forward all these events.
    } while (event.operationType === "kNewShardDetected");
    assertEventMatches(event, expectedEvent, errorMsg);
    csCursor.close();
    return event._id;
}

// Helper function to retrieve change event tokens for all events referred by 'expectedEvents'.
function getTokensForExpectedEvents(expectedEvents, startToken, pipeline = [], options = {}) {
    return expectedEvents
        .reduce(
            (result, expectedEvent) => {
                const lastToken = result[result.length - 1];
                result.push(assertNextChangeEvent(
                    expectedEvent, pipeline, {startAfter: lastToken, ...options}));
                return result;
            },
            [startToken])
        .slice(1);
}

// Generate v1 resume tokens for all expected events on 'last-lts'.
const resumeTokensLastLTS = getTokensForExpectedEvents(
    expectedEvents, testStartV1HWMToken, [], {$_generateV2ResumeTokens: false});
// TODO SERVER-82330: Validate that these tokens are indeed all v1 tokens.

// Upgrade the cluster to 'latest' to allow testing v1 - v2 resume behaviour.
st.upgradeCluster("latest", true);
assert.commandWorked(st.s.adminCommand({setFeatureCompatibilityVersion: latestFCV, confirm: true}));

testDB = st.s.getDB(jsTestName());

// Verify that we can resume from each of the v1 tokens on the new binary version and with the
// $changeStreamSplitLargeEvent stage in the pipeline. When resuming from (i-1)-th event's token we
// expect to get the i-th event. We do not need to test the last token, because it is simply a
// sentinel value that signifies the end of the test.
for (let i = 1; i < expectedEvents.length; ++i) {
    assertNextChangeEvent(expectedEvents[i],
                          [{$changeStreamSplitLargeEvent: {}}],
                          {startAfter: resumeTokensLastLTS[i - 1]});
}

// Test that we can split the update events after 'v1HwmTokenBeforeUpdateWithSplit'.
const expectedSplitEvents = [
    {operationType: "update", splitEvent: {fragment: 1, of: 2}},
    {splitEvent: {fragment: 2, of: 2}},
    {operationType: "update", splitEvent: {fragment: 1, of: 2}},
    {splitEvent: {fragment: 2, of: 2}},
    {operationType: "replace"}
];
// Generate a set of v2 resume tokens for each of the entries in 'expectedSplitEvents'.
const resumeTokensWithSplitLatest = getTokensForExpectedEvents(expectedSplitEvents,
                                                               v1HwmTokenBeforeUpdateWithSplit,
                                                               [{$changeStreamSplitLargeEvent: {}}],
                                                               {fullDocument: "required"});

// Downgrade back to the original version.
assert.commandWorked(
    st.s.adminCommand({setFeatureCompatibilityVersion: lastLTSFCV, confirm: true}));
st.downgradeCluster("latest", "last-lts", true);

testDB = st.s.getDB(jsTestName());

// Test the v2 split 'update' event tokens on the downgraded binary version. When resuming from
// (i-1)-th event token we expect to get the i-th event. This means, after the last 'update' event
// we expect the first 'replace' event.
for (let i = 1; i < expectedSplitEvents.length; ++i) {
    assertNextChangeEvent(
        expectedSplitEvents[i],
        [{$changeStreamSplitLargeEvent: {}}],
        {resumeAfter: resumeTokensWithSplitLatest[i - 1], fullDocument: "required"});
}

st.stop();
