// Copyright ©2015 The gonum Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package testlapack

import (
	"fmt"
	"testing"

	"github.com/gonum/floats"
)

type Dlasq3er interface {
	Dlasq3(i0, n0 int, z []float64, pp int, dmin, sigma, desig, qmax float64, nFail, iter, nDiv int, ttype int, dmin1, dmin2, dn, dn1, dn2, g, tau float64) (
		i0Out, n0Out, ppOut int, dminOut, sigmaOut, desigOut, qmaxOut float64, nFailOut, iterOut, nDivOut, ttypeOut int, dmin1Out, dmin2Out, dnOut, dn1Out, dn2Out, gOut, tauOut float64)
}

func printDlasq3FortranInput(d dlasq3teststruct) {
	z := d.z
	printFortranArray(z, "z")
	fmt.Println("i0 =", d.i0)
	fmt.Println("n0 =", d.n0)
	fmt.Println("pp =", d.pp)

	fmt.Println("dmin =", fortran64(d.dmin))
	fmt.Println("sigma =", fortran64(d.sigma))
	fmt.Println("desig =", fortran64(d.desig))
	fmt.Println("qmax =", fortran64(d.qmax))
	fmt.Println("nFail =", d.nFail)
	fmt.Println("iter =", d.iter)
	fmt.Println("nDiv =", d.nDiv)
	fmt.Println("ttype =", d.ttype)
	fmt.Println("dmin1 =", fortran64(d.dmin1))
	fmt.Println("dmin2 =", fortran64(d.dmin2))
	fmt.Println("dn =", fortran64(d.dn))
	fmt.Println("dn1 =", fortran64(d.dn1))
	fmt.Println("dn2 =", fortran64(d.dn2))
	fmt.Println("g =", fortran64(d.g))
	fmt.Println("tau =", fortran64(d.tau))
}

type dlasq3teststruct struct {
	z                                  []float64
	i0, n0, pp                         int
	dmin, desig, sigma, qmax           float64
	nFail, iter, nDiv, ttype           int
	dmin1, dmin2, dn, dn1, dn2, g, tau float64

	zOut                                                    []float64
	i0Out, n0Out, ppOut                                     int
	dminOut, desigOut, sigmaOut, qmaxOut                    float64
	nFailOut, iterOut, nDivOut, ttypeOut                    int
	dmin1Out, dmin2Out, dnOut, dn1Out, dn2Out, gOut, tauOut float64
}

func Dlasq3Test(t *testing.T, impl Dlasq3er) {
	dTol := 1e-14
	// Tests computed from calling the netlib Dlasq
	for _, test := range []dlasq3teststruct{
		{
			i0:       1,
			n0:       21,
			z:        []float64{0.1914365246180821E+01, 0.1564384297703890E+01, 0.2493389162143899E+00, 0.3499809484769305E+00, 0.1315996513131545E+01, 0.1363862112490627E+01, 0.9898466611970759E-01, 0.2014733168553078E+00, 0.6023973979587287E+00, 0.6465544792741794E+00, 0.2210033410638781E-02, 0.5482758480425683E-01, 0.9861857233678967E-01, 0.2428190810745492E-01, 0.4756321484454819E+00, 0.7654669763997353E-01, 0.2588748143677115E+00, 0.6127784069508770E+00, 0.1078611376690004E+00, 0.1217285558623164E+00, 0.6442896492255246E+00, 0.2293835804898155E+00, 0.6203230486639705E+00, 0.5227672064047094E+00, 0.3695660678607585E+00, 0.7645233184745865E+00, 0.5378838054252265E+00, 0.2253657980501426E+00, 0.3562533181264623E+00, 0.8820486722335483E+00, 0.2222132496436145E-01, 0.1208845131814035E-01, 0.1275094303021685E+01, 0.6548746852163357E+00, 0.1647324354821218E+00, 0.6424409427697111E+00, 0.1007530576543866E+01, 0.3269551736546701E+00, 0.3453881601783118E+00, 0.8453078383713172E+00, 0.2679391719153404E+00, 0.4116714838778281E+00, 0.7328677736683723E+00, 0.2016558482158241E+00, 0.8360828138307410E+00, 0.9737579452195326E+00, 0.4813660709592822E+00, 0.5951926422795808E+00, 0.6495370513676459E+00, 0.6761876248148171E+00, 0.2325475880222648E+00, 0.4547154975121112E+00, 0.1993624802893807E+00, 0.3321819367342255E+00, 0.3782318916911257E+00, 0.9972813157741996E-01, 0.9830449403503746E+00, 0.7561080996844842E+00, 0.4429733864040367E+00, 0.6051687323570161E+00, 0.1173279550602403E+01, 0.7195724480316686E+00, 0.5035524069144587E+00, 0.8966804889747714E+00, 0.3058980395058521E+00, 0.6588832353928662E+00, 0.3014634433415453E+00, 0.1505672110274446E+00, 0.1289422237567578E+01, 0.6124645310993601E+00, 0.7583364305799440E+00, 0.9784211498097629E+00, 0.4977814779461571E+00, 0.9993813577491869E+00, 0.2841468847862598E+00, 0.2567365507769143E+00, 0.9257539794205765E+00, 0.5509268385614666E+00, 0.5231355605450990E-04, 0.6589740256453697E+00, 0.2117869221381033E-04, 0.7349224826832024E-04, 0.0000000000000000E+00, 0.0000000000000000E+00},
			pp:       0,
			dmin:     -0.0000000000000000,
			desig:    0.0000000000000000,
			qmax:     2.1637041623952107,
			ttype:    0,
			dmin1:    0.0000000000000000,
			dmin2:    0.0000000000000000,
			dn:       0.0000000000000000,
			dn1:      0.0000000000000000,
			dn2:      0.0000000000000000,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    0,
			iter:     2,
			sigma:    0.0000000000000000,
			nDiv:     40,
			zOut:     []float64{0.1914365246180821E+01, 0.2163704162395211E+01, 0.2493389162143899E+00, 0.1516515751224039E+00, 0.1315996513131545E+01, 0.1263329604128848E+01, 0.9898466611970759E-01, 0.4719916727467415E-01, 0.6023973979587287E+00, 0.5574082640946934E+00, 0.2210033410638781E-02, 0.3910066531356214E-03, 0.9861857233678967E-01, 0.5738597141291359E+00, 0.4756321484454819E+00, 0.2145632131068746E+00, 0.2588748143677115E+00, 0.1521727389298373E+00, 0.1078611376690004E+00, 0.4566771620366771E+00, 0.6442896492255246E+00, 0.8079355358528180E+00, 0.6203230486639705E+00, 0.2837483186776231E+00, 0.3695660678607585E+00, 0.6237015546083620E+00, 0.5378838054252265E+00, 0.3072349091217998E+00, 0.3562533181264623E+00, 0.7123973396902394E-01, 0.2222132496436145E-01, 0.3977314805803597E+00, 0.1275094303021685E+01, 0.1042095257923447E+01, 0.1647324354821218E+00, 0.1592685164190333E+00, 0.1007530576543866E+01, 0.1193650220303144E+01, 0.3453881601783118E+00, 0.7752942700755104E-01, 0.2679391719153404E+00, 0.9232775185761617E+00, 0.7328677736683723E+00, 0.6636554427529671E+00, 0.8360828138307410E+00, 0.6537934420370561E+00, 0.4813660709592822E+00, 0.4782322339990674E+00, 0.6495370513676459E+00, 0.4038524053908432E+00, 0.2325475880222648E+00, 0.1147975431483785E+00, 0.1993624802893807E+00, 0.4627968288321279E+00, 0.3782318916911257E+00, 0.8034172324482011E+00, 0.9830449403503746E+00, 0.6226010943062101E+00, 0.4429733864040367E+00, 0.8347746582554776E+00, 0.1173279550602403E+01, 0.8420572992613844E+00, 0.5035524069144587E+00, 0.1829278057427913E+00, 0.3058980395058521E+00, 0.4244336771046062E+00, 0.3014634433415453E+00, 0.9158407747236312E+00, 0.1289422237567578E+01, 0.1131917893423890E+01, 0.7583364305799440E+00, 0.3334922359541972E+00, 0.4977814779461571E+00, 0.4484361267782198E+00, 0.2841468847862598E+00, 0.5865943745895725E+00, 0.9257539794205765E+00, 0.3392119183870583E+00, 0.5231355605450990E-04, 0.3266196269153995E-08, 0.2117869221381033E-04, 0.2117542601754118E-04, 0.0000000000000000E+00, 0.3910066531356214E-03},
			i0Out:    1,
			n0Out:    21,
			ppOut:    0,
			dminOut:  2.1175426017541180E-005,
			desigOut: 0.0000000000000000,
			sigmaOut: 0.0000000000000000,
			qmaxOut:  2.1637041623952107,
			nFailOut: 0,
			iterOut:  3,
			nDivOut:  62,
			ttypeOut: -1,
			dmin1Out: 4.4311601260836921E-002,
			dmin2Out: 4.4311601260836921E-002,
			dnOut:    2.1175426017541180E-005,
			dn1Out:   0.33915960483100382,
			dn2Out:   0.16428924199195991,
			gOut:     0.0000000000000000,
			tauOut:   0.0000000000000000,
		},
		{
			i0:       1,
			n0:       21,
			z:        []float64{0.1914365246180821E+01, 0.2163704162395211E+01, 0.2493389162143899E+00, 0.1516515751224039E+00, 0.1315996513131545E+01, 0.1263329604128848E+01, 0.9898466611970759E-01, 0.4719916727467415E-01, 0.6023973979587287E+00, 0.5574082640946934E+00, 0.2210033410638781E-02, 0.3910066531356214E-03, 0.9861857233678967E-01, 0.5738597141291359E+00, 0.4756321484454819E+00, 0.2145632131068746E+00, 0.2588748143677115E+00, 0.1521727389298373E+00, 0.1078611376690004E+00, 0.4566771620366771E+00, 0.6442896492255246E+00, 0.8079355358528180E+00, 0.6203230486639705E+00, 0.2837483186776231E+00, 0.3695660678607585E+00, 0.6237015546083620E+00, 0.5378838054252265E+00, 0.3072349091217998E+00, 0.3562533181264623E+00, 0.7123973396902394E-01, 0.2222132496436145E-01, 0.3977314805803597E+00, 0.1275094303021685E+01, 0.1042095257923447E+01, 0.1647324354821218E+00, 0.1592685164190333E+00, 0.1007530576543866E+01, 0.1193650220303144E+01, 0.3453881601783118E+00, 0.7752942700755104E-01, 0.2679391719153404E+00, 0.9232775185761617E+00, 0.7328677736683723E+00, 0.6636554427529671E+00, 0.8360828138307410E+00, 0.6537934420370561E+00, 0.4813660709592822E+00, 0.4782322339990674E+00, 0.6495370513676459E+00, 0.4038524053908432E+00, 0.2325475880222648E+00, 0.1147975431483785E+00, 0.1993624802893807E+00, 0.4627968288321279E+00, 0.3782318916911257E+00, 0.8034172324482011E+00, 0.9830449403503746E+00, 0.6226010943062101E+00, 0.4429733864040367E+00, 0.8347746582554776E+00, 0.1173279550602403E+01, 0.8420572992613844E+00, 0.5035524069144587E+00, 0.1829278057427913E+00, 0.3058980395058521E+00, 0.4244336771046062E+00, 0.3014634433415453E+00, 0.9158407747236312E+00, 0.1289422237567578E+01, 0.1131917893423890E+01, 0.7583364305799440E+00, 0.3334922359541972E+00, 0.4977814779461571E+00, 0.4484361267782198E+00, 0.2841468847862598E+00, 0.5865943745895725E+00, 0.9257539794205765E+00, 0.3392119183870583E+00, 0.5231355605450990E-04, 0.3266196269153995E-08, 0.2117869221381033E-04, 0.2117542601754118E-04, 0.0000000000000000E+00, 0.3910066531356214E-03},
			pp:       1,
			dmin:     2.1175426017541180E-005,
			desig:    0.0000000000000000,
			qmax:     2.1637041623952107,
			ttype:    -1,
			dmin1:    4.4311601260836921E-002,
			dmin2:    4.4311601260836921E-002,
			dn:       2.1175426017541180E-005,
			dn1:      0.33915960483100382,
			dn2:      0.16428924199195991,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    0,
			iter:     3,
			sigma:    0.0000000000000000,
			nDiv:     62,
			zOut:     []float64{0.2315355737517615E+01, 0.2163704162395211E+01, 0.8274578340618610E-01, 0.1516515751224039E+00, 0.1227782987997336E+01, 0.1263329604128848E+01, 0.2142822156235013E-01, 0.4719916727467415E-01, 0.5363710491854788E+00, 0.5574082640946934E+00, 0.4183353417969536E-03, 0.3910066531356214E-03, 0.7880045918942136E+00, 0.5738597141291359E+00, 0.4143462125464707E-01, 0.2145632131068746E+00, 0.5674152797118673E+00, 0.1521727389298373E+00, 0.6502569120260687E+00, 0.4566771620366771E+00, 0.4414269425043723E+00, 0.8079355358528180E+00, 0.4009140594652070E+00, 0.2837483186776231E+00, 0.5300224042649548E+00, 0.6237015546083620E+00, 0.4129510944388858E-01, 0.3072349091217998E+00, 0.4276761051054951E+00, 0.7123973396902394E-01, 0.9691308092544145E+00, 0.3977314805803597E+00, 0.2322329650880660E+00, 0.1042095257923447E+01, 0.8186215063776209E+00, 0.1592685164190333E+00, 0.4525581409330741E+00, 0.1193650220303144E+01, 0.1581701233715052E+00, 0.7752942700755104E-01, 0.1428762837957623E+01, 0.9232775185761617E+00, 0.3036848136842134E+00, 0.6636554427529671E+00, 0.8283408623519102E+00, 0.6537934420370561E+00, 0.2331591338951825E+00, 0.4782322339990674E+00, 0.2854908146440392E+00, 0.4038524053908432E+00, 0.1860933389154074E+00, 0.1147975431483785E+00, 0.1080120722364922E+01, 0.4627968288321279E+00, 0.4631042046962229E+00, 0.8034172324482011E+00, 0.9942715478654648E+00, 0.6226010943062101E+00, 0.7069779837626068E+00, 0.8347746582554776E+00, 0.3180071212415688E+00, 0.8420572992613844E+00, 0.2441477440283845E+00, 0.1829278057427913E+00, 0.1096126707799853E+01, 0.4244336771046062E+00, 0.9457451890006905E+00, 0.9158407747236312E+00, 0.5196649403773971E+00, 0.1131917893423890E+01, 0.2877815203259632E+00, 0.3334922359541972E+00, 0.7472489810418290E+00, 0.4484361267782198E+00, 0.2662831374385604E+00, 0.5865943745895725E+00, 0.7292878421469419E-01, 0.3392119183870583E+00, 0.9483648767903632E-12, 0.3266196269153995E-08, 0.2117542506917630E-04, 0.2117542601754118E-04, 0.4183353417969536E-03, 0.3910066531356214E-03},
			i0Out:    1,
			n0Out:    21,
			ppOut:    1,
			dminOut:  2.1175425069176302E-005,
			desigOut: 0.0000000000000000,
			sigmaOut: 0.0000000000000000,
			qmaxOut:  2.1637041623952107,
			nFailOut: 0,
			iterOut:  4,
			nDivOut:  84,
			ttypeOut: -4,
			dmin1Out: 2.9944624525135358E-002,
			dmin2Out: 2.9944624525135358E-002,
			dnOut:    2.1175425069176302E-005,
			dn1Out:   7.2928780948497918E-002,
			dn2Out:   0.16065460645225654,
			gOut:     0.0000000000000000,
			tauOut:   0.0000000000000000,
		},
		{
			i0:       1,
			n0:       21,
			z:        []float64{0.2315355737517615E+01, 0.2163704162395211E+01, 0.8274578340618610E-01, 0.1516515751224039E+00, 0.1227782987997336E+01, 0.1263329604128848E+01, 0.2142822156235013E-01, 0.4719916727467415E-01, 0.5363710491854788E+00, 0.5574082640946934E+00, 0.4183353417969536E-03, 0.3910066531356214E-03, 0.7880045918942136E+00, 0.5738597141291359E+00, 0.4143462125464707E-01, 0.2145632131068746E+00, 0.5674152797118673E+00, 0.1521727389298373E+00, 0.6502569120260687E+00, 0.4566771620366771E+00, 0.4414269425043723E+00, 0.8079355358528180E+00, 0.4009140594652070E+00, 0.2837483186776231E+00, 0.5300224042649548E+00, 0.6237015546083620E+00, 0.4129510944388858E-01, 0.3072349091217998E+00, 0.4276761051054951E+00, 0.7123973396902394E-01, 0.9691308092544145E+00, 0.3977314805803597E+00, 0.2322329650880660E+00, 0.1042095257923447E+01, 0.8186215063776209E+00, 0.1592685164190333E+00, 0.4525581409330741E+00, 0.1193650220303144E+01, 0.1581701233715052E+00, 0.7752942700755104E-01, 0.1428762837957623E+01, 0.9232775185761617E+00, 0.3036848136842134E+00, 0.6636554427529671E+00, 0.8283408623519102E+00, 0.6537934420370561E+00, 0.2331591338951825E+00, 0.4782322339990674E+00, 0.2854908146440392E+00, 0.4038524053908432E+00, 0.1860933389154074E+00, 0.1147975431483785E+00, 0.1080120722364922E+01, 0.4627968288321279E+00, 0.4631042046962229E+00, 0.8034172324482011E+00, 0.9942715478654648E+00, 0.6226010943062101E+00, 0.7069779837626068E+00, 0.8347746582554776E+00, 0.3180071212415688E+00, 0.8420572992613844E+00, 0.2441477440283845E+00, 0.1829278057427913E+00, 0.1096126707799853E+01, 0.4244336771046062E+00, 0.9457451890006905E+00, 0.9158407747236312E+00, 0.5196649403773971E+00, 0.1131917893423890E+01, 0.2877815203259632E+00, 0.3334922359541972E+00, 0.7472489810418290E+00, 0.4484361267782198E+00, 0.2662831374385604E+00, 0.5865943745895725E+00, 0.7292878421469419E-01, 0.3392119183870583E+00, 0.9483648767903632E-12, 0.3266196269153995E-08, 0.2117542506917630E-04, 0.2117542601754118E-04, 0.4183353417969536E-03, 0.3910066531356214E-03},
			pp:       0,
			dmin:     2.1175425069176302E-005,
			desig:    0.0000000000000000,
			qmax:     2.1637041623952107,
			ttype:    -4,
			dmin1:    2.9944624525135358E-002,
			dmin2:    2.9944624525135358E-002,
			dn:       2.1175425069176302E-005,
			dn1:      7.2928780948497918E-002,
			dn2:      0.16065460645225654,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    0,
			iter:     4,
			sigma:    0.0000000000000000,
			nDiv:     84,
			zOut:     []float64{0.2315355737517615E+01, 0.2398080345610006E+01, 0.8274578340618610E-01, 0.4236466279397526E-01, 0.1227782987997336E+01, 0.1206825371451915E+01, 0.2142822156235013E-01, 0.9523728911788614E-02, 0.5363710491854788E+00, 0.5272444803016919E+00, 0.4183353417969536E-03, 0.6252320936560726E-03, 0.7880045918942136E+00, 0.8287928057414093E+00, 0.4143462125464707E-01, 0.2836732781232222E-01, 0.5674152797118673E+00, 0.1189283688611819E+01, 0.6502569120260687E+00, 0.2413561400585997E+00, 0.4414269425043723E+00, 0.6009636865971842E+00, 0.4009140594652070E+00, 0.3535878097802652E+00, 0.5300224042649548E+00, 0.2177085286147829E+00, 0.4129510944388858E-01, 0.8112190955144877E-01, 0.4276761051054951E+00, 0.1315663829494665E+01, 0.9691308092544145E+00, 0.1710650671895379E+00, 0.2322329650880660E+00, 0.8797682289623537E+00, 0.8186215063776209E+00, 0.4211038940233675E+00, 0.4525581409330741E+00, 0.1896031949674164E+00, 0.1581701233715052E+00, 0.1191897606932286E+01, 0.1428762837957623E+01, 0.5405288693957555E+00, 0.3036848136842134E+00, 0.4653859482687157E+00, 0.8283408623519102E+00, 0.5960928726645816E+00, 0.2331591338951825E+00, 0.1116684901463164E+00, 0.2854908146440392E+00, 0.3598944880993349E+00, 0.1860933389154074E+00, 0.5585061130503639E+00, 0.1080120722364922E+01, 0.9846976386969850E+00, 0.4631042046962229E+00, 0.4676068229793028E+00, 0.9942715478654648E+00, 0.1233621533334973E+01, 0.7069779837626068E+00, 0.1822471700779458E+00, 0.3180071212415688E+00, 0.3798865198782122E+00, 0.2441477440283845E+00, 0.7044652781161848E+00, 0.1096126707799853E+01, 0.1337385443370563E+01, 0.9457451890006905E+00, 0.3674861422265960E+00, 0.5196649403773971E+00, 0.4399391431629689E+00, 0.2877815203259632E+00, 0.4888049885267526E+00, 0.7472489810418290E+00, 0.5247059546398414E+00, 0.2662831374385604E+00, 0.3701064434002514E-01, 0.7292878421469419E-01, 0.3589696456182207E-01, 0.9483648767903632E-12, 0.5594353069081231E-15, 0.2117542506917630E-04, 0.1112732565966979E-09, 0.4183353417969536E-03, 0.6252320936560726E-03},
			i0Out:    1,
			n0Out:    21,
			ppOut:    0,
			dminOut:  1.1127325659669794E-010,
			desigOut: 0.0000000000000000,
			sigmaOut: 2.1175313795360271E-005,
			qmaxOut:  2.1637041623952107,
			nFailOut: 0,
			iterOut:  5,
			nDivOut:  106,
			ttypeOut: -4,
			dmin1Out: 3.1433071595911154E-002,
			dmin2Out: 3.1433071595911154E-002,
			dnOut:    1.1127325659669794E-010,
			dn1Out:   3.5896964560873705E-002,
			dn2Out:   0.25842281720128102,
			gOut:     0.0000000000000000,
			tauOut:   2.1175313795360271E-005,
		},
		{
			i0:       1,
			n0:       21,
			z:        []float64{0.2315355737517615E+01, 0.2398080345610006E+01, 0.8274578340618610E-01, 0.4236466279397526E-01, 0.1227782987997336E+01, 0.1206825371451915E+01, 0.2142822156235013E-01, 0.9523728911788614E-02, 0.5363710491854788E+00, 0.5272444803016919E+00, 0.4183353417969536E-03, 0.6252320936560726E-03, 0.7880045918942136E+00, 0.8287928057414093E+00, 0.4143462125464707E-01, 0.2836732781232222E-01, 0.5674152797118673E+00, 0.1189283688611819E+01, 0.6502569120260687E+00, 0.2413561400585997E+00, 0.4414269425043723E+00, 0.6009636865971842E+00, 0.4009140594652070E+00, 0.3535878097802652E+00, 0.5300224042649548E+00, 0.2177085286147829E+00, 0.4129510944388858E-01, 0.8112190955144877E-01, 0.4276761051054951E+00, 0.1315663829494665E+01, 0.9691308092544145E+00, 0.1710650671895379E+00, 0.2322329650880660E+00, 0.8797682289623537E+00, 0.8186215063776209E+00, 0.4211038940233675E+00, 0.4525581409330741E+00, 0.1896031949674164E+00, 0.1581701233715052E+00, 0.1191897606932286E+01, 0.1428762837957623E+01, 0.5405288693957555E+00, 0.3036848136842134E+00, 0.4653859482687157E+00, 0.8283408623519102E+00, 0.5960928726645816E+00, 0.2331591338951825E+00, 0.1116684901463164E+00, 0.2854908146440392E+00, 0.3598944880993349E+00, 0.1860933389154074E+00, 0.5585061130503639E+00, 0.1080120722364922E+01, 0.9846976386969850E+00, 0.4631042046962229E+00, 0.4676068229793028E+00, 0.9942715478654648E+00, 0.1233621533334973E+01, 0.7069779837626068E+00, 0.1822471700779458E+00, 0.3180071212415688E+00, 0.3798865198782122E+00, 0.2441477440283845E+00, 0.7044652781161848E+00, 0.1096126707799853E+01, 0.1337385443370563E+01, 0.9457451890006905E+00, 0.3674861422265960E+00, 0.5196649403773971E+00, 0.4399391431629689E+00, 0.2877815203259632E+00, 0.4888049885267526E+00, 0.7472489810418290E+00, 0.5247059546398414E+00, 0.2662831374385604E+00, 0.3701064434002514E-01, 0.7292878421469419E-01, 0.3589696456182207E-01, 0.9483648767903632E-12, 0.5594353069081231E-15, 0.2117542506917630E-04, 0.1112732565966979E-09, 0.4183353417969536E-03, 0.6252320936560726E-03},
			pp:       1,
			dmin:     1.1127325659669794E-010,
			desig:    0.0000000000000000,
			qmax:     2.1637041623952107,
			ttype:    -4,
			dmin1:    3.1433071595911154E-002,
			dmin2:    3.1433071595911154E-002,
			dn:       1.1127325659669794E-010,
			dn1:      3.5896964560873705E-002,
			dn2:      0.25842281720128102,
			g:        0.0000000000000000,
			tau:      2.1175313795360271E-005,
			nFail:    0,
			iter:     5,
			sigma:    2.1175313795360271E-005,
			nDiv:     106,
			zOut:     []float64{0.2440445008292708E+01, 0.2398080345610006E+01, 0.2094976520226600E-01, 0.4236466279397526E-01, 0.1195399335050165E+01, 0.1206825371451915E+01, 0.4200549016048655E-02, 0.9523728911788614E-02, 0.5236691632680260E+00, 0.5272444803016919E+00, 0.9895328911616120E-03, 0.6252320936560726E-03, 0.8561706005512968E+00, 0.8287928057414093E+00, 0.3940429656773515E-01, 0.2836732781232222E-01, 0.1391235531991410E+01, 0.1189283688611819E+01, 0.1042571673718422E+00, 0.2413561400585997E+00, 0.8502943288943339E+00, 0.6009636865971842E+00, 0.9053227710395735E-01, 0.3535878097802652E+00, 0.2082981609510011E+00, 0.2177085286147829E+00, 0.5123864833424303E+00, 0.8112190955144877E-01, 0.9743424132304999E+00, 0.1315663829494665E+01, 0.1544607000116935E+00, 0.1710650671895379E+00, 0.1146411422862754E+01, 0.8797682289623537E+00, 0.6964571542795012E-01, 0.4211038940233675E+00, 0.1311855086360479E+01, 0.1896031949674164E+00, 0.4911023119923957E+00, 0.1191897606932286E+01, 0.5148125055608023E+00, 0.5405288693957555E+00, 0.5388626806938843E+00, 0.4653859482687157E+00, 0.1688986820057405E+00, 0.5960928726645816E+00, 0.2379466412690434E+00, 0.1116684901463164E+00, 0.6804539597693821E+00, 0.3598944880993349E+00, 0.8082246312519304E+00, 0.5585061130503639E+00, 0.6440798303130841E+00, 0.9846976386969850E+00, 0.8956185534970393E+00, 0.4676068229793028E+00, 0.5202501498046066E+00, 0.1233621533334973E+01, 0.1330768347199243E+00, 0.1822471700779458E+00, 0.9512749631631994E+00, 0.3798865198782122E+00, 0.9903988276741268E+00, 0.7044652781161848E+00, 0.7144727578117591E+00, 0.1337385443370563E+01, 0.2262808998212762E+00, 0.3674861422265960E+00, 0.7024632317571722E+00, 0.4399391431629689E+00, 0.3651136124179467E+00, 0.4888049885267526E+00, 0.1966029864506465E+00, 0.5247059546398414E+00, 0.6757627705811050E-02, 0.3701064434002514E-01, 0.2913933674473832E-01, 0.3589696456182207E-01, 0.2136293938333395E-23, 0.5594353069081231E-15, 0.0000000000000000E+00, 0.1112732565966979E-09, 0.9895328911616120E-03, 0.6252320936560726E-03},
			i0Out:    1,
			n0Out:    21,
			ppOut:    1,
			dminOut:  0.0000000000000000,
			desigOut: -5.1698788284564230E-026,
			sigmaOut: 2.1175425068616867E-005,
			qmaxOut:  2.1637041623952107,
			nFailOut: 1,
			iterOut:  7,
			nDivOut:  150,
			ttypeOut: -15,
			dmin1Out: 2.9139336744737766E-002,
			dmin2Out: 4.9426557292086552E-002,
			dnOut:    -2.0808762284537102E-024,
			dn1Out:   2.9139336744737766E-002,
			dn2Out:   0.15959234211062134,
			gOut:     0.0000000000000000,
			tauOut:   1.1127325659669789E-010,
		},
		{
			i0:       1,
			n0:       21,
			z:        []float64{0.2440445008292708E+01, 0.2398080345610006E+01, 0.2094976520226600E-01, 0.4236466279397526E-01, 0.1195399335050165E+01, 0.1206825371451915E+01, 0.4200549016048655E-02, 0.9523728911788614E-02, 0.5236691632680260E+00, 0.5272444803016919E+00, 0.9895328911616120E-03, 0.6252320936560726E-03, 0.8561706005512968E+00, 0.8287928057414093E+00, 0.3940429656773515E-01, 0.2836732781232222E-01, 0.1391235531991410E+01, 0.1189283688611819E+01, 0.1042571673718422E+00, 0.2413561400585997E+00, 0.8502943288943339E+00, 0.6009636865971842E+00, 0.9053227710395735E-01, 0.3535878097802652E+00, 0.2082981609510011E+00, 0.2177085286147829E+00, 0.5123864833424303E+00, 0.8112190955144877E-01, 0.9743424132304999E+00, 0.1315663829494665E+01, 0.1544607000116935E+00, 0.1710650671895379E+00, 0.1146411422862754E+01, 0.8797682289623537E+00, 0.6964571542795012E-01, 0.4211038940233675E+00, 0.1311855086360479E+01, 0.1896031949674164E+00, 0.4911023119923957E+00, 0.1191897606932286E+01, 0.5148125055608023E+00, 0.5405288693957555E+00, 0.5388626806938843E+00, 0.4653859482687157E+00, 0.1688986820057405E+00, 0.5960928726645816E+00, 0.2379466412690434E+00, 0.1116684901463164E+00, 0.6804539597693821E+00, 0.3598944880993349E+00, 0.8082246312519304E+00, 0.5585061130503639E+00, 0.6440798303130841E+00, 0.9846976386969850E+00, 0.8956185534970393E+00, 0.4676068229793028E+00, 0.5202501498046066E+00, 0.1233621533334973E+01, 0.1330768347199243E+00, 0.1822471700779458E+00, 0.9512749631631994E+00, 0.3798865198782122E+00, 0.9903988276741268E+00, 0.7044652781161848E+00, 0.7144727578117591E+00, 0.1337385443370563E+01, 0.2262808998212762E+00, 0.3674861422265960E+00, 0.7024632317571722E+00, 0.4399391431629689E+00, 0.3651136124179467E+00, 0.4888049885267526E+00, 0.1966029864506465E+00, 0.5247059546398414E+00, 0.6757627705811050E-02, 0.3701064434002514E-01, 0.2913933674473832E-01, 0.3589696456182207E-01, 0.2136293938333395E-23, 0.5594353069081231E-15, 0.0000000000000000E+00, 0.1112732565966979E-09, 0.9895328911616120E-03, 0.6252320936560726E-03},
			pp:       0,
			dmin:     0.0000000000000000,
			desig:    -5.1698788284564230E-026,
			qmax:     2.1637041623952107,
			ttype:    -15,
			dmin1:    2.9139336744737766E-002,
			dmin2:    4.9426557292086552E-002,
			dn:       -2.0808762284537102E-024,
			dn1:      2.9139336744737766E-002,
			dn2:      0.15959234211062134,
			g:        0.0000000000000000,
			tau:      1.1127325659669789E-010,
			nFail:    1,
			iter:     7,
			sigma:    2.1175425068616867E-005,
			nDiv:     150,
			zOut:     []float64{0.2440445008292708E+01, 0.2461394773494974E+01, 0.2094976520226600E-01, 0.1017444891892999E-01, 0.1195399335050165E+01, 0.1189425435147283E+01, 0.4200549016048655E-02, 0.1849378635683999E-02, 0.5236691632680260E+00, 0.5228093175235037E+00, 0.9895328911616120E-03, 0.1620493249248586E-02, 0.8561706005512968E+00, 0.8939544038697832E+00, 0.3940429656773515E-01, 0.6132377362967349E-01, 0.1391235531991410E+01, 0.1434168925733579E+01, 0.1042571673718422E+00, 0.6181229879703373E-01, 0.8502943288943339E+00, 0.8790143072012576E+00, 0.9053227710395735E-01, 0.2145324219750511E-01, 0.2082981609510011E+00, 0.6992314020959263E+00, 0.5123864833424303E+00, 0.7139837844669097E+00, 0.9743424132304999E+00, 0.4148193287752837E+00, 0.1544607000116935E+00, 0.4268738185358478E+00, 0.1146411422862754E+01, 0.7891833197548568E+00, 0.6964571542795012E-01, 0.1157716892137957E+00, 0.1311855086360479E+01, 0.1687185709139079E+01, 0.4911023119923957E+00, 0.1498504938454686E+00, 0.5148125055608023E+00, 0.9038246924092180E+00, 0.5388626806938843E+00, 0.1006978425303630E+00, 0.1688986820057405E+00, 0.3061474807444209E+00, 0.2379466412690434E+00, 0.5288684194677825E+00, 0.6804539597693821E+00, 0.9598101715535300E+00, 0.8082246312519304E+00, 0.5423584775195998E+00, 0.6440798303130841E+00, 0.9973399062905237E+00, 0.8956185534970393E+00, 0.4671884516860899E+00, 0.5202501498046066E+00, 0.1861385328384410E+00, 0.1330768347199243E+00, 0.6800991665489665E+00, 0.9512749631631994E+00, 0.1261574624288360E+01, 0.9903988276741268E+00, 0.5608966509936130E+00, 0.7144727578117591E+00, 0.3798570066394223E+00, 0.2262808998212762E+00, 0.4184574968871406E+00, 0.7024632317571722E+00, 0.6491193472879784E+00, 0.3651136124179467E+00, 0.1105843276664904E+00, 0.1966029864506465E+00, 0.9277628648996712E-01, 0.6757627705811050E-02, 0.2122447413720272E-02, 0.2913933674473832E-01, 0.2701688933101806E-01, 0.2136293938333395E-23, 0.0000000000000000E+00, 0.0000000000000000E+00, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    21,
			ppOut:    0,
			dminOut:  0.0000000000000000,
			desigOut: -5.1698788284564230E-026,
			sigmaOut: 2.1175425068616867E-005,
			qmaxOut:  2.1637041623952107,
			nFailOut: 1,
			iterOut:  8,
			nDivOut:  172,
			ttypeOut: -1,
			dmin1Out: 2.7016889331018056E-002,
			dmin2Out: 5.3061698118516694E-002,
			dnOut:    0.0000000000000000,
			dn1Out:   2.7016889331018056E-002,
			dn2Out:   8.6018658784156071E-002,
			gOut:     0.0000000000000000,
			tauOut:   0.0000000000000000,
		},
		{
			i0:       1,
			n0:       21,
			z:        []float64{0.2440445008292708E+01, 0.2461394773494974E+01, 0.2094976520226600E-01, 0.1017444891892999E-01, 0.1195399335050165E+01, 0.1189425435147283E+01, 0.4200549016048655E-02, 0.1849378635683999E-02, 0.5236691632680260E+00, 0.5228093175235037E+00, 0.9895328911616120E-03, 0.1620493249248586E-02, 0.8561706005512968E+00, 0.8939544038697832E+00, 0.3940429656773515E-01, 0.6132377362967349E-01, 0.1391235531991410E+01, 0.1434168925733579E+01, 0.1042571673718422E+00, 0.6181229879703373E-01, 0.8502943288943339E+00, 0.8790143072012576E+00, 0.9053227710395735E-01, 0.2145324219750511E-01, 0.2082981609510011E+00, 0.6992314020959263E+00, 0.5123864833424303E+00, 0.7139837844669097E+00, 0.9743424132304999E+00, 0.4148193287752837E+00, 0.1544607000116935E+00, 0.4268738185358478E+00, 0.1146411422862754E+01, 0.7891833197548568E+00, 0.6964571542795012E-01, 0.1157716892137957E+00, 0.1311855086360479E+01, 0.1687185709139079E+01, 0.4911023119923957E+00, 0.1498504938454686E+00, 0.5148125055608023E+00, 0.9038246924092180E+00, 0.5388626806938843E+00, 0.1006978425303630E+00, 0.1688986820057405E+00, 0.3061474807444209E+00, 0.2379466412690434E+00, 0.5288684194677825E+00, 0.6804539597693821E+00, 0.9598101715535300E+00, 0.8082246312519304E+00, 0.5423584775195998E+00, 0.6440798303130841E+00, 0.9973399062905237E+00, 0.8956185534970393E+00, 0.4671884516860899E+00, 0.5202501498046066E+00, 0.1861385328384410E+00, 0.1330768347199243E+00, 0.6800991665489665E+00, 0.9512749631631994E+00, 0.1261574624288360E+01, 0.9903988276741268E+00, 0.5608966509936130E+00, 0.7144727578117591E+00, 0.3798570066394223E+00, 0.2262808998212762E+00, 0.4184574968871406E+00, 0.7024632317571722E+00, 0.6491193472879784E+00, 0.3651136124179467E+00, 0.1105843276664904E+00, 0.1966029864506465E+00, 0.9277628648996712E-01, 0.6757627705811050E-02, 0.2122447413720272E-02, 0.2913933674473832E-01, 0.2701688933101806E-01, 0.2136293938333395E-23, 0.0000000000000000E+00, 0.0000000000000000E+00, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       1,
			dmin:     0.0000000000000000,
			desig:    -5.1698788284564230E-026,
			qmax:     2.1637041623952107,
			ttype:    -1,
			dmin1:    2.7016889331018056E-002,
			dmin2:    5.3061698118516694E-002,
			dn:       0.0000000000000000,
			dn1:      2.7016889331018056E-002,
			dn2:      8.6018658784156071E-002,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    1,
			iter:     8,
			sigma:    2.1175425068616867E-005,
			nDiv:     172,
			zOut:     []float64{0.2471569222413904E+01, 0.2461394773494974E+01, 0.4896382518051712E-02, 0.1017444891892999E-01, 0.1186378431264915E+01, 0.1189425435147283E+01, 0.8149780515932184E-03, 0.1849378635683999E-02, 0.5236148327211592E+00, 0.5228093175235037E+00, 0.2766627272719901E-02, 0.1620493249248586E-02, 0.9525115502267366E+00, 0.8939544038697832E+00, 0.9233342160256496E-01, 0.6132377362967349E-01, 0.1403647802928048E+01, 0.1434168925733579E+01, 0.3870906568602875E-01, 0.6181229879703373E-01, 0.8617584837127339E+00, 0.8790143072012576E+00, 0.1740717486950262E-01, 0.2145324219750511E-01, 0.1395808011693333E+01, 0.6992314020959263E+00, 0.2121884039551361E+00, 0.7139837844669097E+00, 0.6295047433559955E+00, 0.4148193287752837E+00, 0.5351535485381410E+00, 0.4268738185358478E+00, 0.3698014604305115E+00, 0.7891833197548568E+00, 0.5281978587564573E+00, 0.1157716892137957E+00, 0.1308838344228090E+01, 0.1687185709139079E+01, 0.1034799882693896E+00, 0.1498504938454686E+00, 0.9010425466701916E+00, 0.9038246924092180E+00, 0.3421413441684364E-01, 0.1006978425303630E+00, 0.8008017657953598E+00, 0.3061474807444209E+00, 0.6338813300623194E+00, 0.5288684194677825E+00, 0.8682873190108105E+00, 0.9598101715535300E+00, 0.6229686202966810E+00, 0.5423584775195998E+00, 0.8415597376799326E+00, 0.9973399062905237E+00, 0.1033340463692495E+00, 0.4671884516860899E+00, 0.7629036530181579E+00, 0.1861385328384410E+00, 0.1124645093942705E+01, 0.6800991665489665E+00, 0.6978261813392677E+00, 0.1261574624288360E+01, 0.3053203341720497E+00, 0.5608966509936130E+00, 0.4929941693545132E+00, 0.3798570066394223E+00, 0.5509778292160957E+00, 0.4184574968871406E+00, 0.2087258457383731E+00, 0.6491193472879784E+00, 0.4915348757406203E-01, 0.1105843276664904E+00, 0.4574524632962537E-01, 0.9277628648996712E-01, 0.1253505697055357E-02, 0.2122447413720272E-02, 0.2576338363396270E-01, 0.2701688933101806E-01, 0.8149780515932184E-03, 0.0000000000000000E+00, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    20,
			ppOut:    1,
			dminOut:  2.5763383633962696E-002,
			desigOut: -5.1698788284564230E-026,
			sigmaOut: 2.1175425068616867E-005,
			qmaxOut:  2.1637041623952107,
			nFailOut: 1,
			iterOut:  9,
			nDivOut:  193,
			ttypeOut: -1,
			dmin1Out: 4.3622798915905092E-002,
			dmin2Out: 7.4536672467372611E-002,
			dnOut:    2.5763383633962696E-002,
			dn1Out:   4.3622798915905092E-002,
			dn2Out:   9.8141518071882677E-002,
			gOut:     0.0000000000000000,
			tauOut:   0.0000000000000000,
		},
		{
			i0:       1,
			n0:       20,
			z:        []float64{0.2471569222413904E+01, 0.2461394773494974E+01, 0.4896382518051712E-02, 0.1017444891892999E-01, 0.1186378431264915E+01, 0.1189425435147283E+01, 0.8149780515932184E-03, 0.1849378635683999E-02, 0.5236148327211592E+00, 0.5228093175235037E+00, 0.2766627272719901E-02, 0.1620493249248586E-02, 0.9525115502267366E+00, 0.8939544038697832E+00, 0.9233342160256496E-01, 0.6132377362967349E-01, 0.1403647802928048E+01, 0.1434168925733579E+01, 0.3870906568602875E-01, 0.6181229879703373E-01, 0.8617584837127339E+00, 0.8790143072012576E+00, 0.1740717486950262E-01, 0.2145324219750511E-01, 0.1395808011693333E+01, 0.6992314020959263E+00, 0.2121884039551361E+00, 0.7139837844669097E+00, 0.6295047433559955E+00, 0.4148193287752837E+00, 0.5351535485381410E+00, 0.4268738185358478E+00, 0.3698014604305115E+00, 0.7891833197548568E+00, 0.5281978587564573E+00, 0.1157716892137957E+00, 0.1308838344228090E+01, 0.1687185709139079E+01, 0.1034799882693896E+00, 0.1498504938454686E+00, 0.9010425466701916E+00, 0.9038246924092180E+00, 0.3421413441684364E-01, 0.1006978425303630E+00, 0.8008017657953598E+00, 0.3061474807444209E+00, 0.6338813300623194E+00, 0.5288684194677825E+00, 0.8682873190108105E+00, 0.9598101715535300E+00, 0.6229686202966810E+00, 0.5423584775195998E+00, 0.8415597376799326E+00, 0.9973399062905237E+00, 0.1033340463692495E+00, 0.4671884516860899E+00, 0.7629036530181579E+00, 0.1861385328384410E+00, 0.1124645093942705E+01, 0.6800991665489665E+00, 0.6978261813392677E+00, 0.1261574624288360E+01, 0.3053203341720497E+00, 0.5608966509936130E+00, 0.4929941693545132E+00, 0.3798570066394223E+00, 0.5509778292160957E+00, 0.4184574968871406E+00, 0.2087258457383731E+00, 0.6491193472879784E+00, 0.4915348757406203E-01, 0.1105843276664904E+00, 0.4574524632962537E-01, 0.9277628648996712E-01, 0.1253505697055357E-02, 0.2122447413720272E-02, 0.2576338363396270E-01, 0.2701688933101806E-01, 0.8149780515932184E-03, 0.1620493249248586E-02, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       0,
			dmin:     2.5763383633962696E-002,
			desig:    -5.1698788284564230E-026,
			qmax:     2.4715692224139039,
			ttype:    -1,
			dmin1:    4.3622798915905092E-002,
			dmin2:    7.4536672467372611E-002,
			dn:       2.5763383633962696E-002,
			dn1:      4.3622798915905092E-002,
			dn2:      9.8141518071882677E-002,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    1,
			iter:     9,
			sigma:    2.1175425068616867E-005,
			nDiv:     193,
			zOut:     []float64{0.2471569222413904E+01, 0.2471445466333236E+01, 0.4896382518051712E-02, 0.2350431231346416E-02, 0.1186378431264915E+01, 0.1179822839486443E+01, 0.8149780515932184E-03, 0.3616937915375072E-03, 0.5236148327211592E+00, 0.5209996276036221E+00, 0.2766627272719901E-02, 0.5058054349403302E-02, 0.9525115502267366E+00, 0.1034766778881179E+01, 0.9233342160256496E-01, 0.1252490967185870E+00, 0.1403647802928048E+01, 0.1312087633296770E+01, 0.3870906568602875E-01, 0.2542350442532051E-01, 0.8617584837127339E+00, 0.8487220155581966E+00, 0.1740717486950262E-01, 0.2862783537884150E-01, 0.1395808011693333E+01, 0.1574348441670908E+01, 0.2121884039551361E+00, 0.8484373804386666E-01, 0.6295047433559955E+00, 0.1074794415251550E+01, 0.5351535485381410E+00, 0.1841287608083240E+00, 0.3698014604305115E+00, 0.7088504197799252E+00, 0.5281978587564573E+00, 0.9752771411128711E+00, 0.1308838344228090E+01, 0.4320210527858890E+00, 0.1034799882693896E+00, 0.2158225196628609E+00, 0.9010425466701916E+00, 0.7144140228254550E+00, 0.3421413441684364E-01, 0.3835134583138245E-01, 0.8008017657953598E+00, 0.1391311611427577E+01, 0.6338813300623194E+00, 0.3955915526975877E+00, 0.8682873190108105E+00, 0.1090644248011184E+01, 0.6229686202966810E+00, 0.4806932321292802E+00, 0.8415597376799326E+00, 0.4591804133211825E+00, 0.1033340463692495E+00, 0.1716839812178710E+00, 0.7629036530181579E+00, 0.1710844627144272E+01, 0.1124645093942705E+01, 0.4587247601659613E+00, 0.6978261813392677E+00, 0.5394016167466366E+00, 0.3053203341720497E+00, 0.2790520826393697E+00, 0.4929941693545132E+00, 0.7598997773325197E+00, 0.5509778292160957E+00, 0.1513401067044909E+00, 0.2087258457383731E+00, 0.1015190880092246E+00, 0.4915348757406203E-01, 0.2214892234681356E-01, 0.4574524632962537E-01, 0.1982969108114764E-01, 0.1253505697055357E-02, 0.1628595626045726E-02, 0.2576338363396270E-01, 0.1911464940919745E-01, 0.8149780515932184E-03, 0.3616937915375072E-03, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    20,
			ppOut:    0,
			dminOut:  1.8576185384092288E-002,
			desigOut: 2.6427422784455342E-019,
			sigmaOut: 5.0413140237881371E-003,
			qmaxOut:  2.4715692224139039,
			nFailOut: 2,
			iterOut:  11,
			nDivOut:  235,
			ttypeOut: -15,
			dmin1Out: 1.8576185384092288E-002,
			dmin2Out: 5.2365600435162571E-002,
			dnOut:    1.9114649409197451E-002,
			dn1Out:   1.8576185384092288E-002,
			dn2Out:   5.2365600435162571E-002,
			gOut:     0.0000000000000000,
			tauOut:   5.0201385987195205E-003,
		},
		{
			i0:       1,
			n0:       20,
			z:        []float64{0.2471569222413904E+01, 0.2471445466333236E+01, 0.4896382518051712E-02, 0.2350431231346416E-02, 0.1186378431264915E+01, 0.1179822839486443E+01, 0.8149780515932184E-03, 0.3616937915375072E-03, 0.5236148327211592E+00, 0.5209996276036221E+00, 0.2766627272719901E-02, 0.5058054349403302E-02, 0.9525115502267366E+00, 0.1034766778881179E+01, 0.9233342160256496E-01, 0.1252490967185870E+00, 0.1403647802928048E+01, 0.1312087633296770E+01, 0.3870906568602875E-01, 0.2542350442532051E-01, 0.8617584837127339E+00, 0.8487220155581966E+00, 0.1740717486950262E-01, 0.2862783537884150E-01, 0.1395808011693333E+01, 0.1574348441670908E+01, 0.2121884039551361E+00, 0.8484373804386666E-01, 0.6295047433559955E+00, 0.1074794415251550E+01, 0.5351535485381410E+00, 0.1841287608083240E+00, 0.3698014604305115E+00, 0.7088504197799252E+00, 0.5281978587564573E+00, 0.9752771411128711E+00, 0.1308838344228090E+01, 0.4320210527858890E+00, 0.1034799882693896E+00, 0.2158225196628609E+00, 0.9010425466701916E+00, 0.7144140228254550E+00, 0.3421413441684364E-01, 0.3835134583138245E-01, 0.8008017657953598E+00, 0.1391311611427577E+01, 0.6338813300623194E+00, 0.3955915526975877E+00, 0.8682873190108105E+00, 0.1090644248011184E+01, 0.6229686202966810E+00, 0.4806932321292802E+00, 0.8415597376799326E+00, 0.4591804133211825E+00, 0.1033340463692495E+00, 0.1716839812178710E+00, 0.7629036530181579E+00, 0.1710844627144272E+01, 0.1124645093942705E+01, 0.4587247601659613E+00, 0.6978261813392677E+00, 0.5394016167466366E+00, 0.3053203341720497E+00, 0.2790520826393697E+00, 0.4929941693545132E+00, 0.7598997773325197E+00, 0.5509778292160957E+00, 0.1513401067044909E+00, 0.2087258457383731E+00, 0.1015190880092246E+00, 0.4915348757406203E-01, 0.2214892234681356E-01, 0.4574524632962537E-01, 0.1982969108114764E-01, 0.1253505697055357E-02, 0.1628595626045726E-02, 0.2576338363396270E-01, 0.1911464940919745E-01, 0.8149780515932184E-03, 0.3616937915375072E-03, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       1,
			dmin:     1.8576185384092288E-002,
			desig:    2.6427422784455342E-019,
			qmax:     2.4715692224139039,
			ttype:    -15,
			dmin1:    1.8576185384092288E-002,
			dmin2:    5.2365600435162571E-002,
			dn:       1.9114649409197451E-002,
			dn1:      1.8576185384092288E-002,
			dn2:      5.2365600435162571E-002,
			g:        0.0000000000000000,
			tau:      5.0201385987195205E-003,
			nFail:    2,
			iter:     11,
			sigma:    5.0413140237881371E-003,
			nDiv:     235,
			zOut:     []float64{0.2468318984233055E+01, 0.2471445466333236E+01, 0.1123474100024551E-02, 0.2350431231346416E-02, 0.1173584145846428E+01, 0.1179822839486443E+01, 0.1605699355811189E-03, 0.3616937915375072E-03, 0.5204201986859162E+00, 0.5209996276036221E+00, 0.1005707814522541E-01, 0.5058054349403302E-02, 0.1144481884123012E+01, 0.1034766778881179E+01, 0.1435914304680996E+00, 0.1252490967185870E+00, 0.1188442793922463E+01, 0.1312087633296770E+01, 0.1815610143690141E-01, 0.2542350442532051E-01, 0.8537168361686087E+00, 0.8487220155581966E+00, 0.5279290053521807E-01, 0.2862783537884150E-01, 0.1600922365848029E+01, 0.1574348441670908E+01, 0.5696064828871891E-01, 0.8484373804386666E-01, 0.1196485614439627E+01, 0.1074794415251550E+01, 0.1090859328498209E+00, 0.1841287608083240E+00, 0.1569564714711448E+01, 0.7088504197799252E+00, 0.2684440171930437E+00, 0.9752771411128711E+00, 0.3739226419241781E+00, 0.4320210527858890E+00, 0.4123490187575627E+00, 0.2158225196628609E+00, 0.3349394365677468E+00, 0.7144140228254550E+00, 0.1593084209965356E+00, 0.3835134583138245E-01, 0.1622117829797102E+01, 0.1391311611427577E+01, 0.2659792301064862E+00, 0.3955915526975877E+00, 0.1299881336702450E+01, 0.1090644248011184E+01, 0.1698038973078534E+00, 0.4806932321292802E+00, 0.4555835838996722E+00, 0.4591804133211825E+00, 0.6447216871142054E+00, 0.1716839812178710E+00, 0.1519370786864500E+01, 0.1710844627144272E+01, 0.1628548339973444E+00, 0.4587247601659613E+00, 0.6501219520571339E+00, 0.5394016167466366E+00, 0.3261720586281595E+00, 0.2790520826393697E+00, 0.5795909120773233E+00, 0.7598997773325197E+00, 0.2650819619788820E-01, 0.1513401067044909E+00, 0.9168290082662192E-01, 0.1015190880092246E+00, 0.4790492926791300E-02, 0.2214892234681356E-01, 0.1119088044887405E-01, 0.1982969108114764E-01, 0.2781732372482683E-02, 0.1628595626045726E-02, 0.1085600370518675E-01, 0.1911464940919745E-01, 0.1605699355811189E-03, 0.3616937915375072E-03, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    20,
			ppOut:    1,
			dminOut:  9.5622848228283271E-003,
			desigOut: -6.0308751014385013E-019,
			sigmaOut: 1.0518227355316156E-002,
			qmaxOut:  2.4715692224139039,
			nFailOut: 2,
			iterOut:  12,
			nDivOut:  256,
			ttypeOut: -4,
			dmin1Out: 9.5622848228283271E-003,
			dmin2Out: 6.9533978479808370E-002,
			dnOut:    1.0856003705186750E-002,
			dn1Out:   9.5622848228283271E-003,
			dn2Out:   6.9533978479808370E-002,
			gOut:     0.0000000000000000,
			tauOut:   5.4769133315280185E-003,
		},
		{
			i0:       1,
			n0:       20,
			z:        []float64{0.2468318984233055E+01, 0.2471445466333236E+01, 0.1123474100024551E-02, 0.2350431231346416E-02, 0.1173584145846428E+01, 0.1179822839486443E+01, 0.1605699355811189E-03, 0.3616937915375072E-03, 0.5204201986859162E+00, 0.5209996276036221E+00, 0.1005707814522541E-01, 0.5058054349403302E-02, 0.1144481884123012E+01, 0.1034766778881179E+01, 0.1435914304680996E+00, 0.1252490967185870E+00, 0.1188442793922463E+01, 0.1312087633296770E+01, 0.1815610143690141E-01, 0.2542350442532051E-01, 0.8537168361686087E+00, 0.8487220155581966E+00, 0.5279290053521807E-01, 0.2862783537884150E-01, 0.1600922365848029E+01, 0.1574348441670908E+01, 0.5696064828871891E-01, 0.8484373804386666E-01, 0.1196485614439627E+01, 0.1074794415251550E+01, 0.1090859328498209E+00, 0.1841287608083240E+00, 0.1569564714711448E+01, 0.7088504197799252E+00, 0.2684440171930437E+00, 0.9752771411128711E+00, 0.3739226419241781E+00, 0.4320210527858890E+00, 0.4123490187575627E+00, 0.2158225196628609E+00, 0.3349394365677468E+00, 0.7144140228254550E+00, 0.1593084209965356E+00, 0.3835134583138245E-01, 0.1622117829797102E+01, 0.1391311611427577E+01, 0.2659792301064862E+00, 0.3955915526975877E+00, 0.1299881336702450E+01, 0.1090644248011184E+01, 0.1698038973078534E+00, 0.4806932321292802E+00, 0.4555835838996722E+00, 0.4591804133211825E+00, 0.6447216871142054E+00, 0.1716839812178710E+00, 0.1519370786864500E+01, 0.1710844627144272E+01, 0.1628548339973444E+00, 0.4587247601659613E+00, 0.6501219520571339E+00, 0.5394016167466366E+00, 0.3261720586281595E+00, 0.2790520826393697E+00, 0.5795909120773233E+00, 0.7598997773325197E+00, 0.2650819619788820E-01, 0.1513401067044909E+00, 0.9168290082662192E-01, 0.1015190880092246E+00, 0.4790492926791300E-02, 0.2214892234681356E-01, 0.1119088044887405E-01, 0.1982969108114764E-01, 0.2781732372482683E-02, 0.1628595626045726E-02, 0.1085600370518675E-01, 0.1911464940919745E-01, 0.1605699355811189E-03, 0.3616937915375072E-03, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       0,
			dmin:     9.5622848228283271E-003,
			desig:    -6.0308751014385013E-019,
			qmax:     2.4715692224139039,
			ttype:    -4,
			dmin1:    9.5622848228283271E-003,
			dmin2:    6.9533978479808370E-002,
			dn:       1.0856003705186750E-002,
			dn1:      9.5622848228283271E-003,
			dn2:      6.9533978479808370E-002,
			g:        0.0000000000000000,
			tau:      5.4769133315280185E-003,
			nFail:    2,
			iter:     12,
			sigma:    1.0518227355316156E-002,
			nDiv:     256,
			zOut:     []float64{0.2468318984233055E+01, 0.2464320851971913E+01, 0.1123474100024551E-02, 0.5350323562789559E-03, 0.1173584145846428E+01, 0.1168088077064565E+01, 0.1605699355811189E-03, 0.7153898701552432E-04, 0.5204201986859162E+00, 0.5252841314829605E+00, 0.1005707814522541E-01, 0.2191222436498315E-01, 0.1144481884123012E+01, 0.1261039483864963E+01, 0.1435914304680996E+00, 0.1353250258951489E+00, 0.1188442793922463E+01, 0.1066152263103050E+01, 0.1815610143690141E-01, 0.1453842008528346E-01, 0.8537168361686087E+00, 0.8868497102573779E+00, 0.5279290053521807E-01, 0.9530062900995111E-01, 0.1600922365848029E+01, 0.1557460778765631E+01, 0.5696064828871891E-01, 0.4375878814786067E-01, 0.1196485614439627E+01, 0.1256691152780422E+01, 0.1090859328498209E+00, 0.1362446379077657E+00, 0.1569564714711448E+01, 0.1696642487635560E+01, 0.2684440171930437E+00, 0.5916231430550117E-01, 0.3739226419241781E+00, 0.7219877400150740E+00, 0.4123490187575627E+00, 0.1912940350054112E+00, 0.3349394365677468E+00, 0.2978322161977056E+00, 0.1593084209965356E+00, 0.8676597630518320E+00, 0.1622117829797102E+01, 0.1015315690490590E+01, 0.2659792301064862E+00, 0.3405260456467969E+00, 0.1299881336702450E+01, 0.1124037582002341E+01, 0.1698038973078534E+00, 0.6882320425428856E-01, 0.4555835838996722E+00, 0.1026360460398424E+01, 0.6447216871142054E+00, 0.9544125430154021E+00, 0.1519370786864500E+01, 0.7226914714852769E+00, 0.1628548339973444E+00, 0.1465016632377001E+00, 0.6501219520571339E+00, 0.8246707410864278E+00, 0.3261720586281595E+00, 0.2292385937027206E+00, 0.5795909120773233E+00, 0.3717389082113253E+00, 0.2650819619788820E-01, 0.6537783023029759E-02, 0.9168290082662192E-01, 0.8481400436921797E-01, 0.4790492926791300E-02, 0.6320870478125323E-03, 0.1119088044887405E-01, 0.8218919412378699E-02, 0.2781732372482683E-02, 0.3674266095981827E-02, 0.1085600370518675E-01, 0.2060131248039419E-02, 0.1605699355811189E-03, 0.7153898701552432E-04, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    20,
			ppOut:    0,
			dminOut:  2.0601312480394186E-003,
			desigOut: 0.0000000000000000,
			sigmaOut: 1.5639833716481661E-002,
			qmaxOut:  2.4715692224139039,
			nFailOut: 2,
			iterOut:  13,
			nDivOut:  277,
			ttypeOut: -4,
			dmin1Out: 5.4371870398960158E-003,
			dmin2Out: 8.0023511442426670E-002,
			dnOut:    2.0601312480394186E-003,
			dn1Out:   5.4371870398960158E-003,
			dn2Out:   8.0023511442426670E-002,
			gOut:     0.0000000000000000,
			tauOut:   5.1216063611655054E-003,
		},
		{
			i0:       1,
			n0:       20,
			z:        []float64{0.2468318984233055E+01, 0.2464320851971913E+01, 0.1123474100024551E-02, 0.5350323562789559E-03, 0.1173584145846428E+01, 0.1168088077064565E+01, 0.1605699355811189E-03, 0.7153898701552432E-04, 0.5204201986859162E+00, 0.5252841314829605E+00, 0.1005707814522541E-01, 0.2191222436498315E-01, 0.1144481884123012E+01, 0.1261039483864963E+01, 0.1435914304680996E+00, 0.1353250258951489E+00, 0.1188442793922463E+01, 0.1066152263103050E+01, 0.1815610143690141E-01, 0.1453842008528346E-01, 0.8537168361686087E+00, 0.8868497102573779E+00, 0.5279290053521807E-01, 0.9530062900995111E-01, 0.1600922365848029E+01, 0.1557460778765631E+01, 0.5696064828871891E-01, 0.4375878814786067E-01, 0.1196485614439627E+01, 0.1256691152780422E+01, 0.1090859328498209E+00, 0.1362446379077657E+00, 0.1569564714711448E+01, 0.1696642487635560E+01, 0.2684440171930437E+00, 0.5916231430550117E-01, 0.3739226419241781E+00, 0.7219877400150740E+00, 0.4123490187575627E+00, 0.1912940350054112E+00, 0.3349394365677468E+00, 0.2978322161977056E+00, 0.1593084209965356E+00, 0.8676597630518320E+00, 0.1622117829797102E+01, 0.1015315690490590E+01, 0.2659792301064862E+00, 0.3405260456467969E+00, 0.1299881336702450E+01, 0.1124037582002341E+01, 0.1698038973078534E+00, 0.6882320425428856E-01, 0.4555835838996722E+00, 0.1026360460398424E+01, 0.6447216871142054E+00, 0.9544125430154021E+00, 0.1519370786864500E+01, 0.7226914714852769E+00, 0.1628548339973444E+00, 0.1465016632377001E+00, 0.6501219520571339E+00, 0.8246707410864278E+00, 0.3261720586281595E+00, 0.2292385937027206E+00, 0.5795909120773233E+00, 0.3717389082113253E+00, 0.2650819619788820E-01, 0.6537783023029759E-02, 0.9168290082662192E-01, 0.8481400436921797E-01, 0.4790492926791300E-02, 0.6320870478125323E-03, 0.1119088044887405E-01, 0.8218919412378699E-02, 0.2781732372482683E-02, 0.3674266095981827E-02, 0.1085600370518675E-01, 0.2060131248039419E-02, 0.1605699355811189E-03, 0.7153898701552432E-04, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       1,
			dmin:     2.0601312480394186E-003,
			desig:    0.0000000000000000,
			qmax:     2.4715692224139039,
			ttype:    -4,
			dmin1:    5.4371870398960158E-003,
			dmin2:    8.0023511442426670E-002,
			dn:       2.0601312480394186E-003,
			dn1:      5.4371870398960158E-003,
			dn2:      8.0023511442426670E-002,
			g:        0.0000000000000000,
			tau:      5.1216063611655054E-003,
			nFail:    2,
			iter:     13,
			sigma:    1.5639833716481661E-002,
			nDiv:     277,
			zOut:     []float64{0.2463574096511276E+01, 0.2464320851971913E+01, 0.2536822079344948E-03, 0.5350323562789559E-03, 0.1166624146026729E+01, 0.1168088077064565E+01, 0.3221114082852138E-04, 0.7153898701552432E-04, 0.5458823568901986E+00, 0.5252841314829605E+00, 0.5061929508212644E-01, 0.2191222436498315E-01, 0.1344463426861069E+01, 0.1261039483864963E+01, 0.1073120173669855E+00, 0.1353250258951489E+00, 0.9720968780044319E+00, 0.1066152263103050E+01, 0.1326348631702415E-01, 0.1453842008528346E-01, 0.9676050651333883E+00, 0.8868497102573779E+00, 0.1533962535161303E+00, 0.9530062900995111E-01, 0.1446541525580445E+01, 0.1557460778765631E+01, 0.3801569533217738E-01, 0.4375878814786067E-01, 0.1353638307539094E+01, 0.1256691152780422E+01, 0.1707682473962209E+00, 0.1362446379077657E+00, 0.1583754766727924E+01, 0.1696642487635560E+01, 0.2697037855661164E-01, 0.5916231430550117E-01, 0.8850296086469572E+00, 0.7219877400150740E+00, 0.6437471225190403E-01, 0.1912940350054112E+00, 0.1099835479180717E+01, 0.2978322161977056E+00, 0.8009821360646626E+00, 0.8676597630518320E+00, 0.5535778122558079E+00, 0.1015315690490590E+01, 0.6914368034330997E+00, 0.3405260456467969E+00, 0.5001421950066134E+00, 0.1124037582002341E+01, 0.1412346654806686E+00, 0.6882320425428856E-01, 0.1838256550116241E+01, 0.1026360460398424E+01, 0.3752173792456719E+00, 0.9544125430154021E+00, 0.4926939676603885E+00, 0.7226914714852769E+00, 0.2452143584512202E+00, 0.1465016632377001E+00, 0.8074131885210117E+00, 0.8246707410864278E+00, 0.1055431169003394E+00, 0.2292385937027206E+00, 0.2714517865170992E+00, 0.3717389082113253E+00, 0.2042703660177667E-02, 0.6537783023029759E-02, 0.8212159993993635E-01, 0.8481400436921797E-01, 0.6326073178529442E-04, 0.6320870478125323E-03, 0.1054813695965874E-01, 0.8218919412378699E-02, 0.7176120699696391E-03, 0.3674266095981827E-02, 0.6073136115328898E-04, 0.2060131248039419E-02, 0.3221114082852138E-04, 0.7153898701552432E-04, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    20,
			ppOut:    1,
			dminOut:  6.0731361153288982E-005,
			desigOut: 1.7347234759768071E-018,
			sigmaOut: 1.6921621533398150E-002,
			qmaxOut:  2.4715692224139039,
			nFailOut: 2,
			iterOut:  14,
			nDivOut:  298,
			ttypeOut: -2,
			dmin1Out: 6.8738708636769136E-003,
			dmin2Out: 8.1489512892123819E-002,
			dnOut:    6.0731361153288982E-005,
			dn1Out:   6.8738708636769136E-003,
			dn2Out:   8.1489512892123819E-002,
			gOut:     0.0000000000000000,
			tauOut:   1.2817878169164906E-003,
		},
		{
			i0:       1,
			n0:       20,
			z:        []float64{0.2463574096511276E+01, 0.2464320851971913E+01, 0.2536822079344948E-03, 0.5350323562789559E-03, 0.1166624146026729E+01, 0.1168088077064565E+01, 0.3221114082852138E-04, 0.7153898701552432E-04, 0.5458823568901986E+00, 0.5252841314829605E+00, 0.5061929508212644E-01, 0.2191222436498315E-01, 0.1344463426861069E+01, 0.1261039483864963E+01, 0.1073120173669855E+00, 0.1353250258951489E+00, 0.9720968780044319E+00, 0.1066152263103050E+01, 0.1326348631702415E-01, 0.1453842008528346E-01, 0.9676050651333883E+00, 0.8868497102573779E+00, 0.1533962535161303E+00, 0.9530062900995111E-01, 0.1446541525580445E+01, 0.1557460778765631E+01, 0.3801569533217738E-01, 0.4375878814786067E-01, 0.1353638307539094E+01, 0.1256691152780422E+01, 0.1707682473962209E+00, 0.1362446379077657E+00, 0.1583754766727924E+01, 0.1696642487635560E+01, 0.2697037855661164E-01, 0.5916231430550117E-01, 0.8850296086469572E+00, 0.7219877400150740E+00, 0.6437471225190403E-01, 0.1912940350054112E+00, 0.1099835479180717E+01, 0.2978322161977056E+00, 0.8009821360646626E+00, 0.8676597630518320E+00, 0.5535778122558079E+00, 0.1015315690490590E+01, 0.6914368034330997E+00, 0.3405260456467969E+00, 0.5001421950066134E+00, 0.1124037582002341E+01, 0.1412346654806686E+00, 0.6882320425428856E-01, 0.1838256550116241E+01, 0.1026360460398424E+01, 0.3752173792456719E+00, 0.9544125430154021E+00, 0.4926939676603885E+00, 0.7226914714852769E+00, 0.2452143584512202E+00, 0.1465016632377001E+00, 0.8074131885210117E+00, 0.8246707410864278E+00, 0.1055431169003394E+00, 0.2292385937027206E+00, 0.2714517865170992E+00, 0.3717389082113253E+00, 0.2042703660177667E-02, 0.6537783023029759E-02, 0.8212159993993635E-01, 0.8481400436921797E-01, 0.6326073178529442E-04, 0.6320870478125323E-03, 0.1054813695965874E-01, 0.8218919412378699E-02, 0.7176120699696391E-03, 0.3674266095981827E-02, 0.6073136115328898E-04, 0.2060131248039419E-02, 0.3221114082852138E-04, 0.7153898701552432E-04, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       0,
			dmin:     6.0731361153288982E-005,
			desig:    1.7347234759768071E-018,
			qmax:     2.4715692224139039,
			ttype:    -2,
			dmin1:    6.8738708636769136E-003,
			dmin2:    8.1489512892123819E-002,
			dn:       6.0731361153288982E-005,
			dn1:      6.8738708636769136E-003,
			dn2:      8.1489512892123819E-002,
			g:        0.0000000000000000,
			tau:      1.2817878169164906E-003,
			nFail:    2,
			iter:     14,
			sigma:    1.6921621533398150E-002,
			nDiv:     298,
			zOut:     []float64{0.2463574096511276E+01, 0.2463770941477959E+01, 0.2536822079344948E-03, 0.1201214707955848E-03, 0.1166624146026729E+01, 0.1166479398455512E+01, 0.3221114082852138E-04, 0.1507398544447245E-04, 0.5458823568901986E+00, 0.5964297407456295E+00, 0.5061929508212644E-01, 0.1141052940222717E+00, 0.1344463426861069E+01, 0.1337613312964532E+01, 0.1073120173669855E+00, 0.7798791776646297E-01, 0.9720968780044319E+00, 0.9073156093137420E+00, 0.1326348631702415E-01, 0.1414482062243694E-01, 0.9676050651333883E+00, 0.1106799660785830E+01, 0.1533962535161303E+00, 0.2004825791345134E+00, 0.1446541525580445E+01, 0.1284017804536858E+01, 0.3801569533217738E-01, 0.4007693764646178E-01, 0.1353638307539094E+01, 0.1484272780047602E+01, 0.1707682473962209E+00, 0.1822138285193538E+00, 0.1583754766727924E+01, 0.1428454479523931E+01, 0.2697037855661164E-01, 0.1671007646458111E-01, 0.8850296086469572E+00, 0.9326374071930291E+00, 0.6437471225190403E-01, 0.7591545433480534E-01, 0.1099835479180717E+01, 0.1824845323669324E+01, 0.8009821360646626E+00, 0.2429827519008994E+00, 0.5535778122558079E+00, 0.1001975026546757E+01, 0.6914368034330997E+00, 0.3451350696526060E+00, 0.5001421950066134E+00, 0.2961849535934249E+00, 0.1412346654806686E+00, 0.8765656248686587E+00, 0.1838256550116241E+01, 0.1336851467252003E+01, 0.3752173792456719E+00, 0.1382856239786244E+00, 0.4926939676603885E+00, 0.5995658648917332E+00, 0.2452143584512202E+00, 0.3302211126778973E+00, 0.8074131885210117E+00, 0.5826783555022028E+00, 0.1055431169003394E+00, 0.4916926700063749E-01, 0.2714517865170992E+00, 0.2242683859353883E+00, 0.2042703660177667E-02, 0.7479881396448043E-03, 0.8212159993993635E-01, 0.8138003529082581E-01, 0.6326073178529442E-04, 0.8199589256196194E-05, 0.1054813695965874E-01, 0.1120071219912114E-01, 0.7176120699696391E-03, 0.3890963093641941E-05, 0.6073136115328898E-04, 0.3156808608191942E-08, 0.3221114082852138E-04, 0.1507398544447245E-04, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    20,
			ppOut:    0,
			dminOut:  3.1568086081919418E-009,
			desigOut: 4.7433845046240819E-020,
			sigmaOut: 1.6978458774649190E-002,
			qmaxOut:  2.4715692224139039,
			nFailOut: 2,
			iterOut:  15,
			nDivOut:  319,
			ttypeOut: -2,
			dmin1Out: 1.0483100129151506E-002,
			dmin2Out: 8.1316774559040517E-002,
			dnOut:    3.1568086081919418E-009,
			dn1Out:   1.0483100129151506E-002,
			dn2Out:   8.1316774559040517E-002,
			gOut:     0.0000000000000000,
			tauOut:   5.6837241251038845E-005,
		},
		{
			i0:       1,
			n0:       20,
			z:        []float64{0.2463574096511276E+01, 0.2463770941477959E+01, 0.2536822079344948E-03, 0.1201214707955848E-03, 0.1166624146026729E+01, 0.1166479398455512E+01, 0.3221114082852138E-04, 0.1507398544447245E-04, 0.5458823568901986E+00, 0.5964297407456295E+00, 0.5061929508212644E-01, 0.1141052940222717E+00, 0.1344463426861069E+01, 0.1337613312964532E+01, 0.1073120173669855E+00, 0.7798791776646297E-01, 0.9720968780044319E+00, 0.9073156093137420E+00, 0.1326348631702415E-01, 0.1414482062243694E-01, 0.9676050651333883E+00, 0.1106799660785830E+01, 0.1533962535161303E+00, 0.2004825791345134E+00, 0.1446541525580445E+01, 0.1284017804536858E+01, 0.3801569533217738E-01, 0.4007693764646178E-01, 0.1353638307539094E+01, 0.1484272780047602E+01, 0.1707682473962209E+00, 0.1822138285193538E+00, 0.1583754766727924E+01, 0.1428454479523931E+01, 0.2697037855661164E-01, 0.1671007646458111E-01, 0.8850296086469572E+00, 0.9326374071930291E+00, 0.6437471225190403E-01, 0.7591545433480534E-01, 0.1099835479180717E+01, 0.1824845323669324E+01, 0.8009821360646626E+00, 0.2429827519008994E+00, 0.5535778122558079E+00, 0.1001975026546757E+01, 0.6914368034330997E+00, 0.3451350696526060E+00, 0.5001421950066134E+00, 0.2961849535934249E+00, 0.1412346654806686E+00, 0.8765656248686587E+00, 0.1838256550116241E+01, 0.1336851467252003E+01, 0.3752173792456719E+00, 0.1382856239786244E+00, 0.4926939676603885E+00, 0.5995658648917332E+00, 0.2452143584512202E+00, 0.3302211126778973E+00, 0.8074131885210117E+00, 0.5826783555022028E+00, 0.1055431169003394E+00, 0.4916926700063749E-01, 0.2714517865170992E+00, 0.2242683859353883E+00, 0.2042703660177667E-02, 0.7479881396448043E-03, 0.8212159993993635E-01, 0.8138003529082581E-01, 0.6326073178529442E-04, 0.8199589256196194E-05, 0.1054813695965874E-01, 0.1120071219912114E-01, 0.7176120699696391E-03, 0.3890963093641941E-05, 0.6073136115328898E-04, 0.3156808608191942E-08, 0.3221114082852138E-04, 0.1507398544447245E-04, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       1,
			dmin:     3.1568086081919418E-009,
			desig:    4.7433845046240819E-020,
			qmax:     2.4715692224139039,
			ttype:    -2,
			dmin1:    1.0483100129151506E-002,
			dmin2:    8.1316774559040517E-002,
			dn:       3.1568086081919418E-009,
			dn1:      1.0483100129151506E-002,
			dn2:      8.1316774559040517E-002,
			g:        0.0000000000000000,
			tau:      5.6837241251038845E-005,
			nFail:    2,
			iter:     15,
			sigma:    1.6978458774649190E-002,
			nDiv:     319,
			zOut:     []float64{0.2463891059793043E+01, 0.2463770941477959E+01, 0.5686908130061341E-04, 0.1201214707955848E-03, 0.1166437600203943E+01, 0.1166479398455512E+01, 0.7707718980490818E-05, 0.1507398544447245E-04, 0.7105273238932086E+00, 0.5964297407456295E+00, 0.2148105431436762E+00, 0.1141052940222717E+00, 0.1200790684431606E+01, 0.1337613312964532E+01, 0.5892755169139442E-01, 0.7798791776646297E-01, 0.8625328750890724E+00, 0.9073156093137420E+00, 0.1815059242254727E-01, 0.1414482062243694E-01, 0.1289131644342084E+01, 0.1106799660785830E+01, 0.1996872873596725E+00, 0.2004825791345134E+00, 0.1124407451667935E+01, 0.1284017804536858E+01, 0.5290351604133232E-01, 0.4007693764646178E-01, 0.1613583089369911E+01, 0.1484272780047602E+01, 0.1613081850537457E+00, 0.1822138285193538E+00, 0.1283856367779054E+01, 0.1428454479523931E+01, 0.1213877407087503E-01, 0.1671007646458111E-01, 0.9964140843012472E+00, 0.9326374071930291E+00, 0.1390325207358455E+00, 0.7591545433480534E-01, 0.1928795551678665E+01, 0.1824845323669324E+01, 0.1262252233392066E+00, 0.2429827519008994E+00, 0.1220884869704444E+01, 0.1001975026546757E+01, 0.8372928285471114E-01, 0.3451350696526060E+00, 0.1089021292451660E+01, 0.2961849535934249E+00, 0.1076046951396362E+01, 0.8765656248686587E+00, 0.3990901366785531E+00, 0.1336851467252003E+01, 0.2077509116934600E+00, 0.1382856239786244E+00, 0.7220360627204584E+00, 0.5995658648917332E+00, 0.2664862668525171E+00, 0.3302211126778973E+00, 0.3653613524946110E+00, 0.5826783555022028E+00, 0.3018138637972599E-01, 0.4916926700063749E-01, 0.1948349845395949E+00, 0.2242683859353883E+00, 0.3124249032854923E-03, 0.7479881396448043E-03, 0.8107580682108434E-01, 0.8138003529082581E-01, 0.1132782305976083E-05, 0.8199589256196194E-05, 0.1120346722419663E-01, 0.1120071219912114E-01, 0.1096359327194516E-11, 0.3890963093641941E-05, 0.6968497581336674E-16, 0.3156808608191942E-08, 0.7707718980490818E-05, 0.1507398544447245E-04, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    20,
			ppOut:    1,
			dminOut:  6.9684975813366743E-017,
			desigOut: 1.5445815365207740E-018,
			sigmaOut: 1.6978461930361368E-002,
			qmaxOut:  2.4715692224139039,
			nFailOut: 2,
			iterOut:  16,
			nDivOut:  340,
			ttypeOut: -2,
			dmin1Out: 1.1199576261102989E-002,
			dmin2Out: 8.1067607231828140E-002,
			dnOut:    6.9684975813366743E-017,
			dn1Out:   1.1199576261102989E-002,
			dn2Out:   8.1067607231828140E-002,
			gOut:     0.0000000000000000,
			tauOut:   3.1557121791797713E-009,
		},
		{
			i0:       1,
			n0:       20,
			z:        []float64{0.2463891059793043E+01, 0.2463770941477959E+01, 0.5686908130061341E-04, 0.1201214707955848E-03, 0.1166437600203943E+01, 0.1166479398455512E+01, 0.7707718980490818E-05, 0.1507398544447245E-04, 0.7105273238932086E+00, 0.5964297407456295E+00, 0.2148105431436762E+00, 0.1141052940222717E+00, 0.1200790684431606E+01, 0.1337613312964532E+01, 0.5892755169139442E-01, 0.7798791776646297E-01, 0.8625328750890724E+00, 0.9073156093137420E+00, 0.1815059242254727E-01, 0.1414482062243694E-01, 0.1289131644342084E+01, 0.1106799660785830E+01, 0.1996872873596725E+00, 0.2004825791345134E+00, 0.1124407451667935E+01, 0.1284017804536858E+01, 0.5290351604133232E-01, 0.4007693764646178E-01, 0.1613583089369911E+01, 0.1484272780047602E+01, 0.1613081850537457E+00, 0.1822138285193538E+00, 0.1283856367779054E+01, 0.1428454479523931E+01, 0.1213877407087503E-01, 0.1671007646458111E-01, 0.9964140843012472E+00, 0.9326374071930291E+00, 0.1390325207358455E+00, 0.7591545433480534E-01, 0.1928795551678665E+01, 0.1824845323669324E+01, 0.1262252233392066E+00, 0.2429827519008994E+00, 0.1220884869704444E+01, 0.1001975026546757E+01, 0.8372928285471114E-01, 0.3451350696526060E+00, 0.1089021292451660E+01, 0.2961849535934249E+00, 0.1076046951396362E+01, 0.8765656248686587E+00, 0.3990901366785531E+00, 0.1336851467252003E+01, 0.2077509116934600E+00, 0.1382856239786244E+00, 0.7220360627204584E+00, 0.5995658648917332E+00, 0.2664862668525171E+00, 0.3302211126778973E+00, 0.3653613524946110E+00, 0.5826783555022028E+00, 0.3018138637972599E-01, 0.4916926700063749E-01, 0.1948349845395949E+00, 0.2242683859353883E+00, 0.3124249032854923E-03, 0.7479881396448043E-03, 0.8107580682108434E-01, 0.8138003529082581E-01, 0.1132782305976083E-05, 0.8199589256196194E-05, 0.1120346722419663E-01, 0.1120071219912114E-01, 0.1096359327194516E-11, 0.3890963093641941E-05, 0.6968497581336674E-16, 0.3156808608191942E-08, 0.7707718980490818E-05, 0.1507398544447245E-04, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       0,
			dmin:     6.9684975813366743E-017,
			desig:    1.5445815365207740E-018,
			qmax:     2.4715692224139039,
			ttype:    -2,
			dmin1:    1.1199576261102989E-002,
			dmin2:    8.1067607231828140E-002,
			dn:       6.9684975813366743E-017,
			dn1:      1.1199576261102989E-002,
			dn2:      8.1067607231828140E-002,
			g:        0.0000000000000000,
			tau:      3.1557121791797713E-009,
			nFail:    2,
			iter:     16,
			sigma:    1.6978461930361368E-002,
			nDiv:     340,
			zOut:     []float64{0.2463891059793043E+01, 0.2463947928874343E+01, 0.5686908130061341E-04, 0.2692193042748079E-04, 0.1166437600203943E+01, 0.1166418385992496E+01, 0.7707718980490818E-05, 0.4695180568393632E-05, 0.7105273238932086E+00, 0.9253331718563164E+00, 0.2148105431436762E+00, 0.2787563517334627E+00, 0.1200790684431606E+01, 0.9809618843895378E+00, 0.5892755169139442E-01, 0.5181337969514327E-01, 0.8625328750890724E+00, 0.8288700878164763E+00, 0.1815059242254727E-01, 0.2822939734392020E-01, 0.1289131644342084E+01, 0.1460589534357837E+01, 0.1996872873596725E+00, 0.1537255119449346E+00, 0.1124407451667935E+01, 0.1023585455764333E+01, 0.5290351604133232E-01, 0.8339725654733963E-01, 0.1613583089369911E+01, 0.1691494017876317E+01, 0.1613081850537457E+00, 0.1224340957564512E+00, 0.1283856367779054E+01, 0.1173561046093478E+01, 0.1213877407087503E-01, 0.1030644761994533E-01, 0.9964140843012472E+00, 0.1125140157417147E+01, 0.1390325207358455E+00, 0.2383394688796517E+00, 0.1928795551678665E+01, 0.1816681306138221E+01, 0.1262252233392066E+00, 0.8482856339700598E-01, 0.1220884869704444E+01, 0.1219785589162149E+01, 0.8372928285471114E-01, 0.7475327847832687E-01, 0.1089021292451660E+01, 0.2090314965369696E+01, 0.1076046951396362E+01, 0.2054425921547012E+00, 0.3990901366785531E+00, 0.4013984562173118E+00, 0.2077509116934600E+00, 0.3737026089221466E+00, 0.7220360627204584E+00, 0.6148197206508288E+00, 0.2664862668525171E+00, 0.1583615157552351E+00, 0.3653613524946110E+00, 0.2371812231191019E+00, 0.3018138637972599E-01, 0.2479281399828426E-01, 0.1948349845395949E+00, 0.1703545954445960E+00, 0.3124249032854923E-03, 0.1486904479375115E-03, 0.8107580682108434E-01, 0.8092824915545274E-01, 0.1132782305976083E-05, 0.1568190288260776E-06, 0.1120346722419663E-01, 0.1120331040626409E-01, 0.1096359327194516E-11, 0.6819392699821255E-26, 0.6968497581336674E-16, 0.6162975822039155E-31, 0.7707718980490818E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    20,
			ppOut:    0,
			dminOut:  6.1629758220391547E-032,
			desigOut: -1.6288286479578371E-018,
			sigmaOut: 1.6978461930361441E-002,
			qmaxOut:  2.4715692224139039,
			nFailOut: 2,
			iterOut:  17,
			nDivOut:  361,
			ttypeOut: -2,
			dmin1Out: 1.1203310405167735E-002,
			dmin2Out: 8.0927116373146771E-002,
			dnOut:    6.1629758220391547E-032,
			dn1Out:   1.1203310405167735E-002,
			dn2Out:   8.0927116373146771E-002,
			gOut:     0.0000000000000000,
			tauOut:   6.9684975806547287E-017,
		},
		{
			i0:       1,
			n0:       20,
			z:        []float64{0.2463891059793043E+01, 0.2463947928874343E+01, 0.5686908130061341E-04, 0.2692193042748079E-04, 0.1166437600203943E+01, 0.1166418385992496E+01, 0.7707718980490818E-05, 0.4695180568393632E-05, 0.7105273238932086E+00, 0.9253331718563164E+00, 0.2148105431436762E+00, 0.2787563517334627E+00, 0.1200790684431606E+01, 0.9809618843895378E+00, 0.5892755169139442E-01, 0.5181337969514327E-01, 0.8625328750890724E+00, 0.8288700878164763E+00, 0.1815059242254727E-01, 0.2822939734392020E-01, 0.1289131644342084E+01, 0.1460589534357837E+01, 0.1996872873596725E+00, 0.1537255119449346E+00, 0.1124407451667935E+01, 0.1023585455764333E+01, 0.5290351604133232E-01, 0.8339725654733963E-01, 0.1613583089369911E+01, 0.1691494017876317E+01, 0.1613081850537457E+00, 0.1224340957564512E+00, 0.1283856367779054E+01, 0.1173561046093478E+01, 0.1213877407087503E-01, 0.1030644761994533E-01, 0.9964140843012472E+00, 0.1125140157417147E+01, 0.1390325207358455E+00, 0.2383394688796517E+00, 0.1928795551678665E+01, 0.1816681306138221E+01, 0.1262252233392066E+00, 0.8482856339700598E-01, 0.1220884869704444E+01, 0.1219785589162149E+01, 0.8372928285471114E-01, 0.7475327847832687E-01, 0.1089021292451660E+01, 0.2090314965369696E+01, 0.1076046951396362E+01, 0.2054425921547012E+00, 0.3990901366785531E+00, 0.4013984562173118E+00, 0.2077509116934600E+00, 0.3737026089221466E+00, 0.7220360627204584E+00, 0.6148197206508288E+00, 0.2664862668525171E+00, 0.1583615157552351E+00, 0.3653613524946110E+00, 0.2371812231191019E+00, 0.3018138637972599E-01, 0.2479281399828426E-01, 0.1948349845395949E+00, 0.1703545954445960E+00, 0.3124249032854923E-03, 0.1486904479375115E-03, 0.8107580682108434E-01, 0.8092824915545274E-01, 0.1132782305976083E-05, 0.1568190288260776E-06, 0.1120346722419663E-01, 0.1120331040626409E-01, 0.1096359327194516E-11, 0.6819392699821255E-26, 0.6968497581336674E-16, 0.6162975822039155E-31, 0.7707718980490818E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       1,
			dmin:     6.1629758220391547E-032,
			desig:    -1.6288286479578371E-018,
			qmax:     2.4715692224139039,
			ttype:    -2,
			dmin1:    1.1203310405167735E-002,
			dmin2:    8.0927116373146771E-002,
			dn:       6.1629758220391547E-032,
			dn1:      1.1203310405167735E-002,
			dn2:      8.0927116373146771E-002,
			g:        0.0000000000000000,
			tau:      6.9684975806547287E-017,
			nFail:    2,
			iter:     17,
			sigma:    1.6978461930361441E-002,
			nDiv:     361,
			zOut:     []float64{0.2463974850804771E+01, 0.2463947928874343E+01, 0.1274454348702788E-04, 0.2692193042748079E-04, 0.1166410336629578E+01, 0.1166418385992496E+01, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.1204085798823124E+01, 0.9253331718563164E+00, 0.2271012218143261E+00, 0.2787563517334627E+00, 0.8056740422703550E+00, 0.9809618843895378E+00, 0.5330513126246473E-01, 0.5181337969514327E-01, 0.8037943538979316E+00, 0.8288700878164763E+00, 0.5129615818002433E-01, 0.2822939734392020E-01, 0.1563018888122747E+01, 0.1460589534357837E+01, 0.1006713350698832E+00, 0.1537255119449346E+00, 0.1006311377241790E+01, 0.1023585455764333E+01, 0.1401812239704283E+00, 0.8339725654733963E-01, 0.1673746889662340E+01, 0.1691494017876317E+01, 0.8584564749956700E-01, 0.1224340957564512E+00, 0.1098021846213856E+01, 0.1173561046093478E+01, 0.1056099032774466E-01, 0.1030644761994533E-01, 0.1352918635969054E+01, 0.1125140157417147E+01, 0.3200390963041470E+00, 0.2383394688796517E+00, 0.1581470773231080E+01, 0.1816681306138221E+01, 0.6542811978092533E-01, 0.8482856339700598E-01, 0.1229110747859551E+01, 0.1219785589162149E+01, 0.1271308521106110E+00, 0.7475327847832687E-01, 0.2168626705413786E+01, 0.2090314965369696E+01, 0.3802606466401751E-01, 0.2054425921547012E+00, 0.7370750004754409E+00, 0.4013984562173118E+00, 0.3117182559112661E+00, 0.3737026089221466E+00, 0.4614629804947978E+00, 0.6148197206508288E+00, 0.8139413038408401E-01, 0.1583615157552351E+00, 0.1805799067333021E+00, 0.2371812231191019E+00, 0.2338892446571373E-01, 0.2479281399828426E-01, 0.1471143614268198E+00, 0.1703545954445960E+00, 0.8179526119010886E-04, 0.1486904479375115E-03, 0.8084661071329148E-01, 0.8092824915545274E-01, 0.2173118009582292E-07, 0.1568190288260776E-06, 0.1120328867508400E-01, 0.1120331040626409E-01, 0.3751376363572422E-55, 0.6819392699821255E-26, 0.6162975822039155E-31, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    20,
			ppOut:    1,
			dminOut:  6.1629758220391547E-032,
			desigOut: -1.6288286479578371E-018,
			sigmaOut: 1.6978461930361441E-002,
			qmaxOut:  2.4715692224139039,
			nFailOut: 2,
			iterOut:  18,
			nDivOut:  382,
			ttypeOut: -2,
			dmin1Out: 1.1203288675083998E-002,
			dmin2Out: 8.0846453894262649E-002,
			dnOut:    6.1629758220391547E-032,
			dn1Out:   1.1203288675083998E-002,
			dn2Out:   8.0846453894262649E-002,
			gOut:     0.0000000000000000,
			tauOut:   0.0000000000000000,
		},
		{
			i0:       1,
			n0:       20,
			z:        []float64{0.2463974850804771E+01, 0.2463947928874343E+01, 0.1274454348702788E-04, 0.2692193042748079E-04, 0.1166410336629578E+01, 0.1166418385992496E+01, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.1204085798823124E+01, 0.9253331718563164E+00, 0.2271012218143261E+00, 0.2787563517334627E+00, 0.8056740422703550E+00, 0.9809618843895378E+00, 0.5330513126246473E-01, 0.5181337969514327E-01, 0.8037943538979316E+00, 0.8288700878164763E+00, 0.5129615818002433E-01, 0.2822939734392020E-01, 0.1563018888122747E+01, 0.1460589534357837E+01, 0.1006713350698832E+00, 0.1537255119449346E+00, 0.1006311377241790E+01, 0.1023585455764333E+01, 0.1401812239704283E+00, 0.8339725654733963E-01, 0.1673746889662340E+01, 0.1691494017876317E+01, 0.8584564749956700E-01, 0.1224340957564512E+00, 0.1098021846213856E+01, 0.1173561046093478E+01, 0.1056099032774466E-01, 0.1030644761994533E-01, 0.1352918635969054E+01, 0.1125140157417147E+01, 0.3200390963041470E+00, 0.2383394688796517E+00, 0.1581470773231080E+01, 0.1816681306138221E+01, 0.6542811978092533E-01, 0.8482856339700598E-01, 0.1229110747859551E+01, 0.1219785589162149E+01, 0.1271308521106110E+00, 0.7475327847832687E-01, 0.2168626705413786E+01, 0.2090314965369696E+01, 0.3802606466401751E-01, 0.2054425921547012E+00, 0.7370750004754409E+00, 0.4013984562173118E+00, 0.3117182559112661E+00, 0.3737026089221466E+00, 0.4614629804947978E+00, 0.6148197206508288E+00, 0.8139413038408401E-01, 0.1583615157552351E+00, 0.1805799067333021E+00, 0.2371812231191019E+00, 0.2338892446571373E-01, 0.2479281399828426E-01, 0.1471143614268198E+00, 0.1703545954445960E+00, 0.8179526119010886E-04, 0.1486904479375115E-03, 0.8084661071329148E-01, 0.8092824915545274E-01, 0.2173118009582292E-07, 0.1568190288260776E-06, 0.1120328867508400E-01, 0.1120331040626409E-01, 0.3751376363572422E-55, 0.6819392699821255E-26, 0.6162975822039155E-31, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       0,
			dmin:     6.1629758220391547E-032,
			desig:    -1.6288286479578371E-018,
			qmax:     2.4715692224139039,
			ttype:    -2,
			dmin1:    1.1203288675083998E-002,
			dmin2:    8.0846453894262649E-002,
			dn:       6.1629758220391547E-032,
			dn1:      1.1203288675083998E-002,
			dn2:      8.0846453894262649E-002,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    2,
			iter:     18,
			sigma:    1.6978461930361441E-002,
			nDiv:     382,
			zOut:     []float64{0.2463974850804771E+01, 0.2452784311062345E+01, 0.1274454348702788E-04, 0.6060609239813679E-05, 0.1166410336629578E+01, 0.1155204716501079E+01, 0.3724766654883956E-05, 0.3882375624867434E-05, 0.1204085798823124E+01, 0.1419979853975912E+01, 0.2271012218143261E+00, 0.1288536304732592E+00, 0.8056740422703550E+00, 0.7189222587736472E+00, 0.5330513126246473E-01, 0.5959804835594534E-01, 0.8037943538979316E+00, 0.7842891794360973E+00, 0.5129615818002433E-01, 0.1022287011292917E+00, 0.1563018888122747E+01, 0.1550258237777425E+01, 0.1006713350698832E+00, 0.6534828028921505E-01, 0.1006311377241790E+01, 0.1069941036637090E+01, 0.1401812239704283E+00, 0.2192904838448094E+00, 0.1673746889662340E+01, 0.1529098769031184E+01, 0.8584564749956700E-01, 0.6164441320989392E-01, 0.1098021846213856E+01, 0.1035735139045794E+01, 0.1056099032774466E-01, 0.1379518767882891E-01, 0.1352918635969054E+01, 0.1647959260308459E+01, 0.3200390963041470E+00, 0.3071268139247320E+00, 0.1581470773231080E+01, 0.1328568794801360E+01, 0.6542811978092533E-01, 0.6053010243026304E-01, 0.1229110747859551E+01, 0.1284508213253985E+01, 0.1271308521106110E+00, 0.2146341752620367E+00, 0.2168626705413786E+01, 0.1980815310529854E+01, 0.3802606466401751E-01, 0.1414976019284330E-01, 0.7370750004754409E+00, 0.1023440211907950E+01, 0.3117182559112661E+00, 0.1405518698344743E+00, 0.4614629804947978E+00, 0.3911019567584944E+00, 0.8139413038408401E-01, 0.3758136265851575E-01, 0.1805799067333021E+00, 0.1551841842545868E+00, 0.2338892446571373E-01, 0.2217266343062855E-01, 0.1471143614268198E+00, 0.1138202089714681E+00, 0.8179526119010886E-04, 0.5809925758690545E-04, 0.8084661071329148E-01, 0.6958524890097138E-01, 0.2173118009582292E-07, 0.3498739858072780E-08, 0.1120328867508400E-01, 0.8904308505808256E-09, 0.3751376363572422E-55, 0.3882375624867434E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    19,
			ppOut:    0,
			dminOut:  8.9043085058082561E-010,
			desigOut: 1.7347234759768071E-018,
			sigmaOut: 2.8181746216274728E-002,
			qmaxOut:  2.4715692224139039,
			nFailOut: 2,
			iterOut:  19,
			nDivOut:  402,
			ttypeOut: -7,
			dmin1Out: 6.9585227169791292E-002,
			dmin2Out: 0.11373841371027797,
			dnOut:    8.9043085058082561E-010,
			dn1Out:   6.9585227169791292E-002,
			dn2Out:   0.11373841371027797,
			gOut:     0.0000000000000000,
			tauOut:   1.1203284285913290E-002,
		},
		{
			i0:       1,
			n0:       19,
			z:        []float64{0.2463974850804771E+01, 0.2452784311062345E+01, 0.1274454348702788E-04, 0.6060609239813679E-05, 0.1166410336629578E+01, 0.1155204716501079E+01, 0.3724766654883956E-05, 0.3882375624867434E-05, 0.1204085798823124E+01, 0.1419979853975912E+01, 0.2271012218143261E+00, 0.1288536304732592E+00, 0.8056740422703550E+00, 0.7189222587736472E+00, 0.5330513126246473E-01, 0.5959804835594534E-01, 0.8037943538979316E+00, 0.7842891794360973E+00, 0.5129615818002433E-01, 0.1022287011292917E+00, 0.1563018888122747E+01, 0.1550258237777425E+01, 0.1006713350698832E+00, 0.6534828028921505E-01, 0.1006311377241790E+01, 0.1069941036637090E+01, 0.1401812239704283E+00, 0.2192904838448094E+00, 0.1673746889662340E+01, 0.1529098769031184E+01, 0.8584564749956700E-01, 0.6164441320989392E-01, 0.1098021846213856E+01, 0.1035735139045794E+01, 0.1056099032774466E-01, 0.1379518767882891E-01, 0.1352918635969054E+01, 0.1647959260308459E+01, 0.3200390963041470E+00, 0.3071268139247320E+00, 0.1581470773231080E+01, 0.1328568794801360E+01, 0.6542811978092533E-01, 0.6053010243026304E-01, 0.1229110747859551E+01, 0.1284508213253985E+01, 0.1271308521106110E+00, 0.2146341752620367E+00, 0.2168626705413786E+01, 0.1980815310529854E+01, 0.3802606466401751E-01, 0.1414976019284330E-01, 0.7370750004754409E+00, 0.1023440211907950E+01, 0.3117182559112661E+00, 0.1405518698344743E+00, 0.4614629804947978E+00, 0.3911019567584944E+00, 0.8139413038408401E-01, 0.3758136265851575E-01, 0.1805799067333021E+00, 0.1551841842545868E+00, 0.2338892446571373E-01, 0.2217266343062855E-01, 0.1471143614268198E+00, 0.1138202089714681E+00, 0.8179526119010886E-04, 0.5809925758690545E-04, 0.8084661071329148E-01, 0.6958524890097138E-01, 0.2173118009582292E-07, 0.3498739858072780E-08, 0.1120328867508400E-01, 0.8904308505808256E-09, 0.3751376363572422E-55, 0.3882375624867434E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       1,
			dmin:     8.9043085058082561E-010,
			desig:    1.7347234759768071E-018,
			qmax:     2.4715692224139039,
			ttype:    -7,
			dmin1:    6.9585227169791292E-002,
			dmin2:    0.11373841371027797,
			dn:       8.9043085058082561E-010,
			dn1:      6.9585227169791292E-002,
			dn2:      0.11373841371027797,
			g:        0.0000000000000000,
			tau:      1.1203284285913290E-002,
			nFail:    2,
			iter:     19,
			sigma:    2.8181746216274728E-002,
			nDiv:     402,
			zOut:     []float64{0.2452790370781154E+01, 0.2452784311062345E+01, 0.2854399814229969E-05, 0.6060609239813679E-05, 0.1155205743586459E+01, 0.1155204716501079E+01, 0.4772219324121025E-05, 0.3882375624867434E-05, 0.1548828711339416E+01, 0.1419979853975912E+01, 0.5981019230390531E-01, 0.1288536304732592E+00, 0.7187101139352565E+00, 0.7189222587736472E+00, 0.6503610222645050E-01, 0.5959804835594534E-01, 0.8214817774485077E+00, 0.7842891794360973E+00, 0.1929207566298143E+00, 0.1022287011292917E+00, 0.1422685760546395E+01, 0.1550258237777425E+01, 0.4914564318703864E-01, 0.6534828028921505E-01, 0.1240085876404429E+01, 0.1069941036637090E+01, 0.2703980549150243E+00, 0.2192904838448094E+00, 0.1320345126435623E+01, 0.1529098769031184E+01, 0.4835651195207321E-01, 0.6164441320989392E-01, 0.1001173813882118E+01, 0.1035735139045794E+01, 0.2270725319399537E-01, 0.1379518767882891E-01, 0.1932378820148765E+01, 0.1647959260308459E+01, 0.2111589595024383E+00, 0.3071268139247320E+00, 0.1177939936838754E+01, 0.1328568794801360E+01, 0.6600626338337756E-01, 0.6053010243026304E-01, 0.1433136124242214E+01, 0.1284508213253985E+01, 0.2966575563411978E+00, 0.2146341752620367E+00, 0.1698307513491068E+01, 0.1980815310529854E+01, 0.8526979628348903E-02, 0.1414976019284330E-01, 0.1155465101223645E+01, 0.1023440211907950E+01, 0.4757401262929915E-01, 0.1405518698344743E+00, 0.3811093058972801E+00, 0.3911019567584944E+00, 0.1530278326215801E-01, 0.3758136265851575E-01, 0.1620540635326265E+00, 0.1551841842545868E+00, 0.1557318051836492E-01, 0.2217266343062855E-01, 0.9830512682025924E-01, 0.1138202089714681E+00, 0.4112553872737890E-04, 0.5809925758690545E-04, 0.6954412597055305E-01, 0.6958524890097138E-01, 0.4479725446695418E-16, 0.3498739858072780E-08, 0.1396210563637444E-18, 0.8904308505808256E-09, 0.2854399814229969E-05, 0.3882375624867434E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    19,
			ppOut:    1,
			dminOut:  1.3962105636374437E-019,
			desigOut: -1.5687885635568321E-018,
			sigmaOut: 2.8181747106705537E-002,
			qmaxOut:  2.4715692224139039,
			nFailOut: 2,
			iterOut:  20,
			nDivOut:  422,
			ttypeOut: -2,
			dmin1Out: 6.9544122471813200E-002,
			dmin2Out: 9.8247027562672340E-002,
			dnOut:    1.3962105636374437E-019,
			dn1Out:   6.9544122471813200E-002,
			dn2Out:   9.8247027562672340E-002,
			gOut:     0.0000000000000000,
			tauOut:   8.9043080564395014E-010,
		},
		{
			i0:       1,
			n0:       19,
			z:        []float64{0.2452790370781154E+01, 0.2452784311062345E+01, 0.2854399814229969E-05, 0.6060609239813679E-05, 0.1155205743586459E+01, 0.1155204716501079E+01, 0.4772219324121025E-05, 0.3882375624867434E-05, 0.1548828711339416E+01, 0.1419979853975912E+01, 0.5981019230390531E-01, 0.1288536304732592E+00, 0.7187101139352565E+00, 0.7189222587736472E+00, 0.6503610222645050E-01, 0.5959804835594534E-01, 0.8214817774485077E+00, 0.7842891794360973E+00, 0.1929207566298143E+00, 0.1022287011292917E+00, 0.1422685760546395E+01, 0.1550258237777425E+01, 0.4914564318703864E-01, 0.6534828028921505E-01, 0.1240085876404429E+01, 0.1069941036637090E+01, 0.2703980549150243E+00, 0.2192904838448094E+00, 0.1320345126435623E+01, 0.1529098769031184E+01, 0.4835651195207321E-01, 0.6164441320989392E-01, 0.1001173813882118E+01, 0.1035735139045794E+01, 0.2270725319399537E-01, 0.1379518767882891E-01, 0.1932378820148765E+01, 0.1647959260308459E+01, 0.2111589595024383E+00, 0.3071268139247320E+00, 0.1177939936838754E+01, 0.1328568794801360E+01, 0.6600626338337756E-01, 0.6053010243026304E-01, 0.1433136124242214E+01, 0.1284508213253985E+01, 0.2966575563411978E+00, 0.2146341752620367E+00, 0.1698307513491068E+01, 0.1980815310529854E+01, 0.8526979628348903E-02, 0.1414976019284330E-01, 0.1155465101223645E+01, 0.1023440211907950E+01, 0.4757401262929915E-01, 0.1405518698344743E+00, 0.3811093058972801E+00, 0.3911019567584944E+00, 0.1530278326215801E-01, 0.3758136265851575E-01, 0.1620540635326265E+00, 0.1551841842545868E+00, 0.1557318051836492E-01, 0.2217266343062855E-01, 0.9830512682025924E-01, 0.1138202089714681E+00, 0.4112553872737890E-04, 0.5809925758690545E-04, 0.6954412597055305E-01, 0.6958524890097138E-01, 0.4479725446695418E-16, 0.3498739858072780E-08, 0.1396210563637444E-18, 0.8904308505808256E-09, 0.2854399814229969E-05, 0.3882375624867434E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       0,
			dmin:     1.3962105636374437E-019,
			desig:    -1.5687885635568321E-018,
			qmax:     2.4715692224139039,
			ttype:    -2,
			dmin1:    6.9544122471813200E-002,
			dmin2:    9.8247027562672340E-002,
			dn:       1.3962105636374437E-019,
			dn1:      6.9544122471813200E-002,
			dn2:      9.8247027562672340E-002,
			g:        0.0000000000000000,
			tau:      8.9043080564395014E-010,
			nFail:    2,
			iter:     20,
			sigma:    2.8181747106705537E-002,
			nDiv:     422,
			zOut:     []float64{0.2452790370781154E+01, 0.2452793225180968E+01, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1155205743586459E+01, 0.1155209171453138E+01, 0.4772219324121025E-05, 0.6398278760815103E-05, 0.1548828711339416E+01, 0.1608632505364561E+01, 0.5981019230390531E-01, 0.2672219415054498E-01, 0.7187101139352565E+00, 0.7570240220111620E+00, 0.6503610222645050E-01, 0.7057368234283014E-01, 0.8214817774485077E+00, 0.9438288517354918E+00, 0.1929207566298143E+00, 0.2908001941945216E+00, 0.1422685760546395E+01, 0.1181031209538913E+01, 0.4914564318703864E-01, 0.5160305461093759E-01, 0.1240085876404429E+01, 0.1458880876708516E+01, 0.2703980549150243E+00, 0.2447209773632920E+00, 0.1320345126435623E+01, 0.1123980661024405E+01, 0.4835651195207321E-01, 0.4307304847484577E-01, 0.1001173813882118E+01, 0.9808080186012680E+00, 0.2270725319399537E-01, 0.4473761868138884E-01, 0.1932378820148765E+01, 0.2098800160969815E+01, 0.2111589595024383E+00, 0.1185117935689049E+00, 0.1177939936838754E+01, 0.1125434406653226E+01, 0.6600626338337756E-01, 0.8405284210411720E-01, 0.1433136124242214E+01, 0.1645740838479294E+01, 0.2966575563411978E+00, 0.3061331074057168E+00, 0.1698307513491068E+01, 0.1400701385713701E+01, 0.8526979628348903E-02, 0.7034066989504621E-02, 0.1155465101223645E+01, 0.1196005046863439E+01, 0.4757401262929915E-01, 0.1515955052150448E-01, 0.3811093058972801E+00, 0.3812525386379337E+00, 0.1530278326215801E-01, 0.6504555274179689E-02, 0.1620540635326265E+00, 0.1711226887768117E+00, 0.1557318051836492E-01, 0.8946350111698947E-02, 0.9830512682025924E-01, 0.8939990224728768E-01, 0.4112553872737890E-04, 0.3199152990069927E-04, 0.6954412597055305E-01, 0.6951213444065239E-01, 0.4479725446695418E-16, 0.8997910999570377E-34, 0.1396210563637444E-18, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    19,
			ppOut:    0,
			dminOut:  1.3962105636374430E-019,
			desigOut: -1.5687885635568321E-018,
			sigmaOut: 2.8181747106705537E-002,
			qmaxOut:  2.4715692224139039,
			nFailOut: 2,
			iterOut:  21,
			nDivOut:  442,
			ttypeOut: -2,
			dmin1Out: 6.9512134440652351E-002,
			dmin2Out: 8.9358776708560295E-002,
			dnOut:    1.3962105636374430E-019,
			dn1Out:   6.9512134440652351E-002,
			dn2Out:   8.9358776708560295E-002,
			gOut:     0.0000000000000000,
			tauOut:   0.0000000000000000,
		},
		{
			i0:       1,
			n0:       19,
			z:        []float64{0.2452790370781154E+01, 0.2452793225180968E+01, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1155205743586459E+01, 0.1155209171453138E+01, 0.4772219324121025E-05, 0.6398278760815103E-05, 0.1548828711339416E+01, 0.1608632505364561E+01, 0.5981019230390531E-01, 0.2672219415054498E-01, 0.7187101139352565E+00, 0.7570240220111620E+00, 0.6503610222645050E-01, 0.7057368234283014E-01, 0.8214817774485077E+00, 0.9438288517354918E+00, 0.1929207566298143E+00, 0.2908001941945216E+00, 0.1422685760546395E+01, 0.1181031209538913E+01, 0.4914564318703864E-01, 0.5160305461093759E-01, 0.1240085876404429E+01, 0.1458880876708516E+01, 0.2703980549150243E+00, 0.2447209773632920E+00, 0.1320345126435623E+01, 0.1123980661024405E+01, 0.4835651195207321E-01, 0.4307304847484577E-01, 0.1001173813882118E+01, 0.9808080186012680E+00, 0.2270725319399537E-01, 0.4473761868138884E-01, 0.1932378820148765E+01, 0.2098800160969815E+01, 0.2111589595024383E+00, 0.1185117935689049E+00, 0.1177939936838754E+01, 0.1125434406653226E+01, 0.6600626338337756E-01, 0.8405284210411720E-01, 0.1433136124242214E+01, 0.1645740838479294E+01, 0.2966575563411978E+00, 0.3061331074057168E+00, 0.1698307513491068E+01, 0.1400701385713701E+01, 0.8526979628348903E-02, 0.7034066989504621E-02, 0.1155465101223645E+01, 0.1196005046863439E+01, 0.4757401262929915E-01, 0.1515955052150448E-01, 0.3811093058972801E+00, 0.3812525386379337E+00, 0.1530278326215801E-01, 0.6504555274179689E-02, 0.1620540635326265E+00, 0.1711226887768117E+00, 0.1557318051836492E-01, 0.8946350111698947E-02, 0.9830512682025924E-01, 0.8939990224728768E-01, 0.4112553872737890E-04, 0.3199152990069927E-04, 0.6954412597055305E-01, 0.6951213444065239E-01, 0.4479725446695418E-16, 0.8997910999570377E-34, 0.1396210563637444E-18, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       1,
			dmin:     1.3962105636374430E-019,
			desig:    -1.5687885635568321E-018,
			qmax:     2.4715692224139039,
			ttype:    -2,
			dmin1:    6.9512134440652351E-002,
			dmin2:    8.9358776708560295E-002,
			dn:       1.3962105636374430E-019,
			dn1:      6.9512134440652351E-002,
			dn2:      8.9358776708560295E-002,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    2,
			iter:     21,
			sigma:    2.8181747106705537E-002,
			nDiv:     442,
			zOut:     []float64{0.2384705997428018E+01, 0.2452793225180968E+01, 0.6512368848977837E-06, 0.1344352644992036E-05, 0.1087126346389419E+01, 0.1155209171453138E+01, 0.9467601652019932E-05, 0.6398278760815103E-05, 0.1567256659807859E+01, 0.1608632505364561E+01, 0.1290748567965171E-01, 0.2672219415054498E-01, 0.7466016465687454E+00, 0.7570240220111620E+00, 0.8921689079377818E-01, 0.7057368234283014E-01, 0.1077323583030640E+01, 0.9438288517354918E+00, 0.3187938243378626E+00, 0.2908001941945216E+00, 0.8457518677063927E+00, 0.1181031209538913E+01, 0.8901276181133644E-01, 0.5160305461093759E-01, 0.1546500520154876E+01, 0.1458880876708516E+01, 0.1778606876095877E+00, 0.2447209773632920E+00, 0.9211044497840675E+00, 0.1123980661024405E+01, 0.4586493023634136E-01, 0.4307304847484577E-01, 0.9115921349407204E+00, 0.9808080186012680E+00, 0.1030014605117353E+00, 0.4473761868138884E-01, 0.2046221921921390E+01, 0.2098800160969815E+01, 0.6518220171905387E-01, 0.1185117935689049E+00, 0.1076216474932694E+01, 0.1125434406653226E+01, 0.1285328723941422E+00, 0.8405284210411720E-01, 0.1755252501385274E+01, 0.1645740838479294E+01, 0.2442959445536252E+00, 0.3061331074057168E+00, 0.1095350936043985E+01, 0.1400701385713701E+01, 0.7680442260639305E-02, 0.7034066989504621E-02, 0.1135395583018709E+01, 0.1196005046863439E+01, 0.5090399511302622E-02, 0.1515955052150448E-01, 0.3145781222952156E+00, 0.3812525386379337E+00, 0.3538316586334169E-02, 0.6504555274179689E-02, 0.1084421501965814E+00, 0.1711226887768117E+00, 0.7375387005938499E-02, 0.8946350111698947E-02, 0.1396793466565477E-01, 0.8939990224728768E-01, 0.1592074691534444E-03, 0.3199152990069927E-04, 0.1264354865903830E-02, 0.6951213444065239E-01, 0.6512368848977837E-06, 0.8997910999570377E-34, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    18,
			ppOut:    1,
			dminOut:  1.2643548659038301E-003,
			desigOut: -5.0382355155104463E-018,
			sigmaOut: 9.6270319212300656E-002,
			qmaxOut:  2.4715692224139039,
			nFailOut: 2,
			iterOut:  22,
			nDivOut:  461,
			ttypeOut: -8,
			dmin1Out: 1.3935943135754067E-002,
			dmin2Out: 9.9495800084882416E-002,
			dnOut:    1.2643548659038301E-003,
			dn1Out:   1.3935943135754067E-002,
			dn2Out:   9.9495800084882416E-002,
			gOut:     0.0000000000000000,
			tauOut:   6.8088572105595116E-002,
		},
		{
			i0:       1,
			n0:       18,
			z:        []float64{0.2384705997428018E+01, 0.2452793225180968E+01, 0.6512368848977837E-06, 0.1344352644992036E-05, 0.1087126346389419E+01, 0.1155209171453138E+01, 0.9467601652019932E-05, 0.6398278760815103E-05, 0.1567256659807859E+01, 0.1608632505364561E+01, 0.1290748567965171E-01, 0.2672219415054498E-01, 0.7466016465687454E+00, 0.7570240220111620E+00, 0.8921689079377818E-01, 0.7057368234283014E-01, 0.1077323583030640E+01, 0.9438288517354918E+00, 0.3187938243378626E+00, 0.2908001941945216E+00, 0.8457518677063927E+00, 0.1181031209538913E+01, 0.8901276181133644E-01, 0.5160305461093759E-01, 0.1546500520154876E+01, 0.1458880876708516E+01, 0.1778606876095877E+00, 0.2447209773632920E+00, 0.9211044497840675E+00, 0.1123980661024405E+01, 0.4586493023634136E-01, 0.4307304847484577E-01, 0.9115921349407204E+00, 0.9808080186012680E+00, 0.1030014605117353E+00, 0.4473761868138884E-01, 0.2046221921921390E+01, 0.2098800160969815E+01, 0.6518220171905387E-01, 0.1185117935689049E+00, 0.1076216474932694E+01, 0.1125434406653226E+01, 0.1285328723941422E+00, 0.8405284210411720E-01, 0.1755252501385274E+01, 0.1645740838479294E+01, 0.2442959445536252E+00, 0.3061331074057168E+00, 0.1095350936043985E+01, 0.1400701385713701E+01, 0.7680442260639305E-02, 0.7034066989504621E-02, 0.1135395583018709E+01, 0.1196005046863439E+01, 0.5090399511302622E-02, 0.1515955052150448E-01, 0.3145781222952156E+00, 0.3812525386379337E+00, 0.3538316586334169E-02, 0.6504555274179689E-02, 0.1084421501965814E+00, 0.1711226887768117E+00, 0.7375387005938499E-02, 0.8946350111698947E-02, 0.1396793466565477E-01, 0.8939990224728768E-01, 0.1592074691534444E-03, 0.3199152990069927E-04, 0.1264354865903830E-02, 0.6951213444065239E-01, 0.6512368848977837E-06, 0.1344352644992036E-05, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       0,
			dmin:     1.2643548659038301E-003,
			desig:    -5.0382355155104463E-018,
			qmax:     2.3847059974280183,
			ttype:    -8,
			dmin1:    1.3935943135754067E-002,
			dmin2:    9.9495800084882416E-002,
			dn:       1.2643548659038301E-003,
			dn1:      1.3935943135754067E-002,
			dn2:      9.9495800084882416E-002,
			g:        0.0000000000000000,
			tau:      6.8088572105595116E-002,
			nFail:    2,
			iter:     22,
			sigma:    9.6270319212300656E-002,
			nDiv:     461,
			zOut:     []float64{0.2384705997428018E+01, 0.2383460331210544E+01, 0.6512368848977837E-06, 0.2970373645586866E-06, 0.1087126346389419E+01, 0.1085889199499347E+01, 0.9467601652019932E-05, 0.1366452649899944E-04, 0.1567256659807859E+01, 0.1578904163506652E+01, 0.1290748567965171E-01, 0.6103442048115082E-02, 0.7466016465687454E+00, 0.8284687778600495E+00, 0.8921689079377818E-01, 0.1160157908485996E+00, 0.1077323583030640E+01, 0.1278855299065544E+01, 0.3187938243378626E+00, 0.2108295383723412E+00, 0.8457518677063927E+00, 0.7226887736910288E+00, 0.8901276181133644E-01, 0.1904807262171572E+00, 0.1546500520154876E+01, 0.1532634164092948E+01, 0.1778606876095877E+00, 0.1068932656188067E+00, 0.9211044497840675E+00, 0.8588297969472432E+00, 0.4586493023634136E-01, 0.4868264913684867E-01, 0.9115921349407204E+00, 0.9646646288612479E+00, 0.1030014605117353E+00, 0.2184840619043245E+00, 0.2046221921921390E+01, 0.1891673744281760E+01, 0.6518220171905387E-01, 0.3708364593761750E-01, 0.1076216474932694E+01, 0.1166419383934860E+01, 0.1285328723941422E+00, 0.1934189785315259E+00, 0.1755252501385274E+01, 0.1804883149953014E+01, 0.2442959445536252E+00, 0.1482587897978486E+00, 0.1095350936043985E+01, 0.9535262710524167E+00, 0.7680442260639305E-02, 0.9145359161143373E-02, 0.1135395583018709E+01, 0.1130094305914509E+01, 0.5090399511302622E-02, 0.1416986451145964E-02, 0.3145781222952156E+00, 0.3154531349760448E+00, 0.3538316586334169E-02, 0.1216353924418735E-02, 0.1084421501965814E+00, 0.1133548658237420E+00, 0.7375387005938499E-02, 0.9088178357782586E-03, 0.1396793466565477E-01, 0.1197200684467082E-01, 0.1592074691534444E-03, 0.1681378409852773E-04, 0.1264354865903830E-02, 0.1223627446170173E-05, 0.6512368848977837E-06, 0.2970373645586866E-06, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    18,
			ppOut:    0,
			dminOut:  1.2236274461701734E-006,
			desigOut: -3.2526065174565133E-018,
			sigmaOut: 9.7516636666659787E-002,
			qmaxOut:  2.3847059974280183,
			nFailOut: 2,
			iterOut:  23,
			nDivOut:  480,
			ttypeOut: -2,
			dmin1Out: 1.1812799375517376E-002,
			dmin2Out: 0.10597947881780349,
			dnOut:    1.2236274461701734E-006,
			dn1Out:   1.1812799375517376E-002,
			dn2Out:   0.10597947881780349,
			gOut:     0.0000000000000000,
			tauOut:   1.2463174543591322E-003,
		},
		{
			i0:       1,
			n0:       18,
			z:        []float64{0.2384705997428018E+01, 0.2383460331210544E+01, 0.6512368848977837E-06, 0.2970373645586866E-06, 0.1087126346389419E+01, 0.1085889199499347E+01, 0.9467601652019932E-05, 0.1366452649899944E-04, 0.1567256659807859E+01, 0.1578904163506652E+01, 0.1290748567965171E-01, 0.6103442048115082E-02, 0.7466016465687454E+00, 0.8284687778600495E+00, 0.8921689079377818E-01, 0.1160157908485996E+00, 0.1077323583030640E+01, 0.1278855299065544E+01, 0.3187938243378626E+00, 0.2108295383723412E+00, 0.8457518677063927E+00, 0.7226887736910288E+00, 0.8901276181133644E-01, 0.1904807262171572E+00, 0.1546500520154876E+01, 0.1532634164092948E+01, 0.1778606876095877E+00, 0.1068932656188067E+00, 0.9211044497840675E+00, 0.8588297969472432E+00, 0.4586493023634136E-01, 0.4868264913684867E-01, 0.9115921349407204E+00, 0.9646646288612479E+00, 0.1030014605117353E+00, 0.2184840619043245E+00, 0.2046221921921390E+01, 0.1891673744281760E+01, 0.6518220171905387E-01, 0.3708364593761750E-01, 0.1076216474932694E+01, 0.1166419383934860E+01, 0.1285328723941422E+00, 0.1934189785315259E+00, 0.1755252501385274E+01, 0.1804883149953014E+01, 0.2442959445536252E+00, 0.1482587897978486E+00, 0.1095350936043985E+01, 0.9535262710524167E+00, 0.7680442260639305E-02, 0.9145359161143373E-02, 0.1135395583018709E+01, 0.1130094305914509E+01, 0.5090399511302622E-02, 0.1416986451145964E-02, 0.3145781222952156E+00, 0.3154531349760448E+00, 0.3538316586334169E-02, 0.1216353924418735E-02, 0.1084421501965814E+00, 0.1133548658237420E+00, 0.7375387005938499E-02, 0.9088178357782586E-03, 0.1396793466565477E-01, 0.1197200684467082E-01, 0.1592074691534444E-03, 0.1681378409852773E-04, 0.1264354865903830E-02, 0.1223627446170173E-05, 0.6512368848977837E-06, 0.2970373645586866E-06, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       1,
			dmin:     1.2236274461701734E-006,
			desig:    -3.2526065174565133E-018,
			qmax:     2.3847059974280183,
			ttype:    -2,
			dmin1:    1.1812799375517376E-002,
			dmin2:    0.10597947881780349,
			dn:       1.2236274461701734E-006,
			dn1:      1.1812799375517376E-002,
			dn2:      0.10597947881780349,
			g:        0.0000000000000000,
			tau:      1.2463174543591322E-003,
			nFail:    2,
			iter:     23,
			sigma:    9.7516636666659787E-002,
			nDiv:     480,
			zOut:     []float64{0.2383459406360114E+01, 0.2383460331210544E+01, 0.1353283656358167E-06, 0.2970373645586866E-06, 0.1085901506809686E+01, 0.1085889199499347E+01, 0.1986826396898849E-04, 0.1366452649899944E-04, 0.1584986515403004E+01, 0.1578904163506652E+01, 0.3190255011763210E-02, 0.6103442048115082E-02, 0.9412930918090910E+00, 0.8284687778600495E+00, 0.1576208411525267E+00, 0.1160157908485996E+00, 0.1332062774397564E+01, 0.1278855299065544E+01, 0.1143821023097510E+00, 0.2108295383723412E+00, 0.7987861757106404E+00, 0.7226887736910288E+00, 0.3654761154847582E+00, 0.1904807262171572E+00, 0.1274050092339202E+01, 0.1532634164092948E+01, 0.7205613198290631E-01, 0.1068932656188067E+00, 0.8354550922133908E+00, 0.8588297969472432E+00, 0.5621179414582513E-01, 0.4868264913684867E-01, 0.1126935674731953E+01, 0.9646646288612479E+00, 0.3667472533840468E+00, 0.2184840619043245E+00, 0.1562008914947536E+01, 0.1891673744281760E+01, 0.2769195683500124E-01, 0.3708364593761750E-01, 0.1332145183743590E+01, 0.1166419383934860E+01, 0.2620575140703801E+00, 0.1934189785315259E+00, 0.1691083203792688E+01, 0.1804883149953014E+01, 0.8359650824372877E-01, 0.1482587897978486E+00, 0.8790739000820365E+00, 0.9535262710524167E+00, 0.1175682535061811E-01, 0.9145359161143373E-02, 0.1119753245127243E+01, 0.1130094305914509E+01, 0.3991886785572840E-03, 0.1416986451145964E-02, 0.3162690783341116E+00, 0.3154531349760448E+00, 0.4359567385560517E-03, 0.1216353924418735E-02, 0.1138265050331695E+00, 0.1133548658237420E+00, 0.9558734450579545E-04, 0.9088178357782586E-03, 0.1189201139646886E-01, 0.1197200684467082E-01, 0.1730052806966466E-08, 0.1681378409852773E-04, 0.9598672791582074E-11, 0.1223627446170173E-05, 0.1353283656358167E-06, 0.2970373645586866E-06, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    18,
			ppOut:    1,
			dminOut:  9.5986727915820745E-012,
			desigOut: 6.4755668817591261E-018,
			sigmaOut: 9.7517858554454467E-002,
			qmaxOut:  2.3847059974280183,
			nFailOut: 2,
			iterOut:  24,
			nDivOut:  499,
			ttypeOut: -2,
			dmin1Out: 1.1875197612370336E-002,
			dmin2Out: 0.11291768719739126,
			dnOut:    9.5986727915820745E-012,
			dn1Out:   1.1875197612370336E-002,
			dn2Out:   0.11291768719739126,
			gOut:     0.0000000000000000,
			tauOut:   1.2218877946904154E-006,
		},
		{
			i0:       1,
			n0:       18,
			z:        []float64{0.2383459406360114E+01, 0.2383460331210544E+01, 0.1353283656358167E-06, 0.2970373645586866E-06, 0.1085901506809686E+01, 0.1085889199499347E+01, 0.1986826396898849E-04, 0.1366452649899944E-04, 0.1584986515403004E+01, 0.1578904163506652E+01, 0.3190255011763210E-02, 0.6103442048115082E-02, 0.9412930918090910E+00, 0.8284687778600495E+00, 0.1576208411525267E+00, 0.1160157908485996E+00, 0.1332062774397564E+01, 0.1278855299065544E+01, 0.1143821023097510E+00, 0.2108295383723412E+00, 0.7987861757106404E+00, 0.7226887736910288E+00, 0.3654761154847582E+00, 0.1904807262171572E+00, 0.1274050092339202E+01, 0.1532634164092948E+01, 0.7205613198290631E-01, 0.1068932656188067E+00, 0.8354550922133908E+00, 0.8588297969472432E+00, 0.5621179414582513E-01, 0.4868264913684867E-01, 0.1126935674731953E+01, 0.9646646288612479E+00, 0.3667472533840468E+00, 0.2184840619043245E+00, 0.1562008914947536E+01, 0.1891673744281760E+01, 0.2769195683500124E-01, 0.3708364593761750E-01, 0.1332145183743590E+01, 0.1166419383934860E+01, 0.2620575140703801E+00, 0.1934189785315259E+00, 0.1691083203792688E+01, 0.1804883149953014E+01, 0.8359650824372877E-01, 0.1482587897978486E+00, 0.8790739000820365E+00, 0.9535262710524167E+00, 0.1175682535061811E-01, 0.9145359161143373E-02, 0.1119753245127243E+01, 0.1130094305914509E+01, 0.3991886785572840E-03, 0.1416986451145964E-02, 0.3162690783341116E+00, 0.3154531349760448E+00, 0.4359567385560517E-03, 0.1216353924418735E-02, 0.1138265050331695E+00, 0.1133548658237420E+00, 0.9558734450579545E-04, 0.9088178357782586E-03, 0.1189201139646886E-01, 0.1197200684467082E-01, 0.1730052806966466E-08, 0.1681378409852773E-04, 0.9598672791582074E-11, 0.1223627446170173E-05, 0.1353283656358167E-06, 0.2970373645586866E-06, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       0,
			dmin:     9.5986727915820745E-012,
			desig:    6.4755668817591261E-018,
			qmax:     2.3847059974280183,
			ttype:    -2,
			dmin1:    1.1875197612370336E-002,
			dmin2:    0.11291768719739126,
			dn:       9.5986727915820745E-012,
			dn1:      1.1875197612370336E-002,
			dn2:      0.11291768719739126,
			g:        0.0000000000000000,
			tau:      1.2218877946904154E-006,
			nFail:    2,
			iter:     24,
			sigma:    9.7517858554454467E-002,
			nDiv:     499,
			zOut:     []float64{0.2383459406360114E+01, 0.2383459541678881E+01, 0.1353283656358167E-06, 0.6165545233233256E-07, 0.1085901506809686E+01, 0.1085921313408604E+01, 0.1986826396898849E-04, 0.2899927470478232E-04, 0.1584986515403004E+01, 0.1588147771130463E+01, 0.3190255011763210E-02, 0.1890859942802736E-02, 0.9412930918090910E+00, 0.1097023073009216E+01, 0.1576208411525267E+00, 0.1913914667196325E+00, 0.1332062774397564E+01, 0.1255053409978084E+01, 0.1143821023097510E+00, 0.7279916643176543E-01, 0.7987861757106404E+00, 0.1091463124754034E+01, 0.3654761154847582E+00, 0.4266153094141972E+00, 0.1274050092339202E+01, 0.9194909148983121E+00, 0.7205613198290631E-01, 0.6547064404326042E-01, 0.8354550922133908E+00, 0.8261962423063568E+00, 0.5621179414582513E-01, 0.7667315937770838E-01, 0.1126935674731953E+01, 0.1417009768728692E+01, 0.3667472533840468E+00, 0.4042756034295816E+00, 0.1562008914947536E+01, 0.1185425268343357E+01, 0.2769195683500124E-01, 0.3111938635974581E-01, 0.1332145183743590E+01, 0.1563083311444625E+01, 0.2620575140703801E+00, 0.2835172362390010E+00, 0.1691083203792688E+01, 0.1491162475787817E+01, 0.8359650824372877E-01, 0.4928202642453804E-01, 0.8790739000820365E+00, 0.8415486989985177E+00, 0.1175682535061811E-01, 0.1564347179719430E-01, 0.1119753245127243E+01, 0.1104508961999007E+01, 0.3991886785572840E-03, 0.1143051254380294E-03, 0.3162690783341116E+00, 0.3165907299376309E+00, 0.4359567385560517E-03, 0.1567431614478116E-03, 0.1138265050331695E+00, 0.1137653492066288E+00, 0.9558734450579545E-04, 0.9991845479738400E-05, 0.1189201139646886E-01, 0.1188202127144326E-01, 0.1730052806966466E-08, 0.1397591405272086E-17, 0.9598672791582074E-11, 0.6617412588681544E-21, 0.1353283656358167E-06, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    18,
			ppOut:    0,
			dminOut:  6.6174125886815435E-022,
			desigOut: -5.0906815225379586E-018,
			sigmaOut: 9.7517858564053150E-002,
			qmaxOut:  2.3847059974280183,
			nFailOut: 2,
			iterOut:  25,
			nDivOut:  518,
			ttypeOut: -2,
			dmin1Out: 1.1882019541390456E-002,
			dmin2Out: 0.11366976186212303,
			dnOut:    6.6174125886815435E-022,
			dn1Out:   1.1882019541390456E-002,
			dn2Out:   0.11366976186212303,
			gOut:     0.0000000000000000,
			tauOut:   9.5986713933289272E-012,
		},
		{
			i0:       1,
			n0:       18,
			z:        []float64{0.2383459406360114E+01, 0.2383459541678881E+01, 0.1353283656358167E-06, 0.6165545233233256E-07, 0.1085901506809686E+01, 0.1085921313408604E+01, 0.1986826396898849E-04, 0.2899927470478232E-04, 0.1584986515403004E+01, 0.1588147771130463E+01, 0.3190255011763210E-02, 0.1890859942802736E-02, 0.9412930918090910E+00, 0.1097023073009216E+01, 0.1576208411525267E+00, 0.1913914667196325E+00, 0.1332062774397564E+01, 0.1255053409978084E+01, 0.1143821023097510E+00, 0.7279916643176543E-01, 0.7987861757106404E+00, 0.1091463124754034E+01, 0.3654761154847582E+00, 0.4266153094141972E+00, 0.1274050092339202E+01, 0.9194909148983121E+00, 0.7205613198290631E-01, 0.6547064404326042E-01, 0.8354550922133908E+00, 0.8261962423063568E+00, 0.5621179414582513E-01, 0.7667315937770838E-01, 0.1126935674731953E+01, 0.1417009768728692E+01, 0.3667472533840468E+00, 0.4042756034295816E+00, 0.1562008914947536E+01, 0.1185425268343357E+01, 0.2769195683500124E-01, 0.3111938635974581E-01, 0.1332145183743590E+01, 0.1563083311444625E+01, 0.2620575140703801E+00, 0.2835172362390010E+00, 0.1691083203792688E+01, 0.1491162475787817E+01, 0.8359650824372877E-01, 0.4928202642453804E-01, 0.8790739000820365E+00, 0.8415486989985177E+00, 0.1175682535061811E-01, 0.1564347179719430E-01, 0.1119753245127243E+01, 0.1104508961999007E+01, 0.3991886785572840E-03, 0.1143051254380294E-03, 0.3162690783341116E+00, 0.3165907299376309E+00, 0.4359567385560517E-03, 0.1567431614478116E-03, 0.1138265050331695E+00, 0.1137653492066288E+00, 0.9558734450579545E-04, 0.9991845479738400E-05, 0.1189201139646886E-01, 0.1188202127144326E-01, 0.1730052806966466E-08, 0.1397591405272086E-17, 0.9598672791582074E-11, 0.6617412588681544E-21, 0.1353283656358167E-06, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       1,
			dmin:     6.6174125886815435E-022,
			desig:    -5.0906815225379586E-018,
			qmax:     2.3847059974280183,
			ttype:    -2,
			dmin1:    1.1882019541390456E-002,
			dmin2:    0.11366976186212303,
			dn:       6.6174125886815435E-022,
			dn1:      1.1882019541390456E-002,
			dn2:      0.11366976186212303,
			g:        0.0000000000000000,
			tau:      9.5986713933289272E-012,
			nFail:    2,
			iter:     25,
			sigma:    9.7517858564053150E-002,
			nDiv:     518,
			zOut:     []float64{0.2383459603334333E+01, 0.2383459541678881E+01, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.1085950284592642E+01, 0.1085921313408604E+01, 0.4240998334843304E-04, 0.2899927470478232E-04, 0.1589996221089918E+01, 0.1588147771130463E+01, 0.1304604978030436E-02, 0.1890859942802736E-02, 0.1287109934750819E+01, 0.1097023073009216E+01, 0.1866247058326724E+00, 0.1913914667196325E+00, 0.1141227870577177E+01, 0.1255053409978084E+01, 0.6962466280543776E-01, 0.7279916643176543E-01, 0.1448453771362794E+01, 0.1091463124754034E+01, 0.2708190685256155E+00, 0.4266153094141972E+00, 0.7141424904159570E+00, 0.9194909148983121E+00, 0.7574342769943966E-01, 0.6547064404326042E-01, 0.8271259739846255E+00, 0.8261962423063568E+00, 0.1313543755784947E+00, 0.7667315937770838E-01, 0.1689930996579779E+01, 0.1417009768728692E+01, 0.2835846650840220E+00, 0.4042756034295816E+00, 0.9329599896190804E+00, 0.1185425268343357E+01, 0.5213749144931325E-01, 0.3111938635974581E-01, 0.1794463056234313E+01, 0.1563083311444625E+01, 0.2355970842920850E+00, 0.2835172362390010E+00, 0.1304847417920270E+01, 0.1491162475787817E+01, 0.3178396542921673E-01, 0.4928202642453804E-01, 0.8254082053664953E+00, 0.8415486989985177E+00, 0.2093310277804656E-01, 0.1564347179719430E-01, 0.1083690164346398E+01, 0.1104508961999007E+01, 0.3339325601415243E-04, 0.1143051254380294E-03, 0.3167140798430647E+00, 0.3165907299376309E+00, 0.5630296103885627E-04, 0.1567431614478116E-03, 0.1137190380910697E+00, 0.1137653492066288E+00, 0.1044005669799533E-05, 0.9991845479738400E-05, 0.1188097726577346E-01, 0.1188202127144326E-01, 0.7784240935906335E-37, 0.1397591405272086E-17, 0.6617412588681543E-21, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    18,
			ppOut:    1,
			dminOut:  6.6174125886815426E-022,
			desigOut: -5.0906815225379586E-018,
			sigmaOut: 9.7517858564053150E-002,
			qmaxOut:  2.3847059974280183,
			nFailOut: 2,
			iterOut:  26,
			nDivOut:  537,
			ttypeOut: -2,
			dmin1Out: 1.1880977265773463E-002,
			dmin2Out: 0.11370904624558997,
			dnOut:    6.6174125886815426E-022,
			dn1Out:   1.1880977265773463E-002,
			dn2Out:   0.11370904624558997,
			gOut:     0.0000000000000000,
			tauOut:   0.0000000000000000,
		},
		{
			i0:       1,
			n0:       18,
			z:        []float64{0.2383459603334333E+01, 0.2383459541678881E+01, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.1085950284592642E+01, 0.1085921313408604E+01, 0.4240998334843304E-04, 0.2899927470478232E-04, 0.1589996221089918E+01, 0.1588147771130463E+01, 0.1304604978030436E-02, 0.1890859942802736E-02, 0.1287109934750819E+01, 0.1097023073009216E+01, 0.1866247058326724E+00, 0.1913914667196325E+00, 0.1141227870577177E+01, 0.1255053409978084E+01, 0.6962466280543776E-01, 0.7279916643176543E-01, 0.1448453771362794E+01, 0.1091463124754034E+01, 0.2708190685256155E+00, 0.4266153094141972E+00, 0.7141424904159570E+00, 0.9194909148983121E+00, 0.7574342769943966E-01, 0.6547064404326042E-01, 0.8271259739846255E+00, 0.8261962423063568E+00, 0.1313543755784947E+00, 0.7667315937770838E-01, 0.1689930996579779E+01, 0.1417009768728692E+01, 0.2835846650840220E+00, 0.4042756034295816E+00, 0.9329599896190804E+00, 0.1185425268343357E+01, 0.5213749144931325E-01, 0.3111938635974581E-01, 0.1794463056234313E+01, 0.1563083311444625E+01, 0.2355970842920850E+00, 0.2835172362390010E+00, 0.1304847417920270E+01, 0.1491162475787817E+01, 0.3178396542921673E-01, 0.4928202642453804E-01, 0.8254082053664953E+00, 0.8415486989985177E+00, 0.2093310277804656E-01, 0.1564347179719430E-01, 0.1083690164346398E+01, 0.1104508961999007E+01, 0.3339325601415243E-04, 0.1143051254380294E-03, 0.3167140798430647E+00, 0.3165907299376309E+00, 0.5630296103885627E-04, 0.1567431614478116E-03, 0.1137190380910697E+00, 0.1137653492066288E+00, 0.1044005669799533E-05, 0.9991845479738400E-05, 0.1188097726577346E-01, 0.1188202127144326E-01, 0.7784240935906335E-37, 0.1397591405272086E-17, 0.6617412588681543E-21, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       0,
			dmin:     6.6174125886815426E-022,
			desig:    -5.0906815225379586E-018,
			qmax:     2.3847059974280183,
			ttype:    -2,
			dmin1:    1.1880977265773463E-002,
			dmin2:    0.11370904624558997,
			dn:       6.6174125886815426E-022,
			dn1:      1.1880977265773463E-002,
			dn2:      0.11370904624558997,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    2,
			iter:     26,
			sigma:    9.7517858564053150E-002,
			nDiv:     537,
			zOut:     []float64{0.2383459603334333E+01, 0.2371578799269292E+01, 0.2809066689524106E-07, 0.1286276792433928E-07, 0.1085950284592642E+01, 0.1074111849557515E+01, 0.4240998334843304E-04, 0.6277904232066118E-04, 0.1589996221089918E+01, 0.1579357214869920E+01, 0.1304604978030436E-02, 0.1063198377376995E-02, 0.1287109934750819E+01, 0.1460790610050406E+01, 0.1866247058326724E+00, 0.1457986614708343E+00, 0.1141227870577177E+01, 0.1053173039756072E+01, 0.6962466280543776E-01, 0.9575644420574685E-01, 0.1448453771362794E+01, 0.1611635563526955E+01, 0.2708190685256155E+00, 0.1200044280642223E+00, 0.7141424904159570E+00, 0.6580006578954666E+00, 0.7574342769943966E-01, 0.9521169265880262E-01, 0.8271259739846255E+00, 0.8513878247486099E+00, 0.1313543755784947E+00, 0.2607270439790754E+00, 0.1689930996579779E+01, 0.1700907785529018E+01, 0.2835846650840220E+00, 0.1555482010511415E+00, 0.9329599896190804E+00, 0.8176684478615445E+00, 0.5213749144931325E-01, 0.1144214412274440E+00, 0.1794463056234313E+01, 0.1903757867143247E+01, 0.2355970842920850E+00, 0.1614796988702029E+00, 0.1304847417920270E+01, 0.1163270852323576E+01, 0.3178396542921673E-01, 0.2255256874351997E-01, 0.8254082053664953E+00, 0.8119079072453143E+00, 0.2093310277804656E-01, 0.2794035799797570E-01, 0.1083690164346398E+01, 0.1043902367448729E+01, 0.3339325601415243E-04, 0.1013132519024162E-04, 0.3167140798430647E+00, 0.3048794193232055E+00, 0.5630296103885627E-04, 0.2100082250625822E-04, 0.1137190380910697E+00, 0.1018182491185255E+00, 0.1044005669799533E-05, 0.1218230301111122E-06, 0.1188097726577346E-01, 0.2328703557243073E-07, 0.7784240935906335E-37, 0.1286276792433928E-07, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    17,
			ppOut:    0,
			dminOut:  2.3287035572430725E-008,
			desigOut: 1.7347234759768071E-018,
			sigmaOut: 0.10939869071976092,
			qmaxOut:  2.3847059974280183,
			nFailOut: 2,
			iterOut:  27,
			nDivOut:  555,
			ttypeOut: -7,
			dmin1Out: 0.10181720511285566,
			dmin2Out: 0.30482311636216664,
			dnOut:    2.3287035572430725E-008,
			dn1Out:   0.10181720511285566,
			dn2Out:   0.30482311636216664,
			gOut:     0.0000000000000000,
			tauOut:   1.1880832155707781E-002,
		},
		{
			i0:       1,
			n0:       17,
			z:        []float64{0.2383459603334333E+01, 0.2371578799269292E+01, 0.2809066689524106E-07, 0.1286276792433928E-07, 0.1085950284592642E+01, 0.1074111849557515E+01, 0.4240998334843304E-04, 0.6277904232066118E-04, 0.1589996221089918E+01, 0.1579357214869920E+01, 0.1304604978030436E-02, 0.1063198377376995E-02, 0.1287109934750819E+01, 0.1460790610050406E+01, 0.1866247058326724E+00, 0.1457986614708343E+00, 0.1141227870577177E+01, 0.1053173039756072E+01, 0.6962466280543776E-01, 0.9575644420574685E-01, 0.1448453771362794E+01, 0.1611635563526955E+01, 0.2708190685256155E+00, 0.1200044280642223E+00, 0.7141424904159570E+00, 0.6580006578954666E+00, 0.7574342769943966E-01, 0.9521169265880262E-01, 0.8271259739846255E+00, 0.8513878247486099E+00, 0.1313543755784947E+00, 0.2607270439790754E+00, 0.1689930996579779E+01, 0.1700907785529018E+01, 0.2835846650840220E+00, 0.1555482010511415E+00, 0.9329599896190804E+00, 0.8176684478615445E+00, 0.5213749144931325E-01, 0.1144214412274440E+00, 0.1794463056234313E+01, 0.1903757867143247E+01, 0.2355970842920850E+00, 0.1614796988702029E+00, 0.1304847417920270E+01, 0.1163270852323576E+01, 0.3178396542921673E-01, 0.2255256874351997E-01, 0.8254082053664953E+00, 0.8119079072453143E+00, 0.2093310277804656E-01, 0.2794035799797570E-01, 0.1083690164346398E+01, 0.1043902367448729E+01, 0.3339325601415243E-04, 0.1013132519024162E-04, 0.3167140798430647E+00, 0.3048794193232055E+00, 0.5630296103885627E-04, 0.2100082250625822E-04, 0.1137190380910697E+00, 0.1018182491185255E+00, 0.1044005669799533E-05, 0.1218230301111122E-06, 0.1188097726577346E-01, 0.2328703557243073E-07, 0.7784240935906335E-37, 0.1286276792433928E-07, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       1,
			dmin:     2.3287035572430725E-008,
			desig:    1.7347234759768071E-018,
			qmax:     2.3847059974280183,
			ttype:    -7,
			dmin1:    0.10181720511285566,
			dmin2:    0.30482311636216664,
			dn:       2.3287035572430725E-008,
			dn1:      0.10181720511285566,
			dn2:      0.30482311636216664,
			g:        0.0000000000000000,
			tau:      1.1880832155707781E-002,
			nFail:    2,
			iter:     27,
			sigma:    0.10939869071976092,
			nDiv:     555,
			zOut:     []float64{0.2371578788845052E+01, 0.2371578799269292E+01, 0.5825676764620370E-08, 0.1286276792433928E-07, 0.1074174599487151E+01, 0.1074111849557515E+01, 0.9230392664199863E-04, 0.6277904232066118E-04, 0.1580328086033647E+01, 0.1579357214869920E+01, 0.9827770701659698E-03, 0.1063198377376995E-02, 0.1605606471164067E+01, 0.1460790610050406E+01, 0.9563440497488795E-01, 0.1457986614708343E+00, 0.1053295055699924E+01, 0.1053173039756072E+01, 0.1465159169633776E+00, 0.9575644420574685E-01, 0.1585124051340792E+01, 0.1611635563526955E+01, 0.4981502397231050E-01, 0.1200044280642223E+00, 0.7033973032949510E+00, 0.6580006578954666E+00, 0.1152436546510613E+00, 0.9521169265880262E-01, 0.9968711907896164E+00, 0.8513878247486099E+00, 0.4448645553200346E+00, 0.2607270439790754E+00, 0.1411591407973117E+01, 0.1700907785529018E+01, 0.9010174998427344E-01, 0.1555482010511415E+00, 0.8419881158177074E+00, 0.8176684478615445E+00, 0.2587099684834221E+00, 0.1144214412274440E+00, 0.1806527574243020E+01, 0.1903757867143247E+01, 0.1039810460775319E+00, 0.1614796988702029E+00, 0.1081842351702556E+01, 0.1163270852323576E+01, 0.1692539477932339E-01, 0.2255256874351997E-01, 0.8229228471769590E+00, 0.8119079072453143E+00, 0.3544318396494814E-01, 0.2794035799797570E-01, 0.1008469291521964E+01, 0.1043902367448729E+01, 0.3062892015595061E-05, 0.1013132519024162E-04, 0.3048973339666884E+00, 0.3048794193232055E+00, 0.7013072071892081E-05, 0.2100082250625822E-04, 0.1018113345824760E+00, 0.1018182491185255E+00, 0.2786425742647189E-13, 0.1218230301111122E-06, 0.2696165428114542E-17, 0.2328703557243073E-07, 0.5825676764620370E-08, 0.1286276792433928E-07, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    17,
			ppOut:    1,
			dminOut:  2.6961654281145418E-018,
			desigOut: -3.8251180717391650E-018,
			sigmaOut: 0.10939871400676864,
			qmaxOut:  2.3847059974280183,
			nFailOut: 2,
			iterOut:  28,
			nDivOut:  573,
			ttypeOut: -2,
			dmin1Out: 0.10181121275944585,
			dmin2Out: 0.30487633314418217,
			dnOut:    2.6961654281145418E-018,
			dn1Out:   0.10181121275944585,
			dn2Out:   0.30487633314418217,
			gOut:     0.0000000000000000,
			tauOut:   2.3287007705477136E-008,
		},
		{
			i0:       1,
			n0:       17,
			z:        []float64{0.2371578788845052E+01, 0.2371578799269292E+01, 0.5825676764620370E-08, 0.1286276792433928E-07, 0.1074174599487151E+01, 0.1074111849557515E+01, 0.9230392664199863E-04, 0.6277904232066118E-04, 0.1580328086033647E+01, 0.1579357214869920E+01, 0.9827770701659698E-03, 0.1063198377376995E-02, 0.1605606471164067E+01, 0.1460790610050406E+01, 0.9563440497488795E-01, 0.1457986614708343E+00, 0.1053295055699924E+01, 0.1053173039756072E+01, 0.1465159169633776E+00, 0.9575644420574685E-01, 0.1585124051340792E+01, 0.1611635563526955E+01, 0.4981502397231050E-01, 0.1200044280642223E+00, 0.7033973032949510E+00, 0.6580006578954666E+00, 0.1152436546510613E+00, 0.9521169265880262E-01, 0.9968711907896164E+00, 0.8513878247486099E+00, 0.4448645553200346E+00, 0.2607270439790754E+00, 0.1411591407973117E+01, 0.1700907785529018E+01, 0.9010174998427344E-01, 0.1555482010511415E+00, 0.8419881158177074E+00, 0.8176684478615445E+00, 0.2587099684834221E+00, 0.1144214412274440E+00, 0.1806527574243020E+01, 0.1903757867143247E+01, 0.1039810460775319E+00, 0.1614796988702029E+00, 0.1081842351702556E+01, 0.1163270852323576E+01, 0.1692539477932339E-01, 0.2255256874351997E-01, 0.8229228471769590E+00, 0.8119079072453143E+00, 0.3544318396494814E-01, 0.2794035799797570E-01, 0.1008469291521964E+01, 0.1043902367448729E+01, 0.3062892015595061E-05, 0.1013132519024162E-04, 0.3048973339666884E+00, 0.3048794193232055E+00, 0.7013072071892081E-05, 0.2100082250625822E-04, 0.1018113345824760E+00, 0.1018182491185255E+00, 0.2786425742647189E-13, 0.1218230301111122E-06, 0.2696165428114542E-17, 0.2328703557243073E-07, 0.5825676764620370E-08, 0.1286276792433928E-07, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       0,
			dmin:     2.6961654281145418E-018,
			desig:    -3.8251180717391650E-018,
			qmax:     2.3847059974280183,
			ttype:    -2,
			dmin1:    0.10181121275944585,
			dmin2:    0.30487633314418217,
			dn:       2.6961654281145418E-018,
			dn1:      0.10181121275944585,
			dn2:      0.30487633314418217,
			g:        0.0000000000000000,
			tau:      2.3287007705477136E-008,
			nFail:    2,
			iter:     28,
			sigma:    0.10939871400676864,
			nDiv:     573,
			zOut:     []float64{0.2371578788845052E+01, 0.2371578794670729E+01, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.1074174599487151E+01, 0.1074266900775131E+01, 0.9230392664199863E-04, 0.1357860766428602E-03, 0.1580328086033647E+01, 0.1581175077027170E+01, 0.9827770701659698E-03, 0.9979623676695663E-03, 0.1605606471164067E+01, 0.1700242913771285E+01, 0.9563440497488795E-01, 0.5924520849284005E-01, 0.1053295055699924E+01, 0.1140565764170461E+01, 0.1465159169633776E+00, 0.2036234219705987E+00, 0.1585124051340792E+01, 0.1431315653342504E+01, 0.4981502397231050E-01, 0.2448080089382756E-01, 0.7033973032949510E+00, 0.7941601570521848E+00, 0.1152436546510613E+00, 0.1446598374682775E+00, 0.9968711907896164E+00, 0.1297075908641373E+01, 0.4448645553200346E+00, 0.4841405038964208E+00, 0.1411591407973117E+01, 0.1017552654060970E+01, 0.9010174998427344E-01, 0.7455594793877945E-01, 0.8419881158177074E+00, 0.1026142136362350E+01, 0.2587099684834221E+00, 0.4554599945126984E+00, 0.1806527574243020E+01, 0.1455048625807853E+01, 0.1039810460775319E+00, 0.7731088667813635E-01, 0.1081842351702556E+01, 0.1021456859803743E+01, 0.1692539477932339E-01, 0.1363571444815687E-01, 0.8229228471769590E+00, 0.8447303166937503E+00, 0.3544318396494814E-01, 0.4231334180394086E-01, 0.1008469291521964E+01, 0.9661590126100381E+00, 0.3062892015595061E-05, 0.9665775484099522E-06, 0.3048973339666884E+00, 0.3049033804612119E+00, 0.7013072071892081E-05, 0.2341758973227438E-05, 0.1018113345824760E+00, 0.1018089928235306E+00, 0.2786425742647189E-13, 0.7379175991216932E-30, 0.2696165428114542E-17, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    17,
			ppOut:    0,
			dminOut:  2.6961654281138038E-018,
			desigOut: -3.8251180717391650E-018,
			sigmaOut: 0.10939871400676864,
			qmaxOut:  2.3847059974280183,
			nFailOut: 2,
			iterOut:  29,
			nDivOut:  591,
			ttypeOut: -2,
			dmin1Out: 0.10180899282350273,
			dmin2Out: 0.30489636738914000,
			dnOut:    2.6961654281138038E-018,
			dn1Out:   0.10180899282350273,
			dn2Out:   0.30489636738914000,
			gOut:     0.0000000000000000,
			tauOut:   0.0000000000000000,
		},
		{
			i0:       1,
			n0:       17,
			z:        []float64{0.2371578788845052E+01, 0.2371578794670729E+01, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.1074174599487151E+01, 0.1074266900775131E+01, 0.9230392664199863E-04, 0.1357860766428602E-03, 0.1580328086033647E+01, 0.1581175077027170E+01, 0.9827770701659698E-03, 0.9979623676695663E-03, 0.1605606471164067E+01, 0.1700242913771285E+01, 0.9563440497488795E-01, 0.5924520849284005E-01, 0.1053295055699924E+01, 0.1140565764170461E+01, 0.1465159169633776E+00, 0.2036234219705987E+00, 0.1585124051340792E+01, 0.1431315653342504E+01, 0.4981502397231050E-01, 0.2448080089382756E-01, 0.7033973032949510E+00, 0.7941601570521848E+00, 0.1152436546510613E+00, 0.1446598374682775E+00, 0.9968711907896164E+00, 0.1297075908641373E+01, 0.4448645553200346E+00, 0.4841405038964208E+00, 0.1411591407973117E+01, 0.1017552654060970E+01, 0.9010174998427344E-01, 0.7455594793877945E-01, 0.8419881158177074E+00, 0.1026142136362350E+01, 0.2587099684834221E+00, 0.4554599945126984E+00, 0.1806527574243020E+01, 0.1455048625807853E+01, 0.1039810460775319E+00, 0.7731088667813635E-01, 0.1081842351702556E+01, 0.1021456859803743E+01, 0.1692539477932339E-01, 0.1363571444815687E-01, 0.8229228471769590E+00, 0.8447303166937503E+00, 0.3544318396494814E-01, 0.4231334180394086E-01, 0.1008469291521964E+01, 0.9661590126100381E+00, 0.3062892015595061E-05, 0.9665775484099522E-06, 0.3048973339666884E+00, 0.3049033804612119E+00, 0.7013072071892081E-05, 0.2341758973227438E-05, 0.1018113345824760E+00, 0.1018089928235306E+00, 0.2786425742647189E-13, 0.7379175991216932E-30, 0.2696165428114542E-17, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       1,
			dmin:     2.6961654281138038E-018,
			desig:    -3.8251180717391650E-018,
			qmax:     2.3847059974280183,
			ttype:    -2,
			dmin1:    0.10180899282350273,
			dmin2:    0.30489636738914000,
			dn:       2.6961654281138038E-018,
			dn1:      0.10180899282350273,
			dn2:      0.30489636738914000,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    2,
			iter:     29,
			sigma:    0.10939871400676864,
			nDiv:     591,
			zOut:     []float64{0.2269772292606755E+01, 0.2371578794670729E+01, 0.1248859578385863E-08, 0.2638661645752538E-08, 0.9725961809002787E+00, 0.1074266900775131E+01, 0.2207509801202943E-03, 0.1357860766428602E-03, 0.1480145783712083E+01, 0.1581175077027170E+01, 0.1146359002276934E-02, 0.9979623676695663E-03, 0.1656535258559212E+01, 0.1700242913771285E+01, 0.4079180092843103E-01, 0.5924520849284005E-01, 0.1201590880509993E+01, 0.1140565764170461E+01, 0.2425529321011353E+00, 0.2036234219705987E+00, 0.1111437017432560E+01, 0.1431315653342504E+01, 0.1749237822536809E-01, 0.2448080089382756E-01, 0.8195211115924584E+00, 0.7941601570521848E+00, 0.2289566278084960E+00, 0.1446598374682775E+00, 0.1450453280026662E+01, 0.1297075908641373E+01, 0.3396444831847066E+00, 0.4841405038964208E+00, 0.6506576141124067E+00, 0.1017552654060970E+01, 0.1175810411452473E+00, 0.7455594793877945E-01, 0.1262214585027166E+01, 0.1026142136362350E+01, 0.5250426092262996E+00, 0.4554599945126984E+00, 0.9055103985570538E+00, 0.1455048625807853E+01, 0.8721019179982008E-01, 0.7731088667813635E-01, 0.8460758777494440E+00, 0.1021456859803743E+01, 0.1361402882064932E-01, 0.1363571444815687E-01, 0.7716231249744060E+00, 0.8447303166937503E+00, 0.5298106707064092E-01, 0.4231334180394086E-01, 0.8113724074143097E+00, 0.9661590126100381E+00, 0.3632274887524200E-06, 0.9665775484099522E-06, 0.2030988542900605E+00, 0.3049033804612119E+00, 0.1173872316183806E-05, 0.2341758973227438E-05, 0.1314248578535016E-05, 0.1018089928235306E+00, 0.1248859578385863E-08, 0.7379175991216932E-30, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    16,
			ppOut:    1,
			dminOut:  1.3142485785350155E-006,
			desigOut: 0.0000000000000000,
			sigmaOut: 0.21120521870940451,
			qmaxOut:  2.3847059974280183,
			nFailOut: 2,
			iterOut:  30,
			nDivOut:  608,
			ttypeOut: -7,
			dmin1Out: 0.20309651253108729,
			dmin2Out: 0.57610166617362735,
			dnOut:    1.3142485785350155E-006,
			dn1Out:   0.20309651253108729,
			dn2Out:   0.81137144083676127,
			gOut:     0.0000000000000000,
			tauOut:   0.10180650470263587,
		},
		{
			i0:       1,
			n0:       16,
			z:        []float64{0.2269772292606755E+01, 0.2371578794670729E+01, 0.1248859578385863E-08, 0.2638661645752538E-08, 0.9725961809002787E+00, 0.1074266900775131E+01, 0.2207509801202943E-03, 0.1357860766428602E-03, 0.1480145783712083E+01, 0.1581175077027170E+01, 0.1146359002276934E-02, 0.9979623676695663E-03, 0.1656535258559212E+01, 0.1700242913771285E+01, 0.4079180092843103E-01, 0.5924520849284005E-01, 0.1201590880509993E+01, 0.1140565764170461E+01, 0.2425529321011353E+00, 0.2036234219705987E+00, 0.1111437017432560E+01, 0.1431315653342504E+01, 0.1749237822536809E-01, 0.2448080089382756E-01, 0.8195211115924584E+00, 0.7941601570521848E+00, 0.2289566278084960E+00, 0.1446598374682775E+00, 0.1450453280026662E+01, 0.1297075908641373E+01, 0.3396444831847066E+00, 0.4841405038964208E+00, 0.6506576141124067E+00, 0.1017552654060970E+01, 0.1175810411452473E+00, 0.7455594793877945E-01, 0.1262214585027166E+01, 0.1026142136362350E+01, 0.5250426092262996E+00, 0.4554599945126984E+00, 0.9055103985570538E+00, 0.1455048625807853E+01, 0.8721019179982008E-01, 0.7731088667813635E-01, 0.8460758777494440E+00, 0.1021456859803743E+01, 0.1361402882064932E-01, 0.1363571444815687E-01, 0.7716231249744060E+00, 0.8447303166937503E+00, 0.5298106707064092E-01, 0.4231334180394086E-01, 0.8113724074143097E+00, 0.9661590126100381E+00, 0.3632274887524200E-06, 0.9665775484099522E-06, 0.2030988542900605E+00, 0.3049033804612119E+00, 0.1173872316183806E-05, 0.2341758973227438E-05, 0.1314248578535016E-05, 0.1018089928235306E+00, 0.1248859578385863E-08, 0.2638661645752538E-08, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       0,
			dmin:     1.3142485785350155E-006,
			desig:    0.0000000000000000,
			qmax:     2.2697722926067549,
			ttype:    -7,
			dmin1:    0.20309651253108729,
			dmin2:    0.57610166617362735,
			dn:       1.3142485785350155E-006,
			dn1:      0.20309651253108729,
			dn2:      0.81137144083676127,
			g:        0.0000000000000000,
			tau:      0.10180650470263587,
			nFail:    2,
			iter:     30,
			sigma:    0.21120521870940451,
			nDiv:     608,
			zOut:     []float64{0.2269772292606755E+01, 0.2269770979614632E+01, 0.1248859578385863E-08, 0.5351359530665278E-09, 0.9725961809002787E+00, 0.9728156171042807E+00, 0.2207509801202943E-03, 0.3358741643642203E-03, 0.1480145783712083E+01, 0.1480954954309013E+01, 0.1146359002276934E-02, 0.1282269997958535E-02, 0.1656535258559212E+01, 0.1696043475248702E+01, 0.4079180092843103E-01, 0.2889964597634762E-01, 0.1201590880509993E+01, 0.1415242852393798E+01, 0.2425529321011353E+00, 0.1904848393814713E+00, 0.1111437017432560E+01, 0.9384432420354740E+00, 0.1749237822536809E-01, 0.1527569554079381E-01, 0.8195211115924584E+00, 0.1033200729619178E+01, 0.2289566278084960E+00, 0.3214195289148511E+00, 0.1450453280026662E+01, 0.1468676920055535E+01, 0.3396444831847066E+00, 0.1504703083827628E+00, 0.6506576141124067E+00, 0.6177670326339089E+00, 0.1175810411452473E+00, 0.2402402478867145E+00, 0.1262214585027166E+01, 0.1547015632125768E+01, 0.5250426092262996E+00, 0.3073217441808570E+00, 0.9055103985570538E+00, 0.6853975319350345E+00, 0.8721019179982008E-01, 0.1076549537133787E+00, 0.8460758777494440E+00, 0.7520336386157322E+00, 0.1361402882064932E-01, 0.1396865635082151E-01, 0.7716231249744060E+00, 0.8106342214532430E+00, 0.5298106707064092E-01, 0.5302931309687425E-01, 0.8113724074143097E+00, 0.7583421433039418E+00, 0.3632274887524200E-06, 0.9727942389020712E-07, 0.2030988542900605E+00, 0.2030986166419704E+00, 0.1173872316183806E-05, 0.7596112905317353E-11, 0.1314248578535016E-05, 0.8411884199182457E-17, 0.1248859578385863E-08, 0.5351359530665278E-09, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    16,
			ppOut:    0,
			dminOut:  8.4118841991824567E-018,
			desigOut: 1.0739530738237274E-017,
			sigmaOut: 0.21120653295038691,
			qmaxOut:  2.2697722926067549,
			nFailOut: 2,
			iterOut:  31,
			nDivOut:  625,
			ttypeOut: -2,
			dmin1Out: 0.20309744276965425,
			dmin2Out: 0.50018599148866161,
			dnOut:    8.4118841991824567E-018,
			dn1Out:   0.20309744276965425,
			dn2Out:   0.75834178007645303,
			gOut:     0.0000000000000000,
			tauOut:   1.3142409824136984E-006,
		},
		{
			i0:       1,
			n0:       16,
			z:        []float64{0.2269772292606755E+01, 0.2269770979614632E+01, 0.1248859578385863E-08, 0.5351359530665278E-09, 0.9725961809002787E+00, 0.9728156171042807E+00, 0.2207509801202943E-03, 0.3358741643642203E-03, 0.1480145783712083E+01, 0.1480954954309013E+01, 0.1146359002276934E-02, 0.1282269997958535E-02, 0.1656535258559212E+01, 0.1696043475248702E+01, 0.4079180092843103E-01, 0.2889964597634762E-01, 0.1201590880509993E+01, 0.1415242852393798E+01, 0.2425529321011353E+00, 0.1904848393814713E+00, 0.1111437017432560E+01, 0.9384432420354740E+00, 0.1749237822536809E-01, 0.1527569554079381E-01, 0.8195211115924584E+00, 0.1033200729619178E+01, 0.2289566278084960E+00, 0.3214195289148511E+00, 0.1450453280026662E+01, 0.1468676920055535E+01, 0.3396444831847066E+00, 0.1504703083827628E+00, 0.6506576141124067E+00, 0.6177670326339089E+00, 0.1175810411452473E+00, 0.2402402478867145E+00, 0.1262214585027166E+01, 0.1547015632125768E+01, 0.5250426092262996E+00, 0.3073217441808570E+00, 0.9055103985570538E+00, 0.6853975319350345E+00, 0.8721019179982008E-01, 0.1076549537133787E+00, 0.8460758777494440E+00, 0.7520336386157322E+00, 0.1361402882064932E-01, 0.1396865635082151E-01, 0.7716231249744060E+00, 0.8106342214532430E+00, 0.5298106707064092E-01, 0.5302931309687425E-01, 0.8113724074143097E+00, 0.7583421433039418E+00, 0.3632274887524200E-06, 0.9727942389020712E-07, 0.2030988542900605E+00, 0.2030986166419704E+00, 0.1173872316183806E-05, 0.7596112905317353E-11, 0.1314248578535016E-05, 0.8411884199182457E-17, 0.1248859578385863E-08, 0.5351359530665278E-09, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       1,
			dmin:     8.4118841991824567E-018,
			desig:    1.0739530738237274E-017,
			qmax:     2.2697722926067549,
			ttype:    -2,
			dmin1:    0.20309744276965425,
			dmin2:    0.50018599148866161,
			dn:       8.4118841991824567E-018,
			dn1:      0.20309744276965425,
			dn2:      0.75834178007645303,
			g:        0.0000000000000000,
			tau:      1.3142409824136984E-006,
			nFail:    2,
			iter:     31,
			sigma:    0.21120653295038691,
			nDiv:     625,
			zOut:     []float64{0.2269770980149768E+01, 0.2269770979614632E+01, 0.2293573303077261E-09, 0.5351359530665278E-09, 0.9731514910392876E+00, 0.9728156171042807E+00, 0.5111377954200868E-03, 0.3358741643642203E-03, 0.1481726086511552E+01, 0.1480954954309013E+01, 0.1467737987028945E-02, 0.1282269997958535E-02, 0.1723475383238021E+01, 0.1696043475248702E+01, 0.2373112943910766E-01, 0.2889964597634762E-01, 0.1581996562336162E+01, 0.1415242852393798E+01, 0.1129959536471923E+00, 0.1904848393814713E+00, 0.8407229839290754E+00, 0.9384432420354740E+00, 0.1877296098701645E-01, 0.1527569554079381E-01, 0.1335847297547013E+01, 0.1033200729619178E+01, 0.3533797947109679E+00, 0.3214195289148511E+00, 0.1265767433727330E+01, 0.1468676920055535E+01, 0.7343813202351114E-01, 0.1504703083827628E+00, 0.7845691484971122E+00, 0.6177670326339089E+00, 0.4737063898809231E+00, 0.2402402478867145E+00, 0.1380630986425702E+01, 0.1547015632125768E+01, 0.1525661578238559E+00, 0.3073217441808570E+00, 0.6404863278245572E+00, 0.6853975319350345E+00, 0.1264041760751794E+00, 0.1076549537133787E+00, 0.6395981188913744E+00, 0.7520336386157322E+00, 0.1770404028911661E-01, 0.1396865635082151E-01, 0.8459594942610007E+00, 0.8106342214532430E+00, 0.4753698401003136E-01, 0.5302931309687425E-01, 0.7108052565733343E+00, 0.7583421433039418E+00, 0.2779568135873871E-07, 0.9727942389020712E-07, 0.2030985888538852E+00, 0.2030986166419704E+00, 0.3146138162949754E-27, 0.7596112905317353E-11, 0.8411884198867843E-17, 0.8411884199182457E-17, 0.2293573303077261E-09, 0.5351359530665278E-09, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    16,
			ppOut:    1,
			dminOut:  8.4118841988678429E-018,
			desigOut: 1.0739530738237274E-017,
			sigmaOut: 0.21120653295038691,
			qmaxOut:  2.2697722926067549,
			nFailOut: 2,
			iterOut:  32,
			nDivOut:  642,
			ttypeOut: -2,
			dmin1Out: 0.20309858884628909,
			dmin2Out: 0.53283137411117854,
			dnOut:    8.4118841988678429E-018,
			dn1Out:   0.20309858884628909,
			dn2Out:   0.71080515929391042,
			gOut:     0.0000000000000000,
			tauOut:   0.0000000000000000,
		},
		{
			i0:       1,
			n0:       16,
			z:        []float64{0.2269770980149768E+01, 0.2269770979614632E+01, 0.2293573303077261E-09, 0.5351359530665278E-09, 0.9731514910392876E+00, 0.9728156171042807E+00, 0.5111377954200868E-03, 0.3358741643642203E-03, 0.1481726086511552E+01, 0.1480954954309013E+01, 0.1467737987028945E-02, 0.1282269997958535E-02, 0.1723475383238021E+01, 0.1696043475248702E+01, 0.2373112943910766E-01, 0.2889964597634762E-01, 0.1581996562336162E+01, 0.1415242852393798E+01, 0.1129959536471923E+00, 0.1904848393814713E+00, 0.8407229839290754E+00, 0.9384432420354740E+00, 0.1877296098701645E-01, 0.1527569554079381E-01, 0.1335847297547013E+01, 0.1033200729619178E+01, 0.3533797947109679E+00, 0.3214195289148511E+00, 0.1265767433727330E+01, 0.1468676920055535E+01, 0.7343813202351114E-01, 0.1504703083827628E+00, 0.7845691484971122E+00, 0.6177670326339089E+00, 0.4737063898809231E+00, 0.2402402478867145E+00, 0.1380630986425702E+01, 0.1547015632125768E+01, 0.1525661578238559E+00, 0.3073217441808570E+00, 0.6404863278245572E+00, 0.6853975319350345E+00, 0.1264041760751794E+00, 0.1076549537133787E+00, 0.6395981188913744E+00, 0.7520336386157322E+00, 0.1770404028911661E-01, 0.1396865635082151E-01, 0.8459594942610007E+00, 0.8106342214532430E+00, 0.4753698401003136E-01, 0.5302931309687425E-01, 0.7108052565733343E+00, 0.7583421433039418E+00, 0.2779568135873871E-07, 0.9727942389020712E-07, 0.2030985888538852E+00, 0.2030986166419704E+00, 0.3146138162949754E-27, 0.7596112905317353E-11, 0.8411884198867843E-17, 0.8411884199182457E-17, 0.2293573303077261E-09, 0.5351359530665278E-09, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       0,
			dmin:     8.4118841988678429E-018,
			desig:    1.0739530738237274E-017,
			qmax:     2.2697722926067549,
			ttype:    -2,
			dmin1:    0.20309858884628909,
			dmin2:    0.53283137411117854,
			dn:       8.4118841988678429E-018,
			dn1:      0.20309858884628909,
			dn2:      0.71080515929391042,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    2,
			iter:     32,
			sigma:    0.21120653295038691,
			nDiv:     642,
			zOut:     []float64{0.2269770980149768E+01, 0.2269770980379126E+01, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.9731514910392876E+00, 0.9736626287363720E+00, 0.5111377954200868E-03, 0.7778528033461282E-03, 0.1481726086511552E+01, 0.1482415971695234E+01, 0.1467737987028945E-02, 0.1706410574351102E-02, 0.1723475383238021E+01, 0.1745500102102777E+01, 0.2373112943910766E-01, 0.2150819994097728E-01, 0.1581996562336162E+01, 0.1673484316042377E+01, 0.1129959536471923E+00, 0.5676676764251993E-01, 0.8407229839290754E+00, 0.8027291772735718E+00, 0.1877296098701645E-01, 0.3124068479314146E-01, 0.1335847297547013E+01, 0.1657986407464839E+01, 0.3533797947109679E+00, 0.2697830536296953E+00, 0.1265767433727330E+01, 0.1069422512121146E+01, 0.7343813202351114E-01, 0.5387701498318342E-01, 0.7845691484971122E+00, 0.1204398523394852E+01, 0.4737063898809231E+00, 0.5430210247136315E+00, 0.1380630986425702E+01, 0.9901761195359265E+00, 0.1525661578238559E+00, 0.9868601781741700E-01, 0.6404863278245572E+00, 0.6682044860823195E+00, 0.1264041760751794E+00, 0.1209927124430272E+00, 0.6395981188913744E+00, 0.5363094467374639E+00, 0.1770404028911661E-01, 0.2792585709699239E-01, 0.8459594942610007E+00, 0.8655706211740396E+00, 0.4753698401003136E-01, 0.3903729781186600E-01, 0.7108052565733343E+00, 0.6717679865571495E+00, 0.2779568135873871E-07, 0.8403591378512072E-08, 0.2030985888538852E+00, 0.2030985804502939E+00, 0.3146138162949754E-27, 0.1303059324279677E-43, 0.8411884198867843E-17, 0.8411884198867843E-17, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    16,
			ppOut:    0,
			dminOut:  8.4118841988678429E-018,
			desigOut: 1.0739530738237274E-017,
			sigmaOut: 0.21120653295038691,
			qmaxOut:  2.2697722926067549,
			nFailOut: 2,
			iterOut:  33,
			nDivOut:  659,
			ttypeOut: -2,
			dmin1Out: 0.20309858045029386,
			dmin2Out: 0.51860540644834729,
			dnOut:    8.4118841988678429E-018,
			dn1Out:   0.20309858045029386,
			dn2Out:   0.67176795876146822,
			gOut:     0.0000000000000000,
			tauOut:   0.0000000000000000,
		},
		{
			i0:       1,
			n0:       16,
			z:        []float64{0.2269770980149768E+01, 0.2269770980379126E+01, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.9731514910392876E+00, 0.9736626287363720E+00, 0.5111377954200868E-03, 0.7778528033461282E-03, 0.1481726086511552E+01, 0.1482415971695234E+01, 0.1467737987028945E-02, 0.1706410574351102E-02, 0.1723475383238021E+01, 0.1745500102102777E+01, 0.2373112943910766E-01, 0.2150819994097728E-01, 0.1581996562336162E+01, 0.1673484316042377E+01, 0.1129959536471923E+00, 0.5676676764251993E-01, 0.8407229839290754E+00, 0.8027291772735718E+00, 0.1877296098701645E-01, 0.3124068479314146E-01, 0.1335847297547013E+01, 0.1657986407464839E+01, 0.3533797947109679E+00, 0.2697830536296953E+00, 0.1265767433727330E+01, 0.1069422512121146E+01, 0.7343813202351114E-01, 0.5387701498318342E-01, 0.7845691484971122E+00, 0.1204398523394852E+01, 0.4737063898809231E+00, 0.5430210247136315E+00, 0.1380630986425702E+01, 0.9901761195359265E+00, 0.1525661578238559E+00, 0.9868601781741700E-01, 0.6404863278245572E+00, 0.6682044860823195E+00, 0.1264041760751794E+00, 0.1209927124430272E+00, 0.6395981188913744E+00, 0.5363094467374639E+00, 0.1770404028911661E-01, 0.2792585709699239E-01, 0.8459594942610007E+00, 0.8655706211740396E+00, 0.4753698401003136E-01, 0.3903729781186600E-01, 0.7108052565733343E+00, 0.6717679865571495E+00, 0.2779568135873871E-07, 0.8403591378512072E-08, 0.2030985888538852E+00, 0.2030985804502939E+00, 0.3146138162949754E-27, 0.1303059324279677E-43, 0.8411884198867843E-17, 0.8411884198867843E-17, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       1,
			dmin:     8.4118841988678429E-018,
			desig:    1.0739530738237274E-017,
			qmax:     2.2697722926067549,
			ttype:    -2,
			dmin1:    0.20309858045029386,
			dmin2:    0.51860540644834729,
			dn:       8.4118841988678429E-018,
			dn1:      0.20309858045029386,
			dn2:      0.67176795876146822,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    2,
			iter:     33,
			sigma:    0.21120653295038691,
			nDiv:     659,
			zOut:     []float64{0.2168221690252314E+01, 0.2269770980379126E+01, 0.4415865678637858E-10, 0.9833566024906726E-10, 0.8728911912704126E+00, 0.9736626287363720E+00, 0.1321013925721922E-02, 0.7778528033461282E-03, 0.1381252078118717E+01, 0.1482415971695234E+01, 0.2156405683614187E-02, 0.1706410574351102E-02, 0.1663302606134993E+01, 0.1745500102102777E+01, 0.2163985983955575E-01, 0.2150819994097728E-01, 0.1607061933620194E+01, 0.1673484316042377E+01, 0.2835506195054301E-01, 0.5676676764251993E-01, 0.7040655098910235E+00, 0.8027291772735718E+00, 0.7356791380810462E-01, 0.3124068479314146E-01, 0.1752652257061283E+01, 0.1657986407464839E+01, 0.1646145547572216E+00, 0.2697830536296953E+00, 0.8571356821219610E+00, 0.1069422512121146E+01, 0.7570493055431493E-01, 0.5387701498318342E-01, 0.1570165327329021E+01, 0.1204398523394852E+01, 0.3424393862982660E+00, 0.5430210247136315E+00, 0.6448734608299306E+00, 0.9901761195359265E+00, 0.1022564019526126E+00, 0.9868601781741700E-01, 0.5853915063475871E+00, 0.6682044860823195E+00, 0.1108480973262629E+00, 0.1209927124430272E+00, 0.3518379162830466E+00, 0.5363094467374639E+00, 0.6870152520689508E-01, 0.2792585709699239E-01, 0.7343571035538636E+00, 0.8655706211740396E+00, 0.3571015630515456E-01, 0.3903729781186600E-01, 0.5345085484304394E+00, 0.6717679865571495E+00, 0.3193134112956561E-08, 0.8403591378512072E-08, 0.1015492870320128E+00, 0.2030985804502939E+00, 0.4415865678637858E-10, 0.1303059324279677E-43, 0.2112065329503869E+00, 0.8411884198867843E-17, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    15,
			ppOut:    1,
			dminOut:  0.10154928703201281,
			desigOut: 0.0000000000000000,
			sigmaOut: 0.31275582317553385,
			qmaxOut:  2.2697722926067549,
			nFailOut: 2,
			iterOut:  34,
			nDivOut:  675,
			ttypeOut: -9,
			dmin1Out: 0.32391205918605420,
			dmin2Out: 0.32391205918605420,
			dnOut:    0.10154928703201281,
			dn1Out:   0.53450854002684800,
			dn2Out:   0.69531980574199759,
			gOut:     0.0000000000000000,
			tauOut:   0.10154929022514693,
		},
		{
			i0:       1,
			n0:       15,
			z:        []float64{0.2168221690252314E+01, 0.2269770980379126E+01, 0.4415865678637858E-10, 0.9833566024906726E-10, 0.8728911912704126E+00, 0.9736626287363720E+00, 0.1321013925721922E-02, 0.7778528033461282E-03, 0.1381252078118717E+01, 0.1482415971695234E+01, 0.2156405683614187E-02, 0.1706410574351102E-02, 0.1663302606134993E+01, 0.1745500102102777E+01, 0.2163985983955575E-01, 0.2150819994097728E-01, 0.1607061933620194E+01, 0.1673484316042377E+01, 0.2835506195054301E-01, 0.5676676764251993E-01, 0.7040655098910235E+00, 0.8027291772735718E+00, 0.7356791380810462E-01, 0.3124068479314146E-01, 0.1752652257061283E+01, 0.1657986407464839E+01, 0.1646145547572216E+00, 0.2697830536296953E+00, 0.8571356821219610E+00, 0.1069422512121146E+01, 0.7570493055431493E-01, 0.5387701498318342E-01, 0.1570165327329021E+01, 0.1204398523394852E+01, 0.3424393862982660E+00, 0.5430210247136315E+00, 0.6448734608299306E+00, 0.9901761195359265E+00, 0.1022564019526126E+00, 0.9868601781741700E-01, 0.5853915063475871E+00, 0.6682044860823195E+00, 0.1108480973262629E+00, 0.1209927124430272E+00, 0.3518379162830466E+00, 0.5363094467374639E+00, 0.6870152520689508E-01, 0.2792585709699239E-01, 0.7343571035538636E+00, 0.8655706211740396E+00, 0.3571015630515456E-01, 0.3903729781186600E-01, 0.5345085484304394E+00, 0.6717679865571495E+00, 0.3193134112956561E-08, 0.8403591378512072E-08, 0.1015492870320128E+00, 0.2030985804502939E+00, 0.4415865678637858E-10, 0.9833566024906726E-10, 0.2112065329503869E+00, 0.8411884198867843E-17, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       0,
			dmin:     0.10154928703201281,
			desig:    0.0000000000000000,
			qmax:     2.1682216902523144,
			ttype:    -9,
			dmin1:    0.32391205918605420,
			dmin2:    0.32391205918605420,
			dn:       0.10154928703201281,
			dn1:      0.53450854002684800,
			dn2:      0.69531980574199759,
			g:        0.0000000000000000,
			tau:      0.10154929022514693,
			nFail:    2,
			iter:     34,
			sigma:    0.31275582317553385,
			nDiv:     675,
			zOut:     []float64{0.2168221690252314E+01, 0.2066680684115085E+01, 0.4415865678637858E-10, 0.1865101988102620E-10, 0.8728911912704126E+00, 0.7726711989960955E+00, 0.1321013925721922E-02, 0.2361487308570419E-02, 0.1381252078118717E+01, 0.1279505990312372E+01, 0.2156405683614187E-02, 0.2803234389363145E-02, 0.1663302606134993E+01, 0.1580598225403798E+01, 0.2163985983955575E-01, 0.2200217261925751E-01, 0.1607061933620194E+01, 0.1511873816770092E+01, 0.2835506195054301E-01, 0.1320468740761088E-01, 0.7040655098910235E+00, 0.6628877301101292E+00, 0.7356791380810462E-01, 0.1945110224949301E+00, 0.1752652257061283E+01, 0.1621214783142186E+01, 0.1646145547572216E+00, 0.8703165684534678E-01, 0.8571356821219610E+00, 0.7442679496495411E+00, 0.7570493055431493E-01, 0.1597129865933493E+00, 0.1570165327329021E+01, 0.1651350720852550E+01, 0.3424393862982660E+00, 0.1337269360034139E+00, 0.6448734608299306E+00, 0.5118619205977412E+00, 0.1022564019526126E+00, 0.1169456581236225E+00, 0.5853915063475871E+00, 0.4777529393688394E+00, 0.1108480973262629E+00, 0.8163333047984263E-01, 0.3518379162830466E+00, 0.2373651048287110E+00, 0.6870152520689508E-01, 0.2125478936639627E+00, 0.7343571035538636E+00, 0.4559783600136673E+00, 0.3571015630515456E-01, 0.4186028435717903E-01, 0.5345085484304394E+00, 0.3911072610850064E+00, 0.3193134112956561E-08, 0.8290832843879624E-09, 0.1015492870320128E+00, 0.8280021541434701E-05, 0.4415865678637858E-10, 0.1865101988102620E-10, 0.2112065329503869E+00, 0.8411884198867843E-17, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    15,
			ppOut:    0,
			dminOut:  8.2800215414347011E-006,
			desigOut: -1.3877787807814457E-017,
			sigmaOut: 0.41429682935692197,
			qmaxOut:  2.1682216902523144,
			nFailOut: 2,
			iterOut:  35,
			nDivOut:  691,
			ttypeOut: -4,
			dmin1Out: 0.16866357962181588,
			dmin2Out: 0.16866357962181588,
			dnOut:    8.2800215414347011E-006,
			dn1Out:   0.39110725789187228,
			dn2Out:   0.42026820370851276,
			gOut:     0.0000000000000000,
			tauOut:   0.10154100618138810,
		},
		{
			i0:       1,
			n0:       11,
			z:        []float64{0.1565539912114361E+01, 0.1565539887559990E+01, 0.1019378594629470E-16, 0.7719264500395369E-17, 0.1185508165627851E+01, 0.1209595044804978E+01, 0.2408690373149840E-01, 0.1773879145177052E-01, 0.8908058287637680E+00, 0.9017845275302323E+00, 0.2871751477260568E-01, 0.4376453959373496E-01, 0.1374289696466428E+01, 0.1388278114489740E+01, 0.5775298217141787E-01, 0.3977262621431848E-01, 0.9560626040960629E+00, 0.9795790109656241E+00, 0.6328905763825028E-01, 0.5927283946360358E-01, 0.9174165586530958E+00, 0.1200347187817445E+01, 0.3422034931823232E+00, 0.1225138368952072E+00, 0.4297417838091302E+00, 0.3072280530227507E+00, 0.1306631986684747E-06, 0.1577236754259593E-06, 0.3708552845063717E+00, 0.3753559465594062E+00, 0.4500844331080801E-02, 0.3315910790466669E-02, 0.2765362989488873E+00, 0.2810497954194954E+00, 0.7829431815445537E-02, 0.5406547261581845E-02, 0.1940765355151052E+00, 0.1886700618399915E+00, 0.9814083907923291E-07, 0.1277249852674422E-13, 0.2455438642568072E-07, 0.2764930279233778E-14, 0.1019378594629470E-16, 0.7719264500395369E-17, 0.8241395430971566E+00, 0.2245698748385924E-16, 0.1792267857826344E-16, 0.2548071704976161E-16, 0.7318275341991307E+00, 0.4959852501050381E-23, 0.8676862906242004E-16, 0.1414000300319855E-15, 0.5186198053161721E+00, 0.1503295986001297E-14, 0.3452895805257589E-14, 0.7981257539768321E-14, 0.4143051093784424E+00, 0.1303656847202082E-15, 0.9855809897129084E-12, 0.2614986238618434E-11, 0.2112065329503869E+00, 0.8411884198867843E-17, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       1,
			dmin:     2.7649302792337775E-015,
			desig:    4.8519526217661568E-017,
			qmax:     2.1682216902523144,
			ttype:    -2,
			dmin1:    0.18866996369915248,
			dmin2:    0.27322036360404983,
			dn:       2.7649302792337775E-015,
			dn1:      0.18866996369915248,
			dn2:      0.27322036360404983,
			g:        0.0000000000000000,
			tau:      2.4554370888251911E-008,
			nFail:    2,
			iter:     57,
			sigma:    0.91543762594185796,
			nDiv:     1001,
			zOut:     []float64{0.1565539887559988E+01, 0.1565539887559990E+01, 0.5964194309842799E-17, 0.7719264500395369E-17, 0.1227333836256746E+01, 0.1209595044804978E+01, 0.1303359134714337E-01, 0.1773879145177052E-01, 0.9325154757768211E+00, 0.9017845275302323E+00, 0.6515425651042277E-01, 0.4376453959373496E-01, 0.1362896484193633E+01, 0.1388278114489740E+01, 0.2858649229958118E-01, 0.3977262621431848E-01, 0.1010265358129644E+01, 0.9795790109656241E+00, 0.7042504782685149E-01, 0.5927283946360358E-01, 0.1252435976885798E+01, 0.1200347187817445E+01, 0.3005318297487194E-01, 0.1225138368952072E+00, 0.2771750277715515E+00, 0.3072280530227507E+00, 0.2135925446109612E-06, 0.1577236754259593E-06, 0.3786716437573254E+00, 0.3753559465594062E+00, 0.2461066374135985E-02, 0.3315910790466669E-02, 0.2839952763069384E+00, 0.2810497954194954E+00, 0.3591797791316158E-02, 0.5406547261581845E-02, 0.1850782640486854E+00, 0.1886700618399915E+00, 0.1908115364037247E-27, 0.1277249852674422E-13, 0.3470987982972452E-28, 0.2764930279233778E-14, 0.5964194309842799E-17, 0.7719264500395369E-17, 0.8241395430971566E+00, 0.2245698748385924E-16, 0.1792267857826344E-16, 0.2548071704976161E-16, 0.7318275341991307E+00, 0.4959852501050381E-23, 0.8676862906242004E-16, 0.1414000300319855E-15, 0.5186198053161721E+00, 0.1503295986001297E-14, 0.3452895805257589E-14, 0.7981257539768321E-14, 0.4143051093784424E+00, 0.1303656847202082E-15, 0.9855809897129084E-12, 0.2614986238618434E-11, 0.2112065329503869E+00, 0.8411884198867843E-17, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    11,
			ppOut:    1,
			dminOut:  3.4709879829724519E-029,
			desigOut: 3.7892243888322520E-017,
			sigmaOut: 0.91543762594186073,
			qmaxOut:  2.1682216902523144,
			nFailOut: 2,
			iterOut:  58,
			nDivOut:  1013,
			ttypeOut: -2,
			dmin1Out: 0.18507826404867261,
			dmin2Out: 0.27717487004787600,
			dnOut:    3.4709879829724519E-029,
			dn1Out:   0.18507826404867261,
			dn2Out:   0.27858872904535659,
			gOut:     0.0000000000000000,
			tauOut:   2.7649302792335523E-015,
		},
		{
			i0:       1,
			n0:       11,
			z:        []float64{0.1565539887559988E+01, 0.1565539887559990E+01, 0.5964194309842799E-17, 0.7719264500395369E-17, 0.1227333836256746E+01, 0.1209595044804978E+01, 0.1303359134714337E-01, 0.1773879145177052E-01, 0.9325154757768211E+00, 0.9017845275302323E+00, 0.6515425651042277E-01, 0.4376453959373496E-01, 0.1362896484193633E+01, 0.1388278114489740E+01, 0.2858649229958118E-01, 0.3977262621431848E-01, 0.1010265358129644E+01, 0.9795790109656241E+00, 0.7042504782685149E-01, 0.5927283946360358E-01, 0.1252435976885798E+01, 0.1200347187817445E+01, 0.3005318297487194E-01, 0.1225138368952072E+00, 0.2771750277715515E+00, 0.3072280530227507E+00, 0.2135925446109612E-06, 0.1577236754259593E-06, 0.3786716437573254E+00, 0.3753559465594062E+00, 0.2461066374135985E-02, 0.3315910790466669E-02, 0.2839952763069384E+00, 0.2810497954194954E+00, 0.3591797791316158E-02, 0.5406547261581845E-02, 0.1850782640486854E+00, 0.1886700618399915E+00, 0.1908115364037247E-27, 0.1277249852674422E-13, 0.3470987982972452E-28, 0.2764930279233778E-14, 0.5964194309842799E-17, 0.7719264500395369E-17, 0.8241395430971566E+00, 0.2245698748385924E-16, 0.1792267857826344E-16, 0.2548071704976161E-16, 0.7318275341991307E+00, 0.4959852501050381E-23, 0.8676862906242004E-16, 0.1414000300319855E-15, 0.5186198053161721E+00, 0.1503295986001297E-14, 0.3452895805257589E-14, 0.7981257539768321E-14, 0.4143051093784424E+00, 0.1303656847202082E-15, 0.9855809897129084E-12, 0.2614986238618434E-11, 0.2112065329503869E+00, 0.8411884198867843E-17, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       0,
			dmin:     3.4709879829724519E-029,
			desig:    3.7892243888322520E-017,
			qmax:     2.1682216902523144,
			ttype:    -2,
			dmin1:    0.18507826404867261,
			dmin2:    0.27717487004787600,
			dn:       3.4709879829724519E-029,
			dn1:      0.18507826404867261,
			dn2:      0.27858872904535659,
			g:        0.0000000000000000,
			tau:      2.7649302792335523E-015,
			nFail:    2,
			iter:     58,
			sigma:    0.91543762594186073,
			nDiv:     1013,
			zOut:     []float64{0.1565539887559988E+01, 0.1473000755535651E+01, 0.5964194309842799E-17, 0.4969486576955697E-17, 0.1227333836256746E+01, 0.1147828295579553E+01, 0.1303359134714337E-01, 0.1058871408116432E-01, 0.9325154757768211E+00, 0.8945418861817434E+00, 0.6515425651042277E-01, 0.9926701980086396E-01, 0.1362896484193633E+01, 0.1199676824668014E+01, 0.2858649229958118E-01, 0.2407310226126826E-01, 0.1010265358129644E+01, 0.9640781716708908E+00, 0.7042504782685149E-01, 0.9148932748822959E-01, 0.1252435976885798E+01, 0.1098460700348104E+01, 0.3005318297487194E-01, 0.7583331677723075E-02, 0.2771750277715515E+00, 0.1770527776620367E+00, 0.2135925446109612E-06, 0.4568210735249311E-06, 0.3786716437573254E+00, 0.2885931212860515E+00, 0.2461066374135985E-02, 0.2421856840585221E-02, 0.2839952763069384E+00, 0.1926260852333330E+00, 0.3591797791316158E-02, 0.3451057520197492E-02, 0.1850782640486854E+00, 0.8908807450415157E-01, 0.1908115364037247E-27, 0.4969486576955697E-17, 0.9154376259418607E+00, 0.2764930279233778E-14, 0.5964194309842799E-17, 0.7719264500395369E-17, 0.8241395430971566E+00, 0.2245698748385924E-16, 0.1792267857826344E-16, 0.2548071704976161E-16, 0.7318275341991307E+00, 0.4959852501050381E-23, 0.8676862906242004E-16, 0.1414000300319855E-15, 0.5186198053161721E+00, 0.1503295986001297E-14, 0.3452895805257589E-14, 0.7981257539768321E-14, 0.4143051093784424E+00, 0.1303656847202082E-15, 0.9855809897129084E-12, 0.2614986238618434E-11, 0.2112065329503869E+00, 0.8411884198867843E-17, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    10,
			ppOut:    0,
			dminOut:  8.9088074504151571E-002,
			desigOut: -4.1633363423443370E-017,
			sigmaOut: 1.0079767579661971,
			qmaxOut:  2.1682216902523144,
			nFailOut: 2,
			iterOut:  59,
			nDivOut:  1024,
			ttypeOut: -9,
			dmin1Out: 0.17705256406949207,
			dmin2Out: 0.17705256406949207,
			dnOut:    8.9088074504151571E-002,
			dn1Out:   0.18903428744201686,
			dn2Out:   0.28613205491191551,
			gOut:     0.0000000000000000,
			tauOut:   9.2539132024336307E-002,
		},
		{
			i0:       1,
			n0:       10,
			z:        []float64{0.1565539887559988E+01, 0.1473000755535651E+01, 0.5964194309842799E-17, 0.4969486576955697E-17, 0.1227333836256746E+01, 0.1147828295579553E+01, 0.1303359134714337E-01, 0.1058871408116432E-01, 0.9325154757768211E+00, 0.8945418861817434E+00, 0.6515425651042277E-01, 0.9926701980086396E-01, 0.1362896484193633E+01, 0.1199676824668014E+01, 0.2858649229958118E-01, 0.2407310226126826E-01, 0.1010265358129644E+01, 0.9640781716708908E+00, 0.7042504782685149E-01, 0.9148932748822959E-01, 0.1252435976885798E+01, 0.1098460700348104E+01, 0.3005318297487194E-01, 0.7583331677723075E-02, 0.2771750277715515E+00, 0.1770527776620367E+00, 0.2135925446109612E-06, 0.4568210735249311E-06, 0.3786716437573254E+00, 0.2885931212860515E+00, 0.2461066374135985E-02, 0.2421856840585221E-02, 0.2839952763069384E+00, 0.1926260852333330E+00, 0.3591797791316158E-02, 0.3451057520197492E-02, 0.1850782640486854E+00, 0.8908807450415157E-01, 0.1908115364037247E-27, 0.4969486576955697E-17, 0.9154376259418607E+00, 0.2764930279233778E-14, 0.5964194309842799E-17, 0.7719264500395369E-17, 0.8241395430971566E+00, 0.2245698748385924E-16, 0.1792267857826344E-16, 0.2548071704976161E-16, 0.7318275341991307E+00, 0.4959852501050381E-23, 0.8676862906242004E-16, 0.1414000300319855E-15, 0.5186198053161721E+00, 0.1503295986001297E-14, 0.3452895805257589E-14, 0.7981257539768321E-14, 0.4143051093784424E+00, 0.1303656847202082E-15, 0.9855809897129084E-12, 0.2614986238618434E-11, 0.2112065329503869E+00, 0.8411884198867843E-17, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       1,
			dmin:     8.9088074504151571E-002,
			desig:    -4.1633363423443370E-017,
			qmax:     2.1682216902523144,
			ttype:    -9,
			dmin1:    0.17705256406949207,
			dmin2:    0.17705256406949207,
			dn:       8.9088074504151571E-002,
			dn1:      0.18903428744201686,
			dn2:      0.28613205491191551,
			g:        0.0000000000000000,
			tau:      9.2539132024336307E-002,
			nFail:    2,
			iter:     59,
			sigma:    1.0079767579661971,
			nDiv:     1024,
			zOut:     []float64{0.1397612833703614E+01, 0.1473000755535651E+01, 0.4081328655531061E-17, 0.4969486576955697E-17, 0.1083029087828680E+01, 0.1147828295579553E+01, 0.8745885380967966E-02, 0.1058871408116432E-01, 0.9096750987696016E+00, 0.8945418861817434E+00, 0.1309130515609722E+00, 0.9926701980086396E-01, 0.1017448953536272E+01, 0.1199676824668014E+01, 0.2281033592282574E-01, 0.2407310226126826E-01, 0.9573692414042568E+00, 0.9640781716708908E+00, 0.1049724875218357E+00, 0.9148932748822959E-01, 0.9256836226719531E+00, 0.1098460700348104E+01, 0.1450441494900679E-02, 0.7583331677723075E-02, 0.1002148711561719E+00, 0.1770527776620367E+00, 0.1315527505616969E-05, 0.4568210735249311E-06, 0.2156257407670935E+00, 0.2885931212860515E+00, 0.2163530200698068E-02, 0.2421856840585221E-02, 0.1185256907207948E+00, 0.1926260852333330E+00, 0.2593936112987583E-02, 0.3451057520197492E-02, 0.1110621655912630E-01, 0.8908807450415157E-01, 0.4081328655531061E-17, 0.4969486576955697E-17, 0.9154376259418607E+00, 0.2764930279233778E-14, 0.5964194309842799E-17, 0.7719264500395369E-17, 0.8241395430971566E+00, 0.2245698748385924E-16, 0.1792267857826344E-16, 0.2548071704976161E-16, 0.7318275341991307E+00, 0.4959852501050381E-23, 0.8676862906242004E-16, 0.1414000300319855E-15, 0.5186198053161721E+00, 0.1503295986001297E-14, 0.3452895805257589E-14, 0.7981257539768321E-14, 0.4143051093784424E+00, 0.1303656847202082E-15, 0.9855809897129084E-12, 0.2614986238618434E-11, 0.2112065329503869E+00, 0.8411884198867843E-17, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    10,
			ppOut:    1,
			dminOut:  1.1106216559126303E-002,
			desigOut: -4.1633363423443370E-017,
			sigmaOut: 1.0833646797982348,
			qmaxOut:  2.1682216902523144,
			nFailOut: 2,
			iterOut:  60,
			nDivOut:  1035,
			ttypeOut: -4,
			dmin1Out: 0.10021441433509834,
			dmin2Out: 0.10021441433509834,
			dnOut:    1.1106216559126303E-002,
			dn1Out:   0.11507463320059727,
			dn2Out:   0.21320388392650824,
			gOut:     0.0000000000000000,
			tauOut:   7.5387921832037685E-002,
		},
		{
			i0:       1,
			n0:       10,
			z:        []float64{0.1397612833703614E+01, 0.1473000755535651E+01, 0.4081328655531061E-17, 0.4969486576955697E-17, 0.1083029087828680E+01, 0.1147828295579553E+01, 0.8745885380967966E-02, 0.1058871408116432E-01, 0.9096750987696016E+00, 0.8945418861817434E+00, 0.1309130515609722E+00, 0.9926701980086396E-01, 0.1017448953536272E+01, 0.1199676824668014E+01, 0.2281033592282574E-01, 0.2407310226126826E-01, 0.9573692414042568E+00, 0.9640781716708908E+00, 0.1049724875218357E+00, 0.9148932748822959E-01, 0.9256836226719531E+00, 0.1098460700348104E+01, 0.1450441494900679E-02, 0.7583331677723075E-02, 0.1002148711561719E+00, 0.1770527776620367E+00, 0.1315527505616969E-05, 0.4568210735249311E-06, 0.2156257407670935E+00, 0.2885931212860515E+00, 0.2163530200698068E-02, 0.2421856840585221E-02, 0.1185256907207948E+00, 0.1926260852333330E+00, 0.2593936112987583E-02, 0.3451057520197492E-02, 0.1110621655912630E-01, 0.8908807450415157E-01, 0.4081328655531061E-17, 0.4969486576955697E-17, 0.9154376259418607E+00, 0.2764930279233778E-14, 0.5964194309842799E-17, 0.7719264500395369E-17, 0.8241395430971566E+00, 0.2245698748385924E-16, 0.1792267857826344E-16, 0.2548071704976161E-16, 0.7318275341991307E+00, 0.4959852501050381E-23, 0.8676862906242004E-16, 0.1414000300319855E-15, 0.5186198053161721E+00, 0.1503295986001297E-14, 0.3452895805257589E-14, 0.7981257539768321E-14, 0.4143051093784424E+00, 0.1303656847202082E-15, 0.9855809897129084E-12, 0.2614986238618434E-11, 0.2112065329503869E+00, 0.8411884198867843E-17, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       0,
			dmin:     1.1106216559126303E-002,
			desig:    -4.1633363423443370E-017,
			qmax:     2.1682216902523144,
			ttype:    -4,
			dmin1:    0.10021441433509834,
			dmin2:    0.10021441433509834,
			dn:       1.1106216559126303E-002,
			dn1:      0.11507463320059727,
			dn2:      0.21320388392650824,
			g:        0.0000000000000000,
			tau:      7.5387921832037685E-002,
			nFail:    2,
			iter:     60,
			sigma:    1.0833646797982348,
			nDiv:     1035,
			zOut:     []float64{0.1397612833703614E+01, 0.1388412169967495E+01, 0.4081328655531061E-17, 0.3183635051997810E-17, 0.1083029087828680E+01, 0.1082574309473529E+01, 0.8745885380967966E-02, 0.7349069784991225E-02, 0.9096750987696016E+00, 0.1024038416809464E+01, 0.1309130515609722E+00, 0.1300706547025319E+00, 0.1017448953536272E+01, 0.9009879710204475E+00, 0.2281033592282574E-01, 0.2423774201322422E-01, 0.9573692414042568E+00, 0.1028903323176749E+01, 0.1049724875218357E+00, 0.9444163542020824E-01, 0.9256836226719531E+00, 0.8234917650105269E+00, 0.1450441494900679E-02, 0.1765115496075164E-03, 0.1002148711561719E+00, 0.9083901139795122E-01, 0.1315527505616969E-05, 0.3122684720284652E-05, 0.2156257407670935E+00, 0.2085854845469525E+00, 0.2163530200698068E-02, 0.1229394806594584E-02, 0.1185256907207948E+00, 0.1106895682910690E+00, 0.2593936112987583E-02, 0.2602667681892373E-03, 0.1110621655912630E-01, 0.1645286054818337E-02, 0.4081328655531061E-17, 0.3183635051997810E-17, 0.9154376259418607E+00, 0.2764930279233778E-14, 0.5964194309842799E-17, 0.7719264500395369E-17, 0.8241395430971566E+00, 0.2245698748385924E-16, 0.1792267857826344E-16, 0.2548071704976161E-16, 0.7318275341991307E+00, 0.4959852501050381E-23, 0.8676862906242004E-16, 0.1414000300319855E-15, 0.5186198053161721E+00, 0.1503295986001297E-14, 0.3452895805257589E-14, 0.7981257539768321E-14, 0.4143051093784424E+00, 0.1303656847202082E-15, 0.9855809897129084E-12, 0.2614986238618434E-11, 0.2112065329503869E+00, 0.8411884198867843E-17, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    10,
			ppOut:    0,
			dminOut:  1.6452860548183366E-003,
			desigOut: 7.9797279894933126E-017,
			sigmaOut: 1.0925653435343534,
			qmaxOut:  2.1682216902523144,
			nFailOut: 2,
			iterOut:  61,
			nDivOut:  1046,
			ttypeOut: -4,
			dmin1Out: 9.0837695870445614E-002,
			dmin2Out: 9.0837695870445614E-002,
			dnOut:    1.6452860548183366E-003,
			dn1Out:   0.10809563217808144,
			dn2Out:   0.20642195434625446,
			gOut:     0.0000000000000000,
			tauOut:   9.2006637361187298E-003,
		},
		{
			i0:       1,
			n0:       10,
			z:        []float64{0.1397612833703614E+01, 0.1388412169967495E+01, 0.4081328655531061E-17, 0.3183635051997810E-17, 0.1083029087828680E+01, 0.1082574309473529E+01, 0.8745885380967966E-02, 0.7349069784991225E-02, 0.9096750987696016E+00, 0.1024038416809464E+01, 0.1309130515609722E+00, 0.1300706547025319E+00, 0.1017448953536272E+01, 0.9009879710204475E+00, 0.2281033592282574E-01, 0.2423774201322422E-01, 0.9573692414042568E+00, 0.1028903323176749E+01, 0.1049724875218357E+00, 0.9444163542020824E-01, 0.9256836226719531E+00, 0.8234917650105269E+00, 0.1450441494900679E-02, 0.1765115496075164E-03, 0.1002148711561719E+00, 0.9083901139795122E-01, 0.1315527505616969E-05, 0.3122684720284652E-05, 0.2156257407670935E+00, 0.2085854845469525E+00, 0.2163530200698068E-02, 0.1229394806594584E-02, 0.1185256907207948E+00, 0.1106895682910690E+00, 0.2593936112987583E-02, 0.2602667681892373E-03, 0.1110621655912630E-01, 0.1645286054818337E-02, 0.4081328655531061E-17, 0.3183635051997810E-17, 0.9154376259418607E+00, 0.2764930279233778E-14, 0.5964194309842799E-17, 0.7719264500395369E-17, 0.8241395430971566E+00, 0.2245698748385924E-16, 0.1792267857826344E-16, 0.2548071704976161E-16, 0.7318275341991307E+00, 0.4959852501050381E-23, 0.8676862906242004E-16, 0.1414000300319855E-15, 0.5186198053161721E+00, 0.1503295986001297E-14, 0.3452895805257589E-14, 0.7981257539768321E-14, 0.4143051093784424E+00, 0.1303656847202082E-15, 0.9855809897129084E-12, 0.2614986238618434E-11, 0.2112065329503869E+00, 0.8411884198867843E-17, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       1,
			dmin:     1.6452860548183366E-003,
			desig:    7.9797279894933126E-017,
			qmax:     2.1682216902523144,
			ttype:    -4,
			dmin1:    9.0837695870445614E-002,
			dmin2:    9.0837695870445614E-002,
			dn:       1.6452860548183366E-003,
			dn1:      0.10809563217808144,
			dn2:      0.20642195434625446,
			g:        0.0000000000000000,
			tau:      9.2006637361187298E-003,
			nFail:    2,
			iter:     61,
			sigma:    1.0925653435343534,
			nDiv:     1046,
			zOut:     []float64{0.1386852748028120E+01, 0.1388412169967495E+01, 0.2485138759635906E-17, 0.3183635051997810E-17, 0.1088363957319145E+01, 0.1082574309473529E+01, 0.6914717946174950E-02, 0.7349069784991225E-02, 0.1145634931626446E+01, 0.1024038416809464E+01, 0.1022944500333619E+00, 0.1300706547025319E+00, 0.8213718410609353E+00, 0.9009879710204475E+00, 0.3036175828902931E-01, 0.2423774201322422E-01, 0.1091423778368554E+01, 0.1028903323176749E+01, 0.7125729765473905E-01, 0.9444163542020824E-01, 0.7508515569660207E+00, 0.8234917650105269E+00, 0.2135460001102832E-04, 0.1765115496075164E-03, 0.8926135754328580E-01, 0.9083901139795122E-01, 0.7297073710223142E-05, 0.3122684720284652E-05, 0.2082481603404622E+00, 0.2085854845469525E+00, 0.6534568189162364E-03, 0.1229394806594584E-02, 0.1087369563009673E+00, 0.1106895682910690E+00, 0.3938065757966957E-05, 0.2602667681892373E-03, 0.8192604968568760E-04, 0.1645286054818337E-02, 0.2485138759635906E-17, 0.3183635051997810E-17, 0.9154376259418607E+00, 0.2764930279233778E-14, 0.5964194309842799E-17, 0.7719264500395369E-17, 0.8241395430971566E+00, 0.2245698748385924E-16, 0.1792267857826344E-16, 0.2548071704976161E-16, 0.7318275341991307E+00, 0.4959852501050381E-23, 0.8676862906242004E-16, 0.1414000300319855E-15, 0.5186198053161721E+00, 0.1503295986001297E-14, 0.3452895805257589E-14, 0.7981257539768321E-14, 0.4143051093784424E+00, 0.1303656847202082E-15, 0.9855809897129084E-12, 0.2614986238618434E-11, 0.2112065329503869E+00, 0.8411884198867843E-17, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    10,
			ppOut:    1,
			dminOut:  8.1926049685687600E-005,
			desigOut: -9.9312918999672206E-017,
			sigmaOut: 1.0941247654737283,
			qmaxOut:  2.1682216902523144,
			nFailOut: 2,
			iterOut:  62,
			nDivOut:  1057,
			ttypeOut: -4,
			dmin1Out: 8.9258234858565516E-002,
			dmin2Out: 8.9258234858565516E-002,
			dnOut:    8.1926049685687600E-005,
			dn1Out:   0.10847668953277810,
			dn2Out:   0.20701876553386761,
			gOut:     0.0000000000000000,
			tauOut:   1.5594219393746818E-003,
		},
		{
			i0:       1,
			n0:       10,
			z:        []float64{0.1386852748028120E+01, 0.1388412169967495E+01, 0.2485138759635906E-17, 0.3183635051997810E-17, 0.1088363957319145E+01, 0.1082574309473529E+01, 0.6914717946174950E-02, 0.7349069784991225E-02, 0.1145634931626446E+01, 0.1024038416809464E+01, 0.1022944500333619E+00, 0.1300706547025319E+00, 0.8213718410609353E+00, 0.9009879710204475E+00, 0.3036175828902931E-01, 0.2423774201322422E-01, 0.1091423778368554E+01, 0.1028903323176749E+01, 0.7125729765473905E-01, 0.9444163542020824E-01, 0.7508515569660207E+00, 0.8234917650105269E+00, 0.2135460001102832E-04, 0.1765115496075164E-03, 0.8926135754328580E-01, 0.9083901139795122E-01, 0.7297073710223142E-05, 0.3122684720284652E-05, 0.2082481603404622E+00, 0.2085854845469525E+00, 0.6534568189162364E-03, 0.1229394806594584E-02, 0.1087369563009673E+00, 0.1106895682910690E+00, 0.3938065757966957E-05, 0.2602667681892373E-03, 0.8192604968568760E-04, 0.1645286054818337E-02, 0.2485138759635906E-17, 0.3183635051997810E-17, 0.9154376259418607E+00, 0.2764930279233778E-14, 0.5964194309842799E-17, 0.7719264500395369E-17, 0.8241395430971566E+00, 0.2245698748385924E-16, 0.1792267857826344E-16, 0.2548071704976161E-16, 0.7318275341991307E+00, 0.4959852501050381E-23, 0.8676862906242004E-16, 0.1414000300319855E-15, 0.5186198053161721E+00, 0.1503295986001297E-14, 0.3452895805257589E-14, 0.7981257539768321E-14, 0.4143051093784424E+00, 0.1303656847202082E-15, 0.9855809897129084E-12, 0.2614986238618434E-11, 0.2112065329503869E+00, 0.8411884198867843E-17, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       0,
			dmin:     8.1926049685687600E-005,
			desig:    -9.9312918999672206E-017,
			qmax:     2.1682216902523144,
			ttype:    -4,
			dmin1:    8.9258234858565516E-002,
			dmin2:    8.9258234858565516E-002,
			dn:       8.1926049685687600E-005,
			dn1:      0.10847668953277810,
			dn2:      0.20701876553386761,
			g:        0.0000000000000000,
			tau:      1.5594219393746818E-003,
			nFail:    2,
			iter:     62,
			sigma:    1.0941247654737283,
			nDiv:     1057,
			zOut:     []float64{0.1386852748028120E+01, 0.1386771331083530E+01, 0.2485138759635906E-17, 0.1950383162890474E-17, 0.1088363957319145E+01, 0.1095197258320730E+01, 0.6914717946174950E-02, 0.7233164949324962E-02, 0.1145634931626446E+01, 0.1240614799765893E+01, 0.1022944500333619E+00, 0.6772592167212049E-01, 0.8213718410609353E+00, 0.7839262607332537E+00, 0.3036175828902931E-01, 0.4227125255215914E-01, 0.1091423778368554E+01, 0.1120328406526543E+01, 0.7125729765473905E-01, 0.4775711530437247E-01, 0.7508515569660207E+00, 0.7030343793170689E+00, 0.2135460001102832E-04, 0.2711304941630149E-05, 0.8926135754328580E-01, 0.8918452636746399E-01, 0.7297073710223142E-05, 0.1703885458517271E-04, 0.2082481603404622E+00, 0.2088031613602029E+00, 0.6534568189162364E-03, 0.3402961195615630E-03, 0.1087369563009673E+00, 0.1083191813025733E+00, 0.3938065757966957E-05, 0.2978513750500819E-08, 0.8192604968568760E-04, 0.5061265815246250E-06, 0.2485138759635906E-17, 0.1950383162890474E-17, 0.9154376259418607E+00, 0.2764930279233778E-14, 0.5964194309842799E-17, 0.7719264500395369E-17, 0.8241395430971566E+00, 0.2245698748385924E-16, 0.1792267857826344E-16, 0.2548071704976161E-16, 0.7318275341991307E+00, 0.4959852501050381E-23, 0.8676862906242004E-16, 0.1414000300319855E-15, 0.5186198053161721E+00, 0.1503295986001297E-14, 0.3452895805257589E-14, 0.7981257539768321E-14, 0.4143051093784424E+00, 0.1303656847202082E-15, 0.9855809897129084E-12, 0.2614986238618434E-11, 0.2112065329503869E+00, 0.8411884198867843E-17, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    10,
			ppOut:    0,
			dminOut:  5.0612658152462498E-007,
			desigOut: -9.3553094958342964E-017,
			sigmaOut: 1.0942061824183187,
			qmaxOut:  2.1682216902523144,
			nFailOut: 2,
			iterOut:  63,
			nDivOut:  1068,
			ttypeOut: -4,
			dmin1Out: 8.9177229293753768E-002,
			dmin2Out: 8.9177229293753768E-002,
			dnOut:    5.0612658152462498E-007,
			dn1Out:   0.10831524323681536,
			dn2Out:   0.20814970454128662,
			gOut:     0.0000000000000000,
			tauOut:   8.1416944590412474E-005,
		},
		{
			i0:       1,
			n0:       10,
			z:        []float64{0.1386852748028120E+01, 0.1386771331083530E+01, 0.2485138759635906E-17, 0.1950383162890474E-17, 0.1088363957319145E+01, 0.1095197258320730E+01, 0.6914717946174950E-02, 0.7233164949324962E-02, 0.1145634931626446E+01, 0.1240614799765893E+01, 0.1022944500333619E+00, 0.6772592167212049E-01, 0.8213718410609353E+00, 0.7839262607332537E+00, 0.3036175828902931E-01, 0.4227125255215914E-01, 0.1091423778368554E+01, 0.1120328406526543E+01, 0.7125729765473905E-01, 0.4775711530437247E-01, 0.7508515569660207E+00, 0.7030343793170689E+00, 0.2135460001102832E-04, 0.2711304941630149E-05, 0.8926135754328580E-01, 0.8918452636746399E-01, 0.7297073710223142E-05, 0.1703885458517271E-04, 0.2082481603404622E+00, 0.2088031613602029E+00, 0.6534568189162364E-03, 0.3402961195615630E-03, 0.1087369563009673E+00, 0.1083191813025733E+00, 0.3938065757966957E-05, 0.2978513750500819E-08, 0.8192604968568760E-04, 0.5061265815246250E-06, 0.2485138759635906E-17, 0.1950383162890474E-17, 0.9154376259418607E+00, 0.2764930279233778E-14, 0.5964194309842799E-17, 0.7719264500395369E-17, 0.8241395430971566E+00, 0.2245698748385924E-16, 0.1792267857826344E-16, 0.2548071704976161E-16, 0.7318275341991307E+00, 0.4959852501050381E-23, 0.8676862906242004E-16, 0.1414000300319855E-15, 0.5186198053161721E+00, 0.1503295986001297E-14, 0.3452895805257589E-14, 0.7981257539768321E-14, 0.4143051093784424E+00, 0.1303656847202082E-15, 0.9855809897129084E-12, 0.2614986238618434E-11, 0.2112065329503869E+00, 0.8411884198867843E-17, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       1,
			dmin:     5.0612658152462498E-007,
			desig:    -9.3553094958342964E-017,
			qmax:     2.1682216902523144,
			ttype:    -4,
			dmin1:    8.9177229293753768E-002,
			dmin2:    8.9177229293753768E-002,
			dn:       5.0612658152462498E-007,
			dn1:      0.10831524323681536,
			dn2:      0.20814970454128662,
			g:        0.0000000000000000,
			tau:      8.1416944590412474E-005,
			nFail:    2,
			iter:     63,
			sigma:    1.0942061824183187,
			nDiv:     1068,
			zOut:     []float64{0.1386770825043033E+01, 0.1386771331083530E+01, 0.1540308069724697E-17, 0.1950383162890474E-17, 0.1102429917229558E+01, 0.1095197258320730E+01, 0.8139811288713328E-02, 0.7233164949324962E-02, 0.1300200404108803E+01, 0.1240614799765893E+01, 0.4083380405309871E-01, 0.6772592167212049E-01, 0.7853632031918177E+00, 0.7839262607332537E+00, 0.6030036143936175E-01, 0.4227125255215914E-01, 0.1107784654351057E+01, 0.1120328406526543E+01, 0.3030814137396716E-01, 0.4775711530437247E-01, 0.6727284432075471E+00, 0.7030343793170689E+00, 0.3594413905024206E-06, 0.2711304941630149E-05, 0.8920069974016231E-01, 0.8918452636746399E-01, 0.3988496406087022E-04, 0.1703885458517271E-04, 0.2091030664752072E+00, 0.2088031613602029E+00, 0.1762795624794033E-03, 0.3402961195615630E-03, 0.1081423986781113E+00, 0.1083191813025733E+00, 0.1393999949133917E-13, 0.2978513750500819E-08, 0.8607124697139263E-10, 0.5061265815246250E-06, 0.1540308069724697E-17, 0.1950383162890474E-17, 0.9154376259418607E+00, 0.2764930279233778E-14, 0.5964194309842799E-17, 0.7719264500395369E-17, 0.8241395430971566E+00, 0.2245698748385924E-16, 0.1792267857826344E-16, 0.2548071704976161E-16, 0.7318275341991307E+00, 0.4959852501050381E-23, 0.8676862906242004E-16, 0.1414000300319855E-15, 0.5186198053161721E+00, 0.1503295986001297E-14, 0.3452895805257589E-14, 0.7981257539768321E-14, 0.4143051093784424E+00, 0.1303656847202082E-15, 0.9855809897129084E-12, 0.2614986238618434E-11, 0.2112065329503869E+00, 0.8411884198867843E-17, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    10,
			ppOut:    1,
			dminOut:  8.6071246971392626E-011,
			desigOut: 7.0901951641105278E-017,
			sigmaOut: 1.0942066884588149,
			qmaxOut:  2.1682216902523144,
			nFailOut: 2,
			iterOut:  64,
			nDivOut:  1079,
			ttypeOut: -4,
			dmin1Out: 8.9183660885577137E-002,
			dmin2Out: 8.9183660885577137E-002,
			dnOut:    8.6071246971392626E-011,
			dn1Out:   0.10814239569959758,
			dn2Out:   0.20876277035564564,
			gOut:     0.0000000000000000,
			tauOut:   5.0604049633765406E-007,
		},
		{
			i0:       1,
			n0:       10,
			z:        []float64{0.1386770825043033E+01, 0.1386771331083530E+01, 0.1540308069724697E-17, 0.1950383162890474E-17, 0.1102429917229558E+01, 0.1095197258320730E+01, 0.8139811288713328E-02, 0.7233164949324962E-02, 0.1300200404108803E+01, 0.1240614799765893E+01, 0.4083380405309871E-01, 0.6772592167212049E-01, 0.7853632031918177E+00, 0.7839262607332537E+00, 0.6030036143936175E-01, 0.4227125255215914E-01, 0.1107784654351057E+01, 0.1120328406526543E+01, 0.3030814137396716E-01, 0.4775711530437247E-01, 0.6727284432075471E+00, 0.7030343793170689E+00, 0.3594413905024206E-06, 0.2711304941630149E-05, 0.8920069974016231E-01, 0.8918452636746399E-01, 0.3988496406087022E-04, 0.1703885458517271E-04, 0.2091030664752072E+00, 0.2088031613602029E+00, 0.1762795624794033E-03, 0.3402961195615630E-03, 0.1081423986781113E+00, 0.1083191813025733E+00, 0.1393999949133917E-13, 0.2978513750500819E-08, 0.8607124697139263E-10, 0.5061265815246250E-06, 0.1540308069724697E-17, 0.1950383162890474E-17, 0.9154376259418607E+00, 0.2764930279233778E-14, 0.5964194309842799E-17, 0.7719264500395369E-17, 0.8241395430971566E+00, 0.2245698748385924E-16, 0.1792267857826344E-16, 0.2548071704976161E-16, 0.7318275341991307E+00, 0.4959852501050381E-23, 0.8676862906242004E-16, 0.1414000300319855E-15, 0.5186198053161721E+00, 0.1503295986001297E-14, 0.3452895805257589E-14, 0.7981257539768321E-14, 0.4143051093784424E+00, 0.1303656847202082E-15, 0.9855809897129084E-12, 0.2614986238618434E-11, 0.2112065329503869E+00, 0.8411884198867843E-17, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       0,
			dmin:     8.6071246971392626E-011,
			desig:    7.0901951641105278E-017,
			qmax:     2.1682216902523144,
			ttype:    -4,
			dmin1:    8.9183660885577137E-002,
			dmin2:    8.9183660885577137E-002,
			dn:       8.6071246971392626E-011,
			dn1:      0.10814239569959758,
			dn2:      0.20876277035564564,
			g:        0.0000000000000000,
			tau:      5.0604049633765406E-007,
			nFail:    2,
			iter:     64,
			sigma:    1.0942066884588149,
			nDiv:     1079,
			zOut:     []float64{0.1386770825043033E+01, 0.1386770824956962E+01, 0.1540308069724697E-17, 0.1224486171222500E-17, 0.1102429917229558E+01, 0.1110569728432201E+01, 0.8139811288713328E-02, 0.9529690622753703E-02, 0.1300200404108803E+01, 0.1331504517453077E+01, 0.4083380405309871E-01, 0.2408506071837550E-01, 0.7853632031918177E+00, 0.8215785038267327E+00, 0.6030036143936175E-01, 0.8130667336500198E-01, 0.1107784654351057E+01, 0.1056786122273951E+01, 0.3030814137396716E-01, 0.1929354325655847E-01, 0.6727284432075471E+00, 0.6534352593063080E+00, 0.3594413905024206E-06, 0.4906748310831983E-07, 0.8920069974016231E-01, 0.8924053555066887E-01, 0.3988496406087022E-04, 0.9345605379795243E-04, 0.2091030664752072E+00, 0.2091858898978174E+00, 0.1762795624794033E-03, 0.9113088236382798E-04, 0.1081423986781113E+00, 0.1080512677096902E+00, 0.1393999949133917E-13, 0.1110429488179469E-22, 0.8607124697139263E-10, 0.3167883469916549E-16, 0.1540308069724697E-17, 0.1224486171222500E-17, 0.9154376259418607E+00, 0.2764930279233778E-14, 0.5964194309842799E-17, 0.7719264500395369E-17, 0.8241395430971566E+00, 0.2245698748385924E-16, 0.1792267857826344E-16, 0.2548071704976161E-16, 0.7318275341991307E+00, 0.4959852501050381E-23, 0.8676862906242004E-16, 0.1414000300319855E-15, 0.5186198053161721E+00, 0.1503295986001297E-14, 0.3452895805257589E-14, 0.7981257539768321E-14, 0.4143051093784424E+00, 0.1303656847202082E-15, 0.9855809897129084E-12, 0.2614986238618434E-11, 0.2112065329503869E+00, 0.8411884198867843E-17, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    10,
			ppOut:    0,
			dminOut:  3.1678834699165494E-017,
			desigOut: -8.6057196331979189E-017,
			sigmaOut: 1.0942066885448862,
			qmaxOut:  2.1682216902523144,
			nFailOut: 2,
			iterOut:  65,
			nDivOut:  1090,
			ttypeOut: -4,
			dmin1Out: 8.9200650586607991E-002,
			dmin2Out: 8.9200650586607991E-002,
			dnOut:    3.1678834699165494E-017,
			dn1Out:   0.10805126770967630,
			dn2Out:   0.20900961033533805,
			gOut:     0.0000000000000000,
			tauOut:   8.6071215292546838E-011,
		},
		{
			i0:       1,
			n0:       3,
			z:        []float64{0.1143865217830237E+00, 0.1154808079639322E+00, 0.2932453102768040E-16, 0.1817901582187390E-16, 0.1834406364024603E+00, 0.1845166504084682E+00, 0.1101341475930520E-06, 0.1827217490071947E-04, 0.1127820009512990E-04, 0.1105674515151256E-02, 0.1845166504084682E+00, 0.2816548643828996E+00, 0.2169993432366266E+01, 0.5067642455139780E-26, 0.3415542419024794E-17, 0.4171805735046273E-17, 0.1810817848712033E+01, 0.1972152263052530E-29, 0.6115792910959321E-18, 0.6699979816704878E-18, 0.1681677309759878E+01, 0.5241368559131172E-22, 0.4369342452764876E-18, 0.4084491703488284E-18, 0.1303743375476047E+01, 0.0000000000000000E+00, 0.2325140442247288E-18, 0.2353545363971710E-18, 0.1202161769544433E+01, 0.0000000000000000E+00, 0.2163565864913247E-18, 0.2222159192042978E-18, 0.1183377192742860E+01, 0.2033579915108999E-22, 0.2652482201353177E-18, 0.2438256017573345E-18, 0.1094206688544886E+01, 0.3167883469916549E-16, 0.9806070694382835E-18, 0.1224486171222500E-17, 0.9154376259418607E+00, 0.2764930279233778E-14, 0.5964194309842799E-17, 0.7719264500395369E-17, 0.8241395430971566E+00, 0.2245698748385924E-16, 0.1792267857826344E-16, 0.2548071704976161E-16, 0.7318275341991307E+00, 0.4959852501050381E-23, 0.8676862906242004E-16, 0.1414000300319855E-15, 0.5186198053161721E+00, 0.1503295986001297E-14, 0.3452895805257589E-14, 0.7981257539768321E-14, 0.4143051093784424E+00, 0.1303656847202082E-15, 0.9855809897129084E-12, 0.2614986238618434E-11, 0.2112065329503869E+00, 0.8411884198867843E-17, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       0,
			dmin:     1.1278200095129901E-005,
			desig:    1.1991276027689679E-016,
			qmax:     0.41760242607758735,
			ttype:    -4,
			dmin1:    0.11438652178302365,
			dmin2:    0.11438652178302365,
			dn:       1.1278200095129901E-005,
			dn1:      0.18342236422755959,
			dn2:      0.11438652178302365,
			g:        0.0000000000000000,
			tau:      1.0942861809085330E-003,
			nFail:    3,
			iter:     99,
			sigma:    2.3665909917188244,
			nDiv:     1347,
			zOut:     []float64{0.1143865217830237E+00, 0.1143752525446631E+00, 0.2932453102768040E-16, 0.4703212027287794E-16, 0.1834406364024603E+00, 0.1834294772982473E+00, 0.1101341475930520E-06, 0.6771621290952006E-11, 0.1127820009512990E-04, 0.8954962962003413E-08, 0.1845166504084682E+00, 0.1834406364024603E+00, 0.2169993432366266E+01, 0.5067642455139780E-26, 0.3415542419024794E-17, 0.4171805735046273E-17, 0.1810817848712033E+01, 0.1972152263052530E-29, 0.6115792910959321E-18, 0.6699979816704878E-18, 0.1681677309759878E+01, 0.5241368559131172E-22, 0.4369342452764876E-18, 0.4084491703488284E-18, 0.1303743375476047E+01, 0.0000000000000000E+00, 0.2325140442247288E-18, 0.2353545363971710E-18, 0.1202161769544433E+01, 0.0000000000000000E+00, 0.2163565864913247E-18, 0.2222159192042978E-18, 0.1183377192742860E+01, 0.2033579915108999E-22, 0.2652482201353177E-18, 0.2438256017573345E-18, 0.1094206688544886E+01, 0.3167883469916549E-16, 0.9806070694382835E-18, 0.1224486171222500E-17, 0.9154376259418607E+00, 0.2764930279233778E-14, 0.5964194309842799E-17, 0.7719264500395369E-17, 0.8241395430971566E+00, 0.2245698748385924E-16, 0.1792267857826344E-16, 0.2548071704976161E-16, 0.7318275341991307E+00, 0.4959852501050381E-23, 0.8676862906242004E-16, 0.1414000300319855E-15, 0.5186198053161721E+00, 0.1503295986001297E-14, 0.3452895805257589E-14, 0.7981257539768321E-14, 0.4143051093784424E+00, 0.1303656847202082E-15, 0.9855809897129084E-12, 0.2614986238618434E-11, 0.2112065329503869E+00, 0.8411884198867843E-17, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    3,
			ppOut:    0,
			dminOut:  8.9549629620034135E-009,
			desigOut: -3.3964327119002935E-017,
			sigmaOut: 2.3666022609571851,
			qmaxOut:  0.41760242607758735,
			nFailOut: 3,
			iterOut:  100,
			nDivOut:  1351,
			ttypeOut: -4,
			dmin1Out: 0.11437525254466312,
			dmin2Out: 0.11437525254466312,
			dnOut:    8.9549629620034135E-009,
			dn1Out:   0.18342936716409974,
			dn2Out:   0.11437525254466312,
			gOut:     0.0000000000000000,
			tauOut:   1.1269238360546607E-005,
		},
		{
			i0:       1,
			n0:       3,
			z:        []float64{0.1143865217830237E+00, 0.1143752525446631E+00, 0.2932453102768040E-16, 0.4703212027287794E-16, 0.1834406364024603E+00, 0.1834294772982473E+00, 0.1101341475930520E-06, 0.6771621290952006E-11, 0.1127820009512990E-04, 0.8954962962003413E-08, 0.1845166504084682E+00, 0.1834406364024603E+00, 0.2169993432366266E+01, 0.5067642455139780E-26, 0.3415542419024794E-17, 0.4171805735046273E-17, 0.1810817848712033E+01, 0.1972152263052530E-29, 0.6115792910959321E-18, 0.6699979816704878E-18, 0.1681677309759878E+01, 0.5241368559131172E-22, 0.4369342452764876E-18, 0.4084491703488284E-18, 0.1303743375476047E+01, 0.0000000000000000E+00, 0.2325140442247288E-18, 0.2353545363971710E-18, 0.1202161769544433E+01, 0.0000000000000000E+00, 0.2163565864913247E-18, 0.2222159192042978E-18, 0.1183377192742860E+01, 0.2033579915108999E-22, 0.2652482201353177E-18, 0.2438256017573345E-18, 0.1094206688544886E+01, 0.3167883469916549E-16, 0.9806070694382835E-18, 0.1224486171222500E-17, 0.9154376259418607E+00, 0.2764930279233778E-14, 0.5964194309842799E-17, 0.7719264500395369E-17, 0.8241395430971566E+00, 0.2245698748385924E-16, 0.1792267857826344E-16, 0.2548071704976161E-16, 0.7318275341991307E+00, 0.4959852501050381E-23, 0.8676862906242004E-16, 0.1414000300319855E-15, 0.5186198053161721E+00, 0.1503295986001297E-14, 0.3452895805257589E-14, 0.7981257539768321E-14, 0.4143051093784424E+00, 0.1303656847202082E-15, 0.9855809897129084E-12, 0.2614986238618434E-11, 0.2112065329503869E+00, 0.8411884198867843E-17, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       1,
			dmin:     8.9549629620034135E-009,
			desig:    -3.3964327119002935E-017,
			qmax:     0.41760242607758735,
			ttype:    -4,
			dmin1:    0.11437525254466312,
			dmin2:    0.11437525254466312,
			dn:       8.9549629620034135E-009,
			dn1:      0.18342936716409974,
			dn2:      0.11437525254466312,
			g:        0.0000000000000000,
			tau:      1.1269238360546607E-005,
			nFail:    3,
			iter:     100,
			sigma:    2.3666022609571851,
			nDiv:     1351,
			zOut:     []float64{0.1143752435897560E+00, 0.1143752525446631E+00, 0.7542783706608855E-16, 0.4703212027287794E-16, 0.1834294683501117E+00, 0.1834294772982473E+00, 0.3305882004599510E-18, 0.6771621290952006E-11, 0.5575326929115112E-13, 0.8954962962003413E-08, 0.1834294772982473E+00, 0.1834406364024603E+00, 0.2169993432366266E+01, 0.5067642455139780E-26, 0.3415542419024794E-17, 0.4171805735046273E-17, 0.1810817848712033E+01, 0.1972152263052530E-29, 0.6115792910959321E-18, 0.6699979816704878E-18, 0.1681677309759878E+01, 0.5241368559131172E-22, 0.4369342452764876E-18, 0.4084491703488284E-18, 0.1303743375476047E+01, 0.0000000000000000E+00, 0.2325140442247288E-18, 0.2353545363971710E-18, 0.1202161769544433E+01, 0.0000000000000000E+00, 0.2163565864913247E-18, 0.2222159192042978E-18, 0.1183377192742860E+01, 0.2033579915108999E-22, 0.2652482201353177E-18, 0.2438256017573345E-18, 0.1094206688544886E+01, 0.3167883469916549E-16, 0.9806070694382835E-18, 0.1224486171222500E-17, 0.9154376259418607E+00, 0.2764930279233778E-14, 0.5964194309842799E-17, 0.7719264500395369E-17, 0.8241395430971566E+00, 0.2245698748385924E-16, 0.1792267857826344E-16, 0.2548071704976161E-16, 0.7318275341991307E+00, 0.4959852501050381E-23, 0.8676862906242004E-16, 0.1414000300319855E-15, 0.5186198053161721E+00, 0.1503295986001297E-14, 0.3452895805257589E-14, 0.7981257539768321E-14, 0.4143051093784424E+00, 0.1303656847202082E-15, 0.9855809897129084E-12, 0.2614986238618434E-11, 0.2112065329503869E+00, 0.8411884198867843E-17, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    3,
			ppOut:    1,
			dminOut:  5.5753269291151117E-014,
			desigOut: 1.3632246356731358E-016,
			sigmaOut: 2.3666022699120921,
			qmaxOut:  0.41760242607758735,
			nFailOut: 3,
			iterOut:  101,
			nDivOut:  1355,
			ttypeOut: -4,
			dmin1Out: 0.11437524358975594,
			dmin2Out: 0.11437524358975594,
			dnOut:    5.5753269291151117E-014,
			dn1Out:   0.18342946834334006,
			dn2Out:   0.11437524358975594,
			gOut:     0.0000000000000000,
			tauOut:   8.9549072084035346E-009,
		},
		{
			i0:       1,
			n0:       3,
			z:        []float64{0.1143752435897560E+00, 0.1143752525446631E+00, 0.7542783706608855E-16, 0.4703212027287794E-16, 0.1834294683501117E+00, 0.1834294772982473E+00, 0.3305882004599510E-18, 0.6771621290952006E-11, 0.5575326929115112E-13, 0.8954962962003413E-08, 0.1834294772982473E+00, 0.1834406364024603E+00, 0.2169993432366266E+01, 0.5067642455139780E-26, 0.3415542419024794E-17, 0.4171805735046273E-17, 0.1810817848712033E+01, 0.1972152263052530E-29, 0.6115792910959321E-18, 0.6699979816704878E-18, 0.1681677309759878E+01, 0.5241368559131172E-22, 0.4369342452764876E-18, 0.4084491703488284E-18, 0.1303743375476047E+01, 0.0000000000000000E+00, 0.2325140442247288E-18, 0.2353545363971710E-18, 0.1202161769544433E+01, 0.0000000000000000E+00, 0.2163565864913247E-18, 0.2222159192042978E-18, 0.1183377192742860E+01, 0.2033579915108999E-22, 0.2652482201353177E-18, 0.2438256017573345E-18, 0.1094206688544886E+01, 0.3167883469916549E-16, 0.9806070694382835E-18, 0.1224486171222500E-17, 0.9154376259418607E+00, 0.2764930279233778E-14, 0.5964194309842799E-17, 0.7719264500395369E-17, 0.8241395430971566E+00, 0.2245698748385924E-16, 0.1792267857826344E-16, 0.2548071704976161E-16, 0.7318275341991307E+00, 0.4959852501050381E-23, 0.8676862906242004E-16, 0.1414000300319855E-15, 0.5186198053161721E+00, 0.1503295986001297E-14, 0.3452895805257589E-14, 0.7981257539768321E-14, 0.4143051093784424E+00, 0.1303656847202082E-15, 0.9855809897129084E-12, 0.2614986238618434E-11, 0.2112065329503869E+00, 0.8411884198867843E-17, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       0,
			dmin:     5.5753269291151117E-014,
			desig:    1.3632246356731358E-016,
			qmax:     0.41760242607758735,
			ttype:    -4,
			dmin1:    0.11437524358975594,
			dmin2:    0.11437524358975594,
			dn:       5.5753269291151117E-014,
			dn1:      0.18342946834334006,
			dn2:      0.11437524358975594,
			g:        0.0000000000000000,
			tau:      8.9549072084035346E-009,
			nFail:    3,
			iter:     101,
			sigma:    2.3666022699120921,
			nDiv:     1355,
			zOut:     []float64{0.1143752435897560E+00, 0.1143752435897003E+00, 0.7542783706608855E-16, 0.1209675067575323E-15, 0.1834294683501117E+00, 0.1834294683500558E+00, 0.3305882004599510E-18, 0.1004820715586787E-30, 0.5575326929115112E-13, 0.7669624420364386E-22, 0.1834294772982473E+00, 0.1834294683501117E+00, 0.2169993432366266E+01, 0.5067642455139780E-26, 0.3415542419024794E-17, 0.4171805735046273E-17, 0.1810817848712033E+01, 0.1972152263052530E-29, 0.6115792910959321E-18, 0.6699979816704878E-18, 0.1681677309759878E+01, 0.5241368559131172E-22, 0.4369342452764876E-18, 0.4084491703488284E-18, 0.1303743375476047E+01, 0.0000000000000000E+00, 0.2325140442247288E-18, 0.2353545363971710E-18, 0.1202161769544433E+01, 0.0000000000000000E+00, 0.2163565864913247E-18, 0.2222159192042978E-18, 0.1183377192742860E+01, 0.2033579915108999E-22, 0.2652482201353177E-18, 0.2438256017573345E-18, 0.1094206688544886E+01, 0.3167883469916549E-16, 0.9806070694382835E-18, 0.1224486171222500E-17, 0.9154376259418607E+00, 0.2764930279233778E-14, 0.5964194309842799E-17, 0.7719264500395369E-17, 0.8241395430971566E+00, 0.2245698748385924E-16, 0.1792267857826344E-16, 0.2548071704976161E-16, 0.7318275341991307E+00, 0.4959852501050381E-23, 0.8676862906242004E-16, 0.1414000300319855E-15, 0.5186198053161721E+00, 0.1503295986001297E-14, 0.3452895805257589E-14, 0.7981257539768321E-14, 0.4143051093784424E+00, 0.1303656847202082E-15, 0.9855809897129084E-12, 0.2614986238618434E-11, 0.2112065329503869E+00, 0.8411884198867843E-17, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    3,
			ppOut:    0,
			dminOut:  7.6696244203643861E-023,
			desigOut: -6.5648763085702813E-017,
			sigmaOut: 2.3666022699121481,
			qmaxOut:  0.41760242607758735,
			nFailOut: 3,
			iterOut:  102,
			nDivOut:  1359,
			ttypeOut: -4,
			dmin1Out: 0.11437524358970023,
			dmin2Out: 0.11437524358970023,
			dnOut:    7.6696244203643861E-023,
			dn1Out:   0.18342946835005580,
			dn2Out:   0.11437524358970023,
			gOut:     0.0000000000000000,
			tauOut:   5.5753269214454873E-014,
		},
		{
			i0:       1,
			n0:       3,
			z:        []float64{0.1143752435897560E+00, 0.1143752435897003E+00, 0.7542783706608855E-16, 0.1209675067575323E-15, 0.1834294683501117E+00, 0.1834294683500558E+00, 0.3305882004599510E-18, 0.1004820715586787E-30, 0.5575326929115112E-13, 0.7669624420364386E-22, 0.1834294772982473E+00, 0.1834294683501117E+00, 0.2169993432366266E+01, 0.5067642455139780E-26, 0.3415542419024794E-17, 0.4171805735046273E-17, 0.1810817848712033E+01, 0.1972152263052530E-29, 0.6115792910959321E-18, 0.6699979816704878E-18, 0.1681677309759878E+01, 0.5241368559131172E-22, 0.4369342452764876E-18, 0.4084491703488284E-18, 0.1303743375476047E+01, 0.0000000000000000E+00, 0.2325140442247288E-18, 0.2353545363971710E-18, 0.1202161769544433E+01, 0.0000000000000000E+00, 0.2163565864913247E-18, 0.2222159192042978E-18, 0.1183377192742860E+01, 0.2033579915108999E-22, 0.2652482201353177E-18, 0.2438256017573345E-18, 0.1094206688544886E+01, 0.3167883469916549E-16, 0.9806070694382835E-18, 0.1224486171222500E-17, 0.9154376259418607E+00, 0.2764930279233778E-14, 0.5964194309842799E-17, 0.7719264500395369E-17, 0.8241395430971566E+00, 0.2245698748385924E-16, 0.1792267857826344E-16, 0.2548071704976161E-16, 0.7318275341991307E+00, 0.4959852501050381E-23, 0.8676862906242004E-16, 0.1414000300319855E-15, 0.5186198053161721E+00, 0.1503295986001297E-14, 0.3452895805257589E-14, 0.7981257539768321E-14, 0.4143051093784424E+00, 0.1303656847202082E-15, 0.9855809897129084E-12, 0.2614986238618434E-11, 0.2112065329503869E+00, 0.8411884198867843E-17, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			pp:       1,
			dmin:     7.6696244203643861E-023,
			desig:    -6.5648763085702813E-017,
			qmax:     0.41760242607758735,
			ttype:    -4,
			dmin1:    0.11437524358970023,
			dmin2:    0.11437524358970023,
			dn:       7.6696244203643861E-023,
			dn1:      0.18342946835005580,
			dn2:      0.11437524358970023,
			g:        0.0000000000000000,
			tau:      5.5753269214454873E-014,
			nFail:    3,
			iter:     102,
			sigma:    2.3666022699121481,
			nDiv:     1359,
			zOut:     []float64{0.2550031738262204E+01, 0.1834294683500561E+00, 0.7542783706608855E-16, 0.1209675067575323E-15, 0.2480977513501848E+01, 0.1143752435897001E+00, 0.3305882004599510E-18, 0.1004820715586787E-30, 0.2366602269912148E+01, 0.7669624420364386E-22, 0.1834294772982473E+00, 0.1834294683501117E+00, 0.2169993432366266E+01, 0.5067642455139780E-26, 0.3415542419024794E-17, 0.4171805735046273E-17, 0.1810817848712033E+01, 0.1972152263052530E-29, 0.6115792910959321E-18, 0.6699979816704878E-18, 0.1681677309759878E+01, 0.5241368559131172E-22, 0.4369342452764876E-18, 0.4084491703488284E-18, 0.1303743375476047E+01, 0.0000000000000000E+00, 0.2325140442247288E-18, 0.2353545363971710E-18, 0.1202161769544433E+01, 0.0000000000000000E+00, 0.2163565864913247E-18, 0.2222159192042978E-18, 0.1183377192742860E+01, 0.2033579915108999E-22, 0.2652482201353177E-18, 0.2438256017573345E-18, 0.1094206688544886E+01, 0.3167883469916549E-16, 0.9806070694382835E-18, 0.1224486171222500E-17, 0.9154376259418607E+00, 0.2764930279233778E-14, 0.5964194309842799E-17, 0.7719264500395369E-17, 0.8241395430971566E+00, 0.2245698748385924E-16, 0.1792267857826344E-16, 0.2548071704976161E-16, 0.7318275341991307E+00, 0.4959852501050381E-23, 0.8676862906242004E-16, 0.1414000300319855E-15, 0.5186198053161721E+00, 0.1503295986001297E-14, 0.3452895805257589E-14, 0.7981257539768321E-14, 0.4143051093784424E+00, 0.1303656847202082E-15, 0.9855809897129084E-12, 0.2614986238618434E-11, 0.2112065329503869E+00, 0.8411884198867843E-17, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			i0Out:    1,
			n0Out:    0,
			ppOut:    1,
			dminOut:  7.6696244203643861E-023,
			desigOut: -6.5648763085702813E-017,
			sigmaOut: 2.3666022699121481,
			qmaxOut:  0.41760242607758735,
			nFailOut: 3,
			iterOut:  102,
			nDivOut:  1359,
			ttypeOut: -4,
			dmin1Out: 0.11437524358970023,
			dmin2Out: 0.11437524358970023,
			dnOut:    7.6696244203643861E-023,
			dn1Out:   0.18342946835005580,
			dn2Out:   0.11437524358970023,
			gOut:     0.0000000000000000,
			tauOut:   5.5753269214454873E-014,
		},
		{
			i0:       1,
			n0:       21,
			z:        []float64{0.1648283185136998E+01, 0.1396221235720571E+01, 0.1712714336271993E+00, 0.2520619494164272E+00, 0.1510753432847732E+01, 0.1119977945086946E+01, 0.3792800633372563E+00, 0.5620469213879850E+00, 0.1328441621586708E+01, 0.1019485448443405E+01, 0.1388998759717073E+00, 0.6882362364805590E+00, 0.4466938077796418E+00, 0.2681061628164644E+00, 0.7093206047768255E+00, 0.3174875209348847E+00, 0.6843112870203156E+00, 0.9979892152967577E+00, 0.6490582981441884E+00, 0.3956426765003833E+00, 0.5085572738629487E+00, 0.1122623886995757E+01, 0.9750235054014829E-02, 0.3499168501137979E-01, 0.1364886053450573E+00, 0.1417066070690837E+00, 0.2301225778544498E-01, 0.4532233329988395E-02, 0.1646009972289452E+01, 0.6930161671496210E+00, 0.2362515608142310E+00, 0.9760060629252760E+00, 0.5818602562677768E+00, 0.3984323866837953E+00, 0.1797665269485310E-01, 0.4196794303982125E+00, 0.5600419521166516E+00, 0.2492354636952108E-01, 0.2195137569256029E+00, 0.5530950584419837E+00, 0.4184071984843414E+00, 0.2222708575473020E+00, 0.2727864547293006E+00, 0.4156500978626423E+00, 0.6774373914466536E-01, 0.2745959086613283E+00, 0.1050967099374242E+00, 0.6593428521263771E-01, 0.2040338718098096E+00, 0.1079809097801335E+00, 0.1271971985482246E+00, 0.2011496719671002E+00, 0.4444741998443960E-01, 0.1290210252363728E+00, 0.5776327498150620E+00, 0.4262359329629137E-01, 0.3402556968467140E+00, 0.6023491555328507E+00, 0.1086565805630269E+00, 0.3155392911289253E+00, 0.9382999256694983E+00, 0.1171677238805356E+00, 0.7901625299559836E+00, 0.9297887823519896E+00, 0.6095853796269167E+00, 0.7973955560628040E+00, 0.8018125008387630E+00, 0.6023523535200964E+00, 0.2443177602187348E-01, 0.8114406374558937E+00, 0.2277830378453201E+00, 0.1480363940474286E-01, 0.6969081780841352E+00, 0.3759308106650992E+00, 0.8012406122589412E-01, 0.5487604052643561E+00, 0.9781022865072954E-01, 0.1017549972519246E+00, 0.2961248981181939E-03, 0.7617929262469909E-01, 0.8408398800007548E-04, 0.3802088861182694E-03, 0.0000000000000000E+00, 0.0000000000000000E+00},
			pp:       0,
			dmin:     -0.0000000000000000,
			desig:    0.0000000000000000,
			qmax:     1.8900334961849885,
			ttype:    0,
			dmin1:    0.0000000000000000,
			dmin2:    0.0000000000000000,
			dn:       0.0000000000000000,
			dn1:      0.0000000000000000,
			dn2:      0.0000000000000000,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    0,
			iter:     2,
			sigma:    0.0000000000000000,
			nDiv:     40,
			zOut:     []float64{0.1648283185136998E+01, 0.1819554618764197E+01, 0.1712714336271993E+00, 0.1422045283129674E+00, 0.1510753432847732E+01, 0.1747828967872021E+01, 0.3792800633372563E+00, 0.2882727267008810E+00, 0.1328441621586708E+01, 0.1179068770857534E+01, 0.1388998759717073E+00, 0.5262264257308433E-01, 0.4466938077796418E+00, 0.1103391769983383E+01, 0.7093206047768255E+00, 0.4399127392187891E+00, 0.6843112870203156E+00, 0.8934568459457148E+00, 0.6490582981441884E+00, 0.3694451726238032E+00, 0.5085572738629487E+00, 0.1488623362931603E+00, 0.9750235054014829E-02, 0.8939776288934402E-02, 0.1364886053450573E+00, 0.1505610868415679E+00, 0.2301225778544498E-01, 0.2515816443301624E+00, 0.1646009972289452E+01, 0.1630679888773521E+01, 0.2362515608142310E+00, 0.8429943526342391E-01, 0.5818602562677768E+00, 0.5155374736992060E+00, 0.1797665269485310E-01, 0.1952851185677853E-01, 0.5600419521166516E+00, 0.7600271971854760E+00, 0.2195137569256029E+00, 0.1208458544696003E+00, 0.4184071984843414E+00, 0.5703477987440417E+00, 0.2727864547293006E+00, 0.3240053608004366E-01, 0.6774373914466536E-01, 0.1404399130020459E+00, 0.1050967099374242E+00, 0.1526865702536626E+00, 0.2040338718098096E+00, 0.1785445001043715E+00, 0.1271971985482246E+00, 0.3166486394939177E-01, 0.4444741998443960E-01, 0.5904153058501098E+00, 0.5776327498150620E+00, 0.3328891237445398E+00, 0.3402556968467140E+00, 0.1160231536652011E+00, 0.1086565805630269E+00, 0.8787251358464724E+00, 0.9382999256694983E+00, 0.8497373197790092E+00, 0.7901625299559836E+00, 0.5668475593321608E+00, 0.6095853796269167E+00, 0.8445503211335190E+00, 0.8018125008387630E+00, 0.2319542476253924E-01, 0.2443177602187348E-01, 0.2290193891046544E+00, 0.2277830378453201E+00, 0.6931459494493321E+00, 0.6969081780841352E+00, 0.8388628986069724E-01, 0.8012406122589412E-01, 0.9342352322344821E-01, 0.9781022865072954E-01, 0.4682830325399513E-02, 0.2961248981181939E-03, 0.5317160915449039E-05, 0.8408398800007548E-04, 0.7876682708462645E-04, 0.0000000000000000E+00, 0.8939776288934402E-02},
			i0Out:    1,
			n0Out:    21,
			ppOut:    0,
			dminOut:  7.8766827084626452E-005,
			desigOut: 0.0000000000000000,
			sigmaOut: 0.0000000000000000,
			qmaxOut:  1.8900334961849885,
			nFailOut: 0,
			iterOut:  3,
			nDivOut:  62,
			ttypeOut: -1,
			dmin1Out: 1.2363512593342330E-003,
			dmin2Out: 1.2363512593342330E-003,
			dnOut:    7.8766827084626452E-005,
			dn1Out:   4.3867054272813191E-003,
			dn2Out:   3.7622286348031123E-003,
			gOut:     0.0000000000000000,
			tauOut:   0.0000000000000000,
		},
		{
			i0:       1,
			n0:       21,
			z:        []float64{0.1648283185136998E+01, 0.1819554618764197E+01, 0.1712714336271993E+00, 0.1422045283129674E+00, 0.1510753432847732E+01, 0.1747828967872021E+01, 0.3792800633372563E+00, 0.2882727267008810E+00, 0.1328441621586708E+01, 0.1179068770857534E+01, 0.1388998759717073E+00, 0.5262264257308433E-01, 0.4466938077796418E+00, 0.1103391769983383E+01, 0.7093206047768255E+00, 0.4399127392187891E+00, 0.6843112870203156E+00, 0.8934568459457148E+00, 0.6490582981441884E+00, 0.3694451726238032E+00, 0.5085572738629487E+00, 0.1488623362931603E+00, 0.9750235054014829E-02, 0.8939776288934402E-02, 0.1364886053450573E+00, 0.1505610868415679E+00, 0.2301225778544498E-01, 0.2515816443301624E+00, 0.1646009972289452E+01, 0.1630679888773521E+01, 0.2362515608142310E+00, 0.8429943526342391E-01, 0.5818602562677768E+00, 0.5155374736992060E+00, 0.1797665269485310E-01, 0.1952851185677853E-01, 0.5600419521166516E+00, 0.7600271971854760E+00, 0.2195137569256029E+00, 0.1208458544696003E+00, 0.4184071984843414E+00, 0.5703477987440417E+00, 0.2727864547293006E+00, 0.3240053608004366E-01, 0.6774373914466536E-01, 0.1404399130020459E+00, 0.1050967099374242E+00, 0.1526865702536626E+00, 0.2040338718098096E+00, 0.1785445001043715E+00, 0.1271971985482246E+00, 0.3166486394939177E-01, 0.4444741998443960E-01, 0.5904153058501098E+00, 0.5776327498150620E+00, 0.3328891237445398E+00, 0.3402556968467140E+00, 0.1160231536652011E+00, 0.1086565805630269E+00, 0.8787251358464724E+00, 0.9382999256694983E+00, 0.8497373197790092E+00, 0.7901625299559836E+00, 0.5668475593321608E+00, 0.6095853796269167E+00, 0.8445503211335190E+00, 0.8018125008387630E+00, 0.2319542476253924E-01, 0.2443177602187348E-01, 0.2290193891046544E+00, 0.2277830378453201E+00, 0.6931459494493321E+00, 0.6969081780841352E+00, 0.8388628986069724E-01, 0.8012406122589412E-01, 0.9342352322344821E-01, 0.9781022865072954E-01, 0.4682830325399513E-02, 0.2961248981181939E-03, 0.5317160915449039E-05, 0.8408398800007548E-04, 0.7876682708462645E-04, 0.0000000000000000E+00, 0.8939776288934402E-02},
			pp:       1,
			dmin:     7.8766827084626452E-005,
			desig:    0.0000000000000000,
			qmax:     1.8900334961849885,
			ttype:    -1,
			dmin1:    1.2363512593342330E-003,
			dmin2:    1.2363512593342330E-003,
			dn:       7.8766827084626452E-005,
			dn1:      4.3867054272813191E-003,
			dn2:      3.7622286348031123E-003,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    0,
			iter:     3,
			sigma:    0.0000000000000000,
			nDiv:     62,
			zOut:     []float64{0.1961759147077164E+01, 0.1819554618764197E+01, 0.1266970995487882E+00, 0.1422045283129674E+00, 0.1909404595024114E+01, 0.1747828967872021E+01, 0.1780101349021133E+00, 0.2882727267008810E+00, 0.1053681278528505E+01, 0.1179068770857534E+01, 0.5510526941411123E-01, 0.5262264257308433E-01, 0.1488199239788061E+01, 0.1103391769983383E+01, 0.2641064704009213E+00, 0.4399127392187891E+00, 0.9987955481685968E+00, 0.8934568459457148E+00, 0.5506279200968747E-01, 0.3694451726238032E+00, 0.1027393205724072E+00, 0.1488623362931603E+00, 0.1310094739466220E-01, 0.8939776288934402E-02, 0.3890417837770681E+00, 0.1505610868415679E+00, 0.1054511738587064E+01, 0.2515816443301624E+00, 0.6604675854498806E+00, 0.1630679888773521E+01, 0.6580113672099847E-01, 0.8429943526342391E-01, 0.4692648488349861E+00, 0.5155374736992060E+00, 0.3162862116895929E-01, 0.1952851185677853E-01, 0.8492444304861170E+00, 0.7600271971854760E+00, 0.8115939841327705E-01, 0.1208458544696003E+00, 0.5215889364108083E+00, 0.5703477987440417E+00, 0.8723974284448969E-02, 0.3240053608004366E-01, 0.2844025089712595E+00, 0.1404399130020459E+00, 0.9585480612390133E-01, 0.1526865702536626E+00, 0.1143545579298620E+00, 0.1785445001043715E+00, 0.1634864466429828E+00, 0.3166486394939177E-01, 0.7598179829516669E+00, 0.5904153058501098E+00, 0.5083170815153470E-01, 0.3328891237445398E+00, 0.9439165813601388E+00, 0.1160231536652011E+00, 0.7910503496831139E+00, 0.8787251358464724E+00, 0.6255345294280562E+00, 0.8497373197790092E+00, 0.7653155273545736E+00, 0.5668475593321608E+00, 0.1024302185414846E+00, 0.8445503211335190E+00, 0.5186166821452450E-01, 0.2319542476253924E-01, 0.8703036703394620E+00, 0.2290193891046544E+00, 0.6681052144545734E-01, 0.6931459494493321E+00, 0.1104992916386881E+00, 0.8388628986069724E-01, 0.3959179295799719E-02, 0.9342352322344821E-01, 0.7289681905152429E-03, 0.4682830325399513E-02, 0.5745324691222600E-06, 0.5317160915449039E-05, 0.7819229461550419E-04, 0.7876682708462645E-04, 0.8723974284448969E-02, 0.8939776288934402E-02},
			i0Out:    1,
			n0Out:    21,
			ppOut:    1,
			dminOut:  7.8192294615504193E-005,
			desigOut: 0.0000000000000000,
			sigmaOut: 0.0000000000000000,
			qmaxOut:  1.8900334961849885,
			nFailOut: 0,
			iterOut:  4,
			nDivOut:  84,
			ttypeOut: -4,
			dmin1Out: 7.2365102959979382E-004,
			dmin2Out: 1.7075768415239889E-002,
			dnOut:    7.8192294615504193E-005,
			dn1Out:   7.2365102959979382E-004,
			dn2Out:   1.7075768415239889E-002,
			gOut:     0.0000000000000000,
			tauOut:   0.0000000000000000,
		},
		{
			i0:       1,
			n0:       21,
			z:        []float64{0.1961759147077164E+01, 0.1819554618764197E+01, 0.1266970995487882E+00, 0.1422045283129674E+00, 0.1909404595024114E+01, 0.1747828967872021E+01, 0.1780101349021133E+00, 0.2882727267008810E+00, 0.1053681278528505E+01, 0.1179068770857534E+01, 0.5510526941411123E-01, 0.5262264257308433E-01, 0.1488199239788061E+01, 0.1103391769983383E+01, 0.2641064704009213E+00, 0.4399127392187891E+00, 0.9987955481685968E+00, 0.8934568459457148E+00, 0.5506279200968747E-01, 0.3694451726238032E+00, 0.1027393205724072E+00, 0.1488623362931603E+00, 0.1310094739466220E-01, 0.8939776288934402E-02, 0.3890417837770681E+00, 0.1505610868415679E+00, 0.1054511738587064E+01, 0.2515816443301624E+00, 0.6604675854498806E+00, 0.1630679888773521E+01, 0.6580113672099847E-01, 0.8429943526342391E-01, 0.4692648488349861E+00, 0.5155374736992060E+00, 0.3162862116895929E-01, 0.1952851185677853E-01, 0.8492444304861170E+00, 0.7600271971854760E+00, 0.8115939841327705E-01, 0.1208458544696003E+00, 0.5215889364108083E+00, 0.5703477987440417E+00, 0.8723974284448969E-02, 0.3240053608004366E-01, 0.2844025089712595E+00, 0.1404399130020459E+00, 0.9585480612390133E-01, 0.1526865702536626E+00, 0.1143545579298620E+00, 0.1785445001043715E+00, 0.1634864466429828E+00, 0.3166486394939177E-01, 0.7598179829516669E+00, 0.5904153058501098E+00, 0.5083170815153470E-01, 0.3328891237445398E+00, 0.9439165813601388E+00, 0.1160231536652011E+00, 0.7910503496831139E+00, 0.8787251358464724E+00, 0.6255345294280562E+00, 0.8497373197790092E+00, 0.7653155273545736E+00, 0.5668475593321608E+00, 0.1024302185414846E+00, 0.8445503211335190E+00, 0.5186166821452450E-01, 0.2319542476253924E-01, 0.8703036703394620E+00, 0.2290193891046544E+00, 0.6681052144545734E-01, 0.6931459494493321E+00, 0.1104992916386881E+00, 0.8388628986069724E-01, 0.3959179295799719E-02, 0.9342352322344821E-01, 0.7289681905152429E-03, 0.4682830325399513E-02, 0.5745324691222600E-06, 0.5317160915449039E-05, 0.7819229461550419E-04, 0.7876682708462645E-04, 0.8723974284448969E-02, 0.8939776288934402E-02},
			pp:       0,
			dmin:     7.8192294615504193E-005,
			desig:    0.0000000000000000,
			qmax:     1.8900334961849885,
			ttype:    -4,
			dmin1:    7.2365102959979382E-004,
			dmin2:    1.7075768415239889E-002,
			dn:       7.8192294615504193E-005,
			dn1:      7.2365102959979382E-004,
			dn2:      1.7075768415239889E-002,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    0,
			iter:     4,
			sigma:    0.0000000000000000,
			nDiv:     84,
			zOut:     []float64{0.1961759147077164E+01, 0.2088378163269771E+01, 0.1266970995487882E+00, 0.1158391848322702E+00, 0.1909404595024114E+01, 0.1971497461737776E+01, 0.1780101349021133E+00, 0.9513882222772962E-01, 0.1053681278528505E+01, 0.1013569642358705E+01, 0.5510526941411123E-01, 0.8090970429970110E-01, 0.1488199239788061E+01, 0.1671317922533099E+01, 0.2641064704009213E+00, 0.1578325483874163E+00, 0.9987955481685968E+00, 0.8959477084346864E+00, 0.5506279200968747E-01, 0.6314111623521673E-02, 0.1027393205724072E+00, 0.1094480729873660E+00, 0.1310094739466220E-01, 0.4656834793406785E-01, 0.3890417837770681E+00, 0.1396907091073883E+01, 0.1054511738587064E+01, 0.4985806330739840E+00, 0.6604675854498806E+00, 0.2276100057407134E+00, 0.6580113672099847E-01, 0.1356625794022653E+00, 0.4692648488349861E+00, 0.3651528072454984E+00, 0.3162862116895929E-01, 0.7355942454424362E-01, 0.8492444304861170E+00, 0.8567663209989688E+00, 0.8115939841327705E-01, 0.4940885660487250E-01, 0.5215889364108083E+00, 0.4808259707342031E+00, 0.8723974284448969E-02, 0.5160120970399038E-02, 0.2844025089712595E+00, 0.3750191107685802E+00, 0.9585480612390133E-01, 0.2922900104287094E-01, 0.1143545579298620E+00, 0.2485339201737921E+00, 0.1634864466429828E+00, 0.4998108187459615E+00, 0.7598179829516669E+00, 0.3107607890010585E+00, 0.5083170815153470E-01, 0.1543981540828483E+00, 0.9439165813601388E+00, 0.1580490693604223E+01, 0.7910503496831139E+00, 0.3130858727896049E+00, 0.6255345294280562E+00, 0.1077686100636843E+01, 0.7653155273545736E+00, 0.7274051012980143E-01, 0.1024302185414846E+00, 0.8147329327002600E-01, 0.5186166821452450E-01, 0.5539901283655780E+00, 0.8703036703394620E+00, 0.3830459800631597E+00, 0.6681052144545734E-01, 0.1927318305890360E-01, 0.1104992916386881E+00, 0.9510720451940254E-01, 0.3959179295799719E-02, 0.3034592154998885E-04, 0.7289681905152429E-03, 0.6211134452527084E-03, 0.5745324691222600E-06, 0.7232819130731082E-07, 0.7819229461550419E-04, 0.3661024252896476E-07, 0.8723974284448969E-02, 0.5160120970399038E-02},
			i0Out:    1,
			n0Out:    21,
			ppOut:    0,
			dminOut:  3.6610242528964756E-008,
			desigOut: 0.0000000000000000,
			sigmaOut: 7.8083356181667918E-005,
			qmaxOut:  1.8900334961849885,
			nFailOut: 0,
			iterOut:  5,
			nDivOut:  106,
			ttypeOut: -2,
			dmin1Out: 6.2053891278358614E-004,
			dmin2Out: 2.9611625055501498E-002,
			dnOut:    3.6610242528964756E-008,
			dn1Out:   6.2053891278358614E-004,
			dn2Out:   9.1148025223602810E-002,
			gOut:     0.0000000000000000,
			tauOut:   7.8083356181667918E-005,
		},
		{
			i0:       1,
			n0:       21,
			z:        []float64{0.1961759147077164E+01, 0.2088378163269771E+01, 0.1266970995487882E+00, 0.1158391848322702E+00, 0.1909404595024114E+01, 0.1971497461737776E+01, 0.1780101349021133E+00, 0.9513882222772962E-01, 0.1053681278528505E+01, 0.1013569642358705E+01, 0.5510526941411123E-01, 0.8090970429970110E-01, 0.1488199239788061E+01, 0.1671317922533099E+01, 0.2641064704009213E+00, 0.1578325483874163E+00, 0.9987955481685968E+00, 0.8959477084346864E+00, 0.5506279200968747E-01, 0.6314111623521673E-02, 0.1027393205724072E+00, 0.1094480729873660E+00, 0.1310094739466220E-01, 0.4656834793406785E-01, 0.3890417837770681E+00, 0.1396907091073883E+01, 0.1054511738587064E+01, 0.4985806330739840E+00, 0.6604675854498806E+00, 0.2276100057407134E+00, 0.6580113672099847E-01, 0.1356625794022653E+00, 0.4692648488349861E+00, 0.3651528072454984E+00, 0.3162862116895929E-01, 0.7355942454424362E-01, 0.8492444304861170E+00, 0.8567663209989688E+00, 0.8115939841327705E-01, 0.4940885660487250E-01, 0.5215889364108083E+00, 0.4808259707342031E+00, 0.8723974284448969E-02, 0.5160120970399038E-02, 0.2844025089712595E+00, 0.3750191107685802E+00, 0.9585480612390133E-01, 0.2922900104287094E-01, 0.1143545579298620E+00, 0.2485339201737921E+00, 0.1634864466429828E+00, 0.4998108187459615E+00, 0.7598179829516669E+00, 0.3107607890010585E+00, 0.5083170815153470E-01, 0.1543981540828483E+00, 0.9439165813601388E+00, 0.1580490693604223E+01, 0.7910503496831139E+00, 0.3130858727896049E+00, 0.6255345294280562E+00, 0.1077686100636843E+01, 0.7653155273545736E+00, 0.7274051012980143E-01, 0.1024302185414846E+00, 0.8147329327002600E-01, 0.5186166821452450E-01, 0.5539901283655780E+00, 0.8703036703394620E+00, 0.3830459800631597E+00, 0.6681052144545734E-01, 0.1927318305890360E-01, 0.1104992916386881E+00, 0.9510720451940254E-01, 0.3959179295799719E-02, 0.3034592154998885E-04, 0.7289681905152429E-03, 0.6211134452527084E-03, 0.5745324691222600E-06, 0.7232819130731082E-07, 0.7819229461550419E-04, 0.3661024252896476E-07, 0.8723974284448969E-02, 0.5160120970399038E-02},
			pp:       1,
			dmin:     3.6610242528964756E-008,
			desig:    0.0000000000000000,
			qmax:     1.8900334961849885,
			ttype:    -2,
			dmin1:    6.2053891278358614E-004,
			dmin2:    2.9611625055501498E-002,
			dn:       3.6610242528964756E-008,
			dn1:      6.2053891278358614E-004,
			dn2:      9.1148025223602810E-002,
			g:        0.0000000000000000,
			tau:      7.8083356181667918E-005,
			nFail:    0,
			iter:     5,
			sigma:    7.8083356181667918E-005,
			nDiv:     106,
			zOut:     []float64{0.2204217311496068E+01, 0.2088378163269771E+01, 0.1036089580076783E+00, 0.1158391848322702E+00, 0.1963027289351853E+01, 0.1971497461737776E+01, 0.4912301654839819E-01, 0.9513882222772962E-01, 0.1045356293504034E+01, 0.1013569642358705E+01, 0.1293586117415210E+00, 0.8090970429970110E-01, 0.1699791822573021E+01, 0.1671317922533099E+01, 0.8319236989271818E-01, 0.1578325483874163E+00, 0.8190694135595162E+00, 0.8959477084346864E+00, 0.8437225690290790E-03, 0.6314111623521673E-02, 0.1551726617464312E+00, 0.1094480729873660E+00, 0.4192211096758560E+00, 0.4656834793406785E-01, 0.1476266577866037E+01, 0.1396907091073883E+01, 0.7687090018675194E-01, 0.4985806330739840E+00, 0.2864016483502532E+00, 0.2276100057407134E+00, 0.1729653861709652E+00, 0.1356625794022653E+00, 0.2657468090128033E+00, 0.3651528072454984E+00, 0.2371551996266359E+00, 0.7355942454424362E-01, 0.6690199413712319E+00, 0.8567663209989688E+00, 0.3551024412099299E-01, 0.4940885660487250E-01, 0.4504758109776356E+00, 0.4808259707342031E+00, 0.4295777776785939E-02, 0.5160120970399038E-02, 0.3999522974286917E+00, 0.3750191107685802E+00, 0.1816316160365039E-01, 0.2922900104287094E-01, 0.7301815407101296E+00, 0.2485339201737921E+00, 0.2127164214993764E+00, 0.4998108187459615E+00, 0.2524424849785568E+00, 0.3107607890010585E+00, 0.9666552191416621E+00, 0.1543981540828483E+00, 0.9269213106461920E+00, 0.1580490693604223E+01, 0.3640096408786760E+00, 0.3130858727896049E+00, 0.7864169332819952E+00, 0.1077686100636843E+01, 0.7535963003344312E-02, 0.7274051012980143E-01, 0.6279274220262862E+00, 0.8147329327002600E-01, 0.3379430237022286E+00, 0.5539901283655780E+00, 0.6437610281386123E-01, 0.3830459800631597E+00, 0.2847358698029713E-01, 0.1927318305890360E-01, 0.6666392685468185E-01, 0.9510720451940254E-01, 0.2827355178816361E-06, 0.3034592154998885E-04, 0.6208664319525956E-03, 0.6211134452527084E-03, 0.4264931214133004E-11, 0.7232819130731082E-07, 0.4059351646211308E-14, 0.3661024252896476E-07, 0.8437225690290790E-03, 0.5160120970399038E-02},
			i0Out:    1,
			n0Out:    21,
			ppOut:    1,
			dminOut:  4.0593516462113082E-015,
			desigOut: 4.3344264097778650E-021,
			sigmaOut: 7.8119962155206313E-005,
			qmaxOut:  1.8900334961849885,
			nFailOut: 0,
			iterOut:  6,
			nDivOut:  128,
			ttypeOut: -2,
			dmin1Out: 6.2079410376128833E-004,
			dmin2Out: 4.5102919754957636E-002,
			dnOut:    4.0593516462113082E-015,
			dn1Out:   6.2079410376128833E-004,
			dn2Out:   6.6633580933131861E-002,
			gOut:     0.0000000000000000,
			tauOut:   3.6605973538398975E-008,
		},
		{
			i0:       1,
			n0:       21,
			z:        []float64{0.2204217311496068E+01, 0.2088378163269771E+01, 0.1036089580076783E+00, 0.1158391848322702E+00, 0.1963027289351853E+01, 0.1971497461737776E+01, 0.4912301654839819E-01, 0.9513882222772962E-01, 0.1045356293504034E+01, 0.1013569642358705E+01, 0.1293586117415210E+00, 0.8090970429970110E-01, 0.1699791822573021E+01, 0.1671317922533099E+01, 0.8319236989271818E-01, 0.1578325483874163E+00, 0.8190694135595162E+00, 0.8959477084346864E+00, 0.8437225690290790E-03, 0.6314111623521673E-02, 0.1551726617464312E+00, 0.1094480729873660E+00, 0.4192211096758560E+00, 0.4656834793406785E-01, 0.1476266577866037E+01, 0.1396907091073883E+01, 0.7687090018675194E-01, 0.4985806330739840E+00, 0.2864016483502532E+00, 0.2276100057407134E+00, 0.1729653861709652E+00, 0.1356625794022653E+00, 0.2657468090128033E+00, 0.3651528072454984E+00, 0.2371551996266359E+00, 0.7355942454424362E-01, 0.6690199413712319E+00, 0.8567663209989688E+00, 0.3551024412099299E-01, 0.4940885660487250E-01, 0.4504758109776356E+00, 0.4808259707342031E+00, 0.4295777776785939E-02, 0.5160120970399038E-02, 0.3999522974286917E+00, 0.3750191107685802E+00, 0.1816316160365039E-01, 0.2922900104287094E-01, 0.7301815407101296E+00, 0.2485339201737921E+00, 0.2127164214993764E+00, 0.4998108187459615E+00, 0.2524424849785568E+00, 0.3107607890010585E+00, 0.9666552191416621E+00, 0.1543981540828483E+00, 0.9269213106461920E+00, 0.1580490693604223E+01, 0.3640096408786760E+00, 0.3130858727896049E+00, 0.7864169332819952E+00, 0.1077686100636843E+01, 0.7535963003344312E-02, 0.7274051012980143E-01, 0.6279274220262862E+00, 0.8147329327002600E-01, 0.3379430237022286E+00, 0.5539901283655780E+00, 0.6437610281386123E-01, 0.3830459800631597E+00, 0.2847358698029713E-01, 0.1927318305890360E-01, 0.6666392685468185E-01, 0.9510720451940254E-01, 0.2827355178816361E-06, 0.3034592154998885E-04, 0.6208664319525956E-03, 0.6211134452527084E-03, 0.4264931214133004E-11, 0.7232819130731082E-07, 0.4059351646211308E-14, 0.3661024252896476E-07, 0.8437225690290790E-03, 0.5160120970399038E-02},
			pp:       0,
			dmin:     4.0593516462113082E-015,
			desig:    4.3344264097778650E-021,
			qmax:     1.8900334961849885,
			ttype:    -2,
			dmin1:    6.2079410376128833E-004,
			dmin2:    4.5102919754957636E-002,
			dn:       4.0593516462113082E-015,
			dn1:      6.2079410376128833E-004,
			dn2:      6.6633580933131861E-002,
			g:        0.0000000000000000,
			tau:      3.6605973538398975E-008,
			nFail:    0,
			iter:     6,
			sigma:    7.8119962155206313E-005,
			nDiv:     128,
			zOut:     []float64{0.2204217311496068E+01, 0.2307826269503742E+01, 0.1036089580076783E+00, 0.8812934261040264E-01, 0.1963027289351853E+01, 0.1924020963289845E+01, 0.4912301654839819E-01, 0.2668944646890268E-01, 0.1045356293504034E+01, 0.1148025458776649E+01, 0.1293586117415210E+00, 0.1915312145184878E+00, 0.1699791822573021E+01, 0.1591452977947247E+01, 0.8319236989271818E-01, 0.4281642408847453E-01, 0.8190694135595162E+00, 0.7770967120400667E+00, 0.8437225690290790E-03, 0.1684766835109566E-03, 0.1551726617464312E+00, 0.5742252947387723E+00, 0.4192211096758560E+00, 0.1077768810640642E+01, 0.1476266577866037E+01, 0.4753686674121431E+00, 0.7687090018675194E-01, 0.4631342794110950E-01, 0.2864016483502532E+00, 0.4130536065801048E+00, 0.1729653861709652E+00, 0.1112809541240190E+00, 0.2657468090128033E+00, 0.3916210545154161E+00, 0.2371551996266359E+00, 0.4051405202062471E+00, 0.6690199413712319E+00, 0.2993896652859737E+00, 0.3551024412099299E-01, 0.5343038813025977E-01, 0.4504758109776356E+00, 0.4013412006241577E+00, 0.4295777776785939E-02, 0.4280911574482486E-02, 0.3999522974286917E+00, 0.4138345474578556E+00, 0.1816316160365039E-01, 0.3204760309498118E-01, 0.7301815407101296E+00, 0.9108503591145208E+00, 0.2127164214993764E+00, 0.5895442813598011E-01, 0.2524424849785568E+00, 0.1160143275984235E+01, 0.9666552191416621E+00, 0.7723298847804961E+00, 0.9269213106461920E+00, 0.5186010667443678E+00, 0.3640096408786760E+00, 0.5519914319921668E+00, 0.7864169332819952E+00, 0.2419614642931688E+00, 0.7535963003344312E-02, 0.1955698951896722E-01, 0.6279274220262862E+00, 0.9463134562095434E+00, 0.3379430237022286E+00, 0.2298969194226957E-01, 0.6437610281386123E-01, 0.6985999785188474E-01, 0.2847358698029713E-01, 0.2717093011896993E-01, 0.6666392685468185E-01, 0.3949327947122574E-01, 0.2827355178816361E-06, 0.4444831994804014E-08, 0.6208664319525956E-03, 0.6208619913814727E-03, 0.4264931214133004E-11, 0.2788519153273688E-22, 0.4059351646211308E-14, 0.3786532345060857E-28, 0.8437225690290790E-03, 0.1684766835109566E-03},
			i0Out:    1,
			n0Out:    21,
			ppOut:    0,
			dminOut:  3.7865323450608567E-029,
			desigOut: -5.4012504887129591E-021,
			sigmaOut: 7.8119962159265674E-005,
			qmaxOut:  1.8900334961849885,
			nFailOut: 0,
			iterOut:  7,
			nDivOut:  150,
			ttypeOut: -2,
			dmin1Out: 6.2086198711654151E-004,
			dmin2Out: 3.9492996735707858E-002,
			dnOut:    3.7865323450608567E-029,
			dn1Out:   6.2086198711654151E-004,
			dn2Out:   3.9492996735707858E-002,
			gOut:     0.0000000000000000,
			tauOut:   4.0593516183260787E-015,
		},
		{
			i0:       1,
			n0:       21,
			z:        []float64{0.2204217311496068E+01, 0.2307826269503742E+01, 0.1036089580076783E+00, 0.8812934261040264E-01, 0.1963027289351853E+01, 0.1924020963289845E+01, 0.4912301654839819E-01, 0.2668944646890268E-01, 0.1045356293504034E+01, 0.1148025458776649E+01, 0.1293586117415210E+00, 0.1915312145184878E+00, 0.1699791822573021E+01, 0.1591452977947247E+01, 0.8319236989271818E-01, 0.4281642408847453E-01, 0.8190694135595162E+00, 0.7770967120400667E+00, 0.8437225690290790E-03, 0.1684766835109566E-03, 0.1551726617464312E+00, 0.5742252947387723E+00, 0.4192211096758560E+00, 0.1077768810640642E+01, 0.1476266577866037E+01, 0.4753686674121431E+00, 0.7687090018675194E-01, 0.4631342794110950E-01, 0.2864016483502532E+00, 0.4130536065801048E+00, 0.1729653861709652E+00, 0.1112809541240190E+00, 0.2657468090128033E+00, 0.3916210545154161E+00, 0.2371551996266359E+00, 0.4051405202062471E+00, 0.6690199413712319E+00, 0.2993896652859737E+00, 0.3551024412099299E-01, 0.5343038813025977E-01, 0.4504758109776356E+00, 0.4013412006241577E+00, 0.4295777776785939E-02, 0.4280911574482486E-02, 0.3999522974286917E+00, 0.4138345474578556E+00, 0.1816316160365039E-01, 0.3204760309498118E-01, 0.7301815407101296E+00, 0.9108503591145208E+00, 0.2127164214993764E+00, 0.5895442813598011E-01, 0.2524424849785568E+00, 0.1160143275984235E+01, 0.9666552191416621E+00, 0.7723298847804961E+00, 0.9269213106461920E+00, 0.5186010667443678E+00, 0.3640096408786760E+00, 0.5519914319921668E+00, 0.7864169332819952E+00, 0.2419614642931688E+00, 0.7535963003344312E-02, 0.1955698951896722E-01, 0.6279274220262862E+00, 0.9463134562095434E+00, 0.3379430237022286E+00, 0.2298969194226957E-01, 0.6437610281386123E-01, 0.6985999785188474E-01, 0.2847358698029713E-01, 0.2717093011896993E-01, 0.6666392685468185E-01, 0.3949327947122574E-01, 0.2827355178816361E-06, 0.4444831994804014E-08, 0.6208664319525956E-03, 0.6208619913814727E-03, 0.4264931214133004E-11, 0.2788519153273688E-22, 0.4059351646211308E-14, 0.3786532345060857E-28, 0.8437225690290790E-03, 0.1684766835109566E-03},
			pp:       1,
			dmin:     3.7865323450608567E-029,
			desig:    -5.4012504887129591E-021,
			qmax:     1.8900334961849885,
			ttype:    -2,
			dmin1:    6.2086198711654151E-004,
			dmin2:    3.9492996735707858E-002,
			dn:       3.7865323450608567E-029,
			dn1:      6.2086198711654151E-004,
			dn2:      3.9492996735707858E-002,
			g:        0.0000000000000000,
			tau:      4.0593516183260787E-015,
			nFail:    0,
			iter:     7,
			sigma:    7.8119962159265674E-005,
			nDiv:     150,
			zOut:     []float64{0.2395955612114145E+01, 0.2307826269503742E+01, 0.7077038564739886E-01, 0.8812934261040264E-01, 0.1879940024111348E+01, 0.1924020963289845E+01, 0.1629847954401656E-01, 0.2668944646890268E-01, 0.1323258193751120E+01, 0.1148025458776649E+01, 0.2303502998543534E+00, 0.1915312145184878E+00, 0.1403919102181368E+01, 0.1591452977947247E+01, 0.2369972908607684E-01, 0.4281642408847453E-01, 0.7535654596375009E+00, 0.7770967120400667E+00, 0.1283811140869274E-03, 0.1684766835109566E-03, 0.1651865724265327E+01, 0.5742252947387723E+00, 0.3101568824672333E+00, 0.1077768810640642E+01, 0.2115252128860193E+00, 0.4753686674121431E+00, 0.9043805314343908E-01, 0.4631342794110950E-01, 0.4338965075606848E+00, 0.4130536065801048E+00, 0.1004386157577793E+00, 0.1112809541240190E+00, 0.6963229589638840E+00, 0.3916210545154161E+00, 0.1741934301847783E+00, 0.4051405202062471E+00, 0.1786266232314551E+00, 0.2993896652859737E+00, 0.1200482645536405E+00, 0.5343038813025977E-01, 0.2855738476449996E+00, 0.4013412006241577E+00, 0.6203611145567293E-02, 0.4280911574482486E-02, 0.4396785394072695E+00, 0.4138345474578556E+00, 0.6639071087521134E-01, 0.3204760309498118E-01, 0.9034140763752896E+00, 0.9108503591145208E+00, 0.7570790092830114E-01, 0.5895442813598011E-01, 0.1856765259836430E+01, 0.1160143275984235E+01, 0.2157144528657351E+00, 0.7723298847804961E+00, 0.8548780458707995E+00, 0.5186010667443678E+00, 0.1562335771835849E+00, 0.5519914319921668E+00, 0.1052848766285511E+00, 0.2419614642931688E+00, 0.1757806338135458E+00, 0.1955698951896722E-01, 0.7935225143382671E+00, 0.9463134562095434E+00, 0.2023962522401480E-02, 0.2298969194226957E-01, 0.9500696544845319E-01, 0.6985999785188474E-01, 0.1129463646814219E-01, 0.2717093011896993E-01, 0.2819864744791555E-01, 0.3949327947122574E-01, 0.9786381594178541E-10, 0.4444831994804014E-08, 0.6208618935176568E-03, 0.6208619913814727E-03, 0.1700670967075909E-47, 0.2788519153273688E-22, 0.3786532345060857E-28, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			i0Out:    1,
			n0Out:    21,
			ppOut:    1,
			dminOut:  3.7865323450608567E-029,
			desigOut: -5.4012504887129591E-021,
			sigmaOut: 7.8119962159265674E-005,
			qmaxOut:  1.8900334961849885,
			nFailOut: 0,
			iterOut:  8,
			nDivOut:  172,
			ttypeOut: -2,
			dmin1Out: 6.2086189351765679E-004,
			dmin2Out: 2.8198643003083550E-002,
			dnOut:    3.7865323450608567E-029,
			dn1Out:   6.2086189351765679E-004,
			dn2Out:   2.8198643003083550E-002,
			gOut:     0.0000000000000000,
			tauOut:   0.0000000000000000,
		},
		{
			i0:       1,
			n0:       21,
			z:        []float64{0.2395955612114145E+01, 0.2307826269503742E+01, 0.7077038564739886E-01, 0.8812934261040264E-01, 0.1879940024111348E+01, 0.1924020963289845E+01, 0.1629847954401656E-01, 0.2668944646890268E-01, 0.1323258193751120E+01, 0.1148025458776649E+01, 0.2303502998543534E+00, 0.1915312145184878E+00, 0.1403919102181368E+01, 0.1591452977947247E+01, 0.2369972908607684E-01, 0.4281642408847453E-01, 0.7535654596375009E+00, 0.7770967120400667E+00, 0.1283811140869274E-03, 0.1684766835109566E-03, 0.1651865724265327E+01, 0.5742252947387723E+00, 0.3101568824672333E+00, 0.1077768810640642E+01, 0.2115252128860193E+00, 0.4753686674121431E+00, 0.9043805314343908E-01, 0.4631342794110950E-01, 0.4338965075606848E+00, 0.4130536065801048E+00, 0.1004386157577793E+00, 0.1112809541240190E+00, 0.6963229589638840E+00, 0.3916210545154161E+00, 0.1741934301847783E+00, 0.4051405202062471E+00, 0.1786266232314551E+00, 0.2993896652859737E+00, 0.1200482645536405E+00, 0.5343038813025977E-01, 0.2855738476449996E+00, 0.4013412006241577E+00, 0.6203611145567293E-02, 0.4280911574482486E-02, 0.4396785394072695E+00, 0.4138345474578556E+00, 0.6639071087521134E-01, 0.3204760309498118E-01, 0.9034140763752896E+00, 0.9108503591145208E+00, 0.7570790092830114E-01, 0.5895442813598011E-01, 0.1856765259836430E+01, 0.1160143275984235E+01, 0.2157144528657351E+00, 0.7723298847804961E+00, 0.8548780458707995E+00, 0.5186010667443678E+00, 0.1562335771835849E+00, 0.5519914319921668E+00, 0.1052848766285511E+00, 0.2419614642931688E+00, 0.1757806338135458E+00, 0.1955698951896722E-01, 0.7935225143382671E+00, 0.9463134562095434E+00, 0.2023962522401480E-02, 0.2298969194226957E-01, 0.9500696544845319E-01, 0.6985999785188474E-01, 0.1129463646814219E-01, 0.2717093011896993E-01, 0.2819864744791555E-01, 0.3949327947122574E-01, 0.9786381594178541E-10, 0.4444831994804014E-08, 0.6208618935176568E-03, 0.6208619913814727E-03, 0.1700670967075909E-47, 0.2788519153273688E-22, 0.3786532345060857E-28, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			pp:       0,
			dmin:     3.7865323450608567E-029,
			desig:    -5.4012504887129591E-021,
			qmax:     1.8900334961849885,
			ttype:    -2,
			dmin1:    6.2086189351765679E-004,
			dmin2:    2.8198643003083550E-002,
			dn:       3.7865323450608567E-029,
			dn1:      6.2086189351765679E-004,
			dn2:      2.8198643003083550E-002,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    0,
			iter:     8,
			sigma:    7.8119962159265674E-005,
			nDiv:     172,
			zOut:     []float64{0.2395955612114145E+01, 0.2466725997761544E+01, 0.7077038564739886E-01, 0.5393549207373363E-01, 0.1879940024111348E+01, 0.1842303011581631E+01, 0.1629847954401656E-01, 0.1170659574821484E-01, 0.1323258193751120E+01, 0.1541901897857258E+01, 0.2303502998543534E+00, 0.2097365510789915E+00, 0.1403919102181368E+01, 0.1217882280188454E+01, 0.2369972908607684E-01, 0.1466422291592108E-01, 0.7535654596375009E+00, 0.7390296178356669E+00, 0.1283811140869274E-03, 0.2869551596920546E-03, 0.1651865724265327E+01, 0.1961735651572868E+01, 0.3101568824672333E+00, 0.3344283443049241E-01, 0.2115252128860193E+00, 0.2685204315989660E+00, 0.9043805314343908E-01, 0.1461369444993732E+00, 0.4338965075606848E+00, 0.3881981788190908E+00, 0.1004386157577793E+00, 0.1801598202532681E+00, 0.6963229589638840E+00, 0.6903565688953941E+00, 0.1741934301847783E+00, 0.4507175802324546E-01, 0.1786266232314551E+00, 0.2536031297618502E+00, 0.1200482645536405E+00, 0.1351822623162481E+00, 0.2855738476449996E+00, 0.1565951964743188E+00, 0.6203611145567293E-02, 0.1741812487831326E-01, 0.4396785394072695E+00, 0.4886511254041676E+00, 0.6639071087521134E-01, 0.1227425859208231E+00, 0.9034140763752896E+00, 0.8563793913827676E+00, 0.7570790092830114E-01, 0.1641466407918003E+00, 0.1856765259836430E+01, 0.1908333071910365E+01, 0.2157144528657351E+00, 0.9663383852973971E-01, 0.8548780458707995E+00, 0.9144777845246447E+00, 0.1562335771835849E+00, 0.1798735100772441E-01, 0.1052848766285511E+00, 0.2630781594343725E+00, 0.1757806338135458E+00, 0.5302070335887964E+00, 0.7935225143382671E+00, 0.2653394432718723E+00, 0.2023962522401480E-02, 0.7246963929058098E-03, 0.9500696544845319E-01, 0.1055769055236896E+00, 0.1129463646814219E-01, 0.3016696409481782E-02, 0.2819864744791555E-01, 0.2518195113629758E-01, 0.9786381594178541E-10, 0.2412835834031154E-11, 0.6208618935176568E-03, 0.6208618911048210E-03, 0.1700670967075909E-47, 0.2869551596920546E-03, 0.7811996215926567E-04, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			i0Out:    1,
			n0Out:    20,
			ppOut:    0,
			dminOut:  6.2086189110482101E-004,
			desigOut: -5.4012504887129591E-021,
			sigmaOut: 7.8119962159265674E-005,
			qmaxOut:  1.8900334961849885,
			nFailOut: 0,
			iterOut:  9,
			nDivOut:  193,
			ttypeOut: -7,
			dmin1Out: 2.5181951038433764E-002,
			dmin2Out: 8.7297525620826724E-002,
			dnOut:    6.2086189110482101E-004,
			dn1Out:   2.5181951038433764E-002,
			dn2Out:   9.4282269055547374E-002,
			gOut:     0.0000000000000000,
			tauOut:   0.0000000000000000,
		},
		{
			i0:       1,
			n0:       20,
			z:        []float64{0.2395955612114145E+01, 0.2466725997761544E+01, 0.7077038564739886E-01, 0.5393549207373363E-01, 0.1879940024111348E+01, 0.1842303011581631E+01, 0.1629847954401656E-01, 0.1170659574821484E-01, 0.1323258193751120E+01, 0.1541901897857258E+01, 0.2303502998543534E+00, 0.2097365510789915E+00, 0.1403919102181368E+01, 0.1217882280188454E+01, 0.2369972908607684E-01, 0.1466422291592108E-01, 0.7535654596375009E+00, 0.7390296178356669E+00, 0.1283811140869274E-03, 0.2869551596920546E-03, 0.1651865724265327E+01, 0.1961735651572868E+01, 0.3101568824672333E+00, 0.3344283443049241E-01, 0.2115252128860193E+00, 0.2685204315989660E+00, 0.9043805314343908E-01, 0.1461369444993732E+00, 0.4338965075606848E+00, 0.3881981788190908E+00, 0.1004386157577793E+00, 0.1801598202532681E+00, 0.6963229589638840E+00, 0.6903565688953941E+00, 0.1741934301847783E+00, 0.4507175802324546E-01, 0.1786266232314551E+00, 0.2536031297618502E+00, 0.1200482645536405E+00, 0.1351822623162481E+00, 0.2855738476449996E+00, 0.1565951964743188E+00, 0.6203611145567293E-02, 0.1741812487831326E-01, 0.4396785394072695E+00, 0.4886511254041676E+00, 0.6639071087521134E-01, 0.1227425859208231E+00, 0.9034140763752896E+00, 0.8563793913827676E+00, 0.7570790092830114E-01, 0.1641466407918003E+00, 0.1856765259836430E+01, 0.1908333071910365E+01, 0.2157144528657351E+00, 0.9663383852973971E-01, 0.8548780458707995E+00, 0.9144777845246447E+00, 0.1562335771835849E+00, 0.1798735100772441E-01, 0.1052848766285511E+00, 0.2630781594343725E+00, 0.1757806338135458E+00, 0.5302070335887964E+00, 0.7935225143382671E+00, 0.2653394432718723E+00, 0.2023962522401480E-02, 0.7246963929058098E-03, 0.9500696544845319E-01, 0.1055769055236896E+00, 0.1129463646814219E-01, 0.3016696409481782E-02, 0.2819864744791555E-01, 0.2518195113629758E-01, 0.9786381594178541E-10, 0.2412835834031154E-11, 0.6208618935176568E-03, 0.6208618911048210E-03, 0.1700670967075909E-47, 0.2869551596920546E-03, 0.7811996215926567E-04, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			pp:       1,
			dmin:     6.2086189110482101E-004,
			desig:    -5.4012504887129591E-021,
			qmax:     1.8900334961849885,
			ttype:    -7,
			dmin1:    2.5181951038433764E-002,
			dmin2:    8.7297525620826724E-002,
			dn:       6.2086189110482101E-004,
			dn1:      2.5181951038433764E-002,
			dn2:      9.4282269055547374E-002,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    0,
			iter:     9,
			sigma:    7.8119962159265674E-005,
			nDiv:     193,
			zOut:     []float64{0.2520040627944239E+01, 0.2466725997761544E+01, 0.3943012599746679E-01, 0.5393549207373363E-01, 0.1813958619441340E+01, 0.1842303011581631E+01, 0.9950845630193767E-02, 0.1170659574821484E-01, 0.1741066741415017E+01, 0.1541901897857258E+01, 0.1467114516582776E+00, 0.2097365510789915E+00, 0.1085214189555058E+01, 0.1217882280188454E+01, 0.9986318978978259E-02, 0.1466422291592108E-01, 0.7287093921253419E+00, 0.7390296178356669E+00, 0.7725029665513934E-03, 0.2869551596920546E-03, 0.1993785121145770E+01, 0.1961735651572868E+01, 0.4504038193447841E-02, 0.3344283443049241E-01, 0.4095324760138526E+00, 0.2685204315989660E+00, 0.1385240464077977E+00, 0.1461369444993732E+00, 0.4292130907735224E+00, 0.3881981788190908E+00, 0.2897733504323247E+00, 0.1801598202532681E+00, 0.4450341145952761E+00, 0.6903565688953941E+00, 0.2568418582687495E-01, 0.4507175802324546E-01, 0.3624803443601846E+00, 0.2536031297618502E+00, 0.5840011260368079E-01, 0.1351822623162481E+00, 0.1149923468579125E+00, 0.1565951964743188E+00, 0.7401698075381481E-01, 0.1741812487831326E-01, 0.5367558686801371E+00, 0.4886511254041676E+00, 0.1958324578473516E+00, 0.1227425859208231E+00, 0.8240727124361776E+00, 0.8563793913827676E+00, 0.3801199318200257E+00, 0.1641466407918003E+00, 0.1624226116729040E+01, 0.1908333071910365E+01, 0.5440714051978934E-01, 0.9663383852973971E-01, 0.8774371331215411E+00, 0.9144777845246447E+00, 0.5393069221241477E-02, 0.1798735100772441E-01, 0.7872712619108886E+00, 0.2630781594343725E+00, 0.1786993199393658E+00, 0.5302070335887964E+00, 0.8674395783437358E-01, 0.2653394432718723E+00, 0.8820349511059155E-03, 0.7246963929058098E-03, 0.1070907050910267E+00, 0.1055769055236896E+00, 0.7093640994523618E-03, 0.3016696409481782E-02, 0.2385172514821930E-01, 0.2518195113629758E-01, 0.6280626703238275E-13, 0.2412835834031154E-11, 0.3257377007015450E-14, 0.6208618911048210E-03, 0.7725029665513934E-03, 0.2869551596920546E-03, 0.7811996215926567E-04, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			i0Out:    1,
			n0Out:    20,
			ppOut:    1,
			dminOut:  3.2573770070154495E-015,
			desigOut: 2.1703803823424652E-020,
			sigmaOut: 6.9898185319802297E-004,
			qmaxOut:  1.8900334961849885,
			nFailOut: 0,
			iterOut:  10,
			nDivOut:  214,
			ttypeOut: -2,
			dmin1Out: 2.3851725145806461E-002,
			dmin2Out: 8.6019261441467765E-002,
			dnOut:    3.2573770070154495E-015,
			dn1Out:   2.3851725145806461E-002,
			dn2Out:   0.10407400868154487,
			gOut:     0.0000000000000000,
			tauOut:   6.2086189103875732E-004,
		},
		{
			i0:       1,
			n0:       20,
			z:        []float64{0.2520040627944239E+01, 0.2466725997761544E+01, 0.3943012599746679E-01, 0.5393549207373363E-01, 0.1813958619441340E+01, 0.1842303011581631E+01, 0.9950845630193767E-02, 0.1170659574821484E-01, 0.1741066741415017E+01, 0.1541901897857258E+01, 0.1467114516582776E+00, 0.2097365510789915E+00, 0.1085214189555058E+01, 0.1217882280188454E+01, 0.9986318978978259E-02, 0.1466422291592108E-01, 0.7287093921253419E+00, 0.7390296178356669E+00, 0.7725029665513934E-03, 0.2869551596920546E-03, 0.1993785121145770E+01, 0.1961735651572868E+01, 0.4504038193447841E-02, 0.3344283443049241E-01, 0.4095324760138526E+00, 0.2685204315989660E+00, 0.1385240464077977E+00, 0.1461369444993732E+00, 0.4292130907735224E+00, 0.3881981788190908E+00, 0.2897733504323247E+00, 0.1801598202532681E+00, 0.4450341145952761E+00, 0.6903565688953941E+00, 0.2568418582687495E-01, 0.4507175802324546E-01, 0.3624803443601846E+00, 0.2536031297618502E+00, 0.5840011260368079E-01, 0.1351822623162481E+00, 0.1149923468579125E+00, 0.1565951964743188E+00, 0.7401698075381481E-01, 0.1741812487831326E-01, 0.5367558686801371E+00, 0.4886511254041676E+00, 0.1958324578473516E+00, 0.1227425859208231E+00, 0.8240727124361776E+00, 0.8563793913827676E+00, 0.3801199318200257E+00, 0.1641466407918003E+00, 0.1624226116729040E+01, 0.1908333071910365E+01, 0.5440714051978934E-01, 0.9663383852973971E-01, 0.8774371331215411E+00, 0.9144777845246447E+00, 0.5393069221241477E-02, 0.1798735100772441E-01, 0.7872712619108886E+00, 0.2630781594343725E+00, 0.1786993199393658E+00, 0.5302070335887964E+00, 0.8674395783437358E-01, 0.2653394432718723E+00, 0.8820349511059155E-03, 0.7246963929058098E-03, 0.1070907050910267E+00, 0.1055769055236896E+00, 0.7093640994523618E-03, 0.3016696409481782E-02, 0.2385172514821930E-01, 0.2518195113629758E-01, 0.6280626703238275E-13, 0.2412835834031154E-11, 0.3257377007015450E-14, 0.6208618911048210E-03, 0.7725029665513934E-03, 0.2869551596920546E-03, 0.7811996215926567E-04, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			pp:       0,
			dmin:     3.2573770070154495E-015,
			desig:    2.1703803823424652E-020,
			qmax:     1.8900334961849885,
			ttype:    -2,
			dmin1:    2.3851725145806461E-002,
			dmin2:    8.6019261441467765E-002,
			dn:       3.2573770070154495E-015,
			dn1:      2.3851725145806461E-002,
			dn2:      0.10407400868154487,
			g:        0.0000000000000000,
			tau:      6.2086189103875732E-004,
			nFail:    0,
			iter:     10,
			sigma:    6.9898185319802297E-004,
			nDiv:     214,
			zOut:     []float64{0.2520040627944239E+01, 0.2559470753941703E+01, 0.3943012599746679E-01, 0.2794508075882943E-01, 0.1813958619441340E+01, 0.1795964384312701E+01, 0.9950845630193767E-02, 0.9646675917972324E-02, 0.1741066741415017E+01, 0.1878131517155319E+01, 0.1467114516582776E+00, 0.8477220453173254E-01, 0.1085214189555058E+01, 0.1010428304002300E+01, 0.9986318978978259E-02, 0.7202019583097946E-02, 0.7287093921253419E+00, 0.7222798755087921E+00, 0.7725029665513934E-03, 0.2132421202606784E-02, 0.1993785121145770E+01, 0.1996156738136608E+01, 0.4504038193447841E-02, 0.9240506410060376E-03, 0.4095324760138526E+00, 0.5471324717806409E+00, 0.1385240464077977E+00, 0.1086689918286978E+00, 0.4292130907735224E+00, 0.6103174493771462E+00, 0.2897733504323247E+00, 0.2112982785836522E+00, 0.4450341145952761E+00, 0.2594200218384955E+00, 0.2568418582687495E-01, 0.3588779484774172E-01, 0.3624803443601846E+00, 0.3849926621161204E+00, 0.5840011260368079E-01, 0.1744336104525046E-01, 0.1149923468579125E+00, 0.1715659665664736E+00, 0.7401698075381481E-01, 0.2315671901408358E+00, 0.5367558686801371E+00, 0.5010211363866497E+00, 0.1958324578473516E+00, 0.3221025481782661E+00, 0.8240727124361776E+00, 0.8820900960779340E+00, 0.3801199318200257E+00, 0.6999293195746292E+00, 0.1624226116729040E+01, 0.9787039376741966E+00, 0.5440714051978934E-01, 0.4877761656142095E-01, 0.8774371331215411E+00, 0.8340525857813585E+00, 0.5393069221241477E-02, 0.5090576402208483E-02, 0.7872712619108886E+00, 0.9608800054480426E+00, 0.1786993199393658E+00, 0.1613217694817542E-01, 0.8674395783437358E-01, 0.7149381583730083E-01, 0.8820349511059155E-03, 0.1321201613351008E-02, 0.1070907050910267E+00, 0.1064788675771248E+00, 0.7093640994523618E-03, 0.1589006148839502E-03, 0.2385172514821930E-01, 0.2369282453339490E-01, 0.6280626703238275E-13, 0.8634837515442557E-26, 0.3257377007015450E-14, 0.9466330862652142E-28, 0.7725029665513934E-03, 0.9240506410060376E-03, 0.7811996215926567E-04, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			i0Out:    1,
			n0Out:    20,
			ppOut:    0,
			dminOut:  9.4663308626521417E-029,
			desigOut: 2.1703795093889875E-020,
			sigmaOut: 6.9898185320128035E-004,
			qmaxOut:  1.8900334961849885,
			nFailOut: 0,
			iterOut:  11,
			nDivOut:  235,
			ttypeOut: -2,
			dmin1Out: 2.3692824533332088E-002,
			dmin2Out: 7.0611780886194908E-002,
			dnOut:    9.4663308626521417E-029,
			dn1Out:   2.3692824533332088E-002,
			dn2Out:   0.10576950347767239,
			gOut:     0.0000000000000000,
			tauOut:   3.2573770070067200E-015,
		},
		{
			i0:       1,
			n0:       20,
			z:        []float64{0.2520040627944239E+01, 0.2559470753941703E+01, 0.3943012599746679E-01, 0.2794508075882943E-01, 0.1813958619441340E+01, 0.1795964384312701E+01, 0.9950845630193767E-02, 0.9646675917972324E-02, 0.1741066741415017E+01, 0.1878131517155319E+01, 0.1467114516582776E+00, 0.8477220453173254E-01, 0.1085214189555058E+01, 0.1010428304002300E+01, 0.9986318978978259E-02, 0.7202019583097946E-02, 0.7287093921253419E+00, 0.7222798755087921E+00, 0.7725029665513934E-03, 0.2132421202606784E-02, 0.1993785121145770E+01, 0.1996156738136608E+01, 0.4504038193447841E-02, 0.9240506410060376E-03, 0.4095324760138526E+00, 0.5471324717806409E+00, 0.1385240464077977E+00, 0.1086689918286978E+00, 0.4292130907735224E+00, 0.6103174493771462E+00, 0.2897733504323247E+00, 0.2112982785836522E+00, 0.4450341145952761E+00, 0.2594200218384955E+00, 0.2568418582687495E-01, 0.3588779484774172E-01, 0.3624803443601846E+00, 0.3849926621161204E+00, 0.5840011260368079E-01, 0.1744336104525046E-01, 0.1149923468579125E+00, 0.1715659665664736E+00, 0.7401698075381481E-01, 0.2315671901408358E+00, 0.5367558686801371E+00, 0.5010211363866497E+00, 0.1958324578473516E+00, 0.3221025481782661E+00, 0.8240727124361776E+00, 0.8820900960779340E+00, 0.3801199318200257E+00, 0.6999293195746292E+00, 0.1624226116729040E+01, 0.9787039376741966E+00, 0.5440714051978934E-01, 0.4877761656142095E-01, 0.8774371331215411E+00, 0.8340525857813585E+00, 0.5393069221241477E-02, 0.5090576402208483E-02, 0.7872712619108886E+00, 0.9608800054480426E+00, 0.1786993199393658E+00, 0.1613217694817542E-01, 0.8674395783437358E-01, 0.7149381583730083E-01, 0.8820349511059155E-03, 0.1321201613351008E-02, 0.1070907050910267E+00, 0.1064788675771248E+00, 0.7093640994523618E-03, 0.1589006148839502E-03, 0.2385172514821930E-01, 0.2369282453339490E-01, 0.6280626703238275E-13, 0.8634837515442557E-26, 0.3257377007015450E-14, 0.9466330862652142E-28, 0.7725029665513934E-03, 0.9240506410060376E-03, 0.7811996215926567E-04, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			pp:       1,
			dmin:     9.4663308626521417E-029,
			desig:    2.1703795093889875E-020,
			qmax:     1.8900334961849885,
			ttype:    -2,
			dmin1:    2.3692824533332088E-002,
			dmin2:    7.0611780886194908E-002,
			dn:       9.4663308626521417E-029,
			dn1:      2.3692824533332088E-002,
			dn2:      0.10576950347767239,
			g:        0.0000000000000000,
			tau:      3.2573770070067200E-015,
			nFail:    0,
			iter:     11,
			sigma:    6.9898185320128035E-004,
			nDiv:     235,
			zOut:     []float64{0.2587415834700532E+01, 0.2559470753941703E+01, 0.1939710234687058E-01, 0.2794508075882943E-01, 0.1786213957883803E+01, 0.1795964384312701E+01, 0.1014308840067055E-01, 0.9646675917972324E-02, 0.1952760633286381E+01, 0.1878131517155319E+01, 0.4386417535844126E-01, 0.8477220453173254E-01, 0.9737661482269571E+00, 0.1010428304002300E+01, 0.5342015449359675E-02, 0.7202019583097946E-02, 0.7190702812620393E+00, 0.7222798755087921E+00, 0.5919653562455767E-02, 0.2132421202606784E-02, 0.1991161135215158E+01, 0.1996156738136608E+01, 0.2539111990097616E-03, 0.9240506410060376E-03, 0.6555475524103290E+00, 0.5471324717806409E+00, 0.1011712753337584E+00, 0.1086689918286978E+00, 0.7204444526270400E+00, 0.6103174493771462E+00, 0.7608498315828412E-01, 0.2112982785836522E+00, 0.2192228335279531E+00, 0.2594200218384955E+00, 0.6302508481237896E-01, 0.3588779484774172E-01, 0.3394109383489919E+00, 0.3849926621161204E+00, 0.8817297145618809E-02, 0.1744336104525046E-01, 0.3943158595616905E+00, 0.1715659665664736E+00, 0.2942312715577539E+00, 0.2315671901408358E+00, 0.5288924130071619E+00, 0.5010211363866497E+00, 0.5372046576619481E+00, 0.3221025481782661E+00, 0.1044814757990615E+01, 0.8820900960779340E+00, 0.6556411803358774E+00, 0.6999293195746292E+00, 0.3718403738997403E+00, 0.9787039376741966E+00, 0.1094101127175453E+00, 0.4877761656142095E-01, 0.7297330494660218E+00, 0.8340525857813585E+00, 0.6703044469024726E-02, 0.5090576402208483E-02, 0.9703091379271934E+00, 0.9608800054480426E+00, 0.1188642714683105E-02, 0.1613217694817542E-01, 0.7162637473596872E-01, 0.7149381583730083E-01, 0.1964081696850700E-02, 0.1321201613351008E-02, 0.1046736864951580E+00, 0.1064788675771248E+00, 0.3596705640885345E-04, 0.1589006148839502E-03, 0.2365685747698604E-01, 0.2369282453339490E-01, 0.3455244592226135E-52, 0.8634837515442557E-26, 0.9466330862652142E-28, 0.9466330862652142E-28, 0.2539111990097616E-03, 0.9240506410060376E-03, 0.7811996215926567E-04, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			i0Out:    1,
			n0Out:    20,
			ppOut:    1,
			dminOut:  9.4663308626521417E-029,
			desigOut: 2.1703795093889875E-020,
			sigmaOut: 6.9898185320128035E-004,
			qmaxOut:  1.8900334961849885,
			nFailOut: 0,
			iterOut:  12,
			nDivOut:  256,
			ttypeOut: -2,
			dmin1Out: 2.3656857476986041E-002,
			dmin2Out: 7.0305173122617720E-002,
			dnOut:    9.4663308626521417E-029,
			dn1Out:   2.3656857476986041E-002,
			dn2Out:   0.10451478588027406,
			gOut:     0.0000000000000000,
			tauOut:   0.0000000000000000,
		},
		{
			i0:       1,
			n0:       20,
			z:        []float64{0.2587415834700532E+01, 0.2559470753941703E+01, 0.1939710234687058E-01, 0.2794508075882943E-01, 0.1786213957883803E+01, 0.1795964384312701E+01, 0.1014308840067055E-01, 0.9646675917972324E-02, 0.1952760633286381E+01, 0.1878131517155319E+01, 0.4386417535844126E-01, 0.8477220453173254E-01, 0.9737661482269571E+00, 0.1010428304002300E+01, 0.5342015449359675E-02, 0.7202019583097946E-02, 0.7190702812620393E+00, 0.7222798755087921E+00, 0.5919653562455767E-02, 0.2132421202606784E-02, 0.1991161135215158E+01, 0.1996156738136608E+01, 0.2539111990097616E-03, 0.9240506410060376E-03, 0.6555475524103290E+00, 0.5471324717806409E+00, 0.1011712753337584E+00, 0.1086689918286978E+00, 0.7204444526270400E+00, 0.6103174493771462E+00, 0.7608498315828412E-01, 0.2112982785836522E+00, 0.2192228335279531E+00, 0.2594200218384955E+00, 0.6302508481237896E-01, 0.3588779484774172E-01, 0.3394109383489919E+00, 0.3849926621161204E+00, 0.8817297145618809E-02, 0.1744336104525046E-01, 0.3943158595616905E+00, 0.1715659665664736E+00, 0.2942312715577539E+00, 0.2315671901408358E+00, 0.5288924130071619E+00, 0.5010211363866497E+00, 0.5372046576619481E+00, 0.3221025481782661E+00, 0.1044814757990615E+01, 0.8820900960779340E+00, 0.6556411803358774E+00, 0.6999293195746292E+00, 0.3718403738997403E+00, 0.9787039376741966E+00, 0.1094101127175453E+00, 0.4877761656142095E-01, 0.7297330494660218E+00, 0.8340525857813585E+00, 0.6703044469024726E-02, 0.5090576402208483E-02, 0.9703091379271934E+00, 0.9608800054480426E+00, 0.1188642714683105E-02, 0.1613217694817542E-01, 0.7162637473596872E-01, 0.7149381583730083E-01, 0.1964081696850700E-02, 0.1321201613351008E-02, 0.1046736864951580E+00, 0.1064788675771248E+00, 0.3596705640885345E-04, 0.1589006148839502E-03, 0.2365685747698604E-01, 0.2369282453339490E-01, 0.3455244592226135E-52, 0.8634837515442557E-26, 0.9466330862652142E-28, 0.9466330862652142E-28, 0.2539111990097616E-03, 0.9240506410060376E-03, 0.7811996215926567E-04, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			pp:       0,
			dmin:     9.4663308626521417E-029,
			desig:    2.1703795093889875E-020,
			qmax:     1.8900334961849885,
			ttype:    -2,
			dmin1:    2.3656857476986041E-002,
			dmin2:    7.0305173122617720E-002,
			dn:       9.4663308626521417E-029,
			dn1:      2.3656857476986041E-002,
			dn2:      0.10451478588027406,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    0,
			iter:     12,
			sigma:    6.9898185320128035E-004,
			nDiv:     256,
			zOut:     []float64{0.2587415834700532E+01, 0.2594984508308910E+01, 0.1939710234687058E-01, 0.1335166928493912E-01, 0.1786213957883803E+01, 0.1771176948261041E+01, 0.1014308840067055E-01, 0.1118297285215907E-01, 0.1952760633286381E+01, 0.1973613407054170E+01, 0.4386417535844126E-01, 0.2164225725832272E-01, 0.9737661482269571E+00, 0.9456374776795010E+00, 0.5342015449359675E-02, 0.4062111160297227E-02, 0.7190702812620393E+00, 0.7090993949257048E+00, 0.5919653562455767E-02, 0.1662247097070905E-01, 0.1991161135215158E+01, 0.1962964146704966E+01, 0.2539111990097616E-03, 0.8479567256479229E-04, 0.6555475524103290E+00, 0.7448056033330296E+00, 0.1011712753337584E+00, 0.9786215859981671E-01, 0.7204444526270400E+00, 0.6868388484470145E+00, 0.7608498315828412E-01, 0.2428454015756268E-01, 0.2192228335279531E+00, 0.2461349494442763E+00, 0.6302508481237896E-01, 0.8690924724014965E-01, 0.3394109383489919E+00, 0.2494905595159681E+00, 0.8817297145618809E-02, 0.1393559784278329E-01, 0.3943158595616905E+00, 0.6627831045381681E+00, 0.2942312715577539E+00, 0.2347927793131975E+00, 0.5288924130071619E+00, 0.8194758626174196E+00, 0.5372046576619481E+00, 0.6849248159595133E+00, 0.1044814757990615E+01, 0.1003702693628486E+01, 0.6556411803358774E+00, 0.2428944977310168E+00, 0.3718403738997403E+00, 0.2265275601477758E+00, 0.1094101127175453E+00, 0.3524523689025369E+00, 0.7297330494660218E+00, 0.3721552962940165E+00, 0.6703044469024726E-02, 0.1747664312451059E-01, 0.9703091379271934E+00, 0.9421927087788731E+00, 0.1188642714683105E-02, 0.9036173567869616E-04, 0.7162637473596872E-01, 0.6167166595864771E-01, 0.1964081696850700E-02, 0.3333583884775863E-02, 0.1046736864951580E+00, 0.8954764092829798E-01, 0.3596705640885345E-04, 0.9501841907954501E-05, 0.2365685747698604E-01, 0.1181892689658507E-01, 0.3455244592226135E-52, 0.8479567256479229E-04, 0.6989818532012803E-03, 0.9466330862652142E-28, 0.2539111990097616E-03, 0.9240506410060376E-03, 0.7811996215926567E-04, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			i0Out:    1,
			n0Out:    19,
			ppOut:    0,
			dminOut:  1.1818926896585069E-002,
			desigOut: 3.4696444683954120E-019,
			sigmaOut: 1.2527410591694300E-002,
			qmaxOut:  1.8900334961849885,
			nFailOut: 0,
			iterOut:  13,
			nDivOut:  276,
			ttypeOut: -9,
			dmin1Out: 5.9707584261797009E-002,
			dmin2Out: 5.9707584261797009E-002,
			dnOut:    1.1818926896585069E-002,
			dn1Out:   8.9511673871889130E-002,
			dn2Out:   5.9707584261797009E-002,
			gOut:     0.0000000000000000,
			tauOut:   1.1828428738493020E-002,
		},
		{
			i0:       1,
			n0:       19,
			z:        []float64{0.2587415834700532E+01, 0.2594984508308910E+01, 0.1939710234687058E-01, 0.1335166928493912E-01, 0.1786213957883803E+01, 0.1771176948261041E+01, 0.1014308840067055E-01, 0.1118297285215907E-01, 0.1952760633286381E+01, 0.1973613407054170E+01, 0.4386417535844126E-01, 0.2164225725832272E-01, 0.9737661482269571E+00, 0.9456374776795010E+00, 0.5342015449359675E-02, 0.4062111160297227E-02, 0.7190702812620393E+00, 0.7090993949257048E+00, 0.5919653562455767E-02, 0.1662247097070905E-01, 0.1991161135215158E+01, 0.1962964146704966E+01, 0.2539111990097616E-03, 0.8479567256479229E-04, 0.6555475524103290E+00, 0.7448056033330296E+00, 0.1011712753337584E+00, 0.9786215859981671E-01, 0.7204444526270400E+00, 0.6868388484470145E+00, 0.7608498315828412E-01, 0.2428454015756268E-01, 0.2192228335279531E+00, 0.2461349494442763E+00, 0.6302508481237896E-01, 0.8690924724014965E-01, 0.3394109383489919E+00, 0.2494905595159681E+00, 0.8817297145618809E-02, 0.1393559784278329E-01, 0.3943158595616905E+00, 0.6627831045381681E+00, 0.2942312715577539E+00, 0.2347927793131975E+00, 0.5288924130071619E+00, 0.8194758626174196E+00, 0.5372046576619481E+00, 0.6849248159595133E+00, 0.1044814757990615E+01, 0.1003702693628486E+01, 0.6556411803358774E+00, 0.2428944977310168E+00, 0.3718403738997403E+00, 0.2265275601477758E+00, 0.1094101127175453E+00, 0.3524523689025369E+00, 0.7297330494660218E+00, 0.3721552962940165E+00, 0.6703044469024726E-02, 0.1747664312451059E-01, 0.9703091379271934E+00, 0.9421927087788731E+00, 0.1188642714683105E-02, 0.9036173567869616E-04, 0.7162637473596872E-01, 0.6167166595864771E-01, 0.1964081696850700E-02, 0.3333583884775863E-02, 0.1046736864951580E+00, 0.8954764092829798E-01, 0.3596705640885345E-04, 0.9501841907954501E-05, 0.2365685747698604E-01, 0.1181892689658507E-01, 0.3455244592226135E-52, 0.8479567256479229E-04, 0.6989818532012803E-03, 0.9466330862652142E-28, 0.2539111990097616E-03, 0.9240506410060376E-03, 0.7811996215926567E-04, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			pp:       1,
			dmin:     1.1818926896585069E-002,
			desig:    3.4696444683954120E-019,
			qmax:     1.8900334961849885,
			ttype:    -9,
			dmin1:    5.9707584261797009E-002,
			dmin2:    5.9707584261797009E-002,
			dn:       1.1818926896585069E-002,
			dn1:      8.9511673871889130E-002,
			dn2:      5.9707584261797009E-002,
			g:        0.0000000000000000,
			tau:      1.1828428738493020E-002,
			nFail:    0,
			iter:     13,
			sigma:    1.2527410591694300E-002,
			nDiv:     276,
			zOut:     []float64{0.2596646703688871E+01, 0.2594984508308910E+01, 0.9107195378059658E-02, 0.1335166928493912E-01, 0.1761563251830163E+01, 0.1771176948261041E+01, 0.1252913577120413E-01, 0.1118297285215907E-01, 0.1971037054636311E+01, 0.1973613407054170E+01, 0.1038322923301279E-01, 0.2164225725832272E-01, 0.9276268857018078E+00, 0.9456374776795010E+00, 0.3105171497598932E-02, 0.4062111160297227E-02, 0.7109272204938374E+00, 0.7090993949257048E+00, 0.4589684232723624E-01, 0.1662247097070905E-01, 0.1905462626145317E+01, 0.1962964146704966E+01, 0.3314486004504485E-04, 0.8479567256479229E-04, 0.8309451431678238E+00, 0.7448056033330296E+00, 0.8089045693556918E-01, 0.9786215859981671E-01, 0.6185434577640304E+00, 0.6868388484470145E+00, 0.9663466631053550E-02, 0.2428454015756268E-01, 0.3116912561483949E+00, 0.2461349494442763E+00, 0.6956575230565126E-01, 0.8690924724014965E-01, 0.1821709311481225E+00, 0.2494905595159681E+00, 0.5070116699532772E-01, 0.1393559784278329E-01, 0.8351852429510604E+00, 0.6627831045381681E+00, 0.2303764547900405E+00, 0.2347927793131975E+00, 0.1262334749881915E+01, 0.8194758626174196E+00, 0.5445947541061251E+00, 0.6849248159595133E+00, 0.6903129633483998E+00, 0.1003702693628486E+01, 0.7970630839299049E-01, 0.2428944977310168E+00, 0.4875841467523446E+00, 0.2265275601477758E+00, 0.2690141110044630E+00, 0.3524523689025369E+00, 0.1089283545090865E+00, 0.3721552962940165E+00, 0.1511669372043137E+00, 0.1747664312451059E-01, 0.7794266594052606E+00, 0.9421927087788731E+00, 0.7149818024536710E-05, 0.9036173567869616E-04, 0.5330862612042146E-01, 0.6167166595864771E-01, 0.5599742376476567E-02, 0.3333583884775863E-02, 0.7226792648875177E-01, 0.8954764092829798E-01, 0.1553961492315709E-05, 0.9501841907954501E-05, 0.1278990301151681E-03, 0.1181892689658507E-01, 0.7149818024536710E-05, 0.8479567256479229E-04, 0.6989818532012803E-03, 0.9466330862652142E-28, 0.2539111990097616E-03, 0.9240506410060376E-03, 0.7811996215926567E-04, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			i0Out:    1,
			n0Out:    19,
			ppOut:    1,
			dminOut:  1.2789903011516807E-004,
			desigOut: 0.0000000000000000,
			sigmaOut: 2.4216884496671885E-002,
			qmaxOut:  1.8900334961849885,
			nFailOut: 0,
			iterOut:  14,
			nDivOut:  296,
			ttypeOut: -4,
			dmin1Out: 4.9975042235645591E-002,
			dmin2Out: 4.9975042235645591E-002,
			dnOut:    1.2789903011516807E-004,
			dn1Out:   7.2258424646843816E-002,
			dn2Out:   4.9975042235645591E-002,
			gOut:     0.0000000000000000,
			tauOut:   1.1689473904977585E-002,
		},
		{
			i0:       1,
			n0:       19,
			z:        []float64{0.2596646703688871E+01, 0.2594984508308910E+01, 0.9107195378059658E-02, 0.1335166928493912E-01, 0.1761563251830163E+01, 0.1771176948261041E+01, 0.1252913577120413E-01, 0.1118297285215907E-01, 0.1971037054636311E+01, 0.1973613407054170E+01, 0.1038322923301279E-01, 0.2164225725832272E-01, 0.9276268857018078E+00, 0.9456374776795010E+00, 0.3105171497598932E-02, 0.4062111160297227E-02, 0.7109272204938374E+00, 0.7090993949257048E+00, 0.4589684232723624E-01, 0.1662247097070905E-01, 0.1905462626145317E+01, 0.1962964146704966E+01, 0.3314486004504485E-04, 0.8479567256479229E-04, 0.8309451431678238E+00, 0.7448056033330296E+00, 0.8089045693556918E-01, 0.9786215859981671E-01, 0.6185434577640304E+00, 0.6868388484470145E+00, 0.9663466631053550E-02, 0.2428454015756268E-01, 0.3116912561483949E+00, 0.2461349494442763E+00, 0.6956575230565126E-01, 0.8690924724014965E-01, 0.1821709311481225E+00, 0.2494905595159681E+00, 0.5070116699532772E-01, 0.1393559784278329E-01, 0.8351852429510604E+00, 0.6627831045381681E+00, 0.2303764547900405E+00, 0.2347927793131975E+00, 0.1262334749881915E+01, 0.8194758626174196E+00, 0.5445947541061251E+00, 0.6849248159595133E+00, 0.6903129633483998E+00, 0.1003702693628486E+01, 0.7970630839299049E-01, 0.2428944977310168E+00, 0.4875841467523446E+00, 0.2265275601477758E+00, 0.2690141110044630E+00, 0.3524523689025369E+00, 0.1089283545090865E+00, 0.3721552962940165E+00, 0.1511669372043137E+00, 0.1747664312451059E-01, 0.7794266594052606E+00, 0.9421927087788731E+00, 0.7149818024536710E-05, 0.9036173567869616E-04, 0.5330862612042146E-01, 0.6167166595864771E-01, 0.5599742376476567E-02, 0.3333583884775863E-02, 0.7226792648875177E-01, 0.8954764092829798E-01, 0.1553961492315709E-05, 0.9501841907954501E-05, 0.1278990301151681E-03, 0.1181892689658507E-01, 0.7149818024536710E-05, 0.8479567256479229E-04, 0.6989818532012803E-03, 0.9466330862652142E-28, 0.2539111990097616E-03, 0.9240506410060376E-03, 0.7811996215926567E-04, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			pp:       0,
			dmin:     1.2789903011516807E-004,
			desig:    0.0000000000000000,
			qmax:     1.8900334961849885,
			ttype:    -4,
			dmin1:    4.9975042235645591E-002,
			dmin2:    4.9975042235645591E-002,
			dn:       1.2789903011516807E-004,
			dn1:      7.2258424646843816E-002,
			dn2:      4.9975042235645591E-002,
			g:        0.0000000000000000,
			tau:      1.1689473904977585E-002,
			nFail:    0,
			iter:     14,
			sigma:    2.4216884496671885E-002,
			nDiv:     296,
			zOut:     []float64{0.2596646703688871E+01, 0.2605626003858251E+01, 0.9107195378059658E-02, 0.6157023564192275E-02, 0.1761563251830163E+01, 0.1767807468828494E+01, 0.1252913577120413E-01, 0.1396950250695455E-01, 0.1971037054636311E+01, 0.1967322886153689E+01, 0.1038322923301279E-01, 0.4895872794820515E-02, 0.9276268857018078E+00, 0.9257082891959054E+00, 0.3105171497598932E-02, 0.2384715539127593E-02, 0.7109272204938374E+00, 0.7543114520732653E+00, 0.4589684232723624E-01, 0.1159397984377132E+00, 0.1905462626145317E+01, 0.1789428077358968E+01, 0.3314486004504485E-04, 0.1539126429493388E-04, 0.8309451431678238E+00, 0.9116923136304173E+00, 0.8089045693556918E-01, 0.5488064578914756E-01, 0.6185434577640304E+00, 0.5731983833972557E+00, 0.9663466631053550E-02, 0.5254756712901782E-02, 0.3116912561483949E+00, 0.3758743565324636E+00, 0.6956575230565126E-01, 0.3371567560620648E-01, 0.1821709311481225E+00, 0.1990285273285630E+00, 0.5070116699532772E-01, 0.2127577741907859E+00, 0.8351852429510604E+00, 0.8526760283416343E+00, 0.2303764547900405E+00, 0.3410582621885915E+00, 0.1262334749881915E+01, 0.1465743346590768E+01, 0.5445947541061251E+00, 0.2564847518533230E+00, 0.6903129633483998E+00, 0.5134066246793865E+00, 0.7970630839299049E-01, 0.7569737222001199E-01, 0.4875841467523446E+00, 0.6807729903281149E+00, 0.2690141110044630E+00, 0.4304410555024735E-01, 0.1089283545090865E+00, 0.2169232909544721E+00, 0.1511669372043137E+00, 0.5431576312495270E+00, 0.7794266594052606E+00, 0.2361482827650774E+00, 0.7149818024536710E-05, 0.1614015445872399E-05, 0.5330862612042146E-01, 0.5877885927277143E-01, 0.5599742376476567E-02, 0.6884818375619963E-02, 0.7226792648875177E-01, 0.6525676686594341E-01, 0.1553961492315709E-05, 0.3045663725752605E-08, 0.1278990301151681E-03, 0.7757707209639971E-09, 0.7149818024536710E-05, 0.1614015445872399E-05, 0.6989818532012803E-03, 0.9466330862652142E-28, 0.2539111990097616E-03, 0.9240506410060376E-03, 0.7811996215926567E-04, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			i0Out:    1,
			n0Out:    19,
			ppOut:    0,
			dminOut:  7.7577072096399712E-010,
			desigOut: -5.4210108624275222E-020,
			sigmaOut: 2.4344779705352607E-002,
			qmaxOut:  1.8900334961849885,
			nFailOut: 1,
			iterOut:  16,
			nDivOut:  336,
			ttypeOut: -15,
			dmin1Out: 5.3179116896294863E-002,
			dmin2Out: 5.3179116896294863E-002,
			dnOut:    7.7577072096399712E-010,
			dn1Out:   6.5255212904451090E-002,
			dn2Out:   5.3179116896294863E-002,
			gOut:     0.0000000000000000,
			tauOut:   1.2789520868072135E-004,
		},
		{
			i0:       1,
			n0:       19,
			z:        []float64{0.2596646703688871E+01, 0.2605626003858251E+01, 0.9107195378059658E-02, 0.6157023564192275E-02, 0.1761563251830163E+01, 0.1767807468828494E+01, 0.1252913577120413E-01, 0.1396950250695455E-01, 0.1971037054636311E+01, 0.1967322886153689E+01, 0.1038322923301279E-01, 0.4895872794820515E-02, 0.9276268857018078E+00, 0.9257082891959054E+00, 0.3105171497598932E-02, 0.2384715539127593E-02, 0.7109272204938374E+00, 0.7543114520732653E+00, 0.4589684232723624E-01, 0.1159397984377132E+00, 0.1905462626145317E+01, 0.1789428077358968E+01, 0.3314486004504485E-04, 0.1539126429493388E-04, 0.8309451431678238E+00, 0.9116923136304173E+00, 0.8089045693556918E-01, 0.5488064578914756E-01, 0.6185434577640304E+00, 0.5731983833972557E+00, 0.9663466631053550E-02, 0.5254756712901782E-02, 0.3116912561483949E+00, 0.3758743565324636E+00, 0.6956575230565126E-01, 0.3371567560620648E-01, 0.1821709311481225E+00, 0.1990285273285630E+00, 0.5070116699532772E-01, 0.2127577741907859E+00, 0.8351852429510604E+00, 0.8526760283416343E+00, 0.2303764547900405E+00, 0.3410582621885915E+00, 0.1262334749881915E+01, 0.1465743346590768E+01, 0.5445947541061251E+00, 0.2564847518533230E+00, 0.6903129633483998E+00, 0.5134066246793865E+00, 0.7970630839299049E-01, 0.7569737222001199E-01, 0.4875841467523446E+00, 0.6807729903281149E+00, 0.2690141110044630E+00, 0.4304410555024735E-01, 0.1089283545090865E+00, 0.2169232909544721E+00, 0.1511669372043137E+00, 0.5431576312495270E+00, 0.7794266594052606E+00, 0.2361482827650774E+00, 0.7149818024536710E-05, 0.1614015445872399E-05, 0.5330862612042146E-01, 0.5877885927277143E-01, 0.5599742376476567E-02, 0.6884818375619963E-02, 0.7226792648875177E-01, 0.6525676686594341E-01, 0.1553961492315709E-05, 0.3045663725752605E-08, 0.1278990301151681E-03, 0.7757707209639971E-09, 0.7149818024536710E-05, 0.1614015445872399E-05, 0.6989818532012803E-03, 0.9466330862652142E-28, 0.2539111990097616E-03, 0.9240506410060376E-03, 0.7811996215926567E-04, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			pp:       1,
			dmin:     7.7577072096399712E-010,
			desig:    -5.4210108624275222E-020,
			qmax:     1.8900334961849885,
			ttype:    -15,
			dmin1:    5.3179116896294863E-002,
			dmin2:    5.3179116896294863E-002,
			dn:       7.7577072096399712E-010,
			dn1:      6.5255212904451090E-002,
			dn2:      5.3179116896294863E-002,
			g:        0.0000000000000000,
			tau:      1.2789520868072135E-004,
			nFail:    1,
			iter:     16,
			sigma:    2.4344779705352607E-002,
			nDiv:     336,
			zOut:     []float64{0.2611783026646672E+01, 0.2605626003858251E+01, 0.4167433562238479E-02, 0.6157023564192275E-02, 0.1777609536997439E+01, 0.1767807468828494E+01, 0.1546038171944878E-01, 0.1396950250695455E-01, 0.1956758376453290E+01, 0.1967322886153689E+01, 0.2316152103168096E-02, 0.4895872794820515E-02, 0.9257768518560943E+00, 0.9257082891959054E+00, 0.1943036529261404E-02, 0.2384715539127593E-02, 0.8683082132059464E+00, 0.7543114520732653E+00, 0.2389312083572061E+00, 0.1159397984377132E+00, 0.1550512259490286E+01, 0.1789428077358968E+01, 0.9049975109102588E-05, 0.1539126429493388E-04, 0.9665639086686850E+00, 0.9116923136304173E+00, 0.3254569838994440E-01, 0.5488064578914756E-01, 0.5459074409444424E+00, 0.5731983833972557E+00, 0.3618064437406363E-02, 0.5254756712901782E-02, 0.4059719669254931E+00, 0.3758743565324636E+00, 0.1652917395900484E-01, 0.3371567560620648E-01, 0.3952571267845734E+00, 0.1990285273285630E+00, 0.4589757947481138E+00, 0.2127577741907859E+00, 0.7347584950063413E+00, 0.8526760283416343E+00, 0.6803648845168142E+00, 0.3410582621885915E+00, 0.1041863213151506E+01, 0.1465743346590768E+01, 0.1263898840735783E+00, 0.2564847518533230E+00, 0.4627141120500496E+00, 0.5134066246793865E+00, 0.1113705527974558E+00, 0.7569737222001199E-01, 0.6124465423051357E+00, 0.6807729903281149E+00, 0.1524585149425051E-01, 0.4304410555024735E-01, 0.7448350699339780E+00, 0.2169232909544721E+00, 0.1722069046798406E+00, 0.5431576312495270E+00, 0.6394299132491200E-01, 0.2361482827650774E+00, 0.1483665133446019E-05, 0.1614015445872399E-05, 0.6566219320748727E-01, 0.5877885927277143E-01, 0.6842308575232983E-02, 0.6884818375619963E-02, 0.5841446056060346E-01, 0.6525676686594341E-01, 0.4044780558898965E-16, 0.3045663725752605E-08, 0.1013559914197710E-18, 0.7757707209639971E-09, 0.1483665133446019E-05, 0.1614015445872399E-05, 0.6989818532012803E-03, 0.9466330862652142E-28, 0.2539111990097616E-03, 0.9240506410060376E-03, 0.7811996215926567E-04, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			i0Out:    1,
			n0Out:    19,
			ppOut:    1,
			dminOut:  1.0135599141977102E-019,
			desigOut: 2.7104744119407903E-020,
			sigmaOut: 2.4344780481123287E-002,
			qmaxOut:  1.8900334961849885,
			nFailOut: 2,
			iterOut:  18,
			nDivOut:  376,
			ttypeOut: -15,
			dmin1Out: 5.8414457514939733E-002,
			dmin2Out: 5.8777374831867304E-002,
			dnOut:    1.0135599141977102E-019,
			dn1Out:   5.8414457514939733E-002,
			dn2Out:   5.8777374831867304E-002,
			gOut:     0.0000000000000000,
			tauOut:   7.7577068041483555E-010,
		},
		{
			i0:       1,
			n0:       19,
			z:        []float64{0.2611783026646672E+01, 0.2605626003858251E+01, 0.4167433562238479E-02, 0.6157023564192275E-02, 0.1777609536997439E+01, 0.1767807468828494E+01, 0.1546038171944878E-01, 0.1396950250695455E-01, 0.1956758376453290E+01, 0.1967322886153689E+01, 0.2316152103168096E-02, 0.4895872794820515E-02, 0.9257768518560943E+00, 0.9257082891959054E+00, 0.1943036529261404E-02, 0.2384715539127593E-02, 0.8683082132059464E+00, 0.7543114520732653E+00, 0.2389312083572061E+00, 0.1159397984377132E+00, 0.1550512259490286E+01, 0.1789428077358968E+01, 0.9049975109102588E-05, 0.1539126429493388E-04, 0.9665639086686850E+00, 0.9116923136304173E+00, 0.3254569838994440E-01, 0.5488064578914756E-01, 0.5459074409444424E+00, 0.5731983833972557E+00, 0.3618064437406363E-02, 0.5254756712901782E-02, 0.4059719669254931E+00, 0.3758743565324636E+00, 0.1652917395900484E-01, 0.3371567560620648E-01, 0.3952571267845734E+00, 0.1990285273285630E+00, 0.4589757947481138E+00, 0.2127577741907859E+00, 0.7347584950063413E+00, 0.8526760283416343E+00, 0.6803648845168142E+00, 0.3410582621885915E+00, 0.1041863213151506E+01, 0.1465743346590768E+01, 0.1263898840735783E+00, 0.2564847518533230E+00, 0.4627141120500496E+00, 0.5134066246793865E+00, 0.1113705527974558E+00, 0.7569737222001199E-01, 0.6124465423051357E+00, 0.6807729903281149E+00, 0.1524585149425051E-01, 0.4304410555024735E-01, 0.7448350699339780E+00, 0.2169232909544721E+00, 0.1722069046798406E+00, 0.5431576312495270E+00, 0.6394299132491200E-01, 0.2361482827650774E+00, 0.1483665133446019E-05, 0.1614015445872399E-05, 0.6566219320748727E-01, 0.5877885927277143E-01, 0.6842308575232983E-02, 0.6884818375619963E-02, 0.5841446056060346E-01, 0.6525676686594341E-01, 0.4044780558898965E-16, 0.3045663725752605E-08, 0.1013559914197710E-18, 0.7757707209639971E-09, 0.1483665133446019E-05, 0.1614015445872399E-05, 0.6989818532012803E-03, 0.9466330862652142E-28, 0.2539111990097616E-03, 0.9240506410060376E-03, 0.7811996215926567E-04, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			pp:       0,
			dmin:     1.0135599141977102E-019,
			desig:    2.7104744119407903E-020,
			qmax:     1.8900334961849885,
			ttype:    -15,
			dmin1:    5.8414457514939733E-002,
			dmin2:    5.8777374831867304E-002,
			dn:       1.0135599141977102E-019,
			dn1:      5.8414457514939733E-002,
			dn2:      5.8777374831867304E-002,
			g:        0.0000000000000000,
			tau:      7.7577068041483555E-010,
			nFail:    2,
			iter:     18,
			sigma:    2.4344780481123287E-002,
			nDiv:     376,
			zOut:     []float64{0.2611783026646672E+01, 0.2615950460208911E+01, 0.4167433562238479E-02, 0.2831884532112553E-02, 0.1777609536997439E+01, 0.1790238034184775E+01, 0.1546038171944878E-01, 0.1689844079671380E-01, 0.1956758376453290E+01, 0.1942176087759744E+01, 0.2316152103168096E-02, 0.1104039956008399E-02, 0.9257768518560943E+00, 0.9266158484293474E+00, 0.1943036529261404E-02, 0.1820770257466081E-02, 0.8683082132059464E+00, 0.1105418651305687E+01, 0.2389312083572061E+00, 0.3351361651941490E+00, 0.1550512259490286E+01, 0.1215385144271246E+01, 0.9049975109102588E-05, 0.7197207696703830E-05, 0.9665639086686850E+00, 0.9991024098509327E+00, 0.3254569838994440E-01, 0.1778290067827487E-01, 0.5459074409444424E+00, 0.5317426047035739E+00, 0.3618064437406363E-02, 0.2762300261676148E-02, 0.4059719669254931E+00, 0.4197388406228219E+00, 0.1652917395900484E-01, 0.1556509232613395E-01, 0.3952571267845734E+00, 0.8386678292065533E+00, 0.4589757947481138E+00, 0.4021095747913879E+00, 0.7347584950063413E+00, 0.1013013804731768E+01, 0.6803648845168142E+00, 0.6997408538631263E+00, 0.1041863213151506E+01, 0.4685122433619579E+00, 0.1263898840735783E+00, 0.1248257304047288E+00, 0.4627141120500496E+00, 0.4492589344427766E+00, 0.1113705527974558E+00, 0.1518244930621437E+00, 0.6124465423051357E+00, 0.4758679007372426E+00, 0.1524585149425051E-01, 0.2386301922514691E-01, 0.7448350699339780E+00, 0.8931789553886716E+00, 0.1722069046798406E+00, 0.1232835205710967E-01, 0.6394299132491200E-01, 0.5161612293293578E-01, 0.1483665133446019E-05, 0.1887408451311279E-05, 0.6566219320748727E-01, 0.7250261437426894E-01, 0.6842308575232983E-02, 0.5512763475647510E-02, 0.5841446056060346E-01, 0.5290169708495600E-01, 0.4044780558898965E-16, 0.7749519698096867E-34, 0.1013559914197710E-18, 0.1013559914197709E-18, 0.1483665133446019E-05, 0.1887408451311279E-05, 0.6989818532012803E-03, 0.9466330862652142E-28, 0.2539111990097616E-03, 0.9240506410060376E-03, 0.7811996215926567E-04, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			i0Out:    1,
			n0Out:    19,
			ppOut:    0,
			dminOut:  1.0135599141977093E-019,
			desigOut: 2.7104744119407903E-020,
			sigmaOut: 2.4344780481123287E-002,
			qmaxOut:  1.8900334961849885,
			nFailOut: 2,
			iterOut:  19,
			nDivOut:  396,
			ttypeOut: -2,
			dmin1Out: 5.1614639267802333E-002,
			dmin2Out: 5.1614639267802333E-002,
			dnOut:    1.0135599141977093E-019,
			dn1Out:   5.2901697084955956E-002,
			dn2Out:   6.5660305799035965E-002,
			gOut:     0.0000000000000000,
			tauOut:   0.0000000000000000,
		},
		{
			i0:       1,
			n0:       4,
			z:        []float64{0.6482015258643838E+00, 0.6481999396438409E+00, 0.2115037983241246E-16, 0.2912830939801045E-17, 0.8927011497345719E-01, 0.8926852875357447E-01, 0.6601400172354812E-12, 0.2332164496627946E-11, 0.3153708122363886E+00, 0.3200537757344777E+00, 0.4684549720964035E-02, 0.2356209328895529E-07, 0.1609789066298190E-05, 0.6430148075154844E-11, 0.2115037983241246E-16, 0.2912830939801045E-17, 0.1702794694134603E+01, 0.2545269924803487E-17, 0.5179704803914515E-14, 0.1264160184270297E-13, 0.1077066053646038E+01, 0.7156177943897596E-01, 0.2292199980814605E-03, 0.1899018778701386E-03, 0.1060140274732043E+01, 0.8660746506696473E-01, 0.1357005210961402E-39, 0.1331360138522907E-23, 0.9894235909971354E+00, 0.8804208964992894E-17, 0.8660746506696473E-01, 0.8679736700028205E-01, 0.9539342071687115E+00, 0.2761013168273541E-29, 0.1040577915698430E-22, 0.8107757314001177E-24, 0.8046649468928653E+00, 0.5098326619997980E-22, 0.1416795225784663E-10, 0.1216165317638265E-10, 0.7009142227469247E+00, 0.1183291357831518E-29, 0.4806926318247711E-11, 0.3824647564366386E-11, 0.5502651835254770E+00, 0.1368072596837427E-11, 0.1169503544861386E-10, 0.4068037876491279E-10, 0.4423863025187732E+00, 0.2477754901417239E-20, 0.1737857614720001E-07, 0.5295826057530262E-07, 0.3697086796938907E+00, 0.5143373102040997E-26, 0.1312431380925897E-05, 0.1614374370413396E-05, 0.1446051340026323E+00, 0.9928287808749566E-25, 0.3777515963415321E-05, 0.2773141909621761E-05, 0.1110032523123295E+00, 0.0000000000000000E+00, 0.2532463507333992E-05, 0.2934028940292093E-05, 0.7513603923341917E-01, 0.5394210206791908E-19, 0.6032617175984252E-07, 0.2575740214720034E-06, 0.6860214953971246E-01, 0.7542232825258426E-19, 0.3784397549471832E-09, 0.1131370986389306E-09, 0.2434478048112329E-01, 0.1013559914197709E-18, 0.1483665133446019E-05, 0.1887408451311279E-05, 0.6989818532012803E-03, 0.9466330862652142E-28, 0.2539111990097616E-03, 0.9240506410060376E-03, 0.7811996215926567E-04, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			pp:       1,
			dmin:     6.4301480751548441E-012,
			desig:    -2.0633404957759536E-016,
			qmax:     1.6602870234255678,
			ttype:    -2,
			dmin1:    8.9268528752914328E-002,
			dmin2:    8.9268528752914328E-002,
			dn:       6.4301480751548441E-012,
			dn1:      0.31536922601351364,
			dn2:      8.9268528752914328E-002,
			g:        0.49975000000000003,
			tau:      1.5862205428611591E-006,
			nFail:    5,
			iter:     102,
			sigma:    2.0015106747272648,
			nDiv:     1395,
			zOut:     []float64{0.6481999396374125E+00, 0.6481999396438409E+00, 0.4011480356653261E-18, 0.2912830939801045E-17, 0.8926852874947827E-01, 0.8926852875357447E-01, 0.8361491594360282E-11, 0.2332164496627946E-11, 0.3200537992817811E+00, 0.3200537757344777E+00, 0.4733821287189426E-18, 0.2356209328895529E-07, 0.1787791211067534E-14, 0.6430148075154844E-11, 0.4011480356653261E-18, 0.2912830939801045E-17, 0.1702794694134603E+01, 0.2545269924803487E-17, 0.5179704803914515E-14, 0.1264160184270297E-13, 0.1077066053646038E+01, 0.7156177943897596E-01, 0.2292199980814605E-03, 0.1899018778701386E-03, 0.1060140274732043E+01, 0.8660746506696473E-01, 0.1357005210961402E-39, 0.1331360138522907E-23, 0.9894235909971354E+00, 0.8804208964992894E-17, 0.8660746506696473E-01, 0.8679736700028205E-01, 0.9539342071687115E+00, 0.2761013168273541E-29, 0.1040577915698430E-22, 0.8107757314001177E-24, 0.8046649468928653E+00, 0.5098326619997980E-22, 0.1416795225784663E-10, 0.1216165317638265E-10, 0.7009142227469247E+00, 0.1183291357831518E-29, 0.4806926318247711E-11, 0.3824647564366386E-11, 0.5502651835254770E+00, 0.1368072596837427E-11, 0.1169503544861386E-10, 0.4068037876491279E-10, 0.4423863025187732E+00, 0.2477754901417239E-20, 0.1737857614720001E-07, 0.5295826057530262E-07, 0.3697086796938907E+00, 0.5143373102040997E-26, 0.1312431380925897E-05, 0.1614374370413396E-05, 0.1446051340026323E+00, 0.9928287808749566E-25, 0.3777515963415321E-05, 0.2773141909621761E-05, 0.1110032523123295E+00, 0.0000000000000000E+00, 0.2532463507333992E-05, 0.2934028940292093E-05, 0.7513603923341917E-01, 0.5394210206791908E-19, 0.6032617175984252E-07, 0.2575740214720034E-06, 0.6860214953971246E-01, 0.7542232825258426E-19, 0.3784397549471832E-09, 0.1131370986389306E-09, 0.2434478048112329E-01, 0.1013559914197709E-18, 0.1483665133446019E-05, 0.1887408451311279E-05, 0.6989818532012803E-03, 0.9466330862652142E-28, 0.2539111990097616E-03, 0.9240506410060376E-03, 0.7811996215926567E-04, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			i0Out:    1,
			n0Out:    4,
			ppOut:    1,
			dminOut:  1.7877912110675335E-015,
			desigOut: -3.7836067586147711E-017,
			sigmaOut: 2.0015106747336930,
			qmaxOut:  1.6602870234255678,
			nFailOut: 5,
			iterOut:  103,
			nDivOut:  1400,
			ttypeOut: -4,
			dmin1Out: 8.9268528747146109E-002,
			dmin2Out: 8.9268528747146109E-002,
			dnOut:    1.7877912110675335E-015,
			dn1Out:   0.32005377571968785,
			dn2Out:   8.9268528747146109E-002,
			gOut:     0.49975000000000003,
			tauOut:   6.4283598105616478E-012,
		},
		{
			i0:       1,
			n0:       4,
			z:        []float64{0.6481999396374125E+00, 0.6481999396438409E+00, 0.4011480356653261E-18, 0.2912830939801045E-17, 0.8926852874947827E-01, 0.8926852875357447E-01, 0.8361491594360282E-11, 0.2332164496627946E-11, 0.3200537992817811E+00, 0.3200537757344777E+00, 0.4733821287189426E-18, 0.2356209328895529E-07, 0.1787791211067534E-14, 0.6430148075154844E-11, 0.4011480356653261E-18, 0.2912830939801045E-17, 0.1702794694134603E+01, 0.2545269924803487E-17, 0.5179704803914515E-14, 0.1264160184270297E-13, 0.1077066053646038E+01, 0.7156177943897596E-01, 0.2292199980814605E-03, 0.1899018778701386E-03, 0.1060140274732043E+01, 0.8660746506696473E-01, 0.1357005210961402E-39, 0.1331360138522907E-23, 0.9894235909971354E+00, 0.8804208964992894E-17, 0.8660746506696473E-01, 0.8679736700028205E-01, 0.9539342071687115E+00, 0.2761013168273541E-29, 0.1040577915698430E-22, 0.8107757314001177E-24, 0.8046649468928653E+00, 0.5098326619997980E-22, 0.1416795225784663E-10, 0.1216165317638265E-10, 0.7009142227469247E+00, 0.1183291357831518E-29, 0.4806926318247711E-11, 0.3824647564366386E-11, 0.5502651835254770E+00, 0.1368072596837427E-11, 0.1169503544861386E-10, 0.4068037876491279E-10, 0.4423863025187732E+00, 0.2477754901417239E-20, 0.1737857614720001E-07, 0.5295826057530262E-07, 0.3697086796938907E+00, 0.5143373102040997E-26, 0.1312431380925897E-05, 0.1614374370413396E-05, 0.1446051340026323E+00, 0.9928287808749566E-25, 0.3777515963415321E-05, 0.2773141909621761E-05, 0.1110032523123295E+00, 0.0000000000000000E+00, 0.2532463507333992E-05, 0.2934028940292093E-05, 0.7513603923341917E-01, 0.5394210206791908E-19, 0.6032617175984252E-07, 0.2575740214720034E-06, 0.6860214953971246E-01, 0.7542232825258426E-19, 0.3784397549471832E-09, 0.1131370986389306E-09, 0.2434478048112329E-01, 0.1013559914197709E-18, 0.1483665133446019E-05, 0.1887408451311279E-05, 0.6989818532012803E-03, 0.9466330862652142E-28, 0.2539111990097616E-03, 0.9240506410060376E-03, 0.7811996215926567E-04, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			pp:       0,
			dmin:     1.7877912110675335E-015,
			desig:    -3.7836067586147711E-017,
			qmax:     1.6602870234255678,
			ttype:    -4,
			dmin1:    8.9268528747146109E-002,
			dmin2:    8.9268528747146109E-002,
			dn:       1.7877912110675335E-015,
			dn1:      0.32005377571968785,
			dn2:      8.9268528747146109E-002,
			g:        0.49975000000000003,
			tau:      6.4283598105616478E-012,
			nFail:    5,
			iter:     103,
			sigma:    2.0015106747336930,
			nDiv:     1400,
			zOut:     []float64{0.6481999396374125E+00, 0.6481999396374107E+00, 0.4011480356653261E-18, 0.5524513774965514E-19, 0.8926852874947827E-01, 0.8926852875783797E-01, 0.8361491594360282E-11, 0.2997839428604580E-10, 0.3200537992817811E+00, 0.3200537992518009E+00, 0.4733821287189426E-18, 0.2644269217171004E-32, 0.1787791211067534E-14, 0.2227952244484021E-23, 0.4011480356653261E-18, 0.5524513774965514E-19, 0.1702794694134603E+01, 0.2545269924803487E-17, 0.5179704803914515E-14, 0.1264160184270297E-13, 0.1077066053646038E+01, 0.7156177943897596E-01, 0.2292199980814605E-03, 0.1899018778701386E-03, 0.1060140274732043E+01, 0.8660746506696473E-01, 0.1357005210961402E-39, 0.1331360138522907E-23, 0.9894235909971354E+00, 0.8804208964992894E-17, 0.8660746506696473E-01, 0.8679736700028205E-01, 0.9539342071687115E+00, 0.2761013168273541E-29, 0.1040577915698430E-22, 0.8107757314001177E-24, 0.8046649468928653E+00, 0.5098326619997980E-22, 0.1416795225784663E-10, 0.1216165317638265E-10, 0.7009142227469247E+00, 0.1183291357831518E-29, 0.4806926318247711E-11, 0.3824647564366386E-11, 0.5502651835254770E+00, 0.1368072596837427E-11, 0.1169503544861386E-10, 0.4068037876491279E-10, 0.4423863025187732E+00, 0.2477754901417239E-20, 0.1737857614720001E-07, 0.5295826057530262E-07, 0.3697086796938907E+00, 0.5143373102040997E-26, 0.1312431380925897E-05, 0.1614374370413396E-05, 0.1446051340026323E+00, 0.9928287808749566E-25, 0.3777515963415321E-05, 0.2773141909621761E-05, 0.1110032523123295E+00, 0.0000000000000000E+00, 0.2532463507333992E-05, 0.2934028940292093E-05, 0.7513603923341917E-01, 0.5394210206791908E-19, 0.6032617175984252E-07, 0.2575740214720034E-06, 0.6860214953971246E-01, 0.7542232825258426E-19, 0.3784397549471832E-09, 0.1131370986389306E-09, 0.2434478048112329E-01, 0.1013559914197709E-18, 0.1483665133446019E-05, 0.1887408451311279E-05, 0.6989818532012803E-03, 0.9466330862652142E-28, 0.2539111990097616E-03, 0.9240506410060376E-03, 0.7811996215926567E-04, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			i0Out:    1,
			n0Out:    4,
			ppOut:    0,
			dminOut:  2.2279522444840209E-024,
			desigOut: -2.6401698146816920E-017,
			sigmaOut: 2.0015106747336948,
			qmaxOut:  1.6602870234255678,
			nFailOut: 5,
			iterOut:  104,
			nDivOut:  1405,
			ttypeOut: -4,
			dmin1Out: 8.9268528749476481E-002,
			dmin2Out: 8.9268528749476481E-002,
			dnOut:    2.2279522444840209E-024,
			dn1Out:   0.32005379925180094,
			dn2Out:   8.9268528749476481E-002,
			gOut:     0.49975000000000003,
			tauOut:   1.7877912088395813E-015,
		},
		{
			i0:       1,
			n0:       4,
			z:        []float64{0.6481999396374125E+00, 0.6481999396374107E+00, 0.4011480356653261E-18, 0.5524513774965514E-19, 0.8926852874947827E-01, 0.8926852875783797E-01, 0.8361491594360282E-11, 0.2997839428604580E-10, 0.3200537992817811E+00, 0.3200537992518009E+00, 0.4733821287189426E-18, 0.2644269217171004E-32, 0.1787791211067534E-14, 0.2227952244484021E-23, 0.4011480356653261E-18, 0.5524513774965514E-19, 0.1702794694134603E+01, 0.2545269924803487E-17, 0.5179704803914515E-14, 0.1264160184270297E-13, 0.1077066053646038E+01, 0.7156177943897596E-01, 0.2292199980814605E-03, 0.1899018778701386E-03, 0.1060140274732043E+01, 0.8660746506696473E-01, 0.1357005210961402E-39, 0.1331360138522907E-23, 0.9894235909971354E+00, 0.8804208964992894E-17, 0.8660746506696473E-01, 0.8679736700028205E-01, 0.9539342071687115E+00, 0.2761013168273541E-29, 0.1040577915698430E-22, 0.8107757314001177E-24, 0.8046649468928653E+00, 0.5098326619997980E-22, 0.1416795225784663E-10, 0.1216165317638265E-10, 0.7009142227469247E+00, 0.1183291357831518E-29, 0.4806926318247711E-11, 0.3824647564366386E-11, 0.5502651835254770E+00, 0.1368072596837427E-11, 0.1169503544861386E-10, 0.4068037876491279E-10, 0.4423863025187732E+00, 0.2477754901417239E-20, 0.1737857614720001E-07, 0.5295826057530262E-07, 0.3697086796938907E+00, 0.5143373102040997E-26, 0.1312431380925897E-05, 0.1614374370413396E-05, 0.1446051340026323E+00, 0.9928287808749566E-25, 0.3777515963415321E-05, 0.2773141909621761E-05, 0.1110032523123295E+00, 0.0000000000000000E+00, 0.2532463507333992E-05, 0.2934028940292093E-05, 0.7513603923341917E-01, 0.5394210206791908E-19, 0.6032617175984252E-07, 0.2575740214720034E-06, 0.6860214953971246E-01, 0.7542232825258426E-19, 0.3784397549471832E-09, 0.1131370986389306E-09, 0.2434478048112329E-01, 0.1013559914197709E-18, 0.1483665133446019E-05, 0.1887408451311279E-05, 0.6989818532012803E-03, 0.9466330862652142E-28, 0.2539111990097616E-03, 0.9240506410060376E-03, 0.7811996215926567E-04, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			pp:       1,
			dmin:     2.2279522444840209E-024,
			desig:    -2.6401698146816920E-017,
			qmax:     1.6602870234255678,
			ttype:    -4,
			dmin1:    8.9268528749476481E-002,
			dmin2:    8.9268528749476481E-002,
			dn:       2.2279522444840209E-024,
			dn1:      0.32005379925180094,
			dn2:      8.9268528749476481E-002,
			g:        0.49975000000000003,
			tau:      1.7877912088395813E-015,
			nFail:    5,
			iter:     104,
			sigma:    2.0015106747336948,
			nDiv:     1405,
			zOut:     []float64{0.6258828074500417E+00, 0.6481999396374107E+00, 0.7879513719234823E-20, 0.5524513774965514E-19, 0.6695139660044724E-01, 0.8926852875783797E-01, 0.1433084218388560E-09, 0.2997839428604580E-10, 0.2977366669211234E+00, 0.3200537992518009E+00, 0.8926852875783797E-01, 0.2644269217171004E-32, 0.2001510674733695E+01, 0.2227952244484021E-23, 0.4011480356653261E-18, 0.5524513774965514E-19, 0.1702794694134603E+01, 0.2545269924803487E-17, 0.5179704803914515E-14, 0.1264160184270297E-13, 0.1077066053646038E+01, 0.7156177943897596E-01, 0.2292199980814605E-03, 0.1899018778701386E-03, 0.1060140274732043E+01, 0.8660746506696473E-01, 0.1357005210961402E-39, 0.1331360138522907E-23, 0.9894235909971354E+00, 0.8804208964992894E-17, 0.8660746506696473E-01, 0.8679736700028205E-01, 0.9539342071687115E+00, 0.2761013168273541E-29, 0.1040577915698430E-22, 0.8107757314001177E-24, 0.8046649468928653E+00, 0.5098326619997980E-22, 0.1416795225784663E-10, 0.1216165317638265E-10, 0.7009142227469247E+00, 0.1183291357831518E-29, 0.4806926318247711E-11, 0.3824647564366386E-11, 0.5502651835254770E+00, 0.1368072596837427E-11, 0.1169503544861386E-10, 0.4068037876491279E-10, 0.4423863025187732E+00, 0.2477754901417239E-20, 0.1737857614720001E-07, 0.5295826057530262E-07, 0.3697086796938907E+00, 0.5143373102040997E-26, 0.1312431380925897E-05, 0.1614374370413396E-05, 0.1446051340026323E+00, 0.9928287808749566E-25, 0.3777515963415321E-05, 0.2773141909621761E-05, 0.1110032523123295E+00, 0.0000000000000000E+00, 0.2532463507333992E-05, 0.2934028940292093E-05, 0.7513603923341917E-01, 0.5394210206791908E-19, 0.6032617175984252E-07, 0.2575740214720034E-06, 0.6860214953971246E-01, 0.7542232825258426E-19, 0.3784397549471832E-09, 0.1131370986389306E-09, 0.2434478048112329E-01, 0.1013559914197709E-18, 0.1483665133446019E-05, 0.1887408451311279E-05, 0.6989818532012803E-03, 0.9466330862652142E-28, 0.2539111990097616E-03, 0.9240506410060376E-03, 0.7811996215926567E-04, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			i0Out:    1,
			n0Out:    3,
			ppOut:    1,
			dminOut:  6.6951396570468849E-002,
			desigOut: -1.0061396160665481E-016,
			sigmaOut: 2.0238278069210640,
			qmaxOut:  1.6602870234255678,
			nFailOut: 5,
			iterOut:  105,
			nDivOut:  1409,
			ttypeOut: -9,
			dmin1Out: 6.6951396570468849E-002,
			dmin2Out: 0.62588280745004166,
			dnOut:    0.29773666692112338,
			dn1Out:   6.6951396570468849E-002,
			dn2Out:   0.62588280745004166,
			gOut:     0.49975000000000003,
			tauOut:   2.2317132187369120E-002,
		},
		{
			i0:       1,
			n0:       3,
			z:        []float64{0.6258828074500417E+00, 0.6481999396374107E+00, 0.7879513719234823E-20, 0.5524513774965514E-19, 0.6695139660044724E-01, 0.8926852875783797E-01, 0.1433084218388560E-09, 0.2997839428604580E-10, 0.2977366669211234E+00, 0.3200537992518009E+00, 0.8926852875783797E-01, 0.2644269217171004E-32, 0.2001510674733695E+01, 0.2227952244484021E-23, 0.4011480356653261E-18, 0.5524513774965514E-19, 0.1702794694134603E+01, 0.2545269924803487E-17, 0.5179704803914515E-14, 0.1264160184270297E-13, 0.1077066053646038E+01, 0.7156177943897596E-01, 0.2292199980814605E-03, 0.1899018778701386E-03, 0.1060140274732043E+01, 0.8660746506696473E-01, 0.1357005210961402E-39, 0.1331360138522907E-23, 0.9894235909971354E+00, 0.8804208964992894E-17, 0.8660746506696473E-01, 0.8679736700028205E-01, 0.9539342071687115E+00, 0.2761013168273541E-29, 0.1040577915698430E-22, 0.8107757314001177E-24, 0.8046649468928653E+00, 0.5098326619997980E-22, 0.1416795225784663E-10, 0.1216165317638265E-10, 0.7009142227469247E+00, 0.1183291357831518E-29, 0.4806926318247711E-11, 0.3824647564366386E-11, 0.5502651835254770E+00, 0.1368072596837427E-11, 0.1169503544861386E-10, 0.4068037876491279E-10, 0.4423863025187732E+00, 0.2477754901417239E-20, 0.1737857614720001E-07, 0.5295826057530262E-07, 0.3697086796938907E+00, 0.5143373102040997E-26, 0.1312431380925897E-05, 0.1614374370413396E-05, 0.1446051340026323E+00, 0.9928287808749566E-25, 0.3777515963415321E-05, 0.2773141909621761E-05, 0.1110032523123295E+00, 0.0000000000000000E+00, 0.2532463507333992E-05, 0.2934028940292093E-05, 0.7513603923341917E-01, 0.5394210206791908E-19, 0.6032617175984252E-07, 0.2575740214720034E-06, 0.6860214953971246E-01, 0.7542232825258426E-19, 0.3784397549471832E-09, 0.1131370986389306E-09, 0.2434478048112329E-01, 0.1013559914197709E-18, 0.1483665133446019E-05, 0.1887408451311279E-05, 0.6989818532012803E-03, 0.9466330862652142E-28, 0.2539111990097616E-03, 0.9240506410060376E-03, 0.7811996215926567E-04, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			pp:       0,
			dmin:     6.6951396570468849E-002,
			desig:    -1.0061396160665481E-016,
			qmax:     1.6602870234255678,
			ttype:    -9,
			dmin1:    6.6951396570468849E-002,
			dmin2:    0.62588280745004166,
			dn:       0.29773666692112338,
			dn1:      6.6951396570468849E-002,
			dn2:      0.62588280745004166,
			g:        0.49975000000000003,
			tau:      2.2317132187369120E-002,
			nFail:    5,
			iter:     105,
			sigma:    2.0238278069210640,
			nDiv:     1409,
			zOut:     []float64{0.6258828074500417E+00, 0.5589320748538995E+00, 0.7879513719234823E-20, 0.9438435755776795E-21, 0.6695139660044724E-01, 0.6641476135588615E-06, 0.1433084218388560E-09, 0.6424501268835132E-04, 0.2977366669211234E+00, 0.2307216893122929E+00, 0.8926852875783797E-01, 0.6695139660044724E-01, 0.2001510674733695E+01, 0.2227952244484021E-23, 0.4011480356653261E-18, 0.5524513774965514E-19, 0.1702794694134603E+01, 0.2545269924803487E-17, 0.5179704803914515E-14, 0.1264160184270297E-13, 0.1077066053646038E+01, 0.7156177943897596E-01, 0.2292199980814605E-03, 0.1899018778701386E-03, 0.1060140274732043E+01, 0.8660746506696473E-01, 0.1357005210961402E-39, 0.1331360138522907E-23, 0.9894235909971354E+00, 0.8804208964992894E-17, 0.8660746506696473E-01, 0.8679736700028205E-01, 0.9539342071687115E+00, 0.2761013168273541E-29, 0.1040577915698430E-22, 0.8107757314001177E-24, 0.8046649468928653E+00, 0.5098326619997980E-22, 0.1416795225784663E-10, 0.1216165317638265E-10, 0.7009142227469247E+00, 0.1183291357831518E-29, 0.4806926318247711E-11, 0.3824647564366386E-11, 0.5502651835254770E+00, 0.1368072596837427E-11, 0.1169503544861386E-10, 0.4068037876491279E-10, 0.4423863025187732E+00, 0.2477754901417239E-20, 0.1737857614720001E-07, 0.5295826057530262E-07, 0.3697086796938907E+00, 0.5143373102040997E-26, 0.1312431380925897E-05, 0.1614374370413396E-05, 0.1446051340026323E+00, 0.9928287808749566E-25, 0.3777515963415321E-05, 0.2773141909621761E-05, 0.1110032523123295E+00, 0.0000000000000000E+00, 0.2532463507333992E-05, 0.2934028940292093E-05, 0.7513603923341917E-01, 0.5394210206791908E-19, 0.6032617175984252E-07, 0.2575740214720034E-06, 0.6860214953971246E-01, 0.7542232825258426E-19, 0.3784397549471832E-09, 0.1131370986389306E-09, 0.2434478048112329E-01, 0.1013559914197709E-18, 0.1483665133446019E-05, 0.1887408451311279E-05, 0.6989818532012803E-03, 0.9466330862652142E-28, 0.2539111990097616E-03, 0.9240506410060376E-03, 0.7811996215926567E-04, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			i0Out:    1,
			n0Out:    3,
			ppOut:    0,
			dminOut:  6.6400430513702258E-007,
			desigOut: 1.9428902930940239E-016,
			sigmaOut: 2.0907785395172058,
			qmaxOut:  1.6602870234255678,
			nFailOut: 5,
			iterOut:  106,
			nDivOut:  1413,
			ttypeOut: -4,
			dmin1Out: 6.6400430513702258E-007,
			dmin2Out: 0.55893207485389951,
			dnOut:    0.23072168931229292,
			dn1Out:   6.6400430513702258E-007,
			dn2Out:   0.55893207485389951,
			gOut:     0.49975000000000003,
			tauOut:   6.6950732596142107E-002,
		},
		{
			i0:       1,
			n0:       3,
			z:        []float64{0.6258828074500417E+00, 0.5589320748538995E+00, 0.7879513719234823E-20, 0.9438435755776795E-21, 0.6695139660044724E-01, 0.6641476135588615E-06, 0.1433084218388560E-09, 0.6424501268835132E-04, 0.2977366669211234E+00, 0.2307216893122929E+00, 0.8926852875783797E-01, 0.6695139660044724E-01, 0.2001510674733695E+01, 0.2227952244484021E-23, 0.4011480356653261E-18, 0.5524513774965514E-19, 0.1702794694134603E+01, 0.2545269924803487E-17, 0.5179704803914515E-14, 0.1264160184270297E-13, 0.1077066053646038E+01, 0.7156177943897596E-01, 0.2292199980814605E-03, 0.1899018778701386E-03, 0.1060140274732043E+01, 0.8660746506696473E-01, 0.1357005210961402E-39, 0.1331360138522907E-23, 0.9894235909971354E+00, 0.8804208964992894E-17, 0.8660746506696473E-01, 0.8679736700028205E-01, 0.9539342071687115E+00, 0.2761013168273541E-29, 0.1040577915698430E-22, 0.8107757314001177E-24, 0.8046649468928653E+00, 0.5098326619997980E-22, 0.1416795225784663E-10, 0.1216165317638265E-10, 0.7009142227469247E+00, 0.1183291357831518E-29, 0.4806926318247711E-11, 0.3824647564366386E-11, 0.5502651835254770E+00, 0.1368072596837427E-11, 0.1169503544861386E-10, 0.4068037876491279E-10, 0.4423863025187732E+00, 0.2477754901417239E-20, 0.1737857614720001E-07, 0.5295826057530262E-07, 0.3697086796938907E+00, 0.5143373102040997E-26, 0.1312431380925897E-05, 0.1614374370413396E-05, 0.1446051340026323E+00, 0.9928287808749566E-25, 0.3777515963415321E-05, 0.2773141909621761E-05, 0.1110032523123295E+00, 0.0000000000000000E+00, 0.2532463507333992E-05, 0.2934028940292093E-05, 0.7513603923341917E-01, 0.5394210206791908E-19, 0.6032617175984252E-07, 0.2575740214720034E-06, 0.6860214953971246E-01, 0.7542232825258426E-19, 0.3784397549471832E-09, 0.1131370986389306E-09, 0.2434478048112329E-01, 0.1013559914197709E-18, 0.1483665133446019E-05, 0.1887408451311279E-05, 0.6989818532012803E-03, 0.9466330862652142E-28, 0.2539111990097616E-03, 0.9240506410060376E-03, 0.7811996215926567E-04, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			pp:       1,
			dmin:     6.6400430513702258E-007,
			desig:    1.9428902930940239E-016,
			qmax:     1.6602870234255678,
			ttype:    -4,
			dmin1:    6.6400430513702258E-007,
			dmin2:    0.55893207485389951,
			dn:       0.23072168931229292,
			dn1:      6.6400430513702258E-007,
			dn2:      0.55893207485389951,
			g:        0.49975000000000003,
			tau:      6.6950732596142107E-002,
			nFail:    5,
			iter:     106,
			sigma:    2.0907785395172058,
			nDiv:     1413,
			zOut:     []float64{0.5589315065851642E+00, 0.5589320748538995E+00, 0.1121517486324177E-26, 0.9438435755776795E-21, 0.6434089156657428E-04, 0.6641476135588615E-06, 0.2303778747300831E+00, 0.6424501268835132E-04, 0.3432463134744483E-03, 0.2307216893122929E+00, 0.6641476135588615E-06, 0.6695139660044724E-01, 0.2001510674733695E+01, 0.2227952244484021E-23, 0.4011480356653261E-18, 0.5524513774965514E-19, 0.1702794694134603E+01, 0.2545269924803487E-17, 0.5179704803914515E-14, 0.1264160184270297E-13, 0.1077066053646038E+01, 0.7156177943897596E-01, 0.2292199980814605E-03, 0.1899018778701386E-03, 0.1060140274732043E+01, 0.8660746506696473E-01, 0.1357005210961402E-39, 0.1331360138522907E-23, 0.9894235909971354E+00, 0.8804208964992894E-17, 0.8660746506696473E-01, 0.8679736700028205E-01, 0.9539342071687115E+00, 0.2761013168273541E-29, 0.1040577915698430E-22, 0.8107757314001177E-24, 0.8046649468928653E+00, 0.5098326619997980E-22, 0.1416795225784663E-10, 0.1216165317638265E-10, 0.7009142227469247E+00, 0.1183291357831518E-29, 0.4806926318247711E-11, 0.3824647564366386E-11, 0.5502651835254770E+00, 0.1368072596837427E-11, 0.1169503544861386E-10, 0.4068037876491279E-10, 0.4423863025187732E+00, 0.2477754901417239E-20, 0.1737857614720001E-07, 0.5295826057530262E-07, 0.3697086796938907E+00, 0.5143373102040997E-26, 0.1312431380925897E-05, 0.1614374370413396E-05, 0.1446051340026323E+00, 0.9928287808749566E-25, 0.3777515963415321E-05, 0.2773141909621761E-05, 0.1110032523123295E+00, 0.0000000000000000E+00, 0.2532463507333992E-05, 0.2934028940292093E-05, 0.7513603923341917E-01, 0.5394210206791908E-19, 0.6032617175984252E-07, 0.2575740214720034E-06, 0.6860214953971246E-01, 0.7542232825258426E-19, 0.3784397549471832E-09, 0.1131370986389306E-09, 0.2434478048112329E-01, 0.1013559914197709E-18, 0.1483665133446019E-05, 0.1887408451311279E-05, 0.6989818532012803E-03, 0.9466330862652142E-28, 0.2539111990097616E-03, 0.9240506410060376E-03, 0.7811996215926567E-04, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			i0Out:    1,
			n0Out:    3,
			ppOut:    1,
			dminOut:  9.5878878222950548E-008,
			desigOut: -9.3233975295464906E-017,
			sigmaOut: 2.0907791077859414,
			qmaxOut:  1.6602870234255678,
			nFailOut: 6,
			iterOut:  108,
			nDivOut:  1421,
			ttypeOut: -15,
			dmin1Out: 9.5878878222950548E-008,
			dmin2Out: 0.55893150658516422,
			dnOut:    3.4324631347444829E-004,
			dn1Out:   9.5878878222950548E-008,
			dn2Out:   0.55893150658516422,
			gOut:     0.49975000000000003,
			tauOut:   5.6826873533591094E-007,
		},
		{
			i0:       1,
			n0:       3,
			z:        []float64{0.5589315065851642E+00, 0.5589320748538995E+00, 0.1121517486324177E-26, 0.9438435755776795E-21, 0.6434089156657428E-04, 0.6641476135588615E-06, 0.2303778747300831E+00, 0.6424501268835132E-04, 0.3432463134744483E-03, 0.2307216893122929E+00, 0.6641476135588615E-06, 0.6695139660044724E-01, 0.2001510674733695E+01, 0.2227952244484021E-23, 0.4011480356653261E-18, 0.5524513774965514E-19, 0.1702794694134603E+01, 0.2545269924803487E-17, 0.5179704803914515E-14, 0.1264160184270297E-13, 0.1077066053646038E+01, 0.7156177943897596E-01, 0.2292199980814605E-03, 0.1899018778701386E-03, 0.1060140274732043E+01, 0.8660746506696473E-01, 0.1357005210961402E-39, 0.1331360138522907E-23, 0.9894235909971354E+00, 0.8804208964992894E-17, 0.8660746506696473E-01, 0.8679736700028205E-01, 0.9539342071687115E+00, 0.2761013168273541E-29, 0.1040577915698430E-22, 0.8107757314001177E-24, 0.8046649468928653E+00, 0.5098326619997980E-22, 0.1416795225784663E-10, 0.1216165317638265E-10, 0.7009142227469247E+00, 0.1183291357831518E-29, 0.4806926318247711E-11, 0.3824647564366386E-11, 0.5502651835254770E+00, 0.1368072596837427E-11, 0.1169503544861386E-10, 0.4068037876491279E-10, 0.4423863025187732E+00, 0.2477754901417239E-20, 0.1737857614720001E-07, 0.5295826057530262E-07, 0.3697086796938907E+00, 0.5143373102040997E-26, 0.1312431380925897E-05, 0.1614374370413396E-05, 0.1446051340026323E+00, 0.9928287808749566E-25, 0.3777515963415321E-05, 0.2773141909621761E-05, 0.1110032523123295E+00, 0.0000000000000000E+00, 0.2532463507333992E-05, 0.2934028940292093E-05, 0.7513603923341917E-01, 0.5394210206791908E-19, 0.6032617175984252E-07, 0.2575740214720034E-06, 0.6860214953971246E-01, 0.7542232825258426E-19, 0.3784397549471832E-09, 0.1131370986389306E-09, 0.2434478048112329E-01, 0.1013559914197709E-18, 0.1483665133446019E-05, 0.1887408451311279E-05, 0.6989818532012803E-03, 0.9466330862652142E-28, 0.2539111990097616E-03, 0.9240506410060376E-03, 0.7811996215926567E-04, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			pp:       0,
			dmin:     9.5878878222950548E-008,
			desig:    -9.3233975295464906E-017,
			qmax:     1.6602870234255678,
			ttype:    -15,
			dmin1:    9.5878878222950548E-008,
			dmin2:    0.55893150658516422,
			dn:       3.4324631347444829E-004,
			dn1:      9.5878878222950548E-008,
			dn2:      0.55893150658516422,
			g:        0.49975000000000003,
			tau:      5.6826873533591094E-007,
			nFail:    6,
			iter:     108,
			sigma:    2.0907791077859414,
			nDiv:     1421,
			zOut:     []float64{0.5589315065851642E+00, 0.5589314123732620E+00, 0.1121517486324177E-26, 0.1291024862446124E-30, 0.6434089156657428E-04, 0.2304421214097475E+00, 0.2303778747300831E+00, 0.3431506172718059E-03, 0.3432463134744483E-03, 0.1484300435548105E-08, 0.6641476135588615E-06, 0.6434089156657428E-04, 0.2001510674733695E+01, 0.2227952244484021E-23, 0.4011480356653261E-18, 0.5524513774965514E-19, 0.1702794694134603E+01, 0.2545269924803487E-17, 0.5179704803914515E-14, 0.1264160184270297E-13, 0.1077066053646038E+01, 0.7156177943897596E-01, 0.2292199980814605E-03, 0.1899018778701386E-03, 0.1060140274732043E+01, 0.8660746506696473E-01, 0.1357005210961402E-39, 0.1331360138522907E-23, 0.9894235909971354E+00, 0.8804208964992894E-17, 0.8660746506696473E-01, 0.8679736700028205E-01, 0.9539342071687115E+00, 0.2761013168273541E-29, 0.1040577915698430E-22, 0.8107757314001177E-24, 0.8046649468928653E+00, 0.5098326619997980E-22, 0.1416795225784663E-10, 0.1216165317638265E-10, 0.7009142227469247E+00, 0.1183291357831518E-29, 0.4806926318247711E-11, 0.3824647564366386E-11, 0.5502651835254770E+00, 0.1368072596837427E-11, 0.1169503544861386E-10, 0.4068037876491279E-10, 0.4423863025187732E+00, 0.2477754901417239E-20, 0.1737857614720001E-07, 0.5295826057530262E-07, 0.3697086796938907E+00, 0.5143373102040997E-26, 0.1312431380925897E-05, 0.1614374370413396E-05, 0.1446051340026323E+00, 0.9928287808749566E-25, 0.3777515963415321E-05, 0.2773141909621761E-05, 0.1110032523123295E+00, 0.0000000000000000E+00, 0.2532463507333992E-05, 0.2934028940292093E-05, 0.7513603923341917E-01, 0.5394210206791908E-19, 0.6032617175984252E-07, 0.2575740214720034E-06, 0.6860214953971246E-01, 0.7542232825258426E-19, 0.3784397549471832E-09, 0.1131370986389306E-09, 0.2434478048112329E-01, 0.1013559914197709E-18, 0.1483665133446019E-05, 0.1887408451311279E-05, 0.6989818532012803E-03, 0.9466330862652142E-28, 0.2539111990097616E-03, 0.9240506410060376E-03, 0.7811996215926567E-04, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			i0Out:    1,
			n0Out:    3,
			ppOut:    0,
			dminOut:  1.4843004355481051E-009,
			desigOut: -1.6593817482031092E-016,
			sigmaOut: 2.0907792019978437,
			qmaxOut:  1.6602870234255678,
			nFailOut: 6,
			iterOut:  109,
			nDivOut:  1425,
			ttypeOut: -4,
			dmin1Out: 6.4246679664367451E-005,
			dmin2Out: 0.55893141237326205,
			dnOut:    1.4843004355481051E-009,
			dn1Out:   6.4246679664367451E-005,
			dn2Out:   0.55893141237326205,
			gOut:     0.49975000000000003,
			tauOut:   9.4211902206835373E-008,
		},
		{
			i0:       1,
			n0:       3,
			z:        []float64{0.5589315065851642E+00, 0.5589314123732620E+00, 0.1121517486324177E-26, 0.1291024862446124E-30, 0.6434089156657428E-04, 0.2304421214097475E+00, 0.2303778747300831E+00, 0.3431506172718059E-03, 0.3432463134744483E-03, 0.1484300435548105E-08, 0.6641476135588615E-06, 0.6434089156657428E-04, 0.2001510674733695E+01, 0.2227952244484021E-23, 0.4011480356653261E-18, 0.5524513774965514E-19, 0.1702794694134603E+01, 0.2545269924803487E-17, 0.5179704803914515E-14, 0.1264160184270297E-13, 0.1077066053646038E+01, 0.7156177943897596E-01, 0.2292199980814605E-03, 0.1899018778701386E-03, 0.1060140274732043E+01, 0.8660746506696473E-01, 0.1357005210961402E-39, 0.1331360138522907E-23, 0.9894235909971354E+00, 0.8804208964992894E-17, 0.8660746506696473E-01, 0.8679736700028205E-01, 0.9539342071687115E+00, 0.2761013168273541E-29, 0.1040577915698430E-22, 0.8107757314001177E-24, 0.8046649468928653E+00, 0.5098326619997980E-22, 0.1416795225784663E-10, 0.1216165317638265E-10, 0.7009142227469247E+00, 0.1183291357831518E-29, 0.4806926318247711E-11, 0.3824647564366386E-11, 0.5502651835254770E+00, 0.1368072596837427E-11, 0.1169503544861386E-10, 0.4068037876491279E-10, 0.4423863025187732E+00, 0.2477754901417239E-20, 0.1737857614720001E-07, 0.5295826057530262E-07, 0.3697086796938907E+00, 0.5143373102040997E-26, 0.1312431380925897E-05, 0.1614374370413396E-05, 0.1446051340026323E+00, 0.9928287808749566E-25, 0.3777515963415321E-05, 0.2773141909621761E-05, 0.1110032523123295E+00, 0.0000000000000000E+00, 0.2532463507333992E-05, 0.2934028940292093E-05, 0.7513603923341917E-01, 0.5394210206791908E-19, 0.6032617175984252E-07, 0.2575740214720034E-06, 0.6860214953971246E-01, 0.7542232825258426E-19, 0.3784397549471832E-09, 0.1131370986389306E-09, 0.2434478048112329E-01, 0.1013559914197709E-18, 0.1483665133446019E-05, 0.1887408451311279E-05, 0.6989818532012803E-03, 0.9466330862652142E-28, 0.2539111990097616E-03, 0.9240506410060376E-03, 0.7811996215926567E-04, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			pp:       1,
			dmin:     1.4843004355481051E-009,
			desig:    -1.6593817482031092E-016,
			qmax:     1.6602870234255678,
			ttype:    -4,
			dmin1:    6.4246679664367451E-005,
			dmin2:    0.55893141237326205,
			dn:       1.4843004355481051E-009,
			dn1:      6.4246679664367451E-005,
			dn2:      0.55893141237326205,
			g:        0.49975000000000003,
			tau:      9.4211902206835373E-008,
			nFail:    6,
			iter:     109,
			sigma:    2.0907792019978437,
			nDiv:     1425,
			zOut:     []float64{0.2649710614371106E+01, 0.5589314123732620E+00, 0.1121517486324177E-26, 0.1291024862446124E-30, 0.2321564474027070E+01, 0.2307852720292263E+00, 0.2303778747300831E+00, 0.3431506172718059E-03, 0.2090779203479937E+01, 0.1482093454966231E-08, 0.6641476135588615E-06, 0.6434089156657428E-04, 0.2001510674733695E+01, 0.2227952244484021E-23, 0.4011480356653261E-18, 0.5524513774965514E-19, 0.1702794694134603E+01, 0.2545269924803487E-17, 0.5179704803914515E-14, 0.1264160184270297E-13, 0.1077066053646038E+01, 0.7156177943897596E-01, 0.2292199980814605E-03, 0.1899018778701386E-03, 0.1060140274732043E+01, 0.8660746506696473E-01, 0.1357005210961402E-39, 0.1331360138522907E-23, 0.9894235909971354E+00, 0.8804208964992894E-17, 0.8660746506696473E-01, 0.8679736700028205E-01, 0.9539342071687115E+00, 0.2761013168273541E-29, 0.1040577915698430E-22, 0.8107757314001177E-24, 0.8046649468928653E+00, 0.5098326619997980E-22, 0.1416795225784663E-10, 0.1216165317638265E-10, 0.7009142227469247E+00, 0.1183291357831518E-29, 0.4806926318247711E-11, 0.3824647564366386E-11, 0.5502651835254770E+00, 0.1368072596837427E-11, 0.1169503544861386E-10, 0.4068037876491279E-10, 0.4423863025187732E+00, 0.2477754901417239E-20, 0.1737857614720001E-07, 0.5295826057530262E-07, 0.3697086796938907E+00, 0.5143373102040997E-26, 0.1312431380925897E-05, 0.1614374370413396E-05, 0.1446051340026323E+00, 0.9928287808749566E-25, 0.3777515963415321E-05, 0.2773141909621761E-05, 0.1110032523123295E+00, 0.0000000000000000E+00, 0.2532463507333992E-05, 0.2934028940292093E-05, 0.7513603923341917E-01, 0.5394210206791908E-19, 0.6032617175984252E-07, 0.2575740214720034E-06, 0.6860214953971246E-01, 0.7542232825258426E-19, 0.3784397549471832E-09, 0.1131370986389306E-09, 0.2434478048112329E-01, 0.1013559914197709E-18, 0.1483665133446019E-05, 0.1887408451311279E-05, 0.6989818532012803E-03, 0.9466330862652142E-28, 0.2539111990097616E-03, 0.9240506410060376E-03, 0.7811996215926567E-04, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			i0Out:    1,
			n0Out:    0,
			ppOut:    1,
			dminOut:  1.4843004355481051E-009,
			desigOut: -1.6593817482031092E-016,
			sigmaOut: 2.0907792019978437,
			qmaxOut:  1.6602870234255678,
			nFailOut: 6,
			iterOut:  109,
			nDivOut:  1425,
			ttypeOut: -4,
			dmin1Out: 6.4246679664367451E-005,
			dmin2Out: 0.55893141237326205,
			dnOut:    1.4843004355481051E-009,
			dn1Out:   6.4246679664367451E-005,
			dn2Out:   0.55893141237326205,
			gOut:     0.49975000000000003,
			tauOut:   9.4211902206835373E-008,
		},
	} {
		z := make([]float64, len(test.z))
		copy(z, test.z)
		i0 := test.i0 - 1 // zero index
		n0 := test.n0 - 1 // zero index

		i0Out, n0Out, ppOut, dminOut, sigmaOut, desigOut, qmaxOut, nFailOut, iterOut, nDivOut, ttypeOut, dmin1Out, dmin2Out, dnOut, dn1Out, dn2Out, gOut, tauOut :=
			impl.Dlasq3(i0, n0, z, test.pp, test.dmin, test.sigma, test.desig, test.qmax, test.nFail, test.iter, test.nDiv, test.ttype, test.dmin1, test.dmin2, test.dn, test.dn1, test.dn2, test.g, test.tau)

		if !floats.EqualApprox(z, test.zOut, dTol) {
			t.Error("Z mismatch")
		}
		if i0Out != test.i0Out-1 {
			t.Errorf("i0 mismatch. Want %v, got %v", test.n0Out, n0Out)
		}
		if n0Out != test.n0Out-1 {
			t.Errorf("n0 mismatch. Want %v, got %v", test.n0Out, n0Out)
		}
		if ppOut != test.ppOut {
			t.Errorf("pp mismatch. Want %v, got %v", test.ppOut, ppOut)
		}
		if !floats.EqualWithinAbsOrRel(dminOut, test.dminOut, dTol, dTol) {
			t.Errorf("dmin mismatch. Want %v, got %v", test.dminOut, dminOut)
		}
		if !floats.EqualWithinAbsOrRel(desigOut, test.desigOut, dTol, dTol) {
			t.Errorf("desig mismatch. Want %v, got %v", test.desigOut, desigOut)
		}
		if !floats.EqualWithinAbsOrRel(sigmaOut, test.sigmaOut, dTol, dTol) {
			t.Errorf("sigma mismatch. Want %v, got %v", test.sigmaOut, sigmaOut)
		}
		if !floats.EqualWithinAbsOrRel(qmaxOut, test.qmaxOut, dTol, dTol) {
			t.Errorf("qmax mismatch. Want %v, got %v", test.qmaxOut, qmaxOut)
		}
		if nFailOut != test.nFailOut {
			t.Errorf("nFail mismatch. Want %v, got %v", test.nFailOut, nFailOut)
		}
		if iterOut != test.iterOut {
			t.Errorf("iter mismatch. Want %v, got %v", test.iterOut, iterOut)
		}
		if nDivOut != test.nDivOut {
			t.Errorf("nFail mismatch. Want %v, got %v", test.nDivOut, nDivOut)
		}
		if ttypeOut != test.ttypeOut {
			t.Errorf("ttype mismatch. Want %v, got %v", test.ttypeOut, ttypeOut)
		}
		if !floats.EqualWithinAbsOrRel(dmin1Out, test.dmin1Out, dTol, dTol) {
			t.Errorf("dmin1 mismatch. Want %v, got %v", test.dmin1Out, dmin1Out)
		}
		if !floats.EqualWithinAbsOrRel(dmin2Out, test.dmin2Out, dTol, dTol) {
			t.Errorf("dmin2 mismatch. Want %v, got %v", test.dmin2Out, dmin2Out)
		}
		if !floats.EqualWithinAbsOrRel(dnOut, test.dnOut, dTol, dTol) {
			t.Errorf("dn mismatch. Want %v, got %v", test.dnOut, dnOut)
		}
		if !floats.EqualWithinAbsOrRel(dn1Out, test.dn1Out, dTol, dTol) {
			t.Errorf("dn1 mismatch. Want %v, got %v", test.dn1Out, dn1Out)
		}
		if !floats.EqualWithinAbsOrRel(dn2Out, test.dn2Out, dTol, dTol) {
			t.Errorf("dn2 mismatch. Want %v, got %v", test.dn2Out, dn2Out)
		}
		if !floats.EqualWithinAbsOrRel(gOut, test.gOut, dTol, dTol) {
			t.Errorf("g mismatch. Want %v, got %v", test.gOut, gOut)
		}
		if !floats.EqualWithinAbsOrRel(tauOut, test.tauOut, dTol, dTol) {
			t.Errorf("tau mismatch. Want %v, got %v", test.tauOut, tauOut)
		}
	}
}
