"""Test for encoding helper functions of tortoisehg.util.hglib"""

import pytest

from tortoisehg.util import hglib

import helpers

JAPANESE_KANA_I = '\u30a4'  # Japanese katakana "i"

@pytest.mark.skipif(helpers.QTextCodec is None, reason="Qt6 not supported")
@helpers.with_encoding('utf-8')
def test_none():
    """None shouldn't be touched"""
    for e in ('fromunicode', 'fromutf', 'tounicode', 'toutf'):
        f = getattr(hglib, e)
        assert f(None) is None


@pytest.mark.skipif(helpers.QTextCodec is None, reason="Qt6 not supported")
@helpers.with_encoding('utf-8')
def test_fromunicode():
    assert hglib.fromunicode(JAPANESE_KANA_I) == JAPANESE_KANA_I.encode('utf-8')

@pytest.mark.skipif(helpers.QTextCodec is None, reason="Qt6 not supported")
@helpers.with_encoding('ascii', 'utf-8')
def test_fromunicode_fallback():
    assert hglib.fromunicode(JAPANESE_KANA_I) == JAPANESE_KANA_I.encode('utf-8')

@pytest.mark.skipif(helpers.QTextCodec is None, reason="Qt6 not supported")
@helpers.with_encoding('ascii')
def test_fromunicode_replace():
    assert hglib.fromunicode(JAPANESE_KANA_I, errors='replace') == b'?'

@pytest.mark.skipif(helpers.QTextCodec is None, reason="Qt6 not supported")
@helpers.with_encoding('ascii')
def test_fromunicode_strict():
    with pytest.raises(UnicodeEncodeError):
        hglib.fromunicode(JAPANESE_KANA_I)


@pytest.mark.skipif(helpers.QTextCodec is None, reason="Qt6 not supported")
@helpers.with_encoding('euc-jp')
def test_fromutf():
    assert (hglib.fromutf(JAPANESE_KANA_I.encode('utf-8'))
            == JAPANESE_KANA_I.encode('euc-jp'))

@pytest.mark.skipif(helpers.QTextCodec is None, reason="Qt6 not supported")
@helpers.with_encoding('ascii', 'euc-jp')
def test_fromutf_fallback():
    assert (hglib.fromutf(JAPANESE_KANA_I.encode('utf-8'))
            == JAPANESE_KANA_I.encode('euc-jp'))

@pytest.mark.skipif(helpers.QTextCodec is None, reason="Qt6 not supported")
@helpers.with_encoding('ascii')
def test_fromutf_replace():
    assert hglib.fromutf(JAPANESE_KANA_I.encode('utf-8')) == b'?'


@pytest.mark.skipif(helpers.QTextCodec is None, reason="Qt6 not supported")
@helpers.with_encoding('euc-jp')
def test_tounicode():
    assert hglib.tounicode(JAPANESE_KANA_I.encode('euc-jp')) == JAPANESE_KANA_I

@pytest.mark.skipif(helpers.QTextCodec is None, reason="Qt6 not supported")
@helpers.with_encoding('ascii', 'euc-jp')
def test_tounicode_fallback():
    assert hglib.tounicode(JAPANESE_KANA_I.encode('euc-jp')) == JAPANESE_KANA_I


@pytest.mark.skipif(helpers.QTextCodec is None, reason="Qt6 not supported")
@helpers.with_encoding('euc-jp')
def test_toutf():
    assert (hglib.toutf(JAPANESE_KANA_I.encode('euc-jp'))
            == JAPANESE_KANA_I.encode('utf-8'))

@pytest.mark.skipif(helpers.QTextCodec is None, reason="Qt6 not supported")
@helpers.with_encoding('ascii', 'euc-jp')
def test_toutf_fallback():
    assert (hglib.toutf(JAPANESE_KANA_I.encode('euc-jp'))
            == JAPANESE_KANA_I.encode('utf-8'))


@pytest.mark.skipif(helpers.QTextCodec is None, reason="Qt6 not supported")
@helpers.with_encoding('gbk')
def test_gbk_roundtrip():
    # gbk byte sequence can also be interpreted as utf-8 (issue #3299)
    MOKU = '\u76ee'
    l = bytes(hglib.fromunicode(MOKU))
    assert hglib.tounicode(l) == MOKU


@pytest.mark.skipif(helpers.QTextCodec is None, reason="Qt6 not supported")
@helpers.with_encoding('ascii')
def test_lossless_unicode_replaced():
    l = hglib.fromunicode(JAPANESE_KANA_I, 'replace')
    assert l == b'?'
    assert hglib.tounicode(l) == JAPANESE_KANA_I

@pytest.mark.skipif(helpers.QTextCodec is None, reason="Qt6 not supported")
@helpers.with_encoding('euc-jp')
def test_lossless_unicode_double_mapped():
    YEN = '\u00a5'  # "yen" and "back-slash" are mapped to the same code
    l = hglib.fromunicode(YEN)
    assert l == b'\\'
    assert hglib.tounicode(l) == YEN

@pytest.mark.skipif(helpers.QTextCodec is None, reason="Qt6 not supported")
@helpers.with_encoding('ascii')
def test_lossless_utf_replaced():
    u = JAPANESE_KANA_I.encode('utf-8')
    l = hglib.fromutf(u)
    assert l == b'?'
    assert hglib.toutf(l) == u

@pytest.mark.skipif(helpers.QTextCodec is None, reason="Qt6 not supported")
@helpers.with_encoding('ascii')
def test_lossless_utf_cannot_roundtrip():
    u = JAPANESE_KANA_I.encode('cp932')  # bad encoding
    l = hglib.fromutf(u)
    assert hglib.toutf(l) != u
