/*
Copyright 2022 The Kubernetes Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at
    http://www.apache.org/licenses/LICENSE-2.0
Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package komega

import (
	"fmt"
	"reflect"
	"strings"

	"github.com/google/go-cmp/cmp"
	"github.com/onsi/gomega/format"
	"github.com/onsi/gomega/types"
	"k8s.io/apimachinery/pkg/runtime"
)

// These package variables hold pre-created commonly used options that can be used to reduce the manual work involved in
// identifying the paths that need to be compared for testing equality between objects.
var (
	// IgnoreAutogeneratedMetadata contains the paths for all the metadata fields that are commonly set by the
	// client and APIServer. This is used as a MatchOption for situations when only user-provided metadata is relevant.
	IgnoreAutogeneratedMetadata = IgnorePaths{
		"metadata.uid",
		"metadata.generation",
		"metadata.creationTimestamp",
		"metadata.resourceVersion",
		"metadata.managedFields",
		"metadata.deletionGracePeriodSeconds",
		"metadata.deletionTimestamp",
		"metadata.selfLink",
		"metadata.generateName",
	}
)

type diffPath struct {
	types []string
	json  []string
}

// equalObjectMatcher is a Gomega matcher used to establish equality between two Kubernetes runtime.Objects.
type equalObjectMatcher struct {
	// original holds the object that will be used to Match.
	original runtime.Object

	// diffPaths contains the paths that differ between two objects.
	diffPaths []diffPath

	// options holds the options that identify what should and should not be matched.
	options *EqualObjectOptions
}

// EqualObject returns a Matcher for the passed Kubernetes runtime.Object with the passed Options. This function can be
// used as a Gomega Matcher in Gomega Assertions.
func EqualObject(original runtime.Object, opts ...EqualObjectOption) types.GomegaMatcher {
	matchOptions := &EqualObjectOptions{}
	matchOptions = matchOptions.ApplyOptions(opts)

	return &equalObjectMatcher{
		options:  matchOptions,
		original: original,
	}
}

// Match compares the current object to the passed object and returns true if the objects are the same according to
// the Matcher and MatchOptions.
func (m *equalObjectMatcher) Match(actual interface{}) (success bool, err error) {
	// Nil checks required first here for:
	//     1) Nil equality which returns true
	//     2) One object nil which returns an error
	actualIsNil := reflect.ValueOf(actual).IsNil()
	originalIsNil := reflect.ValueOf(m.original).IsNil()

	if actualIsNil && originalIsNil {
		return true, nil
	}
	if actualIsNil || originalIsNil {
		return false, fmt.Errorf("can not compare an object with a nil. original %v , actual %v", m.original, actual)
	}

	m.diffPaths = m.calculateDiff(actual)
	return len(m.diffPaths) == 0, nil
}

// FailureMessage returns a message comparing the full objects after an unexpected failure to match has occurred.
func (m *equalObjectMatcher) FailureMessage(actual interface{}) (message string) {
	return fmt.Sprintf("the following fields were expected to match but did not:\n%v\n%s", m.diffPaths,
		format.Message(actual, "expected to match", m.original))
}

// NegatedFailureMessage returns a string stating that all fields matched, even though that was not expected.
func (m *equalObjectMatcher) NegatedFailureMessage(actual interface{}) (message string) {
	return "it was expected that some fields do not match, but all of them did"
}

func (d diffPath) String() string {
	return fmt.Sprintf("(%s/%s)", strings.Join(d.types, "."), strings.Join(d.json, "."))
}

// diffReporter is a custom recorder for cmp.Diff which records all paths that are
// different between two objects.
type diffReporter struct {
	stack []cmp.PathStep

	diffPaths []diffPath
}

func (r *diffReporter) PushStep(s cmp.PathStep) {
	r.stack = append(r.stack, s)
}

func (r *diffReporter) Report(res cmp.Result) {
	if !res.Equal() {
		r.diffPaths = append(r.diffPaths, r.currentPath())
	}
}

// currentPath converts the current stack into string representations that match
// the IgnorePaths and MatchPaths syntax.
func (r *diffReporter) currentPath() diffPath {
	p := diffPath{types: []string{""}, json: []string{""}}
	for si, s := range r.stack[1:] {
		switch s := s.(type) {
		case cmp.StructField:
			p.types = append(p.types, s.String()[1:])
			// fetch the type information from the parent struct.
			// Note: si has an offset of 1 compared to r.stack as we loop over r.stack[1:], so we don't need -1
			field := r.stack[si].Type().Field(s.Index())
			p.json = append(p.json, strings.Split(field.Tag.Get("json"), ",")[0])
		case cmp.SliceIndex:
			key := fmt.Sprintf("[%d]", s.Key())
			p.types[len(p.types)-1] += key
			p.json[len(p.json)-1] += key
		case cmp.MapIndex:
			key := fmt.Sprintf("%v", s.Key())
			if strings.ContainsAny(key, ".[]/\\") {
				key = fmt.Sprintf("[%s]", key)
				p.types[len(p.types)-1] += key
				p.json[len(p.json)-1] += key
			} else {
				p.types = append(p.types, key)
				p.json = append(p.json, key)
			}
		}
	}
	// Empty strings were added as the first element. If they're still empty, remove them again.
	if len(p.json) > 0 && len(p.json[0]) == 0 {
		p.json = p.json[1:]
		p.types = p.types[1:]
	}
	return p
}

func (r *diffReporter) PopStep() {
	r.stack = r.stack[:len(r.stack)-1]
}

// calculateDiff calculates the difference between two objects and returns the
// paths of the fields that do not match.
func (m *equalObjectMatcher) calculateDiff(actual interface{}) []diffPath {
	var original interface{} = m.original
	// Remove the wrapping Object from unstructured.Unstructured to make comparison behave similar to
	// regular objects.
	if u, isUnstructured := actual.(runtime.Unstructured); isUnstructured {
		actual = u.UnstructuredContent()
	}
	if u, ok := m.original.(runtime.Unstructured); ok {
		original = u.UnstructuredContent()
	}
	r := diffReporter{}
	cmp.Diff(original, actual, cmp.Reporter(&r))
	return filterDiffPaths(*m.options, r.diffPaths)
}

// filterDiffPaths filters the diff paths using the paths in EqualObjectOptions.
func filterDiffPaths(opts EqualObjectOptions, paths []diffPath) []diffPath {
	result := []diffPath{}

	for _, p := range paths {
		if len(opts.matchPaths) > 0 && !hasAnyPathPrefix(p, opts.matchPaths) {
			continue
		}
		if hasAnyPathPrefix(p, opts.ignorePaths) {
			continue
		}

		result = append(result, p)
	}

	return result
}

// hasPathPrefix compares the segments of a path.
func hasPathPrefix(path []string, prefix []string) bool {
	for i, p := range prefix {
		if i >= len(path) {
			return false
		}
		// return false if a segment doesn't match
		if path[i] != p && (i < len(prefix)-1 || !segmentHasPrefix(path[i], p)) {
			return false
		}
	}
	return true
}

func segmentHasPrefix(s, prefix string) bool {
	return len(s) >= len(prefix) && s[0:len(prefix)] == prefix &&
		// if it is a prefix match, make sure the next character is a [ for array/map access
		(len(s) == len(prefix) || s[len(prefix)] == '[')
}

// hasAnyPathPrefix returns true if path matches any of the path prefixes.
// It respects the name boundaries within paths, so 'ObjectMeta.Name' does not
// match 'ObjectMeta.Namespace' for example.
func hasAnyPathPrefix(path diffPath, prefixes [][]string) bool {
	for _, prefix := range prefixes {
		if hasPathPrefix(path.types, prefix) || hasPathPrefix(path.json, prefix) {
			return true
		}
	}
	return false
}

// EqualObjectOption describes an Option that can be applied to a Matcher.
type EqualObjectOption interface {
	// ApplyToEqualObjectMatcher applies this configuration to the given MatchOption.
	ApplyToEqualObjectMatcher(options *EqualObjectOptions)
}

// EqualObjectOptions holds the available types of EqualObjectOptions that can be applied to a Matcher.
type EqualObjectOptions struct {
	ignorePaths [][]string
	matchPaths  [][]string
}

// ApplyOptions adds the passed MatchOptions to the MatchOptions struct.
func (o *EqualObjectOptions) ApplyOptions(opts []EqualObjectOption) *EqualObjectOptions {
	for _, opt := range opts {
		opt.ApplyToEqualObjectMatcher(o)
	}
	return o
}

// IgnorePaths instructs the Matcher to ignore given paths when computing a diff.
// Paths are written in a syntax similar to Go with a few special cases. Both types and
// json/yaml field names are supported.
//
// Regular Paths:
// * "ObjectMeta.Name"
// * "metadata.name"
// Arrays:
// * "metadata.ownerReferences[0].name"
// Maps, if they do not contain any of .[]/\:
// * "metadata.labels.something"
// Maps, if they contain any of .[]/\:
// * "metadata.labels[kubernetes.io/something]"
type IgnorePaths []string

// ApplyToEqualObjectMatcher applies this configuration to the given MatchOptions.
func (i IgnorePaths) ApplyToEqualObjectMatcher(opts *EqualObjectOptions) {
	for _, p := range i {
		opts.ignorePaths = append(opts.ignorePaths, strings.Split(p, "."))
	}
}

// MatchPaths instructs the Matcher to restrict its diff to the given paths. If empty the Matcher will look at all paths.
// Paths are written in a syntax similar to Go with a few special cases. Both types and
// json/yaml field names are supported.
//
// Regular Paths:
// * "ObjectMeta.Name"
// * "metadata.name"
// Arrays:
// * "metadata.ownerReferences[0].name"
// Maps, if they do not contain any of .[]/\:
// * "metadata.labels.something"
// Maps, if they contain any of .[]/\:
// * "metadata.labels[kubernetes.io/something]"
type MatchPaths []string

// ApplyToEqualObjectMatcher applies this configuration to the given MatchOptions.
func (i MatchPaths) ApplyToEqualObjectMatcher(opts *EqualObjectOptions) {
	for _, p := range i {
		opts.matchPaths = append(opts.ignorePaths, strings.Split(p, "."))
	}
}
