//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
//
/// \file field/field05/src/F05DetectorConstruction.cc
/// \brief Implementation of the F05DetectorConstruction class
//

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......
//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

#include "F05DetectorConstruction.hh"

#include "F05Field.hh"

#include "G4Box.hh"
#include "G4Colour.hh"
#include "G4FieldManager.hh"
#include "G4GeometryManager.hh"
#include "G4LogicalVolume.hh"
#include "G4LogicalVolumeStore.hh"
#include "G4Material.hh"
#include "G4NistManager.hh"
#include "G4PVPlacement.hh"
#include "G4PhysicalVolumeStore.hh"
#include "G4SolidStore.hh"
#include "G4SystemOfUnits.hh"
#include "G4TransportationManager.hh"
#include "G4UserLimits.hh"
#include "G4VisAttributes.hh"

// #include "G4RepleteEofM.hh"
#include "G4ChordFinder.hh"
#include "G4ClassicalRK4.hh"
#include "G4EqEMFieldWithSpin.hh"
#include "G4MagIntegratorStepper.hh"
#include "G4PropagatorInField.hh"

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

F05DetectorConstruction::F05DetectorConstruction() : fVacuum(nullptr)
{
  // materials
  DefineMaterials();
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

F05DetectorConstruction::~F05DetectorConstruction()
{
  delete fField;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void F05DetectorConstruction::DefineMaterials()
{
  G4NistManager* nistMan = G4NistManager::Instance();

  fVacuum = nistMan->FindOrBuildMaterial("G4_Galactic");

  G4cout << *(G4Material::GetMaterialTable()) << G4endl;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

G4VPhysicalVolume* F05DetectorConstruction::Construct()
{
  //
  // World
  //

  G4double worldSizeXY = 20.0 * m;
  G4double worldSizeZ = 1.0 * mm;

  auto solidWorld = new G4Box("World",  // its name
                              worldSizeXY / 2, worldSizeXY / 2, worldSizeZ / 2);  // its size

  auto logicWorld = new G4LogicalVolume(solidWorld,  // its solid
                                        fVacuum,  // its material
                                        "World");  // its name

  auto physiWorld = new G4PVPlacement(nullptr,  // no rotation
                                      G4ThreeVector(),  // at (0,0,0)
                                      logicWorld,  // its logical volume
                                      "World",  // its name
                                      nullptr,  // its mother  volume
                                      false,  // no boolean operation
                                      0);  // copy number

  G4UserLimits* stepLimit;
  stepLimit = new G4UserLimits(5 * mm);

  logicWorld->SetUserLimits(stepLimit);

  //
  // Visualization attributes
  //
  // fLogicWorld->SetVisAttributes (G4VisAttributes::GetInvisible());

  //
  // always return the physical World
  //
  return physiWorld;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

G4ThreadLocal F05Field* F05DetectorConstruction::fField = nullptr;

void F05DetectorConstruction::ConstructSDandField()

{
  if (!fField) {
    fField = new F05Field();

    //     G4RepleteEofM* equation = new G4RepleteEofM(fField);
    auto equation = new G4EqEMFieldWithSpin(fField);
    //     equation->SetBField();
    //     equation->SetEField();
    //     equation->SetSpin();

    G4FieldManager* fieldManager =
      G4TransportationManager::GetTransportationManager()->GetFieldManager();
    fieldManager->SetDetectorField(fField);

    G4MagIntegratorStepper* stepper = new G4ClassicalRK4(equation, 12);

    G4double minStep = 0.01 * mm;

    auto chordFinder = new G4ChordFinder((G4MagneticField*)fField, minStep, stepper);

    // Set accuracy parameters
    G4double deltaChord = 3.0 * mm;
    chordFinder->SetDeltaChord(deltaChord);

    // Maximum allowed integration error in one integration sub-step
    G4double deltaOneStep = 0.01 * mm;
    fieldManager->SetAccuraciesWithDeltaOneStep(deltaOneStep);

    G4double deltaIntersection = 0.1 * mm;
    fieldManager->SetDeltaIntersection(deltaIntersection);

    G4TransportationManager* transportManager = G4TransportationManager::GetTransportationManager();

    G4PropagatorInField* fieldPropagator = transportManager->GetPropagatorInField();

    // Limits for relative accuracy of integration
    G4double epsMin = 2.5e-7;
    G4double epsMax = 0.001;

    fieldPropagator->SetMinimumEpsilonStep(epsMin);
    fieldPropagator->SetMaximumEpsilonStep(epsMax);

    fieldManager->SetChordFinder(chordFinder);
  }
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......
