//! Test vectors for the secp256r1 group.

use hex_literal::hex;

/// Repeated addition of the generator.
///
/// These are the first 20 test vectors from <http://point-at-infinity.org/ecc/nisttv>
pub const ADD_TEST_VECTORS: &[([u8; 32], [u8; 32])] = &[
    (
        hex!("6B17D1F2E12C4247F8BCE6E563A440F277037D812DEB33A0F4A13945D898C296"),
        hex!("4FE342E2FE1A7F9B8EE7EB4A7C0F9E162BCE33576B315ECECBB6406837BF51F5"),
    ),
    (
        hex!("7CF27B188D034F7E8A52380304B51AC3C08969E277F21B35A60B48FC47669978"),
        hex!("07775510DB8ED040293D9AC69F7430DBBA7DADE63CE982299E04B79D227873D1"),
    ),
    (
        hex!("5ECBE4D1A6330A44C8F7EF951D4BF165E6C6B721EFADA985FB41661BC6E7FD6C"),
        hex!("8734640C4998FF7E374B06CE1A64A2ECD82AB036384FB83D9A79B127A27D5032"),
    ),
    (
        hex!("E2534A3532D08FBBA02DDE659EE62BD0031FE2DB785596EF509302446B030852"),
        hex!("E0F1575A4C633CC719DFEE5FDA862D764EFC96C3F30EE0055C42C23F184ED8C6"),
    ),
    (
        hex!("51590B7A515140D2D784C85608668FDFEF8C82FD1F5BE52421554A0DC3D033ED"),
        hex!("E0C17DA8904A727D8AE1BF36BF8A79260D012F00D4D80888D1D0BB44FDA16DA4"),
    ),
    (
        hex!("B01A172A76A4602C92D3242CB897DDE3024C740DEBB215B4C6B0AAE93C2291A9"),
        hex!("E85C10743237DAD56FEC0E2DFBA703791C00F7701C7E16BDFD7C48538FC77FE2"),
    ),
    (
        hex!("8E533B6FA0BF7B4625BB30667C01FB607EF9F8B8A80FEF5B300628703187B2A3"),
        hex!("73EB1DBDE03318366D069F83A6F5900053C73633CB041B21C55E1A86C1F400B4"),
    ),
    (
        hex!("62D9779DBEE9B0534042742D3AB54CADC1D238980FCE97DBB4DD9DC1DB6FB393"),
        hex!("AD5ACCBD91E9D8244FF15D771167CEE0A2ED51F6BBE76A78DA540A6A0F09957E"),
    ),
    (
        hex!("EA68D7B6FEDF0B71878938D51D71F8729E0ACB8C2C6DF8B3D79E8A4B90949EE0"),
        hex!("2A2744C972C9FCE787014A964A8EA0C84D714FEAA4DE823FE85A224A4DD048FA"),
    ),
    (
        hex!("CEF66D6B2A3A993E591214D1EA223FB545CA6C471C48306E4C36069404C5723F"),
        hex!("878662A229AAAE906E123CDD9D3B4C10590DED29FE751EEECA34BBAA44AF0773"),
    ),
    (
        hex!("3ED113B7883B4C590638379DB0C21CDA16742ED0255048BF433391D374BC21D1"),
        hex!("9099209ACCC4C8A224C843AFA4F4C68A090D04DA5E9889DAE2F8EEFCE82A3740"),
    ),
    (
        hex!("741DD5BDA817D95E4626537320E5D55179983028B2F82C99D500C5EE8624E3C4"),
        hex!("0770B46A9C385FDC567383554887B1548EEB912C35BA5CA71995FF22CD4481D3"),
    ),
    (
        hex!("177C837AE0AC495A61805DF2D85EE2FC792E284B65EAD58A98E15D9D46072C01"),
        hex!("63BB58CD4EBEA558A24091ADB40F4E7226EE14C3A1FB4DF39C43BBE2EFC7BFD8"),
    ),
    (
        hex!("54E77A001C3862B97A76647F4336DF3CF126ACBE7A069C5E5709277324D2920B"),
        hex!("F599F1BB29F4317542121F8C05A2E7C37171EA77735090081BA7C82F60D0B375"),
    ),
    (
        hex!("F0454DC6971ABAE7ADFB378999888265AE03AF92DE3A0EF163668C63E59B9D5F"),
        hex!("B5B93EE3592E2D1F4E6594E51F9643E62A3B21CE75B5FA3F47E59CDE0D034F36"),
    ),
    (
        hex!("76A94D138A6B41858B821C629836315FCD28392EFF6CA038A5EB4787E1277C6E"),
        hex!("A985FE61341F260E6CB0A1B5E11E87208599A0040FC78BAA0E9DDD724B8C5110"),
    ),
    (
        hex!("47776904C0F1CC3A9C0984B66F75301A5FA68678F0D64AF8BA1ABCE34738A73E"),
        hex!("AA005EE6B5B957286231856577648E8381B2804428D5733F32F787FF71F1FCDC"),
    ),
    (
        hex!("1057E0AB5780F470DEFC9378D1C7C87437BB4C6F9EA55C63D936266DBD781FDA"),
        hex!("F6F1645A15CBE5DC9FA9B7DFD96EE5A7DCC11B5C5EF4F1F78D83B3393C6A45A2"),
    ),
    (
        hex!("CB6D2861102C0C25CE39B7C17108C507782C452257884895C1FC7B74AB03ED83"),
        hex!("58D7614B24D9EF515C35E7100D6D6CE4A496716E30FA3E03E39150752BCECDAA"),
    ),
    (
        hex!("83A01A9378395BAB9BCD6A0AD03CC56D56E6B19250465A94A234DC4C6B28DA9A"),
        hex!("76E49B6DE2F73234AE6A5EB9D612B75C9F2202BB6923F54FF8240AAA86F640B8"),
    ),
];

/// Scalar multiplication with the generator.
///
/// These are the test vectors from <http://point-at-infinity.org/ecc/nisttv> that are not
/// part of [`ADD_TEST_VECTORS`].
pub const MUL_TEST_VECTORS: &[([u8; 32], [u8; 32], [u8; 32])] = &[
    (
        hex!("000000000000000000000000000000000000000000000000018EBBB95EED0E13"),
        hex!("339150844EC15234807FE862A86BE77977DBFB3AE3D96F4C22795513AEAAB82F"),
        hex!("B1C14DDFDC8EC1B2583F51E85A5EB3A155840F2034730E9B5ADA38B674336A21"),
    ),
    (
        hex!("0000000000000000000000000000000000159D893D4CDD747246CDCA43590E13"),
        hex!("1B7E046A076CC25E6D7FA5003F6729F665CC3241B5ADAB12B498CD32F2803264"),
        hex!("BFEA79BE2B666B073DB69A2A241ADAB0738FE9D2DD28B5604EB8C8CF097C457B"),
    ),
    (
        hex!("41FFC1FFFFFE01FFFC0003FFFE0007C001FFF00003FFF07FFE0007C000000003"),
        hex!("9EACE8F4B071E677C5350B02F2BB2B384AAE89D58AA72CA97A170572E0FB222F"),
        hex!("1BBDAEC2430B09B93F7CB08678636CE12EAAFD58390699B5FD2F6E1188FC2A78"),
    ),
    (
        hex!("7FFFFFC03FFFC003FFFFFC007FFF00000000070000100000000E00FFFFFFF3FF"),
        hex!("878F22CC6DB6048D2B767268F22FFAD8E56AB8E2DC615F7BD89F1E350500DD8D"),
        hex!("714A5D7BB901C9C5853400D12341A892EF45D87FC553786756C4F0C9391D763E"),
    ),
    (
        hex!("0000FFFFF01FFFF8FFFFC00FFFFFFFFFC000000FFFFFC007FFFFFC000FFFE3FF"),
        hex!("659A379625AB122F2512B8DADA02C6348D53B54452DFF67AC7ACE4E8856295CA"),
        hex!("49D81AB97B648464D0B4A288BD7818FAB41A16426E943527C4FED8736C53D0F6"),
    ),
    (
        hex!("4000008000FFFFFC000003F00000FFFFFFFF800003800F8000E0000E000000FF"),
        hex!("CBCEAAA8A4DD44BBCE58E8DB7740A5510EC2CB7EA8DA8D8F036B3FB04CDA4DE4"),
        hex!("4BD7AA301A80D7F59FD983FEDBE59BB7B2863FE46494935E3745B360E32332FA"),
    ),
    (
        hex!("003FFFFFF0001F80000003F80003FFFFC0000000000FFE0000007FF818000F80"),
        hex!("F0C4A0576154FF3A33A3460D42EAED806E854DFA37125221D37935124BA462A4"),
        hex!("5B392FA964434D29EEC6C9DBC261CF116796864AA2FAADB984A2DF38D1AEF7A3"),
    ),
    (
        hex!("000001C000000000001001F803FFFFFF80000000000007FF0000000000000000"),
        hex!("5E6C8524B6369530B12C62D31EC53E0288173BD662BDF680B53A41ECBCAD00CC"),
        hex!("447FE742C2BFEF4D0DB14B5B83A2682309B5618E0064A94804E9282179FE089F"),
    ),
    (
        hex!("7FC0007FFFFFFC0003FFFFFFFFFFFFFE00003FFFFF07FFFFFFFFFFFFC007FFFF"),
        hex!("03792E541BC209076A3D7920A915021ECD396A6EB5C3960024BE5575F3223484"),
        hex!("FC774AE092403101563B712F68170312304F20C80B40C06282063DB25F268DE4"),
    ),
    (
        hex!("7FFFFC03FF807FFFE0001FFFFF800FFF800001FFFF0001FFFFFE001FFFC00000"),
        hex!("2379FF85AB693CDF901D6CE6F2473F39C04A2FE3DCD842CE7AAB0E002095BCF8"),
        hex!("F8B476530A634589D5129E46F322B02FBC610A703D80875EE70D7CE1877436A1"),
    ),
    (
        hex!("00FFFFFFFE03FFFC07FFFC800070000FC0007FFC00000000000FFFE1FBFF81FF"),
        hex!("C1E4072C529BF2F44DA769EFC934472848003B3AF2C0F5AA8F8DDBD53E12ED7C"),
        hex!("39A6EE77812BB37E8079CD01ED649D3830FCA46F718C1D3993E4A591824ABCDB"),
    ),
    (
        hex!("01FFF81FC000000000FF801FFFC0F81F01FFF8001FC005FFFFFF800000FFFFFC"),
        hex!("34DFBC09404C21E250A9B40FA8772897AC63A094877DB65862B61BD1507B34F3"),
        hex!("CF6F8A876C6F99CEAEC87148F18C7E1E0DA6E165FFC8ED82ABB65955215F77D3"),
    ),
    (
        hex!("FFFFFFFF00000000FFFFFFFFFFFFFFFFBCE6FAADA7179E84F3B9CAC2FC63253D"),
        hex!("83A01A9378395BAB9BCD6A0AD03CC56D56E6B19250465A94A234DC4C6B28DA9A"),
        hex!("891B64911D08CDCC5195A14629ED48A360DDFD4596DC0AB007DBF5557909BF47"),
    ),
    (
        hex!("FFFFFFFF00000000FFFFFFFFFFFFFFFFBCE6FAADA7179E84F3B9CAC2FC63253E"),
        hex!("CB6D2861102C0C25CE39B7C17108C507782C452257884895C1FC7B74AB03ED83"),
        hex!("A7289EB3DB2610AFA3CA18EFF292931B5B698E92CF05C1FC1C6EAF8AD4313255"),
    ),
    (
        hex!("FFFFFFFF00000000FFFFFFFFFFFFFFFFBCE6FAADA7179E84F3B9CAC2FC63253F"),
        hex!("1057E0AB5780F470DEFC9378D1C7C87437BB4C6F9EA55C63D936266DBD781FDA"),
        hex!("090E9BA4EA341A246056482026911A58233EE4A4A10B0E08727C4CC6C395BA5D"),
    ),
    (
        hex!("FFFFFFFF00000000FFFFFFFFFFFFFFFFBCE6FAADA7179E84F3B9CAC2FC632540"),
        hex!("47776904C0F1CC3A9C0984B66F75301A5FA68678F0D64AF8BA1ABCE34738A73E"),
        hex!("55FFA1184A46A8D89DCE7A9A889B717C7E4D7FBCD72A8CC0CD0878008E0E0323"),
    ),
    (
        hex!("FFFFFFFF00000000FFFFFFFFFFFFFFFFBCE6FAADA7179E84F3B9CAC2FC632541"),
        hex!("76A94D138A6B41858B821C629836315FCD28392EFF6CA038A5EB4787E1277C6E"),
        hex!("567A019DCBE0D9F2934F5E4A1EE178DF7A665FFCF0387455F162228DB473AEEF"),
    ),
    (
        hex!("FFFFFFFF00000000FFFFFFFFFFFFFFFFBCE6FAADA7179E84F3B9CAC2FC632542"),
        hex!("F0454DC6971ABAE7ADFB378999888265AE03AF92DE3A0EF163668C63E59B9D5F"),
        hex!("4A46C11BA6D1D2E1B19A6B1AE069BC19D5C4DE328A4A05C0B81A6321F2FCB0C9"),
    ),
    (
        hex!("FFFFFFFF00000000FFFFFFFFFFFFFFFFBCE6FAADA7179E84F3B9CAC2FC632543"),
        hex!("54E77A001C3862B97A76647F4336DF3CF126ACBE7A069C5E5709277324D2920B"),
        hex!("0A660E43D60BCE8BBDEDE073FA5D183C8E8E15898CAF6FF7E45837D09F2F4C8A"),
    ),
    (
        hex!("FFFFFFFF00000000FFFFFFFFFFFFFFFFBCE6FAADA7179E84F3B9CAC2FC632544"),
        hex!("177C837AE0AC495A61805DF2D85EE2FC792E284B65EAD58A98E15D9D46072C01"),
        hex!("9C44A731B1415AA85DBF6E524BF0B18DD911EB3D5E04B20C63BC441D10384027"),
    ),
    (
        hex!("FFFFFFFF00000000FFFFFFFFFFFFFFFFBCE6FAADA7179E84F3B9CAC2FC632545"),
        hex!("741DD5BDA817D95E4626537320E5D55179983028B2F82C99D500C5EE8624E3C4"),
        hex!("F88F4B9463C7A024A98C7CAAB7784EAB71146ED4CA45A358E66A00DD32BB7E2C"),
    ),
    (
        hex!("FFFFFFFF00000000FFFFFFFFFFFFFFFFBCE6FAADA7179E84F3B9CAC2FC632546"),
        hex!("3ED113B7883B4C590638379DB0C21CDA16742ED0255048BF433391D374BC21D1"),
        hex!("6F66DF64333B375EDB37BC505B0B3975F6F2FB26A16776251D07110317D5C8BF"),
    ),
    (
        hex!("FFFFFFFF00000000FFFFFFFFFFFFFFFFBCE6FAADA7179E84F3B9CAC2FC632547"),
        hex!("CEF66D6B2A3A993E591214D1EA223FB545CA6C471C48306E4C36069404C5723F"),
        hex!("78799D5CD655517091EDC32262C4B3EFA6F212D7018AE11135CB4455BB50F88C"),
    ),
    (
        hex!("FFFFFFFF00000000FFFFFFFFFFFFFFFFBCE6FAADA7179E84F3B9CAC2FC632548"),
        hex!("EA68D7B6FEDF0B71878938D51D71F8729E0ACB8C2C6DF8B3D79E8A4B90949EE0"),
        hex!("D5D8BB358D36031978FEB569B5715F37B28EB0165B217DC017A5DDB5B22FB705"),
    ),
    (
        hex!("FFFFFFFF00000000FFFFFFFFFFFFFFFFBCE6FAADA7179E84F3B9CAC2FC632549"),
        hex!("62D9779DBEE9B0534042742D3AB54CADC1D238980FCE97DBB4DD9DC1DB6FB393"),
        hex!("52A533416E1627DCB00EA288EE98311F5D12AE0A4418958725ABF595F0F66A81"),
    ),
    (
        hex!("FFFFFFFF00000000FFFFFFFFFFFFFFFFBCE6FAADA7179E84F3B9CAC2FC63254A"),
        hex!("8E533B6FA0BF7B4625BB30667C01FB607EF9F8B8A80FEF5B300628703187B2A3"),
        hex!("8C14E2411FCCE7CA92F9607C590A6FFFAC38C9CD34FBE4DE3AA1E5793E0BFF4B"),
    ),
    (
        hex!("FFFFFFFF00000000FFFFFFFFFFFFFFFFBCE6FAADA7179E84F3B9CAC2FC63254B"),
        hex!("B01A172A76A4602C92D3242CB897DDE3024C740DEBB215B4C6B0AAE93C2291A9"),
        hex!("17A3EF8ACDC8252B9013F1D20458FC86E3FF0890E381E9420283B7AC7038801D"),
    ),
    (
        hex!("FFFFFFFF00000000FFFFFFFFFFFFFFFFBCE6FAADA7179E84F3B9CAC2FC63254C"),
        hex!("51590B7A515140D2D784C85608668FDFEF8C82FD1F5BE52421554A0DC3D033ED"),
        hex!("1F3E82566FB58D83751E40C9407586D9F2FED1002B27F7772E2F44BB025E925B"),
    ),
    (
        hex!("FFFFFFFF00000000FFFFFFFFFFFFFFFFBCE6FAADA7179E84F3B9CAC2FC63254D"),
        hex!("E2534A3532D08FBBA02DDE659EE62BD0031FE2DB785596EF509302446B030852"),
        hex!("1F0EA8A4B39CC339E62011A02579D289B103693D0CF11FFAA3BD3DC0E7B12739"),
    ),
    (
        hex!("FFFFFFFF00000000FFFFFFFFFFFFFFFFBCE6FAADA7179E84F3B9CAC2FC63254E"),
        hex!("5ECBE4D1A6330A44C8F7EF951D4BF165E6C6B721EFADA985FB41661BC6E7FD6C"),
        hex!("78CB9BF2B6670082C8B4F931E59B5D1327D54FCAC7B047C265864ED85D82AFCD"),
    ),
    (
        hex!("FFFFFFFF00000000FFFFFFFFFFFFFFFFBCE6FAADA7179E84F3B9CAC2FC63254F"),
        hex!("7CF27B188D034F7E8A52380304B51AC3C08969E277F21B35A60B48FC47669978"),
        hex!("F888AAEE24712FC0D6C26539608BCF244582521AC3167DD661FB4862DD878C2E"),
    ),
    (
        hex!("FFFFFFFF00000000FFFFFFFFFFFFFFFFBCE6FAADA7179E84F3B9CAC2FC632550"),
        hex!("6B17D1F2E12C4247F8BCE6E563A440F277037D812DEB33A0F4A13945D898C296"),
        hex!("B01CBD1C01E58065711814B583F061E9D431CCA994CEA1313449BF97C840AE0A"),
    ),
];
